%{
  #include "glob.h"
  #include "../src/debug/debug_user.h"

  class adaptor_t {
  public:
    const int memorySize = 65536*8*4;  /* oc12 hack: was 65536*8 */
    uint32 mirror[memorySize/4];
    void ReadSram (range_t &range);
    void WriteSram (range_t &range);
    uint32 ReadAsicReg (uint32 address);
    void WriteAsicReg (uint32 address, uint32 data);
  };

  void ShowPrompt (void);
  void SetUnit (int unit);
  void ShowSramRange (range_t &range);
  void SetSramRange (range_t &range, dynarray_t<int> &list);
  void ShowAsicReg (uint32 address);
  void SetAsicReg (uint32 address, uint32 value);
  void SetWatchDevice (char *name);
  void Watch (range_t &range);

  int debugFd;

  adaptor_t adaptor;
  dynarray_t<int> list;

  int unitNumber;
  uint32 lastAddress;
  bool lastIsAsicReg;

  extern "C" int megaDone;
%}

%union {
  uint32  num;
  char   *text;
  range_t range;
}

%token CR
%token PFX_W
%token <num> NUM
%token <text>   ANYTEXT
%token CMD_TTY, CMD_WATCH

%type <range> range, addressRange;

%%

all     : command
        {
          ShowPrompt();
        }
        | all command
        { 
          ShowPrompt();
        }
        ;

command : CR
          {
            if (lastIsAsicReg) {
              ShowAsicReg (lastAddress);
              lastAddress += 4;
            } else {
              range_t range;
              range.lo = lastAddress;
              range.hi = lastAddress | (7*4);
              ShowSramRange (range);
              lastAddress = range.hi + 4;
            }
          }
        | '/' CR {printf ("welcome to my worllld\n");}
        | NUM '/' CR
          {
            SetUnit ($1); printf ("cow2\n");
          }
        | addressRange CR
          {
            ShowSramRange ($1);
            lastAddress = $1.hi+4;
            lastIsAsicReg = false;
          }
        | NUM '/' addressRange CR
          {
            SetUnit ($1);
            ShowSramRange ($3);
            lastAddress = $3.hi+4;
            lastIsAsicReg = false;
          }
        | addressRange colon list CR
          {
            SetSramRange ($1, list);
            lastAddress = $1.hi+4;
            lastIsAsicReg = false;
          }
        | NUM '/' addressRange colon list CR
          {
            SetUnit ($1);
            SetSramRange ($3, list);
            lastAddress = $3.hi+4;
            lastIsAsicReg = false;
          }
        | '@' NUM CR
          {
            ShowAsicReg ($2);
            lastAddress = $2+4;
            lastIsAsicReg = true;
          }
        | NUM '/' '@' NUM CR
          {
            SetUnit ($1);
            ShowAsicReg ($4);
            lastAddress = $4+4;
            lastIsAsicReg = true;
          }
        | '@' NUM colon NUM CR
          {
            SetAsicReg ($2, $4);
            lastAddress = $2+4;
            lastIsAsicReg = true;
          }
        | NUM '/' '@' NUM colon NUM CR
          {
            SetUnit ($1);
            SetAsicReg ($4, $6);
            lastAddress = $4+4;
            lastIsAsicReg = true;
          }
        | '.' CMD_TTY {lex_anytext=true;} ANYTEXT {lex_anytext=false;} CR
          {
            SetWatchDevice ($4);
          }
        | '.' CMD_WATCH range CR
          {          
            Watch ($3);
          }
        ;

addressRange: range
          {
            $$ = $1;
          }
        | PFX_W range
          {          
            $$.lo = $2.lo * 4;
            $$.hi = $2.hi * 4;
          }
        ;

range   : NUM
          {
            $$.lo = $$.hi = $1;
          }
        | NUM '-' NUM
          { 
            $$.lo = $1;
            $$.hi = $3;
          }
        ;

list    : NUM
          {
            list.size = 1;
            list[0] = $1;
          }
        | list NUM
          {
            list[list.size] = $2;
          }
        ;

colon   : ':' | ';' ;

%%

int main ()
{
  debug_.SetMask("");
  if ((debugFd = open ("/dev/debug0", O_RDONLY)) < 0)
    crash (.form ("Error opening /dev/debug0, code=%d\n", errno));
  while (!megaDone) {
    ShowPrompt();
    yyparse ();
    debug.form ("Restarting\n");
  }
}


void ShowPrompt(void)
{
    printf ("%d* ", unitNumber);
}

void yyerror (char *text)
{
  cout.form ("wow--%s\n", text);
}


void SetUnit (int unit)
{
  unitNumber = unit;
}


void ShowSramRange (range_t &range)
{
  uint32 addr;
  if (!range.Fix())
    return;
  adaptor.ReadSram (range);
  for (addr=range.lo; addr<=range.hi; addr+=4) {
    if ((addr==range.lo) | (addr % (8*4) == 0))
      cout.form ("%06x:", addr);
    cout.form ("%08x ", adaptor.mirror[addr/4]);
    if (addr % (8*4) == (7*4))
      cout.form ("\n");
  }
  if (addr % (8*4) != 0)
    cout.form ("\n");
}

void SetSramRange (range_t &range, dynarray_t<int> &list)
{
  uint32 addr;
  int index;
  if (!range.Fix())
    return;
  index = 0;
  // handle the case where the list is bigger than the range
  uint32 listHi = (list.size-1)*4+range.lo;
  if (range.hi < listHi)
    range.hi = listHi;
  for (addr=range.lo; addr<=range.hi; addr+=4) {
    adaptor.mirror[addr/4] = list[index];
    index++;
    if (index == list.size)
      index = 0;
  }
  adaptor.WriteSram (range);
}


void ShowAsicReg (uint32 address)
{
  cout.form ("@%06x:%08x\n", address, adaptor.ReadAsicReg (address));
}


void SetAsicReg (uint32 address, uint32 value)
{
  adaptor.WriteAsicReg (address, value);
  ShowAsicReg (address);
}


void SetWatchDevice (char *name)
{
  cout.form ("Setting watch device to \"%s\"\n", name);
}


void Watch (range_t &range)
{
  cout.form ("Starting watch on [%8x...%8x]\n", range.lo, range.hi);
}


void adaptor_t::ReadSram (range_t &range)
{
  int error;
  int length = range.hi + 4 - range.lo;
  int parms[4] = {unitNumber, range.lo, length, (int)&mirror[range.lo/4]};
  if ((error = ioctl (debugFd, DEBUG_IOCTL_READ_SRAM, parms)) < 0)
    crash (.form ("ioctl error %d\n", errno));
}


void adaptor_t::WriteSram (range_t &range)
{
  int error;
  int length = range.hi + 4 - range.lo;
  int parms[4] = {unitNumber, range.lo, length, (int)&mirror[range.lo/4]};
  if ((error = ioctl (debugFd, DEBUG_IOCTL_WRITE_SRAM, parms)) < 0)
    crash (.form ("ioctl error %d\n", errno));
}


uint32 adaptor_t::ReadAsicReg (uint32 address)
{
  int error;
  int parms[3] = {unitNumber, address, 0};
  if ((error = ioctl (debugFd, DEBUG_IOCTL_READ_ASIC_REG, parms)) < 0)
    crash (.form ("ioctl error %d\n", errno));
  return parms[2];
}


void adaptor_t::WriteAsicReg (uint32 address, uint32 data)
{
  int error;
  int parms[3] = {unitNumber, address, data};
  if ((error = ioctl (debugFd, DEBUG_IOCTL_WRITE_ASIC_REG, parms)) < 0)
    crash (.form ("ioctl error %d\n", errno));
}


bool range_t::Fix ()
{
  lo &= ~3;
  hi &= ~3;
  if (hi < lo)
    hi = lo;
  if (hi > adaptor_t::memorySize) {
    cout.form ("%08x is out of range\n", hi);
    return false;
  }
  return true;
}

