#!/atm/release/fore/bin/sntclsh

##################################################################################
# This TCL script inserts permanent virtual paths and channels into the CDB via
# SNMP
#
# The following are inserted:
# Permanent Through Paths
# Permanent Originating Paths
# Permanent Terminating Paths
# Permanent Virtual Channels
##################################################################################



##################################################################################
# Open CDB session with cdb.pv2cdb
##################################################################################
set filename /atm/etc/cdb.pv2cdb
set result [catch {cdb open c $filename}]
if { $result != 0 } {
    puts "pv2cdb: ERROR: Could not open CDB, \"$filename\""
    exit
}

##################################################################################
# Determine if we need to restore PVCs.  
# If a netmod flavor exists in CDB, we do not.  The hotswap semantics will
# work correctly
##################################################################################
set netmods [c get /hardware/asx_switch/board0]
foreach netmod $netmods {
    set type [lindex $netmod 1]

    if { $type == "INTEGER" } {
	set line [lindex $netmod 0]
	scan $line "module%d/%s" index var

	if { $var == "flavor" } {
	    exit
	}
    }
}


##################################################################################
# Read all permanent virtual channels and paths
# Close CDB session
##################################################################################
set pvcs [c get /software/pvc]
set pvps [c get /software/pvp/through]
set pvpt [c get /software/pvp/term]
set pvpo [c get /software/pvp/orig]
c close



##################################################################################
# Open SNMP session on switch
# Wait for switch to start accepting SNMP requests
# Set all permanent virtual channels and paths
# Close SNMP session
##################################################################################
set iter 0
while {1} {
    set result [catch {snmp open s localhost}]
    if {$result == 0 } {
	break
    }
    if { $iter > 100 } {
	exit
    } else {
	incr iter
    }
}

set iter 0
while {1} {
    set result [catch {s switch get 0 -uptime}]
    if {$result == 0} {
	break
    }
    if { $iter > 100 } {
	exit
    } else {
	incr iter
    }
}


proc getSwitchType { session } {
    if { [catch { set switch_type \
            [$session switch get -switchType] } result] == 0 } {
        return $switch_type
    }
    
    if { [catch { set adapters [$session asx adapter list] } result] != 0 } {
        set adapters {}
    }

    if { $adapters != {} } {
        set adapter_type [$session asx adapter get \
                [lindex $adapters 0] -adapterType]
        if { $adapter_type == "type-ASX-100" } {
            set adapter_type "asx100"
        } else {
            set adapter_type "asx200"
        }
    } else {
        set adapter_type {}
    }

    return $adapter_type
}


set g_switch_type [getSwitchType s]
if { $g_switch_type == {} } { exit }


proc CellsToBits { cells } {
    set bits [expr $cells * 53 * 8 / 1000]
    return $bits
}


proc LinkExists { link } {
    global g_switch_type

    if { $g_switch_type == "asx100" } {
	if { $link < 4 } {
	    set netmod 0
	} elseif { $link < 8 } {
	    set netmod 1
	    set link [expr $link - 4]
	} elseif { $link < 12 } {
	    set netmod 2
	    set link [expr $link - 8]
	} elseif { $link < 16 } {
	    set netmod 3
	    set link [expr $link - 12]
	} elseif { $link == 16 } {
	    return 1
	} else {
	    return 0
	}
    } else {
	if { $link < 8 } {
	    set netmod 0
	} elseif { $link < 16 } {
	    set netmod 1
	    set link [expr $link - 8]
	} elseif { $link < 24 } {
	    set netmod 2
	    set link [expr $link - 16]
	} elseif { $link < 32 } {
	    set netmod 3
	    set link [expr $link - 24]
	} elseif { $link == 56 } {
	    return 1
	} else {
	    return 0
	}
    }

    set result [catch {s hwport get 0 $netmod $link}]
    if { $result != 0 } {
	return 0
    } else {
	return 1
    }
}

##################################################################################
# Remove originating and terminating paths that don't exist in CDB
##################################################################################
if { $g_switch_type == "asx100" } {
    set iterations 16
} else {
    set iterations 32
}


for { set i 0 } { $i < $iterations } { incr i } {
    set term($i) 0
    set orig($i) 0
}

foreach pvp $pvpt {
    set line [lindex $pvp 0]
    scan $line "%d:%d|%d.%s" board link1 path1 var

    if { $path1 == 0 } {
	set term($link1) 1
    }
}

foreach pvp $pvpo {
    set line [lindex $pvp 0]
    scan $line "%d:%d|%d.%s" board link1 path1 var

    if { $path1 == 0 } {
	set orig($link1) 1
    }
}

for { set i 0 } { $i < $iterations } { incr i } {
    if { $term($i) == 0 && [LinkExists $i] == 1 } {
	set result [catch {s path remove $i 0}]
	if { $result != 0 } {
	    puts "s path remove $i 0"
	    puts "pv2cdb: WARNING: Could not remove terminating path $i:0"
	}
    }
    if { $orig($i) == 0 && [LinkExists $i] == 1 } {
	set result [catch {s opath remove $i 0}]
	if { $result != 0 } {
	    puts "s opath remove $i 0"
	    puts "pv2cdb: WARNING: Could not remove originating path $i:0"
	}
    }
}


##################################################################################
# Enter terminating paths into CDB via SNMP
##################################################################################
foreach pvp $pvpt {
    set line [lindex $pvp 0]
    set type [lindex $pvp 1]
    set value [lindex $pvp 2]
    scan $line "%d:%d|%d.%s" board link1 path1 var
    set path($var) $value

    if { $type == "INTEGER" } {
	if { $var == "qos_new/peak" && [LinkExists $link1] == 1 } {
	    if { $path1 != 0 } {
		set result [catch {s path create $link1 $path1 \
			-bw [CellsToBits $path(qos_new/peak)] -maxvci $path(max_vci)}]
		if { $result != 0 } {
		    puts "s path create $link1 $path1 -bw [CellsToBits $path(qos_new/peak)] -maxvci $path(max_vci)"
		    puts "pv2cdb: WARNING: Could not create terminating path $link1:$path1"
		}
	    } else {
		set result [catch {s path set $link1 $path1 \
			-pathAllocBandwidth $path(qos_new/peak)}]
		if { $result != 0 } {
		    puts "s path set $link1 $path1 -pathAllocBandwidth $path(qos_new/peak)"
		    puts "pv2cdb: WARNING: Could not modify terminating path $link1:$path1"
		}
	    }
	}
    }
}



##################################################################################
# Enter originating paths into CDB via SNMP
##################################################################################
foreach pvp $pvpo {
    set line [lindex $pvp 0]
    set type [lindex $pvp 1]
    set value [lindex $pvp 2]
    scan $line "%d:%d|%d.%s" board link1 path1 var
    set path($var) $value

    if { $type == "INTEGER" } {
	if { $var == "qos_new/peak" && [LinkExists $link1] == 1 } {
	    if { $path1 != 0 } {
		set result [catch {s opath create $link1 $path1 \
			-bw [CellsToBits $path(qos_new/peak)] -maxvci $path(max_vci)}]
		if { $result != 0 } {
		    puts "s opath create $link1 $path1 -bw [CellsToBits $path(qos_new/peak)] -maxvci $path(max_vci)"
		    puts "pv2cdb: WARNING: Could not create originating path $link1:$path1"
		}
	    } else {
		set result [catch {s opath set $link1 $path1 \
			-opathAllocBandwidth $path(qos_new/peak)}]
		if { $result != 0 } {
		    puts "s opath set $link1 $path1 -opathAllocBandwidth $path(qos_new/peak)"
		    puts "pv2cdb: WARNING: Could not modify originating path $link1:$path1"
		}
	    }
	}
    }
}


##################################################################################
# Enter through paths into CDB via SNMP
##################################################################################
set paths {}
foreach pvp $pvps {
    set line [lindex $pvp 0]
    set type [lindex $pvp 1]
    set var {}
    scan $line "%d:%d|%d/%s" board link1 path1 var

    if { $type == "DIRECTORY" && $var != "qos_new" } {
	foreach p $paths {
	    scan $p "%d:%d|%d/%d|%d" board link1 path1 link2 path2
	    if { [LinkExists $link1] == 0 || [LinkExists $link2] == 0 } {
		continue
	    }
	    set result [catch {s pathroute create $link1 $path1 $link2 $path2 \
		    -bw [CellsToBits $path(peak)]}]
	    if { $result != 0 } {
		puts "s pathroute create $link1 $path1 $link2 $path2 -bw [CellsToBits $path(peak)]"
		puts "pv2cdb: WARNING: Could not create through path $link1:$path1 $link2:$path2"
	    }
	}
	set path(peak) 0
	set paths {}

    } elseif { $type == "BOOLEAN" } {
	append paths $line
	append paths " "

    } elseif { $type == "INTEGER" } {
	scan $line "%d:%d|%d/qos_new/%s" board link1 path1 var
	set value [lindex $pvp 2]
	set path($var) $value
    }
}
foreach p $paths {
    scan $p "%d:%d|%d/%d|%d" board link1 path1 link2 path2
    if { [LinkExists $link1] == 0 || [LinkExists $link2] == 0 } {
	continue
    }
    set result [catch {s pathroute create $link1 $path1 $link2 $path2 \
	    -bw [CellsToBits $path(peak)]}]
    if { $result != 0 } {
	puts "s pathroute create $link1 $path1 $link2 $path2 -bw [CellsToBits $path(peak)]"
	puts "pv2cdb: WARNING: Could not create through path $link1:$path1 $link2:$path2"
    }
}



##################################################################################
# Enter channels into CDB via SNMP
##################################################################################
set chans {}
foreach pvc $pvcs {
    set line [lindex $pvc 0]
    set type [lindex $pvc 1]
    set var {}
    scan $line "%d:%d|%d|%d/%s" board link1 path1 chan1 var

    if { $type == "DIRECTORY" && $var != "qos_new" } {
	foreach c $chans {
	    scan $c "%d:%d|%d|%d/%d|%d|%d" board link1 path1 chan1 link2 path2 chan2
	    if { [LinkExists $link1] == 0 || [LinkExists $link2] == 0 } {
		continue
	    }
	    set result [catch {s channel create $link1 $path1 $chan1 \
		    $link2 $path2 $chan2 -bw [CellsToBits $channel(peak)] \
		    -cdvt $channel(cdv)}]
	    if { $result != 0 } {
		puts "s channel create $link1 $path1 $chan1 $link2 $path2 $chan2 -bw [CellsToBits $channel(peak)] -cdvt $channel(cdv)"
		puts "pv2cdb: WARNING: Could not create channel $link1:$path1,$chan1 $link2:$path2,$chan2"
	    }
	}
	set chans {}
	set channel(peak) 0
	set channel(cdv)  0
    }

    if { $type == "BOOLEAN"} {
	append chans $line
	append chans " "

    } elseif { $type == "INTEGER" } {
	scan $line "%d:%d|%d|%d/qos_new/%s" board link1 path1 chan1 var
	set value [lindex $pvc 2]
	set channel($var) $value
    }
}

foreach c $chans {
    scan $c "%d:%d|%d|%d/%d|%d|%d" board link1 path1 chan1 link2 path2 chan2
    if { [LinkExists $link1] == 0 || [LinkExists $link2] == 0 } {
	continue
    }
    set result [catch {s channel create $link1 $path1 $chan1 \
	    $link2 $path2 $chan2 -bw [CellsToBits $channel(peak)] \
	    -cdvt $channel(cdv)}]
    if { $result != 0 } {
	puts "s channel create $link1 $path1 $chan1 $link2 $path2 $chan2 -bw [CellsToBits $channel(peak)] -cdvt $channel(cdv)"
	puts "pv2cdb: WARNING: Could not create channel $link1:$path1,$chan1 $link2:$path2,$chan2"
    }
}
