//------------------------------------------------------------------------------------
//                                                                      
//                   I N T E L   P R O P R I E T A R Y                   
//                                                                       
//      COPYRIGHT (c)  2000 BY  INTEL  CORPORATION.  ALL RIGHTS          
//      RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//      BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//      RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//      LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//      MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//      THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                       
//                         INTEL  CORPORATION                            
//                                                                      
//                      2200 MISSION COLLEGE BLVD                        
//                                                                       
//                SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                       
//------------------------------------------------------------------------------------
// port.uc
// macros to access ports (IX bus) 
//
//
// system: IXP1200
// subsystem: Bus IO microcode
// usage: library macros
// author: dfh	May 1, 2000
//
// revisions:


#ifndef PORT_UC
#define PORT_UC


// API:
//	port_rx_init[_rec_req, _rfifo_addr, UENGINE_ID]
//	port_rx_receive[_exception, _rec_state, _rec_req, USE_XFER, PORT_TYPE]
//	port_rx_fail[_rec_state, _rec_req, _exception]
//	port_rx_error[_rec_state, _rec_req, _exception]
//	port_rx_discard[_rec_state, _rec_req, _exception]
//	port_rx_bytecount_extract[_bytecount, _rec_state]


// port types
#define SLOW_PORT  0x01
#define ETHER_10M  0x01
#define ETHER_100M 0x02
#define FAST_PORT  0x03
#define ETHER_GIG  0x03
#define ATM_OC3	   0x11
#define ATM_OC12   0x12
#define ATM_OC48   0x13


// port exceptions, range is 0-0x7
#define    PORT_RXERROR					0x01
#define	   PORT_RXFAIL					0x02
#define	   PORT_RXCANCEL				0x04
#define    PORT_SHDBE_SOP				0x05
#define    PORT_SHDBE_NOT_SOP			0x06

// bit fields in rec_state
#define REC_STATE_ELE_COUNT 0
#define REC_STATE_DISCARD_BIT 7
#define REC_STATE_SOP_BIT 8
#define REC_STATE_EOP_BIT 9
#define REC_STATE_BYTE_COUNT 10
#define REC_STATE_FREELIST 16
#define REC_STATE_SOPSEQ 20
#define REC_STATE_QSELECT 28
#define REC_STATE_FPORT 31
// from		 +-----+-------+------+------+--------+-------------+---+-----+-------+-------+
// rec_state |FPORT|qselect|inport|F seq |freelist|qw/byte count|eop|qtype|discard|outport|
//			 |     |       |      |      |        |             |   |     |       |  ele# |
//			 | 31  | 30:28 |27:24 |23:20 |  19:16 |     15:10   | 9 |  8  |   7   |  6:0  |
//			 +-----+-------+------+------+--------+-------------+---+-----+-------+-------+




// port_rx_init
// 		description: Initialize a register with receive request format.
//		outputs:
//			_rec_req			receive request with element, thread and port. See PRM
//			_rfifo_qw_addr		quadword addrsss of receive fifo element	
//		inputs:
//			UENGINE_ID			microengine id 0-5
//			CTX0PORT			port number for context 0
//			CTX1PORT			port number for context 1
//			CTX2PORT			port number for context 2
//			CTX3PORT			port number for context 3
//		size: 11 instr
//		latency: 4 cycles
//		see also: stdmac.uc
//		example usage:
//			port_rx_init[rec_req, rfifo_addr, 0]
//
#macro port_rx_init[_rec_req, _rfifo_addr, UENGINE_ID, CTX0PORT, CTX1PORT, CTX2PORT, CTX3PORT]
br=ctx[1, context1#]
br=ctx[2, context2#]
br=ctx[3, context3#]

context0#:
#define_eval MY_TID0 (UENGINE_ID<<2)

#if ((CTX0PORT >= FAST_PORT_LO) & (CTX0PORT <= FAST_PORT_HI))
#define_eval FAST_PORT_SEL0 ((CTX0PORT & 1) +1)
#else
#define_eval FAST_PORT_SEL0 (0)
#endif
#define_eval RREQUEST0 ((CTX0PORT) | (MY_TID0<<6) |(MY_TID0<<18) | (FAST_PORT_SEL0<<16))
#define_eval RFIFO_QW_ADDR0 (MY_TID0<<3)

	immed32[_rec_req, RREQUEST0]
	immed[rfifo_addr, RFIFO_QW_ADDR0]
	sem_init[@req_inflight]							// init semaphore to limit 1 rec req per engine
br[end#]

context1#:
#define_eval MY_TID1 ((UENGINE_ID<<2)+1)

#if ((CTX1PORT >= FAST_PORT_LO) & (CTX1PORT <= FAST_PORT_HI))
#define_eval FAST_PORT_SEL1 ((CTX1PORT & 1) +1)
#else
#define_eval FAST_PORT_SEL1 (0)
#endif

#define_eval RREQUEST1 (CTX1PORT | (MY_TID1<<6) |(MY_TID1<<18) | (FAST_PORT_SEL1<<16))
#define_eval RFIFO_QW_ADDR1 (MY_TID1<<3)

	immed32[_rec_req, RREQUEST1]
	immed[rfifo_addr, RFIFO_QW_ADDR1]
br[end#]

context2#:
#define_eval MY_TID2 ((UENGINE_ID<<2)+2)

#if ((CTX2PORT >= FAST_PORT_LO) & (CTX2PORT <= FAST_PORT_HI))
#define_eval FAST_PORT_SEL2 ((CTX2PORT & 1) +1)
#else
#define_eval FAST_PORT_SEL2 (0)
#endif

#define_eval RREQUEST2 (CTX2PORT | (MY_TID2<<6) |(MY_TID2<<18) | (FAST_PORT_SEL2<<16))
#define_eval RFIFO_QW_ADDR2 (MY_TID2<<3)

	immed32[_rec_req, RREQUEST2]
	immed[rfifo_addr, RFIFO_QW_ADDR2]
br[end#]

context3#:
#define_eval MY_TID3 ((UENGINE_ID<<2)+3)

#if ((CTX3PORT >= FAST_PORT_LO) & (CTX3PORT <= FAST_PORT_HI))
#define_eval FAST_PORT_SEL3 ((CTX3PORT & 1) +1)
#else
#define_eval FAST_PORT_SEL3 (0)
#endif

#define_eval RREQUEST3 (CTX3PORT | (MY_TID3<<6) |(MY_TID3<<18) | (FAST_PORT_SEL3<<16))
#define_eval RFIFO_QW_ADDR3 (MY_TID3<<3)

	immed32[_rec_req, RREQUEST3]
	immed[rfifo_addr, RFIFO_QW_ADDR3]
end#:
#endm


// port_rx_receive
// 		description: Issue receive request and when signaled, read receive control.
//					 If PORT_TYPE is a fast port, issue without checking port ready.
//					 If PORT_TYPE is a slow port, check port ready then issue.
//		outputs:
//			_exception			0 if no exception,
//								otherwise PORT_RXERROR, PORT_RXFAIL, or PORT_RXCANCEL
//			_rec_state			captured receive control info into receive state	
//		inputs:
//			_rec_req			GPR. assembled receive request
//		size: 12 instr
//		example usage:
//			port_rx_receive(exception, rec_state, rec_req, X7, ETHER_100M);	
//
#macro port_rx_receive[_exception, _rec_state, _rec_req, USE_XFER, PORT_TYPE]
#if ((PORT_TYPE == ETHER_10M) || (PORT_TYPE == ETHER_100M))
	port_rx_slow_receive[_exception, _rec_state, _rec_req, USE_XFER]
#else
	port_rx_fast_receive[_exception, _rec_state, _rec_req, USE_XFER]
#endif
#endm

// _port_rx_slow_receive
//		description: internal macro to support port_rx_receive with slow PORT_TYPE
//
#macro port_rx_slow_receive[_exception, _rec_state, _rec_req, USE_XFER]
.local temp
check_port#:
	csr[read, USE_XFER, RCV_RDY_LO], ctx_swap			; get port ready
	alu[--, rec_req, B, 0]								; shift right by lo 5 bits (port number)
	alu[--, 1, AND, USE_XFER, >>indirect]
	br=0[check_port#]

port_rdy1#:
	move[USE_XFER, _rec_req]

	critsect_enter[@req_inflight]						; block other contexts from sending rec_req
	csr[write, USE_XFER, rcv_req],ctx_swap				; send req to FBI

	ctx_arb[start_receive]								; wait for fbi signal when receive control can be read
 
    csr[read, USE_XFER, RCV_CNTL], ctx_swap				; read the rcv_ctrl register
	critsect_exit[@req_inflight]						; allow other contexts to send rec_req

// format of RCV_CTL is
// +-----+-----+-----+------+----+-----+-----+----+----+-----------+---+---+
// | msg |port |seq# |rxfail|err |elem2|elem1|1or2|seq | validbytes|EOP|SOP|
// |31:30|29:24|23:20|  19  | 18 |17:14|13:10| 9  | 8  |    7:2    | 1 | 0 |
// +-----+-----+-----+------+----+-----+-----+----+----+-----------+---+---+
//
	ld_field[_rec_state, 0010, USE_XFER, <<8]				; byte count, eop, sop from rcv_cntl 7:0 to 15:8
	alu[_rec_state, _rec_state, +, 1]					; increment element count
	alu[_exception, 0x3, AND, USE_XFER, >>18]			; error or fail
.endlocal
#endm


// _port_rx_fast_receive
//		description: internal macro to support port_rx_receive with fast PORT_TYPE
//
#macro _port_rx_fast_receive[_exception, _rec_state, _rec_req, USE_XFER]
	move[USE_XFER, _rec_req]

	critsect_enter[@req_inflight]						; block other contexts from sending rec_req
	csr[write, USE_XFER, rcv_req],ctx_swap				; send req to FBI

	ctx_arb[start_receive]								; wait for fbi signal when receive control can be read
 
    csr[read, USE_XFER, RCV_CNTL], ctx_swap				; read the rcv_ctrl register
	critsect_exit[@req_inflight]						; allow other contexts to send rec_req


// format of RCV_CTL is
// +-----+-----+-----+------+----+-----+-----+----+----+-----------+---+---+
// | msg |port |seq# |rxfail|err |elem2|elem1|1or2|seq | validbytes|EOP|SOP|
// |31:30|29:24|23:20|  19  | 18 |17:14|13:10| 9  | 8  |    7:2    | 1 | 0 |
// +-----+-----+-----+------+----+-----+-----+----+----+-----------+---+---+
//
	alu[--, 0x3, -, USE_XFER, >>30]						; test cancel
	br!=0[not_cancel#], defer[2]
	ld_field[_rec_state, 0010, USE_XFER, <<8]			; byte count, eop, sop from rcv_cntl 7:0 to 15:8
	move[_exception, PORT_RXCANCEL]
	br[end#]
not_cancel#:
	alu[_rec_state, _rec_state, +, 1]					; increment element count
	alu[_exception, 0x3, AND, USE_XFER, >>18]			; error or fail (but not both)
end#:
#endm


// port_rx_fail
// 		description: Increment a mib counter and otherwise behave like EOP, except
//					 that the packet is not queued. This is an indication that the
//					 port is terminating this packet and any packet data received
//					 is probably junk.
//		note: _rec_state is presently not used, but we might want to set an indication there
//
#macro port_rx_fail[_rec_state, _rec_req, _exception]
.local tempa
	immed[tempa, EXCEPTION_COUNTERS]					; 11 contexts, 16 locations each
	alu[tempa, tempa, +8,_rec_req, <<4]					; add context id * 16
	scratch[incr, --, tempa, _exception, 1]

	immed[tempa, TOTAL_DISCARDS]
	scratch[incr, --, tempa, 0, 1]
.endlocal
#endm


// port_rx_error
//		description: Increment exception counter
//      note: this could be expanded to read status word and decide based on the type of error whether to set discard
//
#macro port_rx_error[_rec_state, _rec_req, _exception]
.local tempa
	immed[tempa, EXCEPTION_COUNTERS]					; 11 contexts, 16 locations each
	alu[tempa, tempa, +8,_rec_req, <<4]					; add context id * 16
	scratch[incr, --, tempa, _exception, 1]
.endlocal
#endm


// port_rx_discard
// 		description: Increment exception counter, total discards, set discard flag
//
#macro port_rx_discard[_rec_state, _rec_req, _exception]
.local tempa
	immed[tempa, EXCEPTION_COUNTERS]					; 11 contexts, 16 locations each
	alu[tempa, tempa, +8,_rec_req, <<4]					; add context id * 16
	scratch[incr, --, tempa, _exception, 1]

#define_eval DISCARD_BIT_SHF (REC_STATE_DISCARD_BIT)
	immed[tempa, TOTAL_DISCARDS]
	scratch[incr, --, tempa, 0, 1]

	alu[_rec_state, _rec_state, OR, 1, <</**/DISCARD_BIT_SHF/**/]	; set discard
.endlocal
#endm

// port_rx_bytecount_extract
//		description: Extract last mpacket's bytecount from rec_state.
//
#macro port_rx_bytecount_extract[_bytecount, _rec_state]
	alu[_bytecount, 0, +8, _rec_state, >>REC_STATE_BYTE_COUNT]
	alu[_bytecount, _bytecount, AND~, 0x3, <<6]
#endm


#endif 	// PORT_UC