//---------------------------------------------------------------------------
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  1998-2000 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
//---------------------------------------------------------------------------
// stdmac.uc
// standard macros
//
//
// system: IXP1200
// subsystem: common low-level microcode
// usage: library macros
// author: dfh Jan 12, 1999
// revisions:
//		dfh April 28, 2000 add alu macros, move other macros to individual files


#ifndef STDMAC_UC
#define STDMAC_UC

// API:
//		immed32[_dst, val]
// 		balr[_link_pc, target_label]
//		move[_dst, _src]
//		add[_dst, _src_a, _src_b]
//		add_c[_dst, _src_a, _src_b]
//		sub[_dst, _src_a, _src_b]
//		alu_op[_dst, _src_a, op_spec, _src_b]
//		shf_right[_dst, _src, _shift_amt]
//		shf_left[_dst, _src, _shift_amt]		
//		rot_right[_dst, _src, _shift_amt]
//		rot_left[_dst, _src, _shift_amt]		
//		alu_shf_right[_dst, _src_a, op_spec, _src_b, _shift_amt]
//		add_shf_right[_dst, _src_a, _src_b, _shift_amt]
//		sub_shf_right[_dst, _src_a, _src_b, _shift_amt]
//		and_shf_right[_dst, _src_a, _src_b, _shift_amt]
//		or_shf_right[_dst, _src_a, _src_b, _shift_amt]
//		alu_shf_left[_dst, _src_a, op_spec, _src_b, _shift_amt]
//		add_shf_left[_dst, _src_a, _src_b, _shift_amt]
//		sub_shf_left[_dst, _src_a, _src_b, _shift_amt]
//		and_shf_left[_dst, _src_a, _src_b, _shift_amt]
//		or_shf_left[_dst, _src_a, _src_b, _shift_amt]
//		alu_rot_right[_dst, _src_a, op_spec, _src_b, _shift_amt]
//		alu_rot_left[_dst, _src_a, op_spec, _src_b, _shift_amt]
//		br_bset_ind[_src, _src_bit_pos, target_label]
//		br_bclr_ind[_src, _src_bit_pos, target_label]
//		bits_set_ind[_reg, _reg_start_bit, _reg_mask]
//		bits_clr_ind[_reg, _reg_start_bit, _reg_mask]
//		thread_id[_tid, UENG_ID]


// immed32[_dst, val]
//		description:load _dst with 32 bit constant
//					if constant fits in 16 bits, expand to a single immed 
//					if constant doesn't, expand into a immed_w0 and immed_w1
//					if it is a name (not a number), assume that it is being IMPORTED
//		ouptuts:
//			_dst	GPR. destination register. 
//		inputs:
//			val		constant
//
#macro immed32[_dst, val]
#if (!isnum(val))
    immed_w0[_dst, val]
    immed_w1[_dst, val>>16]
#elif (((val & 0xFFFF0000) == 0) || ((val & 0xFFFF0000) == 0xFFFF0000))
    immed[_dst,val]
#elif (((val & 0xFF0000FF) == 0) || ((val & 0xFF0000FF) == 0xFF0000FF))
    immed[_dst, (val >> 8), <<8]
#elif ((val & 0x0000FFFF) == 0)
    /* make sure that number is not seen as negative */
    immed[_dst, (0x0000FFFF & (val >> 16)), <<16]
#elif ((val & 0x0000FFFF) == 0x0000FFFF)
    /* make sure that number is seen as negative */
    immed[_dst, (0xFFFF0000 | (val >> 16)), <<16]
#else
    immed_w0[_dst, (val & 0xFFFF)]
    immed_w1[_dst, ((val >> 16) & 0xFFFF)]
#endif
#endm


// balr[_link_pc, target_label]
//	 	description: load next pc to _link_pc, branch to target_label
//		inputs:
//			_link_pc		GPR. to hold next pc for use by rtn instruction
//			target_label	subroutine start
//
#macro balr[_link_pc, target_label]
    br [target_label], defer[1]
    load_addr[_link_pc, ret_lab#]
ret_lab#:
#endm


// move[_dst, _src]
//		description: move the source value into the destination register
//		outputs:
//			-dst	destination GPR or write xfer
//		inputs:
//			_src	source GPR or read xfer
//
#macro move[_dst, _src]
    alu[_dst, --, b, _src]
#endm


// add
//		description: 32 bit add _src_a + _src_b 
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src_a		GPR or constant	
//			_src_b		GPR or constant	
//
#macro add[_dst, _src_a, _src_b]
#if (isnum(_src_b))
#if (_src_b > 31)
.local temp
	immed32[temp, _src_b]
	alu[_dst, _src_a, +, temp]
.endlocal
#else
	alu[_dst, _src_a, +, _src_b]

#endif					// _src_b  not > 31
#elif (isnum(_src_a))
#if (_src_a > 31)
.local temp
	immed32[temp, _src_a]
	alu[_dst, temp, +, _src_b]
.endlocal
#else
	alu[_dst, _src_a, +, _src_b]

#endif	// _src_b  not > 31
#else
	alu[_dst, _src_a, +, _src_b]
#endif
#endm


// add_c
// 		description: 32 bit add _src_a + _src_b + previous carry
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src_a		GPR or constant	
//			_src_b		GPR or constant	
//
#macro add_c[_dst, _src_a, _src_b]
#if (isnum(_src_b))
#if (_src_b > 31)
.local temp
	immed32[temp, _src_b]
	alu[_dst, _src_a, +carry, temp]
.endlocal
#else
	alu[_dst, _src_a, +carry, _src_b]

#endif	// _src_b  not > 31
#elif (isnum(_src_a))
#if (_src_a > 31)
.local temp
	immed32[temp, _src_a]
	alu[_dst, temp, +carry, _src_b]
.endlocal
#else
	alu[_dst, _src_a, +carry, _src_b]

#endif	// _src_b  not > 31
#else
	alu[_dst, _src_a, +carry, _src_b]
#endif
#endm


// sub
//	 	description: subtract _src_b from _src_a
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src_a		GPR or constant	
//			_src_b		GPR or constant	
//
#macro sub[_dst, _src_a, _src_b]
#if (isnum(_src_b))
#if (_src_b > 31)
.local temp
	immed32[temp, _src_b]
	alu[_dst, _src_a, -, temp]
.endlocal
#else
	alu[_dst, _src_a, -, _src_b]
#endif
#elif (isnum(_src_a))
#if (_src_a > 31)
.local temp
	immed32[temp, _src_a]
	alu[_dst, temp, -, _src_b]
.endlocal
#else
	alu[_dst, _src_a, -, _src_b]
#endif
#else
	alu[_dst, _src_a, -, _src_b]
#endif
#endm


// alu_op
//	 	description: perform alu operation _src_a op_spec _src_b
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src_a		GPR or constant	
//			op_spec		ALU operator
//			_src_b		GPR or constant	
//
#macro alu_op[_dst, _src_a, op_spec, _src_b]
#if (isnum(_src_b))
#if (_src_b > 31)
.local temp
	immed32[temp, _src_b]
	alu[_dst, _src_a, op_spec, temp]
.endlocal
#else
	alu[_dst, _src_a, op_spec, _src_b]
#endif
#elif (isnum(_src_a))
#if(_src_a > 31)
.local temp
	immed32[temp, _src_a]
	alu[_dst, temp, op_spec, _src_b]
.endlocal
#else
	alu[_dst, _src_a, op_spec, _src_b]
#endif
#else
	alu[_dst, _src_a, op_spec, _src_b]
#endif
#endm




// shf_right
//		description: shift right _src by _shift_amt bit positions
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src		GPR	
//			_shift_amt	GPR or constant	
//
#macro shf_right[_dst, _src, _shift_amt]
#if (isnum(_shift_amt))
#define_eval SHF_AMT _shift_amt
	alu[_dst, --, B, _src, >>/**/SHF_AMT/**/] 
#else
	alu[--, _shift_amt, B, 0]
	alu[_dst, --, B, _src, >>indirect] 
#endif
#endm


// shf_left
//		description: shift left _src by _shift_amt bit positions
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src		GPR	
//			_shift_amt	GPR or constant	
//
#macro shf_left[_dst, _src, _shift_amt]
#if (isnum(_shift_amt))
#define_eval SHF_AMT _shift_amt
	alu[_dst, --, B, _src, <</**/SHF_AMT/**/] 
#else
.local temp
	alu[temp, _shift_amt, B-A, 1, <<5]
	alu[--, temp, B, 0]
	alu[_dst, --, B, _src, <<indirect]
.endlocal
#endif
#endm


// rot_right
//		description: shift right _src by _shift_amt bit positions
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src		GPR	
//			_shift_amt	constant	
//
#macro rot_right[_dst, _src, _shift_amt]
	alu[_dst, --, B, _src, >>rot /**/SHF_AMT/**/] 
#endm


// rot_left
//		description: shift left _src by _shift_amt bit positions
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src		GPR	
//			_shift_amt	constant	
//
#macro rot_left[_dst, _src, _shift_amt]
	alu[_dst, --, B, _src, <<rot /**/SHF_AMT/**/] 
#endm


// alu_shf_right
//		description: shift right _src_b by _shift_amt bit positions, then perform
//					 operation op_spec against_src_a and shift result
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src_a		GPR or constant	
//			op_spec		alu operation
//			_src_b		GPR	
//			_shift_amt	GPR or constant	
//
#macro alu_shf_right[_dst, _src_a, op_spec, _src_b, _shift_amt]
#if (isnum(_shift_amt))
#define_eval SHF_AMT _shift_amt
#if (isnum(_src_a))
#if (_src_a > 31)
.local tempa
	immed32[tempa, _src_a]
	alu[_dst, tempa, op_spec, _src_b, >>/**/SHF_AMT/**/]
.endlocal		// temp
#else
	alu[_dst, _src_a, op_spec, _src_b, >>/**/SHF_AMT/**/]
#endif
#else
	alu[_dst, _src_a, op_spec, _src_b, >>/**/SHF_AMT/**/]
#endif 
#else		// not isnum _shift_amt
	alu[--, _shift_amt, B, 0]
#if (isnum(_src_a)) 
#if (_src_a > 31)
.local tempa
	immed32[tempa, _src_a]
	alu[_dst, tempa, op_spec, _src_b, >>indirect]
.endlocal		// tempa
#else
	alu[_dst, _src_a, op_spec, _src_b, >>indirect]
#endif
#else
	alu[_dst, _src_a, op_spec, _src_b, >>indirect]
#endif
#endif
#endm

// commonly used alu ops
#macro add_shf_right[_dst, _src_a, _src_b, _shift_amt]
	alu_shf_right[_dst, _src_a, +, _src_b, _shift_amt]
#endm
#macro sub_shf_right[_dst, _src_a, _src_b, _shift_amt]
	alu_shf_right[_dst, _src_a, -, _src_b, _shift_amt]
#endm
#macro and_shf_right[_dst, _src_a, _src_b, _shift_amt]
	alu_shf_right[_dst, _src_a, AND, _src_b, _shift_amt]
#endm
#macro or_shf_right[_dst, _src_a, _src_b, _shift_amt]
	alu_shf_right[_dst, _src_a, OR, _src_b, _shift_amt]
#endm




// alu_shf_left
//		description: shift left _src_b by _shift_amt bit positions, then perform
//					 operation op_spec against_src_a and shift result
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src_a		GPR or constant	
//			op_spec		alu operation
//			_src_b		GPR	
//			_shift_amt	GPR or constant	
//
#macro alu_shf_left[_dst, _src_a, op_spec, _src_b, _shift_amt]
#if (isnum(_shift_amt))
#define_eval SHF_AMT _shift_amt
#if (isnum(_src_a))
#if (_src_a > 31)
.local tempa
	immed32[tempa, _src_a]
	alu[_dst, tempa, op_spec, _src_b, <</**/SHF_AMT/**/]
.endlocal		// tempa
#else		// not (_src_a > 31)
	alu[_dst, _src_a, op_spec, _src_b, <</**/SHF_AMT/**/]
#endif		// not (_src_a > 31)
#else		// not (isnum(_src_a))
	alu[_dst, _src_a, op_spec, _src_b, <</**/SHF_AMT/**/]
#endif 		// not (isnum(_src_a))
#else		// not (isnum(_shift_amt))
.local temp
	alu[temp, _shift_amt, B-A, 1, <<5]
	alu[--, temp, B, 0]
.endlocal		// temp
#if (isnum(_src_a))
#if (_src_a > 31)
.local tempa
	immed32[tempa, _src_a]
	alu[_dst, tempa, op_spec, _src_b, <<indirect]
.endlocal		// tempa
#else
	alu[_dst, _src_a, op_spec, _src_b, <<indirect]
#endif
#else
	alu[_dst, _src_a, op_spec, _src_b, <<indirect]
#endif
#endif
#endm

// commonly used alu ops
#macro add_shf_left[_dst, _src_a, _src_b, _shift_amt]
	alu_shf_left[_dst, _src_a, +, _src_b, _shift_amt]
#endm
#macro sub_shf_left[_dst, _src_a, _src_b, _shift_amt]
	alu_shf_left[_dst, _src_a, -, _src_b, _shift_amt]
#endm
#macro and_shf_left[_dst, _src_a, _src_b, _shift_amt]
	alu_shf_left[_dst, _src_a, AND, _src_b, _shift_amt]
#endm
#macro or_shf_left[_dst, _src_a, _src_b, _shift_amt]
	alu_shf_left[_dst, _src_a, OR, _src_b, _shift_amt]
#endm


// alu_rot_right
//		description: rotate right _src_b by _shift_amt bit positions, then perform
//					 operation op_spec against_src_a and shift result
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src_a		GPR or constant	
//			op_spec		alu operation
//			_src_b		GPR	
//			_shift_amt	constant	
//
#macro alu_rot_right[_dst, _src_a, op_spec, _src_b, _shift_amt]
#if (isnum(_src_a))
#if (_src_a > 31)
.local tempa
	immed32[tempa, _src_a]
	alu[_dst, tempa, op_spec, _src_b, >>rot /**/SHF_AMT/**/]
.endlocal		// temp
#else
	alu[_dst, _src_a, op_spec, _src_b, >>rot /**/SHF_AMT/**/]
#endif
#else
	alu[_dst, _src_a, op_spec, _src_b, >>rot /**/SHF_AMT/**/]
#endif 
#endm


// alu_rot_left
//		description: rotate left _src_b by _shift_amt bit positions, then perform
//					 operation op_spec against_src_a and shift result
//	 	outputs:
//			_dst		GPR
//		inputs:
//			_src_a		GPR or constant	
//			op_spec		alu operation
//			_src_b		GPR	
//			_shift_amt	constant	
//
#macro alu_rot_left[_dst, _src_a, op_spec, _src_b, _shift_amt]
#if (isnum(_src_a))
#if (_src_a > 31)
.local tempa
	immed32[tempa, _src_a]
	alu[_dst, tempa, op_spec, _src_b, <<rot /**/SHF_AMT/**/]
.endlocal		// temp
#else
	alu[_dst, _src_a, op_spec, _src_b, <<rot /**/SHF_AMT/**/]
#endif
#else
	alu[_dst, _src_a, op_spec, _src_b, <<rot /**/SHF_AMT/**/]
#endif 
#endm


// br_bset_ind
//		description:	branch to target_label if bit position _src_bit_pos of _src is 1
//		inputs:
//			_src			register under test
//			_src_bit_pos	GPR. bit position to test
//			target_label	if bit position set, branch to this label
//
//
#macro br_bset_ind[_src, _src_bit_pos, target_label]
	alu[--, _src_bit_pos, B, 0]
	alu[--, 1, AND, _src, >>indirect]
	br>0[target_label]
#endm


// br_bclr_ind
//		description:	branch to target_label if bit position _src_bit_pos of _src is 0
//		inputs:
//			_src			register under test
//			_src_bit_pos	GPR. bit position to test
//			target_label	if bit position clear, branch to this label
//
#macro br_bclr_ind[_src, _src_bit_pos, target_label]
	alu[--, _src_bit_pos, B, 0]
	alu[--, 1, AND, _src, >>indirect]
	br=0[target_label]
#endm


// bits_set_ind[_reg, _reg_start_bit, _reg_mask]
//		description:		set bits indicated by mask at starting position _reg_start_bit
//		output/input:
//			_reg			register to modify
//		inputs:
//			_reg_start_bit	register or constant<32, starting bit position
//			_reg_mask		register or constant<32, bit mask							
//
#macro bits_set_ind[_reg, _reg_start_bit, _reg_mask]
.local temp
	alu[temp, _reg_start_bit, B-A, 1, <<5]
	alu[--, temp, B, 0]
	alu_shf[_reg, _reg, OR, _reg_mask, <<indirect]
.endlocal
#endm


// bits_clr_ind[_reg, _reg_start_bit, _reg_mask]
//		description:		clear bits indicated by mask at starting position _reg_start_bit
//		output/input:
//			_reg			register to modify
//		inputs:
//			_reg_start_bit	register or constant<32, starting bit position
//			_reg_mask		register or constant<32, bit mask							
//
#macro bits_clr_ind[_reg, _reg_start_bit, _reg_mask]
.local temp
	alu[temp, _reg_start_bit, B-A, 1, <<5]
	alu[--, temp, B, 0]
	alu_shf[_reg, _reg, AND~, _reg_mask, <<indirect]
.endlocal
#endm


// thread_id
//		description:		given UENGINE_ID, get thread id of a context
//		output:
//			_tid			GPR thread id, 0-23
//		input:
//			UENG_ID			constant microengine identifier, 0-5
//
#macro thread_id[_tid, UENG_ID]
br=ctx[1, context1#]
br=ctx[2, context2#]
br=ctx[3, context3#]
context0#:
#define_eval MY_TID0 (UENG_ID<<2)
immed32[_tid, MY_TID0]
br[end#]

context1#:
#define_eval MY_TID1 ((UENG_ID<<2)+1)
immed32[_tid, MY_TID1]
br[end#]

context2#:	
#define_eval MY_TID2 ((UENG_ID<<2)+2)
immed32[_tid, MY_TID2]
br[end#]

context3#:
#define_eval MY_TID3 ((UENG_ID<<2)+3)
immed32[_tid, MY_TID3]
end#:
#endm


#endif // STDMAC_UC