//------------------------------------------------------------------------------------
//                                                                      
//                   I N T E L   P R O P R I E T A R Y                   
//                                                                       
//      COPYRIGHT (c)  2000 BY  INTEL  CORPORATION.  ALL RIGHTS          
//      RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//      BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//      RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//      LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//      MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//      THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                       
//                         INTEL  CORPORATION                            
//                                                                      
//                      2200 MISSION COLLEGE BLVD                        
//                                                                       
//                SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                       
//------------------------------------------------------------------------------------
// ip.uc
// macros to verify, modify and extract fields from ip header 
//
//
// system: IXP1200
// subsystem: IP microcode
// usage: library macros
// author: dfh	Apr 3, 2000
//
// revisions:


#ifndef IP_UC
#define IP_UC


// API:
//		ip_proc[_exception, const_dwbuf_ip_start, const_srbuf_ip_start]
//		ip_verify[_exception, const_srbuf_ip_start]
//
//      ip_verslen_extract[_ip_verslen, const_srbuf_ip_start]
//      ip_tos_extract[_ip_tos, const_srbuf_ip_start]
//      ip_totallen_extract[_ip_totallen, const_srbuf_ip_start]
//      ip_id_extract[_ip_id, const_srbuf_ip_start]
//      ip_flagfrag_extract[_ip_flagfrag, const_srbuf_ip_start]
//      ip_ttl_extract[_ip_ttl, const_srbuf_ip_start]
//      ip_prot_extract[_ip_prot, const_srbuf_ip_start]
//		ip_sa_extract[_ip_sa, const_srbuf_ip_start]
//		ip_da_extract[_ip_da, const_srbuf_ip_start]
//      ip_source_port_extract[_ip_source_port, const_srbuf_ip_start]
//      ip_dest_port_extract[_ip_dest_port, const_srbuf_ip_start]

//		ip_modify[const_dwbuf_ip_start, const_srbuf_ip_start]
//		ip_nat_modify[const_dwbuf_ip_start, _new_ip_da, const_srbuf_ip_start]
//
//      ip_flow_lookup[_rt_ptr, const_srbuf_ip_start, const_lookup_base, const_use_xfers]
//		ip_trie5_lookup[_rt_ptr, _ip_da, const_lookup_base, const_use_xfers]
//		ip_trie7_lookup[_rt_ptr, _ip_da, const_lookup_base, const_use_xfers]


// IP exceptions, range is 0x8-0xf
#define    IP_BAD_TOTAL_LENGTH			0x08
#define    IP_BAD_TTL					0x09
#define    IP_BAD_CHECKSUM				0x0a
#define    IP_NO_ROUTE					0x0b
#define	   FIREWALL_DESTROY				0x0c
#define	   IPV4_NO_OPTIONS				0x45			// version 4, header length 5 longwords 

// ip_proc
//		description: Process ip header that starts at read xfer byte pointer. 
//					 1. Verify ip total length, time-to-live and checksum.
//					 2. Modify time-to-live and checksum. Write modified header to sdram write xfers.
//					 Note: ip frag and ip options are not supported
//					 The ipheader must be aligned to a halfword boundary (start bytes 0 or 2 at xfer regs)
//		outputs:
//			exception				0 if no exception, otherwise ip exception code	
//		inputs:
//			const_dwbuf_ip_start	constant start byte of sdram write xfer registers.
//									this buffer of write data ends at end of ip hdr or 
//									physical xfer7, whichever comes first.	
//			const_srbuf_ip_start	constant start byte of ip header at sram read xfer registers
//									the header can wrap from physical xfer 7 to physical xfer 0					
//		size: 28 big-endian, 34 little-endian instr
//		see also: stdmac.uc, endian.uc
//		example usage:
//			
//
#macro ip_proc[exception, const_dwbuf_ip_start, const_srbuf_ip_start]
	ip_verify[exception, const_srbuf_ip_start]					; 18-23 instr
	br>0[end#]
	ctx_arb[voluntary]
	ip_modify[const_dwbuf_ip_start, const_srbuf_ip_start]		; 9-11 instr
	alu[exception, --, B, exception]							; set exit condition code
end#:
#endm


// ip_verify
//		description: Verify ip total length, time-to live and checksum.
//			note: ip header halfword READ_ALIGNs only are supported
//		outputs:
//			exception				0 if no exception, otherwise ip exception code	
//		inputs:
//			const_srbuf_ip_start	constant start byte of ip header at sram read xfer registers
//									the header can wrap from physical xfer 7 to physical xfer 0					
//		size: 22-26 instr
//		latency: 19-23 instr
//		see also: rec.h(ip exception codes), stdmac.uc, endian.uc
//		example usage:
//			ip_verify[exception, 14]		; ip header starts at sram read xfer byte 14
//
#macro ip_verify[exception, const_srbuf_ip_start]
#if ((const_srbuf_ip_start & 0x3) == 0)
#define_eval TOTAL_LEN_XFER (const_srbuf_ip_start /4)
#define TOTAL_LEN0 2
#define TOTAL_LEN1 3

#elif ((const_srbuf_ip_start & 0x3) == 2)
#define_eval TOTAL_LEN_XFER (((const_srbuf_ip_start /4) +1) &0x7)
#define TOTAL_LEN0 0
#define TOTAL_LEN1 1
#else
#error 2 	"Error: IP Header non-halfword READ_ALIGN. Not supported."
#endif

// check that total ip packet length in $xfer2 bytes 0-1 is at least 20 bytes
	field_comp[$xfer/**/TOTAL_LEN_XFER/**/, TOTAL_LEN0, TOTAL_LEN1, 0x14]
	br>0[ttl_verify#], guess_branch
	immed[exception, IP_BAD_TOTAL_LENGTH]
	br[end#]

ttl_verify#:
#define_eval TTL_XFER ((const_srbuf_ip_start /4) +2)
#define_eval TTL_BYTE (const_srbuf_ip_start & 0x3)

// Check that TTL in byte 8 of ip header is greater than 1.  
	field_comp[$xfer/**/TTL_XFER/**/, TTL_BYTE, TTL_BYTE, 1]
	br>0[cksum_verify#], defer[1]
	immed[exception, 0]
	immed[exception, IP_BAD_TTL]
	br[end#]

cksum_verify#:
	ip_cksum_verify[accum, const_srbuf_ip_start]
	br=0[end#]
	immed[exception, IP_BAD_CHECKSUM]
end#:
#endm


// ip_cksum_verify
//		description: Verify ip checksum 
//		outputs:
//			exception				GPR, 0 if good checksum, non-0 if bad checksum.	
//		inputs:
//			const_srbuf_ip_start	constant start byte of ip header at sram read xfer registers
//									the header can wrap from physical xfer 7 to physical xfer 0			
//		size: 10-14 instr (depending on alignment and endian mode)
//		latency: 10-14 cycles
//		see also: stdmac.uc, endian.uc
//		example usage:
//			ip_cksum_verify[exception, 14]		; ip header starts at sram read xfer byte 14
//
#macro ip_cksum_verify[exception, const_srbuf_ip_start]
#ifdef LITTLE_ENDIAN
_ip_checksum_verify_le[exception, const_srbuf_ip_start]
#else
_ip_checksum_verify_be[exception, const_srbuf_ip_start]
#endif
#endm


// _ip_checksum_verify_le
//		description: Verify ip header  checksum, little-endian, halfword aligned
//					 Internal macro used by ip_cksum_verify.
//					 Uses halfword adds to reduce total instructions.
//		outputs:
//			accum					checksum +1. If good, should be 0, otherwise is bad checksum.	
//		inputs:
//			const_srbuf_ip_start	constant start byte of ip header at sram read xfer registers
//									the header can wrap from physical xfer 7 to physical xfer 0
//		size: 14 instr
//		latency: 14 cycles
//		example usage:
//			_ip_cksum_verify_le[exception, 14]		; ip header starts at sram read xfer byte 14
//
#macro _ip_checksum_verify_le[accum, const_srbuf_ip_start]
.local temp
#define_eval READ_XFER0 (const_srbuf_ip_start >>2)
#define_eval READ_XFER1 ((READ_XFER0 +1) &0x7)
#define_eval READ_XFER2 ((READ_XFER1 +1) &0x7)
#define_eval READ_XFER3 ((READ_XFER2 +1) &0x7)
#define_eval READ_XFER4 ((READ_XFER3 +1) &0x7)
#define_eval READ_ALIGN (const_srbuf_ip_start & 0x3)
#if (READ_ALIGN == 0)
	alu[accum, --, B, $xfer/**/READ_XFER0/**/, >>16]
	alu[accum, accum, +16, $xfer/**/READ_XFER0/**/]
#elif (READ_ALIGN == 2)
#define_eval READ_XFER5 ((READ_XFER4 +1) &0x7)
	alu[accum, 0, +, $xfer/**/READ_XFER0/**/, >>16]
	alu[accum, accum, +16, $xfer/**/READ_XFER5/**/]
#else
#error 2 	"Error: IP Header non-halfword READ_ALIGN. Not supported."
#endif
	alu[accum, accum, +, $xfer/**/READ_XFER1/**/, >>16]
	alu[accum, accum, +16, $xfer/**/READ_XFER1/**/]
	alu[accum, accum, +, $xfer/**/READ_XFER2/**/, >>16]
	alu[accum, accum, +16, $xfer/**/READ_XFER2/**/]
	alu[accum, accum, +, $xfer/**/READ_XFER3/**/, >>16]
	alu[accum, accum, +16, $xfer/**/READ_XFER3/**/]
	alu[accum, accum, +, $xfer/**/READ_XFER4/**/, >>16]
	alu[accum, accum, +16, $xfer/**/READ_XFER4/**/]
    ld_field_w_clr[temp, 1100, accum]			; get high 16 of the total
    alu_shf[accum, temp, +, accum, <<16]		; add low 16 bits to upper 16
	alu[temp, 1, +carry, temp, <<16]			; add last carry +1, temp B op=0
	alu[accum, accum, +, temp, <<16]			; add 1<<16 to 0xffff to get zero result
.endlocal
#endm


// _ip_checksum_verify_be
//		description: Verify ip header  checksum, big-endian, halfword aligned
//					 Internal macro used by ip_cksum_verify.
//					 Uses 32 bit add with carry to reduce the total number of instructions.
//		outputs:
//			accum					checksum +1. If good, should be 0, otherwise is bad checksum.	
//		inputs:
//			const_srbuf_ip_start	constant start byte of ip header at sram read xfer registers
//									the header can wrap from physical xfer 7 to physical xfer 0
//		size: 10-11 instr
//		latency: 10-11 cycles
//		example usage:
//			_ip_cksum_verify_be[exception, 14]		; ip header starts at sram read xfer byte 14
//
#macro _ip_checksum_verify_be[accum, const_srbuf_ip_start]
.local temp accum
#define_eval READ_XFER0 (const_srbuf_ip_start >>2)
#define_eval READ_XFER1 ((READ_XFER0 +1) &0x7)
#define_eval READ_XFER2 ((READ_XFER1 +1) &0x7)
#define_eval READ_XFER3 ((READ_XFER2 +1) &0x7)
#define_eval READ_XFER4 ((READ_XFER3 +1) &0x7)
#define_eval READ_ALIGN (const_srbuf_ip_start & 0x3)
#if (READ_ALIGN == 0)
	alu[accum, --, B, $xfer/**/READ_XFER0/**/]
	alu[accum, accum, +, $xfer/**/READ_XFER1/**/]
#elif (READ_ALIGN == 2)
#define_eval READ_XFER5 ((READ_XFER4 +1) &0x7)
	alu[accum, 0, +16, $xfer/**/READ_XFER0/**/]
	alu[accum, accum, +, $xfer/**/READ_XFER1/**/]
	alu[accum, accum, +carry, $xfer/**/READ_XFER5/**/, >>16]
#else
#error 2 	"Error: IP Header non-halfword READ_ALIGN. Not supported."
#endif
	alu[accum, accum, +carry, $xfer/**/READ_XFER2/**/]
	alu[accum, accum, +carry, $xfer/**/READ_XFER3/**/]
	alu[accum, accum, +carry, $xfer/**/READ_XFER4/**/]
	alu[accum, accum, +carry, 0]				; add in previous carry
    ld_field_w_clr[temp, 1100, accum]			; get high 16 of the total
    alu_shf[accum, temp, +, accum, <<16]		; add low 16 bits to upper 16
	alu[temp, 1, +carry, temp, <<16]			; add last carry +1, temp B op=0
	alu[accum, accum, +, temp, <<16]			; add 1<<16 to 0xffff to get zero result
.endlocal
#endm


// ip_modify
//		description: Modify ip header, write modified ip header to write transfer registers.
//		outputs:
//		inputs:
//			const_dwbuf_ip_start	constant start byte of sdram write xfer registers.
//									the header can wrap from physical xfer 7 to physical xfer 0.
//			const_srbuf_ip_start	constant start byte of ip header at sram read xfer registers.
//									the header can wrap from physical xfer 7 to physical xfer 0.
//		size: 9-11 instr
//		latency: 9-11 ccyles
//		see also: stdmac.uc, endian.uc
//		example usage:
//			ip_modify[14, 14]		; ip header starts at sram read xfer byte 14
//
#macro 	ip_modify[const_dwbuf_ip_start, const_srbuf_ip_start]
.local temp
#define_eval IP_RD_WD0 ((const_srbuf_ip_start /4) &0x7)
#define_eval IP_RD_WD1 ((IP_RD_WD0 +1) & 0x7)
#define_eval IP_RD_WD2 ((IP_RD_WD1 +1) & 0x7)
#define_eval IP_RD_WD3 ((IP_RD_WD2 +1) & 0x7)
#define_eval IP_RD_WD4 ((IP_RD_WD3 +1) & 0x7)
#define_eval IP_RD_WD5 ((IP_RD_WD4 +1) & 0x7)
#define_eval IP_WR_WD0 ((const_dwbuf_ip_start /4) &0x7)
#define_eval IP_WR_WD1 ((IP_WR_WD0 +1) & 0x7)
#define_eval IP_WR_WD2 ((IP_WR_WD1 +1) & 0x7)
#define_eval IP_WR_WD3 ((IP_WR_WD2 +1) & 0x7)
#define_eval IP_WR_WD4 ((IP_WR_WD3 +1) & 0x7)
#define_eval IP_WR_WD5 ((IP_WR_WD4 +1) & 0x7)
#define_eval READ_ALIGN (const_srbuf_ip_start & 0x3)
#if (READ_ALIGN == 0)												; read/modify IPWD 2
	move[$$xfer/**/IP_WR_WD0/**/, $xfer/**/IP_RD_WD0/**/]
	move[$$xfer/**/IP_WR_WD1/**/, $xfer/**/IP_RD_WD1/**/]
	field_decr[temp, $$xfer/**/IP_RD_WD2/**/, 0, 0]					; iphdr byte 8 (see endian.uc)
	ip_cksum_modify[$$xfer/**/IP_WR_WD2/**/, temp, 2]				; iphdr byte 10-11
#if (IP_WR_WD3 < 8)
	move[$$xfer/**/IP_WR_WD3/**/, $xfer/**/IP_RD_WD3/**/]
#endif
#if (IP_WR_WD4 < 8)
	move[$$xfer/**/IP_WR_WD4/**/, $xfer/**/IP_RD_WD4/**/]
#endif
#elif(READ_ALIGN == 2)												; read/modify IPWDs 2 and 						3
	move[$$xfer/**/IP_WR_WD0/**/, $xfer/**/IP_RD_WD0/**/]
	move[$$xfer/**/IP_WR_WD1/**/, $xfer/**/IP_RD_WD1/**/]
	field_decr[$$xfer/**/IP_WR_WD2/**/, $xfer/**/IP_RD_WD2/**/, 2, 2]	; (see endian.uc)
	ip_cksum_modify[$$xfer/**/IP_WR_WD3/**/, $xfer/**/IP_RD_WD3/**/, 0]
#if (IP_WR_WD4 < 8)
	move[$$xfer/**/IP_WR_WD4/**/, $xfer/**/IP_RD_WD4/**/]
#endif
#if (IP_WR_WD5 < 8)
	move[$$xfer/**/IP_WR_WD5/**/, $xfer/**/IP_RD_WD5/**/]			; max words containing iphdr no options
#endif
#else
#error 2 	"Error: IP Header non-halfword READ_ALIGN. Not supported."
#endif
.endlocal
#endm


#macro ip_nat_modify[const_dwbuf_ip_start, _new_ip_da, const_srbuf_ip_start]
//tbd
#endm


// ip_cksum_modify
// 		description: Increment checksum to compensate for subtracting 1 from ttl.
// 				 checksum = bytes 10-11 of ip header
//		outputs:
//			dest					GPR destination	
//		inputs:
//			const_byte_offset		constant byte offset 0,1,2, or 3
//		size: 6 instr
//		latency: 6 cycles
//		see also: stdmac.uc, endian.uc. RFC 1141
//		example usage:
//			ip_cksum_modify[new_ip_wd2, prev_ip_wd2, 2]		; cksum at bytes 2-3
//
#macro ip_cksum_modify[dest, source, const_byte_offset]
.local temp0 temp1
#define_eval READ_BYTE0 const_byte_offset
#define_eval READ_BYTE1 (const_byte_offset + 1)
	field_incr[temp0, source, READ_BYTE0, READ_BYTE0]
	field_extract[temp1, temp0, READ_BYTE0, READ_BYTE0]
	br=0[add_carry#]
	br[end#], defer[1]
	move[dest, temp0]									; typically dest is a write xfer reg
add_carry#:
	field_incr[dest, temp0, READ_BYTE0, READ_BYTE1]
end#:
.endlocal
#endm


// ip_verslen_extract
// 		description: Extract ip version and header length from iphdr byte 0.
//		outputs:
//			_ip_verslen				GPR containing 4 bit version, 4 bit header length, right justified	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1 instr
//		see also: field.uc, endian.uc
//		example usage:
//			ip_verslen_extract[verslen, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_verslen_extract[_ip_verslen, const_srbuf_ip_start]
	field_srbuf_extract[_ip_verslen, const_srbuf_ip_start, 0, 0]
#endm


// ip_tos_extract
// 		description: Extract type of service from iphdr byte 1.
//		outputs:
//			_ip_tos					GPR containing 1 byte type of service, right justified	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1 instr
//		see also: stdmac.uc, endian.uc
//		example usage:
//			ip_tos_extract[ip_tos, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_tos_extract[_ip_tos, const_srbuf_ip_start]
	field_srbuf_extract[_ip_tos, const_srbuf_ip_start, 1, 1]
#endm


// ip_totallen_extract
// 		description: Extract total length from iphdr bytes 2-3.
//		outputs:
//			_ip_totallen		GPR containing 2 byte total length, right justiffied	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1 instr
//		see also: field.uc, endian.uc
//		example usage:
//			ip_totallen_extract[ip_totallen, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_totallen_extract[_ip_totallen, const_srbuf_ip_start]
	field_srbuf_extract[_ip_totallen, const_srbuf_ip_start, 2, 3]
#endm


// ip_id_extract
// 		description: Extract identification from iphdr bytes 4-5.
//		outputs:
//			_ip_id				GPR containing 2 byte identification, right justified	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1 instr
//		see also: field.uc, endian.uc
//		example usage:
//			ip_id_extract[ip_id, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_id_extract[_ip_id, const_srbuf_ip_start]
	field_srbuf_extract[_ip_id, const_srbuf_ip_start, 4, 5]
#endm


// ip_flagfrag_extract
// 		description: Extract flags and fragment offset from iphdr bytes 6-7.
//		outputs:
//			_ip_flagfrag		GPR containing 3 bit flags/13 bit fragment offset, right justified	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1 instr
//		see also: field.uc, endian.uc
//		example usage:
//			ip_flagfrag_extract[flagfrag, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_flagfrag_extract[_ip_flagfrag, const_srbuf_ip_start]
	field_srbuf_extract[_ip_flagfrag, const_srbuf_ip_start, 6, 7]
#endm


// ip_ttl_extract
// 		description: Extract time to live from iphdr byte 8.
//		outputs:
//			_ip_ttl				GPR containing 1 byte time to live, right justified	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1 instr
//		see also: field.uc, endian.uc
//		example usage:
//			ip_ttl_extract[ip_ttl, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_ttl_extract[_ip_ttl, const_srbuf_ip_start]
	field_srbuf_extract[_ip_ttl, const_srbuf_ip_start, 8, 8]
#endm


// ip_prot_extract
// 		description: Extract protocol field from iphdr byte 9.
//		outputs:
//			_ip_prot				GPR containing 1 byte protocol, right justified	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1 instr
//		see also: field.uc, endian.uc
//		example usage:
//			ip_prot_extract[ip_prot, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_prot_extract[_ip_prot, const_srbuf_ip_start]
	field_srbuf_extract[_ip_prot, const_srbuf_ip_start, 9, 9]
#endm


// ip_sa_extract
// 		description: Extract ip destination address  from iphdr bytes 12-15.
//		outputs:
//			_ip_sa					GPR containing 32 bit source address	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1 instr
//		see also: stdmac.uc, endian.uc
//		example usage:
//			ip_sa_extract[ip_sa, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_sa_extract[_ip_sa, const_srbuf_ip_start]
	field_srbuf_extract[_ip_sa, const_srbuf_ip_start, 12, 15]
#endm


// ip_da_extract
// 		description: Extract ip destination address from iphdr bytes 16-19.
//		outputs:
//			_ip_da					GPR containing 32 bit destination address	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1-2 instr
//		see also: stdmac.uc, endian.uc
//		example usage:
//			ip_sa_extract[ip_da, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_da_extract[_ip_da, const_srbuf_ip_start]
	field_srbuf_extract[_ip_da, const_srbuf_ip_start, 16, 19]
#endm


// ip_source_port_extract
// 		description: Extract source port from UDP or TCP header, bytes 20-21 from start of ip header.
//		outputs:
//			_ip_source_port			GPR containing 2 byte source port, right justified	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1-2 instr
//		see also: stdmac.uc, endian.uc
//		example usage:
//			ip_source_port_extract[ip_ource_port, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_source_port_extract[_ip_source_port, const_srbuf_ip_start]
	field_srbuf_extract[_ip_source_port, const_srbuf_ip_start, 20, 21]
#endm


// ip_dest_port_extract
// 		description: Extract dest port from UDP or TCP header, bytes 22-23 from start of ip header.
//		outputs:
//			_ip_dest_port			GPR containing 2 byte source port, right justified	
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//		size: 1-2 instr
//		see also: stdmac.uc, endian.uc
//		example usage:
//			ip_dest_port_extract[ip_ource_port, BYTEOFFSET14]		; ip header starts at byte 14 of sram read xfer
//
#macro	ip_dest_port_extract[_ip_dest_port, const_srbuf_ip_start]
	field_srbuf_extract[_ip_dest_port, const_srbuf_ip_start, 22, 23]
#endm

// hbhanoo: extract TCP control bits...

#macro tcp_control_extract[_control_bits, const_srbuf_ip_start] 
	field_srbuf_extract[_control_bits, const_srbuf_ip_start, 33, 33]
#endm

// ip_flow_lookup
//		description: Perform 5-tuple lookup using ip source address, ip dest address, ip protocol, 
//                   tcp or udp source port, tcp or udp dest port.
//		outputs:
//			_rt_ptr					GPR resulting route pointer. index to route entry. 0 if no route entry.
//		inputs:
//			const_srbuf_ip_start	constant byte address of ip header in sram read xfer regs
//			const_lookup_base		constant base address of lookup tables
//			const_use_xfers			constant longword index of 2 read/write transfer registers to be used
//
#macro ip_flow_lookup[_rt_ptr, const_srbuf_ip_start, const_lookup_base, const_use_xfers]
// tbd
//	
#endm


// ip_trie5_lookup
//		description: Perform ip longest prefix match lookup. Uses tables written by core Route Table Manager.
//					 This uses a dual lookup of hi64k entry table and trie block lookups.
//					 It uses a maximum of 5 sram reference signals.
//		outputs:
//			_rt_ptr					GPR resulting route pointer. index to route entry. 0 if no route entry.
//		inputs:
//			_ip_da					GPR with IP destination address
//			const_lookup_base		constant base address of route lookup tables
//			const_use_xfers			constant longword index of 2 read/write transfer registers to be used
//		size: 77 instr
//		latency:
//		see also: IXP Software Reference Manual(SRM) for algorithm description.
//		example usage:
//			ip_trie5_lookup[route_entry_ptr, ip_da, 0x20000, 0]
//
//
#macro ip_trie5_lookup[_rt_ptr, _ip_da, const_lookup_base, const_use_xfers]
.local temp_base2 temp_base3, offset, lookup_short, lookup_long, prev_rt_short, prev_rt_long
#define_eval CONST_XFER0 const_use_xfers
#define_eval CONST_XFER1 ((const_use_xfers +1) & 0x7)


// macro used by route_lookup
#macro next_trie[ipaddr, const_lsb, prev_rt_ptr, lookup, xfer_reg, trie_base]
#define_eval SHIFT_AMT (const_lsb)
	alu_shf[lookup, lookup, +4, ipaddr, >>SHIFT_AMT]	; form offset from ipaddr and trie block addr
	sram[read, xfer_reg, trie_base, lookup, 1], ctx_swap, optimize_mem, defer[1]
	ld_field_w_clr[prev_rt_ptr, 1100, xfer_reg]	
	ld_field_w_clr[lookup, 0011, xfer_reg], load_cc		; get trie pointer, set cond code
#endm

// tables_base can be either a static constant or from an .import_var
	immed32(tables_base, const_lookup_base)					; if static constant
	alu_shf[temp_base2, tables_base, +, 1, <<16]		; add 0x10000, 256 entry table
	alu_shf[temp_base3, temp_base2, +, 1, <<8]			; add 0x100, multiple 16 entry tables

	alu_shf[offset, 0, B, _ip_da, >>16]					; form offset from 31:16
first_lookup#:
 	sram[read, $xfer/**/CONST_XFER0/**/, tables_base, offset, 1], optimize_mem	; direct lookup off addr 31:16
	alu_shf[offset, 0, B, _ip_da, >>24]					; form offset from 31:24
	sram[read, $xfer/**/CONST_XFER1/**/, temp_base2, offset, 1],ctx_swap, optimize_mem, defer[1]	; direct lookup off addr 31:24
	immed[prev_rt_long, 0]
	ld_field_w_clr[lookup_short, 0011, $xfer/**/CONST_XFER1/**/], load_cc	; get trie pointer, set cond code
	br=0[long_path_only#]								; lookup_short = 0, no short path
	ld_field_w_clr[lookup_long, 0011, $xfer/**/CONST_XFER0/**/], load_cc	; get trie pointer, set cond code
	br=0[short_path_only#]								; lookup_long = 0, short path only
	br[both_paths#]										; follow both paths

short_path_only#:

second_lookup_short#:
	next_trie[_ip_da, 20, prev_rt_short, lookup_short, $xfer/**/CONST_XFER1/**/, temp_base3]
	br=0[set_route_ptr#]
third_lookup_short#:
	next_trie[_ip_da, 16, prev_rt_short, lookup_short, $xfer/**/CONST_XFER1/**/, temp_base3]
	br=0[set_route_ptr#]
	br[set_route_ptr#]

long_path_only#:
	ld_field_w_clr[lookup_long, 0011, $xfer/**/CONST_XFER0/**/], load_cc		; get trie pointer, set cond code
	br=0[set_route_ptr#]
second_lookup_long#:
	next_trie[_ip_da, 12, prev_rt_long, lookup_long, $xfer/**/CONST_XFER0/**/, temp_base3]
	br=0[set_route_ptr#]
third_lookup_long#:
	next_trie[_ip_da, 8, prev_rt_long, lookup_long, $xfer/**/CONST_XFER0/**/, temp_base3]
	br=0[set_route_ptr#]
fourth_lookup_long#:
	next_trie[_ip_da, 4, prev_rt_long, lookup_long, $xfer/**/CONST_XFER0/**/, temp_base3]
	br=0[set_route_ptr#]
fifth_lookup_long#:
	next_trie[_ip_da, 0, prev_rt_long, lookup_long, $xfer/**/CONST_XFER0/**/, temp_base3]
	br[set_route_ptr#]

both_paths#:
	alu_shf[lookup_short, lookup_short, +4, _ip_da, >>20]	; form offset from ipaddr and trie block addr
	sram[read, $xfer/**/CONST_XFER1/**/, temp_base3, lookup_short, 1], optimize_mem
	ld_field_w_clr[prev_rt_short, 1100, $xfer/**/CONST_XFER1/**/]	
	alu_shf[lookup_long, lookup_long, +4, _ip_da, >>12]	; form offset from ipaddr and trie block addr
	sram[read, $xfer/**/CONST_XFER0/**/, temp_base3, lookup_long, 1], ctx_swap, optimize_mem, defer[1]
	ld_field_w_clr[prev_rt_long, 1100, $xfer/**/CONST_XFER0/**/]	
	ld_field_w_clr[lookup_long, 0011, $xfer/**/CONST_XFER0/**/], load_cc		; get trie pointer, set cond code
	br=0[second_both_no_long#]
second_both_long#:
	ld_field_w_clr[lookup_short, 0011, $xfer/**/CONST_XFER1/**/], load_cc		; get trie pointer, set cond code
	br=0[third_lookup_long#]
	br[third_lookup_both#]
second_both_no_long#:
	ld_field_w_clr[lookup_short, 0011, $xfer/**/CONST_XFER1/**/], load_cc		; get trie pointer, set cond code
	br=0[set_route_ptr#]
	br[third_lookup_short#]
	
third_lookup_both#:
	alu_shf[lookup_short, lookup_short, +4, _ip_da, >>16]	; form offset from ipaddr and trie block addr
	sram[read, $xfer/**/CONST_XFER1/**/, temp_base3, lookup_short, 1], optimize_mem
	ld_field_w_clr[prev_rt_short, 1100, $xfer/**/CONST_XFER1/**/]	
	alu_shf[lookup_long, lookup_long, +4, _ip_da, >>8]	; form offset from ipaddr and trie block addr
	sram[read, $xfer/**/CONST_XFER0/**/, temp_base3, lookup_long, 1], ctx_swap, optimize_mem, defer[1]
	ld_field_w_clr[prev_rt_long, 1100, $xfer/**/CONST_XFER0/**/]	
	ld_field_w_clr[lookup_long, 0011, $xfer/**/CONST_XFER0/**/], load_cc		; get trie pointer, set cond code
	br=0[set_route_ptr#]
	br[fourth_lookup_long#]

set_route_ptr#:
	alu[_rt_ptr, --, B, $xfer/**/CONST_XFER0/**/, >>17]				; long match
	br!=0[end#]
	alu[_rt_ptr, --, B, prev_rt_long, >>17]		; long match at previous trie
	br!=0[end#]
    alu[_rt_ptr, --, B, $xfer/**/CONST_XFER1/**/, >>17]				; short match
	br!=0[end#]
	alu[_rt_ptr, --, B, prev_rt_short, >>17]		; short match at previous trie
end#:
.endlocal
#endm
#define ip_route_lookup ip_trie5_lookup


// ip_trie7_lookup
//		description: Perform ip longest prefix match lookup. Uses tables written by core Route Table Manager.
//					 This uses trie block lookups, 8 bit index lookup, followed by series of 4 bit index lookups.
//					 It uses a maximum of 7 sram references.				 
//		outputs:
//			_rt_ptr					GPR resulting route pointer. index to route entry. 0 if no route entry.
//		inputs:
//			_ip_da					GPR with IP destination address
//			const_lookup_base		constant base address of route lookup tables
//			const_use_xfers			constant longword index of read/write transfer register to be used
//		size: ? instr
//		latency: ?
//		see also: IXP Software Reference Manual(SRM) for algorithm description..
//		example usage:
//			ip_trie_lookup[route_entry_ptr, ip_dest, 0x20000, 0]
//
#macro ip_trie7_lookup[_rt_ptr, _ip_da, const_lookup_base, const_use_xfers]

// tbd

#endm


#endif // IP_UC