//------------------------------------------------------------------------------------
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  1998-2000 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
//-----------------------------------------------------------------------------------
// endian.uc
// IXPblock big/little-endian related macros
//
//
// system: IXP1200
// subsystem: IO microcode
// usage: library macros
// author: dfh Dec 30, 1998
//
// revisions:

#ifndef ENDIAN_UC
#define ENDIAN_UC


// endian-independent macros
// default is BIG-ENDIAN
// by defining LITTLE_ENDIAN, the appropriate underlying swaps will be inserted
// this will allow you to write code that runs big- or little-endian
//

// explicit endian macros
// these macros perform swaps on operators and result according to the following naming convention

// operator naming convention:
//
// swap :== e
// no_character :== 
// result_op :== e | <no_character>
// all_bytes :== <no_character>
// left_to_right_bytes :== 01 | 12 | 23 | <all_bytes>
// function :== add | comp
// aoperand :== _ea<left_to_right_bytes> | <no_character>
// boperand :== _eb<left_to_right_bytes> | <no_character>
// prev_carry :== _c | <no_charcter>

// unary operator = <result_op><function><left_to_right_bytes>
// operator = <result_op><function><aoperand><boperand><prev_carry>


// list of macros with explicit operand or result swaps
// operator naming convention:
//
// swap :== e
// no_character :== 
// result_op :== e | <no_character>
// all_bytes :== <no_character>
// left_to_right_bytes :== 01 | 12 | 23 | <all_bytes>
// function :== add | comp
// aoperand :== _ea<left_to_right_bytes> | <no_character>
// boperand :== _eb<left_to_right_bytes> | <no_character>
// prev_carry :== _c | <no_charcter>

// unary operator = <result_op><function><left_to_right_bytes>
// operator = <result_op><function><aoperand><boperand><prev_carry>


// swap									endian swap 4 bytes. res = swap(aop)
// swap32, swap01						endian swap 2 bytes. res = swap(left 2 bytes)	
// swap21, swap12						endian swap 2 bytes. res = swap(mid 2 bytes)
// swap10, swap23						endian swap 2 bytes. res = swap(right 2 bytes)
// comp									compare. aop - bop
// comp_ea_eb							compare. swap(aop) - swap(bop)
// comp_ea								compare. swap(aop) - bop
// comp_ea32, comp_ea01					compare. swap(left 2 bytes aop) - bop
// comp_ea21, comp_ea12					compare. swap(mid 2 bytes aop) - bop
// comp_ea10, comp_ea23					compare. swap(right 2 bytes aop) - bop
// eadd_ea_eb							add. res = swap( swap(aop) + swap(bop) )
// add_ea_eb							add. res = swap(aop) + swap(bop)		
// eadd_ea								add. res = swap( swap(aop) + bop )
// add_ea								add. res = swap(aop) + bop
// add_ea10_eb, add_ea23_eb				add. res = swap(right 2 bytes aop) + swap(bop) 
// add_ea32_eb, add_ea01_eb				add. res = swap(left 2 bytes aop) + swap(bop) 
// add_ea_c								add. res = swap(aop) + bop + prev_carry 
// add_ea10_c, add_ea23_c				add. res = swap(right 2 bytes aop) + bop + prev_carry
// add_ea32_c, add_ea01_c				add. res = swap(left 2 bytes aop) + bop + prev_carry




// swap
//		4 byte endian swap. 
//
#macro swap[res, aop]

	ld_field[res, 1111, aop, <<rot8]			; res = aop BBCCDDAA
	ld_field[res, 1010, aop, <<rot24], load_cc	; res = aop rot DDAABBCC insert DDxxBBxx

#endm

// swap32, swap01
//		2 byte endian swap. swap aop left two bytes and place in result
//
#define swap32 swap01
#macro swap01[res, aop]

	ld_field_w_clr[res, 0001, aop, >>24]		; res 000000AA = aop AABBCCDD
	ld_field[res, 0010, aop, >>8], load_cc		; res 0000BBAA = insert aop xxxxBBxx into res

#endm


// swap21, swap12
//		2 byte endian swap. swap aop middle two bytes and place in result
//
#define swap21 swap12
#macro swap12[res, aop]

	ld_field_w_clr[res, 0001, aop, >>16]		; res 000000BB = aop AABBCCDD
	ld_field[res, 0010, aop], load_cc			; res 0000CCBB = insert aop xxxxCCxx into res

#endm


// swap10, swap23
//		2 byte endian swap. swap aop middle two bytes and place in result
//
#define swap10 swap23
#macro swap23[res, aop]

	ld_field_w_clr[res, 0001, aop, >>8]			; res 000000CC = aop AABBCCDD
	ld_field[res, 0010, aop, <<8], load_cc		; res 0000DDCC = insert aop xxxxDDxx into res

#endm


#macro DEF_1BYTE_LOAD_MASK
#if (SHIFT == 0)
#define LOAD_MASK 0001
#elif (SHIFT == 8)
#define LOAD_MASK 0010
#elif (SHIFT == 16)
#define LOAD_MASK 0100
#elif (SHIFT == 24)
#define LOAD_MASK 1000
#endif
#endm


// extract0
// extract byte 0 from source and put it in dest<<shift_amt
//
#macro extract0[dest, source, shift_amt]
#define_eval SHIFT shift_amt
DEF_1BYTE_LOAD_MASK

#ifdef LITTLE_ENDIAN
#define_eval ROTATE (0-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc  
#else
#define_eval ROTATE (24-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc 
#endif

#undef LOAD_MASK
#endm


// extract1
// extract byte 1 from source and put it in dest<<shift_amt
//
#macro extract1[dest, source, shift_amt]
#define_eval SHIFT shift_amt
DEF_1BYTE_LOAD_MASK

#ifdef LITTLE_ENDIAN
#define_eval ROTATE (8-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc  
#else
#define_eval ROTATE (16-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc 
#endif

#undef LOAD_MASK
#endm


// extract2
// extract byte 2 from source and put it in dest<<shift_amt
//
#macro extract2[dest, source, shift_amt]
#define_eval SHIFT shift_amt
DEF_1BYTE_LOAD_MASK

#ifdef LITTLE_ENDIAN
#define_eval ROTATE (16-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc  
#else
#define_eval ROTATE (8-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc 
#endif

#undef LOAD_MASK
#endm


// extract3
// extract byte 3 from source and put it in dest<<shift_amt
//
#macro extract3[dest, source, shift_amt]
#define_eval SHIFT shift_amt
DEF_1BYTE_LOAD_MASK

#ifdef LITTLE_ENDIAN
#define_eval ROTATE (24-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc  
#else
#define_eval ROTATE (0-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc 
#endif

#undef LOAD_MASK
#endm


#macro DEF_RBYTE_LOAD_MASK
#if (SHIFT == 0)
#define RLOAD_MASK 0001
#elif (SHIFT == 8)
#define RLOAD_MASK 0010
#elif (SHIFT == 16)
#define RLOAD_MASK 0100
#else
#define RLOAD_MASK 0000
#endif
#endm


#macro DEF_LBYTE_LOAD_MASK
#if (SHIFT == 0)
#define LLOAD_MASK 0010
#elif (SHIFT == 8)
#define LLOAD_MASK 0100
#elif (SHIFT == 16)
#define LLOAD_MASK 1000
#else
#define LLOAD_MASK 0000

#endif
#endm


#macro DEF_2BYTE_LOAD_MASK
#if (SHIFT == 0)
#define LOAD_MASK 0011
#elif (SHIFT == 8)
#define LOAD_MASK 0110
#elif (SHIFT == 16)
#define LOAD_MASK 1100
#else
#define LOAD_MASK 0000
#endif
#endm


// extract01
//	place bytes 0 and 1 of source in dest<<shift_amt
//
#macro extract01[dest, source, shift_amt]
#define_eval SHIFT shift_amt
DEF_RBYTE_LOAD_MASK
DEF_LBYTE_LOAD_MASK
DEF_2BYTE_LOAD_MASK

#ifdef LITTLE_ENDIAN
#define_eval ROTATE (8-SHIFT&0x1f)
	ld_field_w_clr[dest, RLOAD_MASK, source, >>rot/**/ROTATE]		;if SHIFT==0, 000000CC = byte1 AABBCCDD
#define_eval ROTATE (24-SHIFT&0x1f)
	ld_field[dest, LLOAD_MASK, source, >>rot/**/ROTATE], load_cc		;if SHIFT==0, 0000DDCC = insert byte0 xxxxBBxx
#else
#define_eval ROTATE (16-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc
#endif

#undef RLOAD_MASK
#undef LLOAD_MASK
#undef LOAD_MASK
#endm


// extract12
//	place mid 2 bytes of source in dest<<shift_amt
//
#macro extract12[dest, source, shift_amt]
#define_eval SHIFT shift_amt
DEF_RBYTE_LOAD_MASK
DEF_LBYTE_LOAD_MASK
DEF_2BYTE_LOAD_MASK

#ifdef LITTLE_ENDIAN
#define_eval ROTATE (16-SHIFT&0x1f)
	ld_field_w_clr[dest, RLOAD_MASK, source, >>rot/**/ROTATE]		;if SHIFT==0, 000000BB = byte2 AABBCCDD
#define_eval ROTATE (0-SHIFT&0x1f)
	ld_field[dest, LLOAD_MASK, source, >>rot/**/ROTATE], load_cc	;if SHIFT==0, 0000CCBB = insert byte1 xxxxCCxx
#else
#define_eval ROTATE (8-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc 
#endif
 
#undef RLOAD_MASK
#undef LLOAD_MASK
#undef LOAD_MASK
#endm


// extract23
//	place bytes 2 and 3 of source in dest<<shift_amt
//
#macro extract23[dest, source, shift_amt]
#define_eval SHIFT shift_amt
DEF_RBYTE_LOAD_MASK
DEF_LBYTE_LOAD_MASK
DEF_2BYTE_LOAD_MASK

#ifdef LITTLE_ENDIAN
#define_eval ROTATE (24-SHIFT&0x1f)
	ld_field_w_clr[dest, RLOAD_MASK, source, >>rot/**/ROTATE]		;if SHIFT==0, 000000AA = byte3 AABBCCDD
#define_eval ROTATE (8-SHIFT&0x1f)
	ld_field[dest, LLOAD_MASK, source, >>rot/**/ROTATE], load_cc	;if SHIFT==0, 0000BBAA = insert byte2 xxxxBBxx
#else
#define_eval ROTATE (0-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc 
#endif 

#undef RLOAD_MASK
#undef LLOAD_MASK
#undef LOAD_MASK
#endm


#macro DEF_3BYTE_LOAD_MASK
#if (SHIFT == 0)
#define LOAD_MASK 0111
#elif (SHIFT == 8)
#define LOAD_MASK 1110
#endif
#endm

// extract02
//	place bytes 0-2 of source in dest, endian swapped if little-endian
//
#macro extract02[dest, source, shift_amt]
#define_eval SHIFT shift_amt
#ifdef LITTLE_ENDIAN
	extract12[dest, source, shift_amt]					; bytes 1-2
#if (SHIFT == 8)
	ld_field[dest, 1000, source, <<24], load_cc 		; byte 0
#else
	ld_field[dest, 0100, source, <<16], load_cc 		; byte 0
#endif

// else big-endian
#else
DEF_3BYTE_LOAD_MASK
#define_eval ROTATE (8-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc
#undef LOAD_MASK
#endif
#endm

// extract13
//	place bytes 1-3 of source in dest, endian swapped if little-endian
//
#macro extract13[dest, source, shift_amt]
#define_eval SHIFT shift_amt
#ifdef LITTLE_ENDIAN
	extract23[dest, source, shift_amt]					; bytes 2-3	
#if (SHIFT == 8)
	ld_field[dest, 1000, source, <<16], load_cc 		; byte 1
#else
	ld_field[dest, 0100, source, <<8], load_cc 			; byte 1
#endif

// else big-endian
#else		
DEF_3BYTE_LOAD_MASK
#define_eval ROTATE (0-SHIFT&0x1f)
    ld_field_w_clr[dest, LOAD_MASK, source, >>rot/**/ROTATE], load_cc 
#undef LOAD_MASK
#endif
#endm


// extract03
//	place bytes 2 and 3 of source in dest
//
//	since this is called by extract_field like other extracts above, dummySHIFT arg is added
// however, dummyShift should never be called
//
#macro extract03[dest, source, dummySHIFT]
#ifdef LITTLE_ENDIAN
	swap[dest, source]	
#else
    alu[dest, --, B, source]
#endif 
#endm

// merge_extract0
//	insert byte 0 of source in 3 of dest
//
#macro merge_extract0[dest, source]
#ifdef LITTLE_ENDIAN
    ld_field[dest, 0001, source], load_cc 
#else
    ld_field[dest, 0001, source, >>24], load_cc 
#endif
#endm


// merge_extract1
//	insert byte 1 of source in 3 of dest
//
#macro merge_extract1[dest, source]
#ifdef LITTLE_ENDIAN
    ld_field[dest, 0001, source, >>8], load_cc 
#else
    ld_field[dest, 0001, source, >>16], load_cc 
#endif
#endm


// merge_extract2
//	insert byte 2 of source in 3 of dest
//
#macro merge_extract2[dest, source]
#ifdef LITTLE_ENDIAN
    ld_field[dest, 0001, source, >>16], load_cc 
#else
    ld_field[dest, 0001, source, >>8], load_cc 
#endif
#endm


// merge_extract3
//	insert byte 3 of source in 3 of dest
//
#macro merge_extract3[dest, source]
#ifdef LITTLE_ENDIAN
    ld_field[dest, 0001, source, >>24], load_cc 
#else
    ld_field[dest, 0001, source], load_cc 
#endif
#endm


// merge_extract01[dest, source]
//	insert bytes 0-1 of source in 2-3 of dest, source endian swapped if little-endian
//
#macro merge_extract01[dest, source]
#ifdef LITTLE_ENDIAN
	ld_field[dest, 0001, source, >>8]				; byte 1
	ld_field[dest, 0010, source, <<8], load_cc		; byte 0
#else
    ld_field[dest, 0011, source, >>16], load_cc
#endif
#endm


// merge_extract12[dest, source]
//	insert bytes 1-2 of source in 2-3 of dest, source endian swapped if little-endian
//
#macro merge_extract12[dest, source]
#ifdef LITTLE_ENDIAN
	ld_field[dest, 0001, source, >>16]				; byte 2
	ld_field[dest, 0010, source], load_cc 			; byte 1
#else
    ld_field[dest, 0011, source, >>8], load_cc 
#endif 
#endm


// merge_extract23[dest, source]
//	insert bytes 2-3 of source in 2-3 of dest, source endian swapped if little-endian
//
#macro merge_extract23[dest, source]
#ifdef LITTLE_ENDIAN
	ld_field[dest, 0001, source, >>24]				; byte 3
	ld_field[dest, 0010, source, >>8], load_cc 		; byte 2
#else
    ld_field[dest, 0011, source], load_cc 
#endif 
#endm


// merge_extract02[dest, source]
//	insert bytes 0-2 of source in 1-3 of dest, source endian swapped if little-endian
//
#macro merge_extract02[dest, source]
#ifdef LITTLE_ENDIAN
	ld_field[dest, 0001, source, >>16]				; byte 2
	ld_field[dest, 0010, source]					; byte 1
	ld_field[dest, 0100, source, <<16], load_cc		; byte 0

// else big-endian
#else
    ld_field[dest, 0111, source, >>8], load_cc
#endif
#endm


// merge_extract13[dest, source]
//	insert bytes 1-3 of source in 1-3 of dest, source endian swapped if little-endian
//
#macro merge_extract13[dest, source]
#ifdef LITTLE_ENDIAN
	ld_field[dest, 0001, source, >>24]				; byte 3
	ld_field[dest, 0010, source, >>8]				; byte 2
	ld_field[dest, 0100, source, <<8], load_cc		; byte 1

// else big-endian
#else
    ld_field[dest, 0111, source], load_cc
#endif
#endm


// comp
//		compare aop to bop
//		afterward, branch on resulting condition code
//
#macro comp[aop, bop]
	alu[--, aop, -, bop]
#endm


// comp_ea_eb
//		4 byte endian compare swap(aop) to swap(bop)
//
#macro comp_ea_eb[aop, bop]
.local aswap bswap

	swap[aswap, aop]
	swap[bswap, bop]
	alu[--, aswap, -, bswap]

.endlocal
#endm


// comp_ea
//		4 byte endian compare swap(aop) to bop
//
#macro comp_ea[aop, bop]
.local aswap

	swap[aswap, aop]
	alu[--, aswap, -, bop]

.endlocal
#endm


// comp_ea32, comp_ea01
//		16 bit big-endian compare aop to big-endian bop, 
//		compare left 2 bytes of aop to bop
//
#define comp_ea32 comp_ea01
#macro comp_ea01[aop, bop]
.local aswap

	swap01[aswap, aop]
	alu[--, aswap, -, bop]

.endlocal
#endm


// comp_ea21, comp_ea12
//		16 bit big-endian compare aop to big-endian bop, 
//		compare middle 2 bytes of aop to bop
//
#define comp_ea21 comp_ea12
#macro comp_ea12[aop, bop]
.local aswap

	swap12[aswap, aop]
	alu[--, aswap, -, bop]

.endlocal
#endm


// comp_ea10, comp_ea23
//		16 bit big-endian compare aop to big-endian bop, 
//		compare right 2 bytes of aop to bop
//
#define comp_ea10 comp_ea23
#macro comp_ea23[aop, bop]
.local aswap

	swap23[aswap, aop]
	alu[--, aswap, -, bop]

.endlocal
#endm


// eadd_ea_eb
//		4 byte endian add
//		res = swap( swap(aop) + swap(bop) )
//
#macro eadd_ea_eb[res, aop, bop]
.local aswap

	swap[aswap, aop]
	swap[res, bop]
	alu[aswap, aswap, +, res]
	swap[res, aswap]

.endlocal
#endm


// add_ea_eb
//		4 byte endian add
//		res = swap(aop) + swap(bop)
//
#macro add_ea_eb[res, aop, bop]
.local aswap

	swap[aswap, aop]
	swap[res, bop]
	alu[res, aswap, +, res]

.endlocal
#endm


// eadd_ea
//		4 byte endian add
//		res = swap( swap(aop) + bop )
//		bop can be same register as res
//
#macro eadd_ea[res, aop, bop]
.local aswap

	swap[aswap, aop]
	alu[aswap, aswap, +, bop]
	swap[res, aswap]

.endlocal
#endm


// add_ea
//		4 byte endian add
//		res = swap(aop) + bop
//		bop can be same register as res
//
#macro add_ea[res, aop, bop]
.local aswap

	swap[aswap, aop]
	alu[res, aswap, +, bop]

.endlocal
#endm


// note: these could be easily expanded to support all variations

// add_ea32_eb, add_ea01_eb
//		swap(left 2 bytes aop) + swap(bop)
//
#define add_ea32_eb add_ea01_eb
#macro add_ea01_eb[res, aop, bop]
.local aswap bswap

	swap01[aswap, aop]							; halfword 0, len_version
	swap[bswap, bop]
	alu[res, aswap, +, bswap]

.endlocal
#endm


// add_ea10_eb, add_ea23_eb
//		swap(right 2 bytes aop) + swap(bop)
//
#define add_ea10_eb add_ea23_eb
#macro add_ea23_eb[res, aop, bop]
.local aswap bswap

	swap23[aswap, aop]							; halfword 0, len_version
	swap[bswap, bop]
	alu[res, aswap, +, bswap]

.endlocal
#endm


// add_ea_c
//		4 byte endian add
//		res = swap(aop) + bop + previous carry
//		bop can be same register as res
//
#macro add_ea_c[res, aop, bop]
.local aswap

	swap[aswap, aop]
	alu[res, aswap, +carry, bop]

.endlocal
#endm


// add_ea10_c, add_ea23_c
//		swap(right 2 bytes aop) + bop + previous carry
//		bop can be same register as res
//
#define add_ea10_c add_ea23_c
#macro add_ea23_c[res, aop, bop]
.local aswap

	swap23[aswap, aop]							; halfword 0, len_version
	alu[res, aswap, +carry, bop]

.endlocal
#endm      


// add_ea32_c, add_ea01_c
//		swap(left 2 bytes aop) + bop + previous carry
//		bop can be same register as res
//
#define add_ea32_c add_ea01_c
#macro add_ea01_c[res, aop, bop]
.local aswap

	swap01[aswap, aop]							; halfword 0, len_version
	alu[res, aswap, +carry, bop]

.endlocal
#endm      


// note only 1 byte and 2 byte field increments are supported
// assuming that network protocols never carry counts in 3 byte fields

// incr0
//	increment byte 0
//
//  source can be a read transfer register
//  dest can be a write transfer register
// 
#macro incr0[dest, source]
#ifdef LITTLE_ENDIAN
.local tempa
	alu_shf[tempa, source, +, 1]				; add 1 to byte 0
	ld_field[tempa, 1110, source]
	alu[dest, --, B, tempa]
.endlocal	
#else
	alu[dest, source, +, 1, <<24]				; add 1 to byte 0
#endif
#endm


// incr1
//	increment byte 1
//
//  source can be a read transfer register
//  dest can be a write transfer register
// 
#macro incr1[dest, source]
.local tempa
#ifdef LITTLE_ENDIAN
	alu_shf[tempa, source, +, 1, <<8]			; add 1 to byte 1
	ld_field[tempa, 1101, source]
	alu[dest, --, B, tempa]
#else
	alu[tempa, source, +, 1, <<16]				; add 1 to byte 2
	ld_field[tempa, 1011, source]
	alu[dest, --, B, tempa]
#endif
.endlocal
#endm


// incr2
//	increment byte 2
//
//  source can be a read transfer register
//  dest can be a write transfer register
// 
#macro incr2[dest, source]
.local tempa
#ifdef LITTLE_ENDIAN
	alu_shf[tempa, source, +, 1, <<16]			; add 1 to byte 2
	ld_field[tempa, 1011, source]
	alu[dest, --, B, tempa]
#else
	alu[tempa, source, +, 1, <<8]				; add 1 to byte 2
	ld_field[tempa, 1101, source]
	alu[dest, --, B, tempa]
#endif
.endlocal
#endm


// incr3
//	increment byte 3
//
//  source can be a read transfer register
//  dest can be a write transfer register
// 
#macro incr3[dest, source]
.local tempa
#ifdef LITTLE_ENDIAN
	alu_shf[dest, source, +, 1, <<24]			; add 1 to byte 3
#else
	alu[tempa, source, +, 1]					; add 1 to byte 3
	ld_field[tempa, 1110, source]
	alu[dest, --, B, tempa]
#endif
.endlocal
#endm


// incr01
//	increment bytes 0-1
//
//  source can be a read transfer register
//  dest can be a write transfer register
// 
#macro incr01[dest, source]
#ifdef LITTLE_ENDIAN
.local tempa tempb
	alu_shf[tempa, 1, +8, source, >>8]				; add 1 to byte 1, truncated source
	alu_shf[tempb, tempa, +16, source, <<8]			; shift byte 0 and add
	alu_shf[tempa, 0, +16, tempb, <<8]				; reload byte 1
	alu_shf[tempb, tempa, +8, tempb, >>8]			; reload byte 0
	ld_field_w_clr[tempa, 1100, source]				; get bytes 3-2 source
	alu[dest, tempa, +16, tempb]					; merge
.endlocal	
#else
	alu[dest, source, +, 1, <<16]
#endif
#endm


// incr12
//	increment bytes 1-2
//
//  source can be a read transfer register
//  dest can be a write transfer register
// 
#macro incr12[dest, source]
#ifdef LITTLE_ENDIAN
.local tempa tempb
	extract12[tempa, source, 0]
	alu_shf[tempb, tempa, +16, 1]					; add 1
	alu[tempa, 0, +16, tempb, <<8]					; load byte 2 only
	alu[tempb, tempa, +8, tempb, >>8]					; insert byte 1
	ld_field_w_clr[tempa, 1001, source]				; get bytes 2-1 source
	alu[dest, tempa, +, tempb, <<8]					; merge	
.endlocal	
#else
.local tempa
	alu_shf[tempa, source, +, 1, <<8]
	ld_field[tempa, 1001, source]
	alu[dest, --, B, tempa]
.endlocal
#endif
#endm


// incr23
//	increment bytes 2-3
//
//  source can be a read transfer register
//  dest can be a write transfer register
// 
#macro incr23[dest, source]
#ifdef LITTLE_ENDIAN
.local tempa tempb
	extract23[tempa, source, 0]
	alu_shf[tempb, tempa, +16, 1]					; add 1
	alu[tempa, 0, +16, tempb, <<8]					; load byte 3 only
	alu[tempb, tempa, +8, tempb, >>8]				; insert byte 2
	ld_field_w_clr[tempa, 0011, source]				; get bytes 1-0 source
	alu[dest, tempa, +, tempb, <<16]				; merge	
.endlocal	
#else
.local tempa
	alu_shf[tempa, source, +, 1]
	ld_field[tempa, 1100, source]
	alu[dest, --, B, tempa]
.endlocal
#endif
#endm



// note only 1 byte and 2 byte field decrements are supported
// assuming that network protocols never carry counts in 3 byte fields

// decr0
//	decrement byte 0
//
//	source can be a read transfer register
//	dest can be a write transfer register
//
#macro decr0[dest, source]
.local tempa
#ifdef LITTLE_ENDIAN
	alu_shf[tempa, source, -, 1]
	ld_field[tempa, 1110, source]
#else
	alu_shf[tempa, tempa, -, 1, <<24]
	ld_field[tempa, 0111, source]
#endif
	alu[dest, --, B, tempa]
.endlocal
#endm


// decr1
//	decrement byte 1
//
//	source can be a read transfer register
//	dest can be a write transfer register
//
#macro decr1[dest, source]
.local tempa
#ifdef LITTLE_ENDIAN
	alu_shf[tempa, source, -, 1, <<8]
	ld_field[tempa, 1101, source]
#else
	alu_shf[tempa, source, -, 1, <<16]
	ld_field[tempa, 1011, source]
#endif
	alu[dest, --, B, tempa]
.endlocal
#endm


// decr2
//	decrement byte 2
//
//	source can be a read transfer register
//	dest can be a write transfer register
//
#macro decr2[dest, source]
.local tempa
#ifdef LITTLE_ENDIAN
	alu_shf[tempa, source, -, 1, <<16]
	ld_field[tempa, 1011, source]
#else
	alu_shf[tempa, source, -, 1, <<8]
	ld_field[tempa, 1101, source]
#endif
	alu[dest, --, B, tempa]
.endlocal
#endm


// decr3
//	decrement byte 3
//
//	source can be a read transfer register
//	dest can be a write transfer register
//
#macro decr3[dest, source]
.local tempa
#ifdef LITTLE_ENDIAN
	alu_shf[tempa, source, -, 1, <<24]
	ld_field[tempa, 0111, source]
#else
	alu_shf[tempa, source, -, 1]
	ld_field[tempa, 1110, source]
#endif
	alu[dest, --, B, tempa]
.endlocal
#endm


// decr01
//	decrement byte 0-1
//
//	source can be a read transfer register
//	dest can be a write transfer register
//
#macro decr01[dest, source]
#ifdef LITTLE_ENDIAN
.local tempa tempb
	extract01[tempa, source, 0]
	alu_shf[tempb, tempa, -, 1]
	alu[tempa, 0, +16, tempb, <<8]					; load byte 2 only
	alu[tempb, tempa, +8, tempb, >>8]				; insert byte 1
	ld_field_w_clr[tempa, 1100, source]				; get bytes 2-1 source
	alu[dest, tempa, +, tempb]						; merge	
.endlocal
#else
.local tempa
	alu_shf[tempa, source, -, 1, <<16]
	ld_field[tempa, 0011, source]
	alu[dest, --, B, tempa]
.endlocal
#endif
#endm


// decr12
//	decrement byte 1-2
//
//	source can be a read transfer register
//	dest can be a write transfer register
//
#macro decr12[dest, source]
#ifdef LITTLE_ENDIAN
.local tempa tempb
	extract12[tempa, source, 0]
	alu_shf[tempb, tempa, -, 1]
	alu[tempa, 0, +16, tempb, <<8]					; load byte 2 only
	alu[tempb, tempa, +8, tempb, >>8]				; insert byte 1
	ld_field_w_clr[tempa, 1001, source]				; get bytes 2-1 source
	alu[dest, tempa, +, tempb, <<8]					; merge	
.endlocal
#else
.local tempa
	alu_shf[tempa, source, -, 1, <<8]
	ld_field[tempa, 1001, source]
	alu[dest, --, B, tempa]
.endlocal
#endif
#endm


// decr23
//	decrement byte 2-3
//
//	source can be a read transfer register
//	dest can be a write transfer register
//
#macro decr23[dest, source]
#ifdef LITTLE_ENDIAN
.local tempa tempb
	extract23[tempa, source, 0]
	alu_shf[tempb, tempa, -, 1]
	alu[tempa, 0, +16, tempb, <<8]					; load byte 2 only
	alu[tempb, tempa, +8, tempb, >>8]				; insert byte 1
	ld_field_w_clr[tempa, 0011, source]				; get bytes 2-1 source
	alu[dest, tempa, +, tempb, <<16]				; merge	
.endlocal
#else
.local tempa
	alu_shf[tempa, source, -, 1]
	ld_field[tempa, 1100, source]
	alu[dest, --, B, tempa]
.endlocal
#endif
#endm


// extract_field
// extract byte field from source and place in dest
//
#macro extract_field[dest, source, start_byte, end_byte]
#define_eval SBYTE start_byte
#define_eval EBYTE end_byte

#if (start_byte == end_byte)
	extract/**/SBYTE[dest, source, 0]
#else
	extract/**/SBYTE/**/EBYTE[dest, source, 0]
#endif
#endm


// incr_field
// increment byte field from source and place result in dest
//
#macro incr_field[dest, source, start_byte, end_byte]
#define_eval SBYTE start_byte
#define_eval EBYTE end_byte

#if (start_byte == end_byte)
	incr/**/SBYTE[dest, source]
#else
	incr/**/SBYTE/**/EBYTE[dest, source]
#endif
#endm


// decr_field
// increment byte field from source and place result in dest
//
#macro decr_field[dest, source, start_byte, end_byte]
#define_eval SBYTE start_byte
#define_eval EBYTE end_byte

#if (start_byte == end_byte)
	decr/**/SBYTE[dest, source]
#else
	decr/**/SBYTE/**/EBYTE[dest, source]
#endif
#endm


// comp_field
// extract byte field based on endian mode
// compare byte field to byte_compare_val
//
#macro comp_field[source, start_byte, end_byte, byte_compare_val]
.local tempa tempb
#define_eval SBYTE start_byte
#define_eval EBYTE end_byte

#if (start_byte == end_byte)
	extract/**/SBYTE[tempa, source, 0]
#else
	extract/**/SBYTE/**/EBYTE[tempa, source, 0]
#endif
#if (byte_compare_val < 32)
	alu[--, tempa, -, byte_compare_val]
#else
	immed[tempb, byte_compare_val]
	alu[--, tempa, -, tempb]
#endif
.endlocal
#endm


#macro br_eq_field[source, start_byte, end_byte, byte_compare_val, target_label]
	comp_field[source, start_byte, end_byte, byte_compare_val]
	br=0[target_label]
#endm


#macro br_gtr_field[source, start_byte, end_byte, byte_compare_val, target_label]
	comp_field[source, start_byte, end_byte, byte_compare_val]
	br>0[target_label]
#endm


#macro br_less_field[source, start_byte, end_byte, byte_compare_val, target_label]
	comp_field[source, start_byte, end_byte, byte_compare_val]
	br<0[target_label]
#endm


#macro br_gtreq_field[source, start_byte, end_byte, byte_compare_val, target_label]
	comp_field[source, start_byte, end_byte, byte_compare_val]
	br>=0[target_label]
#endm


#macro br_lesseq_field[source, start_byte, end_byte, byte_compare_val, target_label]
	comp_field[source, start_byte, end_byte, byte_compare_val]
	br<=0[target_label]
#endm



// dbl_extract_field
//	get field that spans 2 source regs
//
//	dest cannot be a write transfer reg
//
#macro dbl_extract_field[dest,  source0, start_byte, source1, end_byte]
.local temp
#define_eval SBYTE start_byte
#define_eval EBYTE end_byte
#define_eval SHIFT (8+end_byte*8)

	extract/**/SBYTE/**/3[dest, source0, SHIFT]
	merge_extract0/**/EBYTE[dest, source1]

.endlocal	
#endm

#endif // ENDIAN_UC