//------------------------------------------------------------------------------------
//                                                                      
//                   I N T E L   P R O P R I E T A R Y                   
//                                                                       
//      COPYRIGHT (c)  2000 BY  INTEL  CORPORATION.  ALL RIGHTS          
//      RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//      BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//      RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//      LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//      MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//      THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                       
//                         INTEL  CORPORATION                            
//                                                                      
//                      2200 MISSION COLLEGE BLVD                        
//                                                                       
//                SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                       
//------------------------------------------------------------------------------------
// sdram.uc
// macros to access sdram
//
//
// system: IXP1200
// subsystem: utility microcode
// usage: library macros
// author: dfh	May 1, 2000
//
// revisions:


#ifndef SDRAM_UC
#define SDRAM_UC


// API:
//	sdram_read[xfer, _base_qw_addr, _qw_offset, const_qw_count, sig_spec]
//	sdram_write[xfer, _base_qw_addr, _qw_offset, const_qw_count, sig_spec]
//	sdram_rfifo_read[_sd_qw_addr, _rfifo_base_addr, const_rfifo_offset, qw_count, sig_spec]



// sdram_read
//		description: Read 1-4 quadwords from sdram to xfer registers.
//		outputs:
//		inputs:
//			xfer			xfer register with first longword of write data
//			_base_qw_addr	quadword base address to be added to form address
//			_qw_offset		quadword offset to be added to form address
//			const_qw_count	number of quadwords to write
//			sig_spec		pend:  waiting on signal, context swap
//							sig:   requesting signal, but continuing
//							nosig: not requesting signal, continuing
//		size: 1 instr
//		see also: constants.uc
//		example usage:
//			sdram_read(DX0, packet_buf_addr, QWOFFSET0, QWCOUNT4, nosig);
//
#macro sdram_read[xfer, _base_qw_addr, _qw_offset, const_qw_count, sig_spec]
#if (isnum(sig_spec))
	sdram[read, xfer, _base_qw_addr, _qw_offset, const_qw_count]
#else
	sdram[read, xfer, _base_qw_addr, _qw_offset, const_qw_count], sig_spec
#endif
#endm


// sdram_write
// 		description: Write 1-4 quadwords from xfer registers to sdram.
//		outputs:
//			xfer			xfer register with first longword of read data
//		inputs:
//			_base_qw_addr	quadword base address to be added to form address
//			_qw_offset		quadword offset to be added to form address
//			const_qw_count	number of quadwords to write
//			sig_spec		pend:  waiting on signal, context swap
//							sig:   requesting signal, but continuing
//							nosig: not requesting signal, continuing
//		size: 1 instr
//		see also: constants.uc
//		example usage:
//			sdram_read(DX0, packet_buf_addr, QWOFFSET0, QWCOUNT4, nosig);
//
#macro sdram_write[xfer, _base_qw_addr, _qw_offset, const_qw_count, sig_spec]
#if (isnum(sig_spec))
	sdram[write, xfer, _base_qw_addr, _qw_offset, const_qw_count]
#else
	sdram[write, xfer, _base_qw_addr, _qw_offset, const_qw_count], sig_spec
#endif
#endm


// sdram_rfifo_read
// 		description: Copy 1-16 quadwords from rfifo(base+offset) to sdram.
//		outputs:
//		inputs:
//			_sd_qw_addr			starting quadword address at sdram
//			_rfifo_base_addr	quadword base address to be added to form rfifo address
//			const_rfifo_offset	quadword offset to be added to form rfifo address
//			qw_count			GPR or constant. number of quadwords to write
//			sig_spec			pend:  waiting on signal, context swap
//								sig:   requesting signal, but continuing
//								nosig: not requesting signal, continuing
//		size: 1-5 instr
//		see also: 
//		example usage:
//
#macro sdram_rfifo_read[_sd_qw_addr, _sd_offset, _rfifo_base_addr, const_rfifo_offset, qw_count, sig_spec]
.local tempa

#if (isnum(qw_count))
#if (const_rfifo_offset == 0)
	alu_shf[--, --, B, _rfifo_base_addr, <<4]
#else
	alu_shf[tempa, --, B, const_rfifo_offset, <<4]
	alu_shf[--, tempa, +, _rfifo_base_addr, <<4]
#endif
#else	// qw_count is a register
// use indirect count (bit 20 overide, 16-19 count)
	alu[tempa, --, B, 1, <<20]
	alu[tempa, tempa, +, qw_count, <<16]
#if (const_rfifo_offset == 0)
	alu_shf[--, tempa, OR, _rfifo_base_addr, <<4]
#else
	alu_shf[tempa, tempa, OR, const_rfifo_offset, <<4]
	alu_shf[--, tempa, +, _rfifo_base_addr, <<4]
#endif
#endif	// qw_count is a register
#if (isnum(qw_count))

#if (isnum(sig_spec))
	sdram[r_fifo_rd, --, _sd_qw_addr, _sd_offset, qw_count], indirect_ref
#else
	sdram[r_fifo_rd, --, _sd_qw_addr, _sd_offset, qw_count], indirect_ref, sig_spec
#endif
#else
#if (isnum(sig_spec))
	sdram[r_fifo_rd, --, _sd_qw_addr, _sd_offset, 1], indirect_ref
#else
	sdram[r_fifo_rd, --, _sd_qw_addr, _sd_offset, 1], indirect_ref, sig_spec
#endif

#endif
.endlocal
#endm



// DO NOT USE. pre v1.0. This will be removed in v1.1
//
//sdram_r_fifo_rd
//		description: Transfer r_fifo fixed count to sdram.
//		inputs:
//			sd_addr			sdram quadword address
//			r_fifo_addr		r_fifo quadword address
//			const_offset	quadowrd offset applied to both addresses
//			const_qw_count	number of quadwords to transfer
//			const_sig_spec	SYNC	ctx_swap
//							ASYNC	sig_done
//						    NOSYNC	(no signal)
//
//		example usage:
//			// transfer 8 quadwords to/from address offset 4
//			sdram_r_fifo_rd[sdaddr, raddr, 4, 8, SYNC]
//
#macro sdram_r_fifo_rd[sd_addr, r_fifo_addr, const_offset, const_qw_count, const_sig_spec]

#if (const_offset == 0)
	alu_shf[--, --, B, r_fifo_addr, <<4]
#else
.local tempa
	alu_shf[tempa, --, B, const_offset, <<4]
	alu_shf[--, tempa, +, r_fifo_addr, <<4]
.endlocal
#endif

#if (const_sig_spec == SYNC)
	sdram[r_fifo_rd, --, sd_addr, const_offset, const_qw_count], indirect_ref, ctx_swap
#elif (const_sig_spec == ASYNC)	
	sdram[r_fifo_rd, --, sd_addr, const_offset, const_qw_count], indirect_ref, sig_done
#else
	sdram[r_fifo_rd, --, sd_addr, const_offset, const_qw_count], indirect_ref
#endif

#endm


#endif // SDRAM_UC