//------------------------------------------------------------------------------------
//                                                                      
//                   I N T E L   P R O P R I E T A R Y                   
//                                                                       
//      COPYRIGHT (c)  2000 BY  INTEL  CORPORATION.  ALL RIGHTS          
//      RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//      BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//      RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//      LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//      MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//      THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                       
//                         INTEL  CORPORATION                            
//                                                                      
//                      2200 MISSION COLLEGE BLVD                        
//                                                                       
//                SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                       
//------------------------------------------------------------------------------------
// packetq.uc
// macros for queueing packets 
//
//
// system: IXP1200
// subsystem: forwarding microcode
// usage: library macros
// author: dfh	May 1, 2000
//
// revisions:


#ifndef PACKETQ_UC
#define PACKETQ_UC


// API:
//	packetq_send[_rec_state, _desc_addr, _output_intf, USE_3XFERS]
//	packetq_slow_send[_rec_state, _desc_addr, _output_intf, USE_3XFERS]
//	packetq_fast_send[_rec_state, _desc_addr, _output_intf, USE_3XFERS]

#include "port.uc"

// if fast port numbers are not defined, default to no fast port, -1
// 
#ifndef FAST_PORT_LO
#define FAST_PORT_LO -1
#endif
#ifndef FAST_PORT_HI
#define FAST_PORT_HI -1
#endif

// core stack interfaces
//
#define CORE_STACK_INTF1 18
#define CORE_STACK_INTF2 19
#define CORE_ATMSTACK_INTF1 20
#define CORE_ATMSTACK_INTF2 21


// packetq_send
// 		description: Enqueue a packet. Depending on interface, it could be for transmit, core stack 
//					 or other thread
//		inputs:
//			_rec_state			state information, e.g.
//							    freelist id, discard flag, byte count, enqueue seq#, fast or slow			
//			_desc_addr			packet descriptor sram address
//			_output_intf		interface selected by route lookup or packet classification
//			Q_BASE				base address of queue head array
//			DESC_BASE			base address of descriptor buffers
//			USE_3XFERS			index of 3 sram transfer regs to use for descriptor
//		size: 
//		example usage:
//			packetq_xmit(rec_state, output_intf, packet_buf_addr, descriptor_addr, X6);
//
#macro packetq_send[_rec_state, _output_intf, _desc_addr, Q_BASE, DESC_BASE, USE_3XFERS]
#define_eval SEQ_XFER (USE_3XFERS)
	.if (bit(rec_state, REC_STATE_FPORT) == 1)
		packetq_fast_seq[_rec_state, _output_intf, $xfer/**/SEQ_XFER/**/]
	.endif
	// note : need assembler change to support richer .if expression
	.if (_output_intf > FAST_PORT_HI)			// queue to core
		packetq_slow_send[_rec_state, _output_intf, _desc_addr, Q_BASE, DESC_BASE, USE_3XFERS]
	.elif (_output_intf >= FAST_PORT_LO)		// queue to fast port
		packetq_fast_send[_rec_state, _output_intf, _desc_addr, Q_BASE, DESC_BASE, USE_3XFERS]
	.else										// queue to slow port
		packetq_slow_send[_rec_state, _output_intf, _desc_addr, Q_BASE, DESC_BASE, USE_3XFERS]
	.endif
#endm


#macro packetq_slow_send[_rec_state, _output_intf, _desc_addr, Q_BASE, DESC_BASE, USE_4XFERS]
.local queue_packet_count tempa tail_ptr head_ptr descriptor_base queue_descriptor_addr relative_buf_addr
#define_eval WRITE_XFER0 (USE_4XFERS)
#define_eval WRITE_XFER1 ((USE_4XFERS + 1) & 0x7)
#define_eval WRITE_XFER2 ((USE_4XFERS + 2) & 0x7)
#define_eval WRITE_XFER3 ((USE_4XFERS + 3) & 0x7)
	immed32[queue_descriptor_addr, Q_BASE]
	immed32[descriptor_base, DESC_BASE]
	alu[tempa, 0x7, AND, _rec_state, >>REC_STATE_QSELECT]														; hard code q select
	alu[queue_descriptor_addr, queue_descriptor_addr, +4, tempa]				; add qselect
	alu[queue_descriptor_addr, queue_descriptor_addr, +, _output_intf, <<4]		; add output port

	sram [read_lock, $xfer/**/WRITE_XFER0/**/, queue_descriptor_addr, 0, 2], 
				optimize_mem, ctx_swap, defer[1]								; read the queue descriptor 2 words
	alu[relative_buf_addr, _desc_addr, -, descriptor_base]
	alu_shf[queue_packet_count, 1, +16, $xfer/**/WRITE_XFER1/**/]	; update queue element count
	.if (queue_packet_count == 1)									; if queue was empty before
		scratch_bset_ind[$xfer/**/WRITE_XFER3/**/, _output_intf, XMIT_PWP_VECTOR, NOSYNC]	; set packet queued bit for output port
		alu_shf [tempa, 0, b, relative_buf_addr, <<16]	
		alu [$xfer/**/WRITE_XFER0/**/, tempa, or, relative_buf_addr]	; set head pointer and tail pointer to new buffer descr
	.else															; if queue had one or more packet
		ld_field_w_clr[tail_ptr, 0011, $xfer/**/WRITE_XFER0/**/]
		ld_field_w_clr[head_ptr, 1100, $xfer/**/WRITE_XFER0/**/]
		alu [$xfer/**/WRITE_XFER0/**/, head_ptr, or, relative_buf_addr]	; OR in the new tail pointer for queue descr
		alu [$xfer/**/WRITE_XFER2/**/, --, B, relative_buf_addr]					; relative_buf_addr is the new tail
		sram [write, $xfer/**/WRITE_XFER2/**/, descriptor_base, tail_ptr, 1], ctx_swap	; rewrite the current (now old) tail
	.endif
	ld_field_w_clr[$xfer/**/WRITE_XFER1/**/, 0111, queue_packet_count]
	sram [write_unlock, $xfer/**/WRITE_XFER0/**/, queue_descriptor_addr, 0, 2], priority, ctx_swap ; write the updated queue descriptor and release the lock - ctx_swap required

#ifdef PROFILE
	immed[tempa, TOTAL_RECEIVES]							; total enqueues
	scratch[incr, --, tempa, 0, 1]

#endif
.endlocal
#endm

// insure correct enqueue sequence for fast ports
//
#macro packetq_fast_seq[_rec_state, _output_intf, USE_XFER]
.local tempa
	alu_shf[tempa, 0xf, AND, rec_state, >>20]					; extract seq number
	.if (_output_intf == FAST_PORT_LO)
		_packetq_fast_seq_check[tempa, ENQUEUE_SEQ1, USE_XFER]	; wait for my seq no. to come up
		fast_wr[3, INCR_ENQ_NUM1]								; increment fast port 1 seq number
	.elif (_output_intf == FAST_PORT_HI)
		_packetq_fast_seq_check[tempa, ENQUEUE_SEQ2, USE_XFER]	; wait for my seq no. to come up
		fast_wr[3, INCR_ENQ_NUM2]								; increment fast port 1 seq number
	.endif
.endlocal
#endm


#macro packetq_fast_send[_rec_state, _output_intf, _desc_addr, Q_BASE, DESC_BASE, USE_3XFERS]
//TBD get code from rec_enqueue_f.uc
#endm


// enqueue packets from FAST PORT in order
// read the fbi fast port sequence number until it matches our sequence number
//
#macro _packetq_fast_seq_check[seq_num, CSR_NAME, USE_XFER]
sleep_wait_for_seq_change#:
	csr[read, USE_XFER, CSR_NAME], ctx_swap				; read the current seq number
	alu[--, seq_num, -, USE_XFER]							; is my number up?
	br=0[seq_check_done#]
	br[sleep_wait_for_seq_change#]
seq_check_done#:
#endm


#endif  // PACKETQ_UC