//------------------------------------------------------------------------------------
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  1999-2000 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
//-----------------------------------------------------------------------------------
// mem.uc
// IXPblock memory allocation macros
//
//
// system: IXP1200
// subsystem: IO microcode
// usage: library macros
// author: dfh Jan 12, 1999	
//
// revisions:	dfh Apr 6, 2000		separated from stdmac.uc


#ifndef MEM_UC
#define MEM_UC

// API

// mem_freelist_create
// mem_pop
// mem_wait
// mem_ds_alloc
// mem_alloc
// mem_free



// mem_freelist_create
//		description: Create freelist of available memory buffers using push/pop regs.
//
//		inputs:
//			NUM_BUFFERS			number of buffers put on the freelist
//			USE_XFER			transfer register to use
//			STACK_ID			0-7, to identify one of 8 hardware push/pop stacks
//			SD_BASE
//			SD_SIZE
//			S_BASE
//			S_SIZE
//		size:	9 instr
//		example usage:
//			#define FREELIST0_HANDLE ,0,100,2048,500,4
//			mem_freelist_create(300, X0, FREELIST0_HANDLE);
///
#macro mem_freelist_create[NUM_BUFFERS, STACK_ID, SD_BASE, SD_SIZE, S_BASE, S_SIZE, USE_XFER]
.local  buf_addr ii count

	immed32[buf_addr, S_BASE]						; base address for memory region
	immed[ii, 0]
	immed32[count, NUM_BUFFERS]						; num entries

freelist_loop#:

	sram[push, --, buf_addr, ii, STACK_ID]			; push to freelist
	alu[ii, ii, +, S_SIZE]							; add sram buffer size
	alu[count, count, -, 1]
	br>0[freelist_loop#]

	immed[USE_XFER, 0]
	sram[write, USE_XFER, buf_addr, 0, 1], ctx_swap	; end of list is set to 0

.endlocal
#endm


// mem_pop
// 		description: Pop a pointer from the push/pop stack identified by STACK_ID.
//		outputs:
//			USE_XFER			which sram read xfer to place the popped pointer, 0-7
//		inputs:
//			STACK_ID			which push/pop stack(freelist) to pop from
//			sig_spec		pend:  waiting on signal, context swap
//							sig:   requesting signal, but continuing
//							nosig: not allowed. error
//		size: 1 instr
//		example usage:
//			mem_pop(X6, FREELIST0, sig);
//
#macro mem_pop[USE_XFER, STACK_ID, sig_spec]
#if (isnum (sig_spec))
#error 2  "Error: mem_pop. pend or sig is required."
#endif
	// note: STACK_ID is a dummy unused argument, it prevents an assembler warning
	sram[pop, USE_XFER, USE_XFER, 0, STACK_ID], sig_spec
#endm


// mem_wait
//		description: Wait on a signal.
//		inputs:
//			mem_type	sram, sdram, or other signal
//
#macro mem_wait[mem_type]
	ctx_arb[mem_type]
#endm
// sig_wait
// later: to be moved to sig.uc
#macro sig_wait[mem_type]
	ctx_arb[mem_type]
#endm


// mem_ds_alloc
// 		description: Given a freelist handle and popped ptr, calculate SDRAM and SRAM buffer addresses
//		outputs:
//			_sd_addr			quadword address of buffer in SDRAM
//			_s_addr				longword address of buffer in SRAM
//		inputs:
//			STACK_ID			which freelist to use for retry
//			SD_BASE				base address of sdram buffers
//			SD_SIZE				size of sdram buffers
//			S_BASE				base address of sram buffers
//			S_SIZE				size of sram buffers	
//			_pop_xfer			popped pointer (from mem_pop), which transfer reg to use for retry
//		size: 9 instr
//		example usage:
//			#define FREELIST0 0
//			#define FREELIST0_HANDLE 0,100,2048,500,4
//			mem_pop(X6, FREELIST0);
//			mem_ds_alloc(packet_buf_addr, descriptor_addr, FREELIST0_HANDLE, X6);
//
#macro mem_ds_alloc[_sd_addr, _s_addr, STACK_ID, SD_BASE, SD_SIZE, S_BASE, S_SIZE, _pop_xfer]
.local buffer_base descriptor_base relative_addr
started#:
// test free buffer pool delivered by the pop 
//
    alu[--, --, B, _pop_xfer]									; if zero, the free list is empty
    br=0[no_buf_available#], defer[1]							; wait for buffer to free up

    immed32[buffer_base, SD_BASE]
	immed32[descriptor_base, S_BASE]
	alu[relative_addr, _pop_xfer, -, descriptor_base]
	alu[relative_addr, relative_addr, +, 1]				; add 1 to not use pointer (temp for debugging)

// note: SD_SIZE is quadwords, S_SIZE is longwords
// calculate ratio, to get correct shift amount
// this will support up to 64kB sdram buffer with 4 word sram buffer
#define_eval SD_MULTIPLIER (SD_SIZE / S_SIZE)
#if (SD_MULTIPLIER == 1)
#define_eval SD_SHIFT (0)
#elif (SD_MULTIPLIER == 2)
#define_eval SD_SHIFT (1)
#elif (SD_MULTIPLIER == 4)
#define_eval SD_SHIFT (2)
#elif (SD_MULTIPLIER == 8)
#define_eval SD_SHIFT (3)
#elif (SD_MULTIPLIER == 16)
#define_eval SD_SHIFT (4)
#elif (SD_MULTIPLIER == 32)
#define_eval SD_SHIFT (5)
#elif (SD_MULTIPLIER == 64)
#define_eval SD_SHIFT (6)
#elif (SD_MULTIPLIER == 128)
#define_eval SD_SHIFT (7)
#elif (SD_MULTIPLIER == 256)
#define_eval SD_SHIFT (8)
#elif (SD_MULTIPLIER == 512)
#define_eval SD_SHIFT (9)
#elif (SD_MULTIPLIER == 1024)
#define_eval SD_SHIFT (10)
#elif (SD_MULTIPLIER == 2048)
#define_eval SD_SHIFT (11)
#else
#error 2 "Error: mem_ds_alloc. sdram buffer size/ sram buffer size too large"
#endif
    alu[_sd_addr, buffer_base, +, relative_addr, <</**/SD_SHIFT/**/]	; calculate sdram address

	br[done#], defer[1]
	alu[_s_addr, descriptor_base, +, relative_addr]		; calculate sram address
no_buf_available#:
// retry until buffer is available
// note: third argument here is ignored for sram pop
	sram[pop, _pop_xfer, _pop_xfer, 0, STACK_ID], ctx_swap
	br[started#]
done#:
.endlocal
nop
#endm


// pre v1.0. THIS IS TO BE REMOVED in v1.1
// freelist_create
//		description: Create freelist of available memory buffers using push/pop regs.
//
//		inputs:
//			STACK_ID_id		0-7, identify which of 8 hardware push/pop stacks
//			const_base_addr			starting address in sram for the memory region
//			const_stride			number of longwords per entry, including a first longword
//									which hardware uses as the node pointer
//			const_num_entries		number of nodes to be created
//		size:	9 instr
//		example usage:
//			mem_freelist_create(FREELIST0, SRAM_BUFFdescriptor_base, LWCOUNT4, 300);
//
#macro freelist_create[STACK_ID_id, const_base_addr, const_stride, const_num_entries]
.local  buf_addr ii count

	immed32[buf_addr, const_base_addr]					; base address for memory region
	immed[ii, 0]
	immed32[count, const_num_entries]					; num entries

freelist_loop#:

	sram[push, --, buf_addr, ii, STACK_ID_id]		; push to freelist
	alu[ii, ii, +, const_stride]						; add stride
	alu[count, count, -, 1]
	br>0[freelist_loop#]

	immed[$xfer7, 0]
	sram[write, $xfer7, buf_addr, 0, 1], ctx_swap		; end of list is set to 0

.endlocal
#endm

// pre v1.0. THIS IS TO BE REMOVED in v1.1
// malloc
//		description: Get next available address (mem_ptr) from freelist (freelist_id)
//					 using the read transfer reg read_xfer_reg
//
#macro malloc[mem_ptr, freelist_id, read_xfer_reg]
.local tempa tempb
	sram[pop, read_xfer_reg, tempa, 0, freelist_id], ctx_swap

#define_eval FREELIST_IDENT freelist_id
#if (FREELIST/**/FREELIST_IDENT/**/_TYPE == STORAGE_CLASS_SDRAM)
	immed32[tempa, FREELIST/**/FREELIST_IDENT/**/_PTR_BASE]						; get sram ptr base
	alu[tempa, read_xfer_reg, -, tempa]											; calculate pointer offset
	immed32[tempb, FREELIST/**/FREELIST_IDENT/**/_BASE]							; get sdram base
	alu[mem_ptr, tempb, +, tempa, <<FREELIST/**/FREELIST_IDENT/**/_PTR_SHIFT]	; add shifted offset to base
#else
	move[mem_ptr, read_xfer_reg]
#endif
.endlocal
#endm

// pre v1.0. THIS IS TO BE REMOVED in v1.1
// free
//		description: Free an address (mem_ptr) by pushing it onto freelist (freelist_id)
//					 using the write transfer reg write_xfer_reg
//
#macro free[mem_ptr, freelist_id]
.local tempa tempb
#define_eval FREELIST_IDENT freelist_id
#if (FREELIST/**/FREELIST_IDENT/**/_TYPE == STORAGE_CLASS_SDRAM)
	immed32[tempa, FREELIST/**/FREELIST_IDENT/**/_BASE]							; get sdram base
	alu[tempa, mem_ptr, -, tempa]												; subtract base from sdram memptr
	alu[tempa, --, B, tempa, >>FREELIST/**/FREELIST_IDENT/**/_PTR_SHIFT]		; shift offset
	immed32[tempb, FREELIST/**/FREELIST_IDENT/**/_PTR_BASE]						; get sram ptr base
	sram[push, --, tempa, tempb, freelist_id], ctx_swap							; push ptr base + shifted offset
#else
	sram[push, --, mem_ptr, 0, freelist_id], ctx_swap
#endif

.endlocal
#endm


#endif // MEM_UC