// ***********************************************************************************************
// * 
// * 
// * 
// * 
// * 
// * 
// *  firewall.uc
// *  helper macros for the firewall
//
//
// *  system: IXP1200
// *  subsystem: IP microcode
// *  usage: library macros
// *  author: hbhanoo, dec 2000
// * 
// *  revisions:
// ***********************************************************************************************


#ifndef FIREWALL_UC
#define FIREWALL_UC

// *  copyCacheEntry
// * 		description: Just copies a fixed number of words from one location to another				 
// * 		outputs:
// * 			_target					
// * 		inputs:
// * 			_source				(if source = 0, then do nothing!)
// * 		example usage:
// * 			copyCacheEntry[_source, _target]
// * 

#macro copyCacheEntrytoHead[_target, _source]
// * we're allowed to play with xfer registers 5,6,7 only
// * 

alu[--,--,B,_source]
br=0[finish_copyCacheEntry#]

sram[read, $xfer5, _source, 0x0, 3], ctx_swap ; read first 3 words
sram[write, $xfer5, _target, 0x0, 3], ctx_swap ; write first 3 words
sram[read, $xfer5, _source, 0x3, 3], ctx_swap ; read next 3 words
sram[write, $xfer5, _target, 0x3, 3], ctx_swap ; write next 3 words
sram[read, $xfer5, _source, 0x6, 1], ctx_swap ; read second last word
sram[write, $xfer5, _target, 0x6, 1], ctx_swap ; write second last word

; skip the 'previous' pointer - this should still == 0 if we're at the head
finish_copyCacheEntry#:

#endm

// *  freeCacheEntry
// * 		description: Frees an entry from the cache. Preserves the integerity of all 
// * 					 chains in the cache.
// * 					 NOTE: The memory associated with this entry is NOT freed.
// * 						   This task is reserved for the control plane, in the
// * 							event that it needs to perform further actions
// * 							(such as logging the end of a flow).
// * 		outputs:
// * 		inputs:
// * 			_addr
// * 		example usage:
// * 			copyCacheEntry[_source, _target]
// * 
#macro freeCacheEntry[_addr]
// * we're allowed to play with xfer registers 5,6,7 only
// * 
.local previous next ptr
alu[--, --, B, _addr]
br=0[freeCacheEntryDone#]
alu[next, _addr, +, 0x6]
sram[read, $xfer5, next, 0x0, 2], ctx_swap

alu[next, --, B, $xfer5]
alu[previous, --, B, $xfer6]

alu[$xfer6, --, B, next]
alu[$xfer5, --, B, previous]

alu[--,--,B,next] ; if next
br=0[skip_nextptr#] ; ==0 then skip, else:
alu[ptr, next, +, 0x7]
sram[write, $xfer5, ptr, 0x0, 1], ctx_swap
skip_nextptr#:


alu[--,--,B,previous] ; if previous
br=0[freeCacheEntryDone#]         ; ==0 then skip, else:
alu[ptr, previous, +, 0x6]
sram[write, $xfer6, ptr, 0x0, 1], ctx_swap
freeCacheEntryDone#:

.endlocal
#endm


// *  addCacheEntry
// * 		description: Adds a cache entry by grabbing memory from the free pool
// *					 and chaining it on to the appropriate place.
// * 		outputs:
// *			_newEntry			;address of the new entry we added
// * 		inputs:
// *			_thisEntry			;address of current entry
// * 			_chaining			;=1 -> chain on to the end
// *								;=0 -> just replace current entry with fields 
// * 									   (if we are at the first element in the cache)
// * 		example usage:
// * 			addCacheEntry[newAddr, currAddr, 1]
// * 

#macro addCacheEntry[_newEntry, _thisEntry, _chaining]
// * we're allowed to play with xfer registers 5,6,7 only
// * 
.local next_free_ptr addrptr

alu[--, --, B, _chaining]
br=0[new_entry_not_required#], defer[1]
alu[_newEntry, --, B, _thisEntry]
immed[addrptr, 0x2000]

sram[read, $xfer5, addrptr, 0x0, 1], ctx_swap
alu[_newEntry, --, B, $xfer5]				;our entry can be stored at the next free location
alu[next_free_ptr, _newEntry, +, 0x6]			;read in the next free entry 
sram[read, $xfer5, next_free_ptr, 0x0, 1], ctx_swap ;so we can update the freeptr
alu[next_free_ptr, --, B, $xfer5]			;(just make sure that 
alu[$xfer5, --, B, next_free_ptr]	;write bank = read bank)
sram[write, $xfer5, addrptr, 0x0, 1], ctx_swap ;finally update the free pointer for future use
// * 
// * do the fwd/reverse chaining
// * 
alu[addrptr, _thisEntry, +, 0x6]
alu[$xfer5, --, B, _newEntry]; writebank = nextptr
sram[write, $xfer5, addrptr, 0x0, 1], ctx_swap; write the next ptr

alu[addrptr, _newEntry, +, 0x7]
alu[$xfer5, --, B, _thisEntry]; writebank = prevptr
sram[write, $xfer5, addrptr, 0x0, 1], ctx_swap; write the prev ptr

.endlocal // next_free_ptr addrptr

new_entry_not_required#:

.local addrptr outbyte1
alu[addrptr, --, B, _newEntry]; copy the new entry's address
// * 
// * WORD 0:
// *  char bitvector; (1 byte)
// *  char protocol; //(1 byte)
// *  short int counter; //(2 bytes) - packet counter.
// * 
immed[outbyte1, 0x0]
alu_shf[outbyte1, outbyte1, +, 0x1, <<31]; set the 'used' bit
alu_shf[outbyte1, outbyte1, OR, ip_prot, <<16]; copy the protcol
alu[$xfer5, --, B, outbyte1]
sram[write, $xfer5, addrptr, 0x0, 1], ctx_swap
// * 
// * WORDS 1/2:
// *  int src_addr; //(4 bytes)
// *  int dst_addr; //(4 bytes)
// * 
alu[addrptr, addrptr, +, 1]
alu[$xfer5, --, B, ip_sa]
alu[$xfer6, --, B, ip_da]
sram[write, $xfer5, addrptr, 0x0, 2], ctx_swap
// * 
// * WORD 3:
// *  short int src_port; //(2 bytes)
// *  short int dst_port; //(2 bytes)
// * 
alu[addrptr, addrptr, +, 2]
alu_shf[outbyte1, 0x0, +, ip_source_port, <<16]
alu[outbyte1, outbyte1, OR, ip_dest_port]
alu[$xfer5, --, B, outbyte1]
sram[write, $xfer5, addrptr, 0x0, 1], ctx_swap
.endlocal // addrptr outbyte1 outbyte2

#endm





#endif // FIREWALL_UC