// ----------------------------------------------------------------------
// cbswitching - WRR switching at layer 2/3 cbswitching action code file
//version 1.0
//author: Servio Lima Reina
//Carnegie Mellon University
//Last modified: Feb 13 2001  9:32PM
//Description:
//This program implements a layer 2/3 switch that load balance a farm
//of servers according to the WRR algorithm. We assume that the weight 
//for each backend server is the same. 
//The basic purpose of this switch is to do a hash lookup according
//to the src port of the requester (client) for getting the MAC address
//of the destination backend server
//If the packet is a SYN packet,a new hash entry is created, the MAC
//address of the destination host is changed to match the selected
//backend (according to WRR) and the packet is forwarded
//if the packet is ACK, FIN or FIN ACK, a hash lookup is performed, the MAC
//address of the destination host is changed to match the selected backend (according
//to WRR) and the packet is forwarded.
//
//This program implements an incomplete TCP stack, just for handling 
//three way handshake (SYN) and ACK, RST pkts
//There is no consideration for packet fragmentation, checksum errors,
//duplicated packets
//Moreover, the TCP implementation considers only the case when the client
//finishes the connection with a RST packet. Other cases are not taken 
//into account.
//The program is expected to run as the CONTROL PLANE in STRONGARM processor
//part of the PA100 Network processor architecture
//PRESENT WORK
//1. Develop of a light TCP layer with support to some TCP states (not all states supported)
//2. ACE search and matching capabilities fully used
//3. Use of hash tables for fast lookup (port to MAC address matching)
//4. Host stack is not used: all work is done in the STRONGARM  processor
//5. Use of data plane functionality for avoinding to keep TCPSessionHandlers in memory. Basically
// this is a hash table in the ACL associated with a rule in the NCL that does not need to call
//any TCPSessionHandler for handling a packet that belongs to handedoff connection. Therefore less
//memory consumption in the frontend is needed,making it capable of handling a big number of 
//requests/sec
//FUTURE WORK
//1. ARP support, ARP table
//2. IP or TCP options support
//3. IP segmentation and reasembly
//4. Remove assumption that the URL (http://*\n\n) fits in a single ethernet frame's payload
//5. All backends have the same thigh and tload because they have the same hardware architecture:
//this assumption should be relief in future versions.
//(1500bytes-54bytes(headers)= 1446bytes payload) which is almost always the case.
//7. TCP sliding window mechanism
// ----------------------------------------------------------------------

//LIBRARIES
#include "nbaction/NBaction.h"
#include "nbaction/NBbacklog.h"
#include "nbswap.h"
#include "Elt_ethernetset.h"
#include <map>
#include "string.h"
#include "defalloc.h"
#include "stdlib.h"

//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
// G L O B A L    V A R I A B L E S
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------

//CLOCKING DEFINITIONS for PROFILING
//#define DEBUG

#ifdef DEBUG
#define PA100_TIMESTAMP		0x0c000d00
volatile unsigned *timer = (volatile unsigned*)PA100_TIMESTAMP;
unsigned start,stop;

unsigned clockCycles(void);

unsigned clockCycles(void) {
	return *timer;
}
#endif


#define ETHER_FLAG		0x0800

//FRONTEND
#define FRONTEND_IPADDR 0xa000002
#define NUMBER_BACKENDS  4
#define FRONTEND_ETHER1	0x0090D700
#define FRONTEND_ETHER2	0x0513
//ROUTER
#define ROUTER_ETHER1_SRC	0x00a0248d
#define ROUTER_ETHER2_SRC	0x4f3c
//BACKENDS
//10.0.0.19
#define BACKEND_ETHER1	0x000000c0       //everglades  Pentium III 600Mhz  256Mb 
#define BACKEND_ETHER2	0x4f0deed0		 //FreeBSD 4.1 REL 0
//10.0.0.20
#define BACKEND_ETHER1_2	0x000000a0  //timberline  Pentium II 266Mhz 64Mb RAM
#define BACKEND_ETHER2_2	0x2412690d	//FreeBSD 4.1 REL 0
//10.0.0.21
#define BACKEND_ETHER1_3	0x000000a0	//dutt  Pentium II 266 Mhz 128 Mb RAM fxp0
#define BACKEND_ETHER2_3	0xc99de40a
//10.0.0.22
#define BACKEND_ETHER1_4	0x00000010	//vail Pentium II 266 Mhz 128 Mb RAM xl0
#define BACKEND_ETHER2_4	0x4b267f13



//hardcoded list of the ip address of the backends
//would be improved if list is passed through a downcall from the host
uint32 backends[NUMBER_BACKENDS*2]={BACKEND_ETHER1,BACKEND_ETHER2
,
BACKEND_ETHER1_2,BACKEND_ETHER2_2
,
BACKEND_ETHER1_3,BACKEND_ETHER2_3
,BACKEND_ETHER1_4,BACKEND_ETHER2_4};
									


//backend data structure
typedef struct backend{
	nuint16 ether1;
	nuint32 ether2;
}backend;

		
//declarations of main classes
	class Ccbswitching;
	class EthernetHashTable;
	class port_B_target;
	class Backend_server;
	class EventClass;


//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
//				  M A I N      C L A S S E S 
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------


// ----------------------------------------------------------------------
//  Ccbswitching ACE CLASS
//  acts as a multiplexor for incoming packets
//  always running
//  EthernetHashTable is a class part of Ccbswitching ACE
// ----------------------------------------------------------------------
class Ccbswitching : public Ace {
public:
   Ccbswitching(ModuleId id, char* name, Image* obj);  
   ~Ccbswitching();  
	bool lookupsettable(Buffer *buf);
	EthernetHashTable *ethert;
	Backend_server *beserver[NUMBER_BACKENDS];
	Set_ethernetset ethernetset_list;
	Backend_server * select_be_using_wrr(void);
	Target *port_B;
	
};

// ----------------------------------------------------------------------
//EthernetHashTable CLASS
//shared data structure for storing srcport to Backend's pointer mapping
//This class stores a associative  table (map in C++) that relates
//the hash value of <srcport> with the pointer
//to the Backend object related to that value.
//A hash function built in the map class used, allows for fast lookup and
//retrieval of the object needed
//This is a shared resource. It should provide an interface for ensure
//atomic transactions over it.
// ----------------------------------------------------------------------
class EthernetHashTable{
	public:
	//map table url to server mapping
	typedef Backend_server* besp;
	typedef map<uint32,besp> ethernethashtable;

	ethernethashtable::iterator itr;
	ethernethashtable ethertable;
	EthernetHashTable();
	~EthernetHashTable();
};




// ----------------------------------------------------------------------
// Target class for handling packets from/to port B
// ----------------------------------------------------------------------
class port_B_target:public Target{
public:
port_B_target(ModuleId id, Ace* ace, char *name);
~port_B_target();
};



class Backend_server{
public:
	Backend_server(nuint16 eth1, nuint32 eth2);
	~Backend_server();
//data structure for storing backend info
	backend bkend;
	Backend_server* Backend_server_pointer();
};


//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
// C O N S T R U C T O R S    A N D   D E S T R U C T O R S
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------


// ----------------------------------------------------------------------
//Ccbswitching CONSTRUCTOR
//creates referenced objects
// ----------------------------------------------------------------------
Ccbswitching::Ccbswitching(
   ModuleId id, 
   char     *name, 
   Image    *obj ):
	Ace( id, name, obj),
	ethernetset_list(id,this,"ethernetset")
{
		
	//backend server initialization for LARD hash table use
		for(int i=0;i<NUMBER_BACKENDS;i++){
			if((beserver[i] = new Backend_server(backends[(i*2)]&0x0000ffff,backends[(i*2)+1]))==NULL)
			printf("ERROR>>Can not allocate memory for Backend_server[%d]\n",i);
		}

   if((ethert = new EthernetHashTable())==NULL)
	   printf("ERROR>>Can not allocate memory for EthernetHashTable\n");
	
   //target object created
	port_B =new port_B_target(id, this, "port_B");

   printf("ACE (1) ");
   printf("cbswitching ");
   printf("has been successfully constructed\r\n" );
} 


// ----------------------------------------------------------------------
//Ccbswitching DESTRUCTOR
//creates referenced objects
// ----------------------------------------------------------------------
Ccbswitching::~Ccbswitching()
{
	int status;
   // TODO:    Add additional destruction code here
   printf("ACE (1) ");
   printf("cbswitching ");
   printf("is destructed\r\n" );

   // ethernetset_list.clear();
	for (Elt_ethernetset* p = (Elt_ethernetset*) ethernetset_list.first();
	     p != NULL;
		 p = (Elt_ethernetset*) ethernetset_list.next(p))
	{
		delete p;
	}

	for(int i=0;i<NUMBER_BACKENDS;i++)
				delete beserver[i];

	//delete EthernetHashTable();
	delete this->ethert;
	//delete port_B_target();
	delete this->port_B;
	
} 




// ----------------------------------------------------------------------
//EthernetHashTable CONSTRUCTOR
// ----------------------------------------------------------------------
EthernetHashTable::EthernetHashTable(){
;
}

// ----------------------------------------------------------------------
//EthernetHashTable DESTRUCTOR
// ----------------------------------------------------------------------
EthernetHashTable::~EthernetHashTable(){
;
}



// ----------------------------------------------------------------------
//port B target CONSTRUCTOR
// ----------------------------------------------------------------------
port_B_target::port_B_target(ModuleId id, Ace *ace, char *name):
Target(id,ace,name)
{

}


// ----------------------------------------------------------------------
//port B target DESTRUCTOR
// ----------------------------------------------------------------------
port_B_target::~port_B_target(){

}




// ----------------------------------------------------------------------
//Backend_server CONSTRUCTOR
// ----------------------------------------------------------------------
Backend_server::Backend_server(nuint16 eth1, nuint32 eth2){
bkend.ether1 = eth1.raw_;
bkend.ether2 = eth2.raw_;
}


// ----------------------------------------------------------------------
//Backend_server DESTRUCTOR
// ----------------------------------------------------------------------
Backend_server::~Backend_server(){
}

Backend_server* Backend_server::Backend_server_pointer(){
	return this;
}



//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
//M I S C E L A N E O U S      F U N C T I O N S
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------




// ----------------------------------------------------------------------
//Insertion of key values ipsrc, ipdst, srcport, dstport in Set table
// ----------------------------------------------------------------------
bool Ccbswitching::lookupsettable(Buffer *buf){
	Search ethernetsearcher;

	ethernetsearcher = ethernetset_list.locate((nuint32)((buf->b.pkt_[33]&0xffff0000)>>16)); //Search the list
	
	if(ethernetsearcher.hit()){
		return true;
	}
	else{
			Elt_ethernetset * elt;
			if((elt = new Elt_ethernetset((nuint32)((buf->b.pkt_[33]&0xffff0000)>>16)))!=NULL){
				ethernetsearcher.insert(elt);
		//		printf("STATUS>>Ethernet info inserted\n");
				return true;
			}
			else {
				return false;
			}
	
	} //else
}

// ----------------------------------------------------------------------
//where everything begins
// ----------------------------------------------------------------------
INITF init_actions(void* id, char* name, Image* obj)
{ 
	printf("init_actions received\n");
    return new Ccbswitching(id, name, obj);  
} 


// ----------------------------------------------------------------------
//action drop: in case packet is of type UNKNOWN
// ----------------------------------------------------------------------
ACTNF action_drop(Buffer* buf, Ccbswitching* ace){
	return ace->drop(buf);
}



// ----------------------------------------------------------------------
// F R O N T      E N D       A C T I O N S
// ----------------------------------------------------------------------
// ----------------------------------------------------------------------
//Handler of the initial three way handshake sent by the client (SYN packet)
// ----------------------------------------------------------------------
ACTNF action_new_tcpsession_syn(Buffer* buf, Ccbswitching* ace, Search stream)
{
//	printf("STATUS>>SYN rcvd\n");
	#ifdef DEBUG
	start = clockCycles();
	#endif
	static int cter=0;
	int tcpsk = (buf->b.pkt_[31]&0x0000ffff)|((buf->b.pkt_[33]&0xffff0000));		
//	printf("tcpsk=0x%x\n",tcpsk);
	printf("%d\n",++cter);
	ace->lookupsettable(buf);  //insert in stream table
			
	ace->ethert->ethertable[tcpsk]=ace->select_be_using_wrr();
//	printf("STATUS>WRR executed\n");

				buf->b.pkt_[24] = ace->ethert->ethertable[tcpsk]->bkend.ether1.raw_&0x0000ffff;//ether part 1
				buf->b.pkt_[25] = ace->ethert->ethertable[tcpsk]->bkend.ether2.raw_;//ether part 2
				buf->b.pkt_[26] = ROUTER_ETHER1_SRC;
				buf->b.pkt_[27] = (ROUTER_ETHER2_SRC&0x0000ffff)<<16|ETHER_FLAG;

//	printf("STATUS>>PKT has been forwarded\n");

				ace->port_B->take(buf);
#ifdef DEBUG
				stop = clockCycles();
				printf("%d cycl after forwarded SYN pkt to be\n",(stop-start)>>3);
#endif
				return RULE_DONE;

} 




ACTNF action_exist_tcpsession_ack10(Buffer* buf, Ccbswitching* ace, Search stream){
//just forward ack packets that belong to pre existent TCP sessions
//	printf("STATUS>> pkt with flag 0x%x received\n",ntohl((buf->b.pkt_[36]&0xffff0000)>>16));
	#ifdef DEBUG
	start = clockCycles();
	#endif
	int tcpsk;
	if(stream.hit()){
		//build the tcpsessionkey key
		tcpsk =(buf->b.pkt_[31]&0x0000ffff)|((buf->b.pkt_[33]&0xffff0000));		
//printf("tcpsk ack=0x%x\n",tcpsk);
	if((ace->ethert->itr = ace->ethert->ethertable.find(tcpsk))!=ace->ethert->ethertable.end()){

				buf->b.pkt_[24] = ((*ace->ethert->itr).second)->bkend.ether1.raw_&0x0000ffff;//ether part 1
				buf->b.pkt_[25] = ((*ace->ethert->itr).second)->bkend.ether2.raw_;//ether part 2
				buf->b.pkt_[26] = ROUTER_ETHER1_SRC;
				buf->b.pkt_[27] = (ROUTER_ETHER2_SRC&0x0000ffff)<<16|ETHER_FLAG;

				ace->port_B->take(buf);
#ifdef DEBUG
				stop = clockCycles();
				printf("%d cycl after forwarded a pkt to be\n",(stop-start)>>3);
#endif

				return RULE_DONE;
			}
			else {
				ace->drop(buf);
			}
	}
	else
		printf("STATUS>>NO HIT\n");
}



Backend_server * Ccbswitching::select_be_using_wrr(void){
//WRR algorithm
static int backend_turn=0;

	backend_turn = (++backend_turn)%NUMBER_BACKENDS;

		return beserver[backend_turn];


}
