;;; -*- Mode: Lisp; Package: CLIM-UTILS; Base: 10.; Syntax: Common-Lisp; Lowercase: Yes -*-

;; $fiHeader: utilities.lisp,v 1.4 91/03/26 12:03:19 cer Exp $

;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "CLIM-UTILS")

;;;
;;; Various UTILITIES
;;; 

;;;
;;; COMMONLISP Extensions
;;; 

(deftype boolean nil '(member nil t))

(defmacro compile-time-warn (warn-string)
  (warn warn-string)
  nil)

(eval-when (compile load eval)
  (defmacro with-collection (&body body)
    `(let (($with-collection-result$ nil)
	   $with-collection-tail$)
       (macrolet
	   ((collect (form)
	      ;;  The FORM is evaluated first so that COLLECT nests
	      ;; properly, i.e., The test to determine if this is
	      ;; the first value collected should be done after the
	      ;; value itself is generated in case it does
	      ;; collection as well.
	      `(let (($collectable$ ,form))
		 (if $with-collection-tail$
		     (rplacd $with-collection-tail$
			     (setq $with-collection-tail$
				   (list $collectable$)))
		     (setq $with-collection-result$
			   (setq $with-collection-tail$
				 (list $collectable$))))
		 $with-collection-tail$)))
	 ,@body $with-collection-result$))))

(defmacro with-gensyms ((&rest vars) &body body)
  `(let ,(mapcar #'(lambda (symbol)
		     `(,symbol (gensymbol ',symbol)))
		 vars)
     ,@body))

;;; These macros are only used by INTEGERIZE-COORDINATE below.
(defmacro integerize-single-float-coordinate (coord)
  `(the fixnum (values (floor (+ (the single-float ,coord) .5f0)))))

(defmacro integerize-double-float-coordinate (coord)
  `(the fixnum (values (floor (+ (the double-float ,coord) .5d0)))))

(defmacro integerize-float-coordinate (coord)
  `(the fixnum (values (floor (+ (the float ,coord) .5)))))

;; replaces my-round
(defun integerize-coordinate (coord)
  (etypecase coord
    (fixnum coord)
    (single-float
      (integerize-single-float-coordinate coord))
    (double-float
      (integerize-double-float-coordinate coord))
    #-Imach
    (float
      (integerize-float-coordinate coord))
    (ratio
      (values (floor (+ coord 1/2))))
    ;; disallow bignums and other types of numbers
    ))

;;; Two uses
(defmacro multiple-value-second (form)
  `(multiple-value-bind (one two) ,form
     (declare (ignore one))
     two))

;;; Unused in CLIM TWO prototype as of 2/7/92 
(defmacro with-fast-vector-references ((&rest macros-and-arrays) &body body)
  (flet ((simple-part-accessor (array)
	   ;; Allegro only allows SVREF on simple T vectors.
	   #+excl `(let ((temp ,array))
		     (etypecase temp
		       (simple-vector temp)
		       ((vector t) (let ((temp2 (excl::ah_data temp)))
				     (setq temp2 (if (consp temp2)
						     (cdr temp2)
						   temp2))
				     (assert (and 
					       (zerop (the fixnum
							   (excl::ah_displacement temp)))
					       (typep temp2 'simple-vector))
					     ()
					     "Arrays passed to ~S must be non-displaced"
					     'with-fast-vector-references)
				     temp2))))
	   #+Genera array
	   #-(or excl Genera) array)
	 (internal-binding-declarations (variables)
	   #+excl `(declare (simple-vector ,@variables))
	   #+Genera `(declare (sys:array-register ,@variables))
	   #-(or excl Genera) `(declare)))
    (let* ((aref #+(or excl Genera) 'svref
		 #-(or excl Genera) 'aref)
	   (macro-names (mapcar #'first macros-and-arrays))
	   (internal-variables (mapcar #'gensymbol macro-names))
	   (arrays (mapcar #'second macros-and-arrays))
	   (bindings (mapcar #'(lambda (variable array)
				 `(,variable ,(simple-part-accessor array)))
			     internal-variables arrays))
	   (macros (mapcar #'(lambda (macro-name variable)
			       `(,macro-name (index) 
				  `(,',aref ,',variable (the fixnum ,index))))
			   macro-names internal-variables)))
  `(let ,bindings
     ,(internal-binding-declarations internal-variables)
     (progn ,@internal-variables nil)
     (macrolet ,macros
       ,@body)))))


;;;
;;; Generates macros for accessing a slot used as a property list.
;;; Used to define port-prop, frame-manager-prop, pane-prop, etc.
;;

(defmacro def-property-slot-macros (name (class) slot-name)
  (declare (ignore class))
  `(progn
     (defmacro ,name (object key)
       `(getf (slot-value ,object ',',slot-name) ,key))
     (defsetf ,name (object key) (val)
       `(setf 
	 (getf (slot-value ,object ',',slot-name) ,key) 
	 ,val))))

(defmacro def-property-slot-accessors (name (class) slot-name)
  `(progn
     (defmethod ,name ((object ,class) key)
       (getf (slot-value object ',slot-name) key))
     (defmethod (setf ,name) (val (object ,class) key)
       (setf (getf (slot-value object ',slot-name) key) val))))

;;;
;;; Stub Sealers 
;;;

(defun unimplemented (&optional (string "So who you gun-na call..."))
  (error "Unimplemented: ~s." string))

(defmacro define-unimplemented-protocol-method (name protocol-name lambda-list)
  (let ((variables (with-collection
		       (dolist (var lambda-list)
			 (unless (member var lambda-list-keywords)
			   (collect (if (consp var) (car var) var))))))
	(protocol-var (first (find-if #'(lambda (lambda-var)
					  (and (consp lambda-var)
					       (eql (second lambda-var)
						    protocol-name)))
				      lambda-list))))
    `(defmethod ,name ,lambda-list
		#+Genera (declare 
			  (sys:function-parent ,name
					       define-unimplemented-protocol-method))
       (progn ,@variables)		;Ignore these variables...
       (error "The required operation ~S is not defined for the~
	       ~@[ ~S implementation of~] protocol ~S"
	      ',name
	      ,(when protocol-var `(class-name (class-of ,protocol-var)))
	      ',protocol-name))))

(defun warn-obsolete (fn)
  (warn "Call to obsolete function ~S" fn))


;;;
;;; PROCESS & SYNCHRONIZATION
;;;

(defvar *multiprocessing-p* 
  #{
    (or excl Genera Lucid Lispworks Minima) T
    otherwise nil
    }
    )
  
#+excl
(unless (excl::scheduler-running-p)
  (mp:start-scheduler))

(defmacro with-lockf ((place &optional state) &body forms)
  #+(or excl Xerox Genera ccl Minima)
  (declare (ignore state #+ccl place))
  #{
    excl	`(mp:with-process-lock (,place) ,@forms)
    Lucid	`(lcl:with-process-lock (,place ,@(if state (cons state nil)))
		   ,@forms)
    lispworks	`(mp::with-lock (,place) ,@forms)
    Xerox	`(il:with.monitor ,place ,@forms)
    Cloe-Runtime `(progn ,@forms)
    Genera	`(process:with-lock (,place) ,@forms)
    Coral	`(progn ,@forms)
    Minima	`(minima:with-lock (,place) ,@forms)
    }
  )

(defun initial-lock-value (&optional (lock-name "a Silica lock"))
  #-(or Genera Minima) (declare (ignore lock-name))
  #{
    excl	(mp::make-process-lock)
    lispworks	(mp::make-lock)
    Lucid	nil
    Coral	nil
    Xerox	(il:create.monitorlock)
    Cloe-Runtime nil
    Genera	(process:make-lock lock-name)
    Minima	(minima:make-lock lock-name)
   }
  )

;;; A lock that CAN be relocked by the same process.
#-Genera
(defmacro with-simple-recursive-lock ((lock &optional (state "Unlock")) &body forms)
  `(flet ((foo () ,@forms))
     (declare (dynamic-extent #'foo))
     (invoke-with-simple-recursive-lock
       ,lock
       ,state
       #'foo)))

#-Genera
(defun invoke-with-simple-recursive-lock (place state continuation)
  (let ((store-value (current-process))
	(place-value (first place)))
    (if (and place-value (eql place-value store-value))
	(funcall continuation)
	(progn
	  (unless (null place-value)
	    (flet ((waiter ()
		     (null (first place))))
	      (declare (dynamic-extent #'waiter))
	      (process-wait state #'waiter)))
	  (unwind-protect
	      (progn (rplaca place store-value)
		     (funcall continuation))
	    (rplaca place nil))))))

(defmacro with-recursive-lockf ((place &optional state) &body forms)
  #+(or excl Xerox Genera ccl)
  (declare (ignore state #+ccl place))
  #{Genera `(process:with-lock (,place) ,@forms)
    Coral `(progn ,@forms)
    otherwise `(with-simple-recursive-lock (,place ,state) ,@forms)
    }
  )

(defun initial-recursive-lock-value (&optional (lock-name "a recursive Silica lock"))
  #-Genera (declare (ignore lock-name))
  #{coral nil
    Genera (process:make-lock lock-name :recursive T)
    otherwise (cons nil nil)
   }
  )

(defmacro without-scheduling (&body forms)
  "Evaluate the forms w/o letting any other process run."
  #{
    excl       `(mp:without-scheduling ,@forms)
    lispworks  `(sys::without-scheduling ,@forms)
    Lucid      `(lcl:with-scheduling-inhibited ,@forms)
    Xerox      `(progn ,@forms)
    Cloe-Runtime `(progn ,@forms)
    ;; should be process:with-no-other-processes if this is used as
    ;; a global locking mechanism
    Genera     `(scl:without-interrupts ,@forms)
    Coral      `(ccl:without-interrupts ,@forms) ; slh
    Minima     `(minima:with-no-other-processes ,@forms)
   }
   )

(defun make-process (function &key name)
  #+(or ccl) (declare (ignore function name))
  (when *multiprocessing-p*
    #{
    lispworks  (mp:process-run-function name nil function)
    Lucid      (lcl:make-process :function function :name name)
    excl       (mp:process-run-function name function)
    Xerox      (il:add.process (funcall function) 'il:name name)
    Genera     (scl:process-run-function name function)
    Minima     (minima:make-process name :initial-function function)
    otherwise  (warn "No implementation of MAKE-PROCESS for this system.")
    }))

(eval-when (compile load eval)
  (proclaim '(inline processp)))
(defun processp (object)
  #{
  ccl        (member object '(:user :event :interrupt))
  Lucid	     (lcl:processp object)
  excl	     (mp::process-p object)
  lispworks  (mp::process-p object)
  ;; In 7.3 and after it is `(process:process-p ,object)
  Genera     (scheduler-compatibility:process-p object)
  otherwise  (progn (warn "No implementation of PROCESSP for this system.")
		    nil)
  }
  )

(defun destroy-process (p)
  #+(or ccl) (declare (ignore p))
  #{
  Lucid      (lcl:kill-process p)
  excl	     (mp:process-kill p)
  lispworks  (mp:process-kill p)
  Xerox	     (il:del.process p)
  Genera     (scl:process-kill p)
  Coral	     nil
  Minima     (minima:process-kill p)
  otherwise  (warn "No implementation of DESTROY-PROCESS for this system.")
  }
  )

#+coral
(defvar *current-process* :user)

(eval-when (compile load eval)
  (proclaim '(inline current-process)))
(defun current-process ()
  #{
  Lucid      lcl:*current-process*
  excl	     mp:*current-process*
  lispworks  mp:*current-process*
  Xerox	     (il:this.process)
  Genera     scl:*current-process*
  coral	     *current-process*
  Minima     (minima:current-process)
  Cloe-Runtime nil
  }
  )

(eval-when (compile load eval)
  (proclaim '(inline all-processes)))
(defun all-processes ()
  #{
  Lucid      lcl:*all-processes*
  excl	     mp:*all-processes*
  lispworks  (mp::list-all-processes)
  Genera     sys:all-processes
  Coral	     (adjoin *current-process* '(:user))
  Cloe-Runtime nil
  }
  )

(defun show-processes ()
  #{
       Lucid	  (lcl::show-processes)
       Genera	  (si:com-show-processes)
       otherwise  (all-processes)
  }
  )
  
(eval-when (compile load eval)
  (proclaim '(inline process-yield)))
(defun process-yield ()
  #{
  Lucid      (lcl:process-allow-schedule)
  excl	     (mp:process-allow-schedule)
  lispworks  (mp::process-allow-scheduling)
  Xerox	     (il:block)
  Genera     (scl:process-allow-schedule)
  Coral	     (ccl:event-dispatch)
  Cloe-Runtime nil
  }
  )

(defun process-wait (wait-reason predicate)
  "Cause the current process to go to sleep until the predicate returns TRUE."
  #{
  Lucid      (lcl:process-wait wait-reason predicate)
  excl	     (mp:process-wait wait-reason predicate)
  lispworks  (mp:process-wait wait-reason predicate)
  Xerox	     (let ((il:*who-line-state* wait-reason))
	       (loop
		 (il:block)
		 (when (and (funcall predicate))
		   (return))))
  Genera     (scl:process-wait wait-reason predicate)
  Coral	     (ccl::process-wait wait-reason predicate)
  Cloe-Runtime nil
  otherwise  (progn (compile-time-warn "Need an implementation for PROCESS-WAIT")
		    (error "~S doesn't have a definition.  Args are ~S ~S"
			   'process-wait wait-reason predicate))
  }
  )

(defun process-wait-with-timeout (wait-reason timeout predicate)
  "Cause the current process to go to sleep until the predicate returns TRUE or
   timeout seconds have gone by." 
  (when (null timeout)
    ;; ensure genera semantics, timeout = NIL means indefinite timeout
    (return-from process-wait-with-timeout
      (process-wait wait-reason predicate)))
  #{
  excl	     (mp:process-wait-with-timeout wait-reason timeout predicate)
  lispworks  (mp:process-wait-with-timeout wait-reason timeout predicate)
  Lucid	     (lcl:process-wait-with-timeout wait-reason timeout predicate)
  Genera     (sys:process-wait-with-timeout wait-reason (* timeout 60.) predicate)
  Coral	     (ccl::process-wait-with-timeout wait-reason timeout predicate)
  otherwise  (progn (compile-time-warn "Need an implementation for process-wait-with-timeout")
		    (error "~S doesn't have a definition.  Args are ~S ~S ~S"
			   'process-wait-with-timeout timeout wait-reason predicate))
  }
  )

(defun process-interrupt (process closure)
  (declare #+Coral (ignore process))
  #{
  lucid     (lcl:interrupt-process process closure)
  excl	    (mp:process-interrupt process closure)
  ;; ---    Is Lispworks' the same as Allegro?
  ;; ---    It is for everything else except ALL-PROCESSES.
  Genera    (scl:process-interrupt process closure)
  Coral     (let ((*current-process* :interrupt))
	      (funcall closure))
  otherwise (progn
	      (compile-time-warn "Need an implementation for process-interrupt")
	      (error "~S doesn't have a definition.  Args are ~S ~S"
		     'process-interrupt process closure))
  }
  )
