;;; -*- Mode: LISP; Syntax: Common-lisp; Package: PCL; Base: 10; Patch-File: T -*-

;;; A potpourri of fixes to PCL for CLIM.  Some of these have been
;;; patched into the official PCL from Parc.

;;; =================================================================

;;; Portable changes:
;;;
;;; Allow changing the number of arguments on generic functions with no
;;; methods.  We should probably allow this even if there are methods, by
;;; removing all the non-conforming methods after a CERROR, but that's not
;;; what it currently does.
;;;
;;; Fix a bug in the calling of NO-APPLICABLE-METHOD everywhere.  It used
;;; to be FUNCALLed with the generic function and its arguments; now it is
;;; APPLied to the GF and arguments.
;;;
;;; Use CERROR in the default method for NO-APPLICABLE-METHOD; the
;;; continuation retries the application of the generic function to the
;;; arguments.  [It would be useful to allow a way of continuing whereby
;;; you could specify different arguments, but that requires the portable
;;; error system.]
;;;
;;; Fix a number of bugs in method expansion.  CALL-NEXT-METHOD is a macro
;;; in cases where nobody tries to make a closure out of it, which makes
;;; it a lot faster in some cases.  IWBNI we could declare it to be a
;;; DYNAMIC-EXTENT closure when that would work, but that would require
;;; more flow analysis of the method body than we are willing to perform.
;;; Also fix a bug in which (DECLARE (IGNORE ARGUMENT)) for
;;; non-specialized arguments was flagged as incorrect even if the
;;; argument was not lexically referenced, because the PCL expansion of
;;; the method would reference the variable.
;;;
;;; Make DEFCONSTRUCTOR generate `(PROCLAIM '(FUNCTION ,constructor-name))
;;; instead of `(DEFUN ,constructor-name ,args (ERROR ...)); this prevents
;;; being warned about defining the function twice in the same file.

;;; =================================================================

;;; Genera-only changes:
;;;
;;; Added INVISIBLE-FRAME declarations to a number of functions
;;;
;;; Made *PCL-SYSTEM-DATE* appear in Genera bug reports.
;;;
;;; Made FDEFINE and FUNDEFINE work on PCL method function-specs.  Makes
;;; m-. work most of the time, as well as m-X Kill Definition.
;;;
;;; Made SETFBOUNDP work for Genera (also for Allegro).
;;;
;;; Make the walker know how to handle Genera's extension to PROG.
;;;

;;; =================================================================

#+Ignore
(SYSTEM-INTERNALS:FILES-PATCHED-IN-THIS-PATCH-FILE 
  "PCL:MAY-DAY-PCL;FIN.LISP.2"
  "PCL:MAY-DAY-PCL;METHODS.LISP.2"
  "PCL:MAY-DAY-PCL;DFUN.LISP.2"
  "PCL:MAY-DAY-PCL;DEFSYS.LISP.2"
  "PCL:MAY-DAY-PCL;GENERA-LOW.LISP.2"
  "PCL:MAY-DAY-PCL;BOOT.LISP.2"
  "PCL:MAY-DAY-PCL;DEFS.LISP.2"
  "PCL:MAY-DAY-PCL;DEFCLASS.LISP.2"
  "PCL:MAY-DAY-PCL;CONSTRUCT.LISP.2")

(in-package "PCL")

#+Genera
(pushnew ':pcl-internals dbg:*all-invisible-frame-types*)

;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "PCL:MAY-DAY-PCL;FIN.LISP.2")
#+Genera (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
	   "-*-Mode:LISP; Package: PCL; Base:10; Syntax:Common-lisp -*-")
#+Genera
;;;
;;; The inner closure of this function will have its code vector replaced
;;;  by a hand-coded fast jump to the function that is stored in the 
;;;  captured-lexical variable.  In effect, that code is a hand-
;;;  optimized version of the code for this inner closure function.
;;;
(defun make-trampoline (function)
  (declare (optimize (speed 3) (safety 0)))
  #'(lambda (&rest args)
      #+Genera (declare (dbg:invisible-frame :pcl-internals))
      (apply function args)))


;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "PCL:MAY-DAY-PCL;METHODS.LISP.2")
#+Genera (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
	   "-*-Mode:LISP; Package: PCL; Base:10; Syntax:Common-lisp -*-")


(defun add-arg-info (generic-function method arg-info)
  (flet ((lose (string &rest args)
	   (error "Attempt to add the method ~S to the generic function ~S.~%~
                   But ~A"
		  method
		  generic-function
		  (apply #'format nil string args)))
	 (compare (x y)
	   (if (> x y) "more" "fewer")))
    (multiple-value-bind (nreq nopt keysp restp allow-other-keys-p keywords)
	(analyze-lambda-list (method-lambda-list method))
      (let ((gf-nreq (arg-info-number-required arg-info))
	    (gf-nopt (arg-info-number-optional arg-info))
	    (gf-key/rest-p (arg-info-key/rest-p arg-info))
	    (gf-keywords (arg-info-keywords arg-info)))

	;; Don't do any error checking if we're replacing the only method for this GF
	(when (null (generic-function-methods generic-function))
	  (return-from add-arg-info (new-arg-info-from-method method)))

	(unless (= nreq gf-nreq)
	  (lose "the method has ~A required arguments than the generic function."
		(compare nreq gf-nreq)))
	(unless (= nopt gf-nopt)
	  (lose "the method has ~A optional arguments than the generic function."
		(compare nopt gf-nopt)))
	(unless (eq (or keysp restp) gf-key/rest-p)
	  (error "the method and generic function differ in whether they accept~%~
                  rest or keyword arguments."))
	(when gf-keywords
	  (unless (or (and restp (not keysp))
		      allow-other-keys-p
		      (every #'(lambda (k) (memq k keywords)) gf-keywords))
	    (error
	      "the generic function requires each method to accept the keyword arguments~%~
               ~S.  The method does not all of accept these."
	      gf-keywords)))

	(make-arg-info (arg-info-precedence arg-info)
		       (mapcar #'raise-metatype (arg-info-metatypes arg-info)
			       (method-specializers method))
		       gf-nopt
		       gf-key/rest-p
		       gf-keywords)))))


(defmacro protect-cache-miss-code (gf args &body body)
  (let ((function (gensym)) (appl (gensym)))
    (once-only (gf args)
      `(if (memq ,gf *invalid-dfuns-on-stack*)
	   (multiple-value-bind (,function ,appl)
	       (get-secondary-dispatch-function ,gf ,args)
	     (if (null ,appl)
		 (apply #'no-applicable-method ,gf ,args)
		 (apply ,function ,args)))
	   (let ((*invalid-dfuns-on-stack* (cons ,gf *invalid-dfuns-on-stack*)))
	     ,@body)))))

(defmethod no-applicable-method (generic-function &rest args)
  (cerror "Retry call to ~S"
	  "No matching method for the generic-function ~S,~@
          when called with arguments ~S."
	  generic-function args)
  (let ((*invalid-dfuns-on-stack* (remove generic-function *invalid-dfuns-on-stack*)))
    (invalidate-discriminating-function generic-function)
    (apply generic-function args)))


;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "PCL:MAY-DAY-PCL;DFUN.LISP.2")
#+Genera (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
	   "-*- Mode:LISP; Package:PCL; Base:10; Syntax:Common-Lisp -*-")


(defun make-initial-dfun (generic-function)
  #'(lambda (&rest args)
      #+Genera (declare (dbg:invisible-frame :pcl-internals))
      (initial-dfun args generic-function)))

(defun invalidate-dfun-internal (generic-function)
  ;;
  ;; Set the funcallable instance function to something that just calls
  ;; invalid-dfun, that is, arrange to use lazy evaluation to update the
  ;; dfun later.
  ;; 
  (set-funcallable-instance-function
    generic-function
    #'(lambda (&rest args)
	#+Genera (declare (dbg:invisible-frame :pcl-internals))
	(invalid-dfun generic-function args)))
  ;;
  ;; Except that during bootstrapping, we would like to update the dfun
  ;; right away, and this arranges for that.
  ;;
  (when *invalidate-discriminating-function-force-p*    
    (let ((*invalid-dfuns-on-stack*
	    (cons generic-function *invalid-dfuns-on-stack*)))
      (set-funcallable-instance-function
	generic-function
	(compute-discriminating-function generic-function)))))

(defun invalid-dfun (gf args)
  #+Genera (declare (dbg:invisible-frame :pcl-internals))
  (protect-cache-miss-code gf args
    (let ((new-dfun (compute-discriminating-function gf)))
      (set-funcallable-instance-function gf new-dfun)
      (apply gf args))))


(defun accessor-miss (gf ostate otype new object oindex ow0 ow1 field cache)
  (declare (ignore ow1))
  (let ((args (ecase otype			;The congruence rules assure
		(reader (list object))		;us that this is safe despite
		(writer (list new object)))))	;not knowing the new type yet.
    
    (protect-cache-miss-code gf
			     args
      (multiple-value-bind (wrappers invalidp nfunction applicable)
	  (cache-miss-values gf args)
	(multiple-value-bind (ntype nindex)
	    (accessor-miss-values gf applicable args)
	  ;;
	  ;; The following lexical functions change the state of the
	  ;; dfun to that which is their name.  They accept arguments
	  ;; which are the parameters of the new state, and get other
	  ;; information from the lexical variables bound above.
	  ;; 
	  (flet ((two-class (index w0 w1)
		   (when (zerop (random 2)) (psetf w0 w1 w1 w0))
		   (ecase ntype
		     (reader (update-to-two-class-readers-dfun gf w0 w1 index))
		     (writer (update-to-two-class-writers-dfun gf w0 w1 index))
		     ))
		 (one-index (index &optional field cache)
		   (ecase ntype
		     (reader
		       (update-to-one-index-readers-dfun gf index field cache))
		     (writer
		       (update-to-one-index-writers-dfun gf index field cache))
		     ))
		 (n-n (&optional field cache)
		   (ecase ntype
		     (reader (update-to-n-n-readers-dfun gf field cache))
		     (writer (update-to-n-n-writers-dfun gf field cache))))
		 (checking ()
		   (update-to-checking-dfun gf nfunction))
		 ;;
		 ;;
		 ;;		 
		 (do-fill (valuep limit-fn update-fn)
		   (multiple-value-bind (nfield ncache)
		       (fill-cache field cache
				   1 valuep
				   limit-fn wrappers nindex)
		     (unless (and (= nfield field)
				  (eq ncache cache))
		       (funcall update-fn nfield ncache)))))

	    (cond ((null nfunction)
                   (apply #'no-applicable-method gf args))
		  ((null ntype)
		   (checking)
		   (apply nfunction args))
                  ((or invalidp
                       (null nindex))
                   (apply nfunction args))
		  ((not (or (std-instance-p object)
			    (fsc-instance-p object)))
		   (checking)
		   (apply nfunction args))
		  ((neq ntype otype)
		   (checking)
		   (apply nfunction args))
		  (t
		   (ecase ostate
		     (one-class
		       (if (eql nindex oindex)
			   (two-class nindex ow0 wrappers)
			   (n-n)))
		     (two-class
		       (if (eql nindex oindex)
			   (one-index nindex)
			   (n-n)))
		     (one-index
		       (if (eql nindex oindex)
			   (do-fill nil
				    #'one-index-limit-fn
				    #'(lambda (nfield ncache)
					(one-index nindex nfield ncache)))
			   (n-n)))
		     (n-n
		       (unless (consp nindex)
			 (do-fill t
				  #'n-n-accessors-limit-fn
				  #'n-n))))
		   (apply nfunction args)))))))))

(defun checking-miss (generic-function args ofunction field cache)
  (protect-cache-miss-code generic-function
			   args
    (let* ((arg-info (gf-arg-info generic-function))
	   (nkeys (arg-info-nkeys arg-info)))
      (multiple-value-bind (wrappers invalidp nfunction)
	  (cache-miss-values generic-function args)
	(cond (invalidp
	       (apply nfunction args))
	      ((null nfunction)
	       (apply #'no-applicable-method generic-function args))
	      ((eq ofunction nfunction)
	       (multiple-value-bind (nfield ncache)
		   (fill-cache field cache nkeys nil #'checking-limit-fn wrappers nil)
		 (unless (and (= nfield field)
			      (eq ncache cache))
		   (update-to-checking-dfun generic-function
					    nfunction nfield ncache)))
	       (apply nfunction args))
	      (t
	       (update-to-caching-dfun generic-function)
	       (apply nfunction args)))))))

(defun caching-miss (generic-function args ofield ocache)
  (protect-cache-miss-code generic-function
			   args
    (let* ((arg-info (gf-arg-info generic-function))
	   (nkeys (arg-info-nkeys arg-info)))
      (multiple-value-bind (wrappers invalidp function)
	  (cache-miss-values generic-function args)
	(cond (invalidp
	       (apply function args))
	      ((null function)
	       (apply #'no-applicable-method generic-function args))
	      (t
	       (multiple-value-bind (nfield ncache)
		   (fill-cache ofield ocache nkeys t #'caching-limit-fn wrappers function)
		 (unless (and (= nfield ofield)
			      (eq ncache ocache))
		   (update-to-caching-dfun generic-function nfield ncache)))
	       (apply function args)))))))


;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "PCL:MAY-DAY-PCL;DFUN.LISP.2")
#+Genera (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
	   "-*- Mode:LISP; Package:PCL; Base:10; Syntax:Common-Lisp -*-")


;;;
;;; The dynamically adaptive method lookup algorithm is implemented is
;;; implemented as a kind of state machine.  The kinds of discriminating
;;; function is the state, the various kinds of reasons for a cache miss
;;; are the state transitions.
;;;
;;; The code which implements the transitions is all in the miss handlers
;;; for each kind of dfun.  Those appear here.
;;;
;;; Note that within the states that cache, there are dfun updates which
;;; simply select a new cache or cache field.  Those are not considered
;;; as state transitions.
;;; 
(defun initial-dfun (args generic-function)
  #+Genera (declare (dbg:invisible-frame :pcl-internals))
  (protect-cache-miss-code generic-function
			   args
    (multiple-value-bind (wrappers invalidp nfunction applicable)
	(cache-miss-values generic-function args)
      (multiple-value-bind (ntype nindex)
	  (accessor-miss-values generic-function applicable args)
	(cond ((null applicable)
	       (apply #'no-applicable-method generic-function args))
	      (invalidp
	       (apply nfunction args))
	      ((and ntype nindex)
	       (ecase ntype
		 (reader (update-to-one-class-readers-dfun generic-function wrappers nindex))
		 (writer (update-to-one-class-writers-dfun generic-function wrappers nindex)))
	       (apply nfunction args))
	      (ntype
	       (apply nfunction args))
	      (t
	       (update-to-checking-dfun generic-function nfunction)
	       (apply nfunction args)))))))

;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "PCL:MAY-DAY-PCL;DEFSYS.LISP.2")
#+Genera (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
	   "-*-Mode:LISP; Package:(PCL LISP 1000); Base:10; Syntax:Common-lisp -*-")

#+Genera
;;; Make sure Genera bug mail contains the PCL bug data.  A little
;;; kludgy, but what the heck.  If they didn't mean for people to do
;;; this, they wouldn't have made private patch notes be flavored
;;; objects, right?  Right.
(progn
  (scl:defflavor pcl-private-patch-info ((description)) ())
  (scl:defmethod (sct::private-patch-info-description pcl-private-patch-info) ()
    (or description
	(setf description (string-append "PCL version: " pcl:*pcl-system-date*))))
  (scl:defmethod (sct::private-patch-info-pathname pcl-private-patch-info) ()
    *pcl-directory*)
  (unless (find-if #'(lambda (x) (typep x 'pcl-private-patch-info)) sct::*private-patch-info*)
    (push (scl:make-instance 'pcl-private-patch-info) sct::*private-patch-info*)))


;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "PCL:MAY-DAY-PCL;GENERA-LOW.LISP.2")
#+Genera (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
	   "-*- Mode:LISP; Package:(PCL Lisp 1000); Base:10.; Syntax:Common-lisp; Patch-File: Yes -*-")

#+Genera
(si:define-function-spec-handler method (op spec &optional arg1 arg2)
  (if (eq op 'sys:validate-function-spec)
      (and (let ((gspec (cadr spec)))
	     (or (symbolp gspec)
		 (and (listp gspec)
		      (eq (car gspec) 'setf)
		      (symbolp (cadr gspec))
		      (null (cddr gspec)))))
	   (let ((tail (cddr spec)))
	     (loop (cond ((null tail) (return nil))
			 ((listp (car tail)) (return t))
			 ((atom (pop tail)))			 
			 (t (return nil))))))
      (let ((table *method-htable*)
	    (key spec))
	(case op
	  ((si:fdefinedp si:fdefinition)
	   (car (gethash key table nil)))
	  (si:fundefine
	    (let* ((generic-function-name (second spec))
		   (generic-function (and (gboundp generic-function-name)
					  (gdefinition generic-function-name)))
		   (method (and generic-function
				(find spec (generic-function-methods generic-function)
				      :test #'equal
				      :key #'(lambda (method)
					       (let ((function (method-function method)))
						 (and function
						      (sys:function-name function))))))))
	      (when method
		(remove-method generic-function method)))
	    (remhash key table))
	  (si:fdefine
	    (let ((old (gethash key table nil))
;		  (gspec (cadr spec))
		  (quals nil)
		  (specs nil)
		  (ptr (cddr spec)))
	      (setq specs
		    (loop (cond ((null ptr) (return nil))
				((listp (car ptr)) (return (car ptr)))
				(t (push (pop ptr) quals)))))
;	      (pcl-fdefine-helper gspec (nreverse quals) specs arg1)
	      (setf (gethash key table) (cons arg1 (cdr old)))))
	  (si:get
	    (let ((old (gethash key table nil)))
	      (getf (cdr old) arg1)))
	  (si:plist
	    (let ((old (gethash key table nil)))
	      (cdr old)))
	  (si:putprop
	    (let ((old (gethash key table nil)))
	      (unless old
		(setf old (cons nil nil))
		(setf (gethash key table) old))
	      (setf (getf (cdr old) arg2) arg1)))
	  (si:remprop
	    (let ((old (gethash key table nil)))
	      (when old
		(remf (cdr old) arg1))))
	  (otherwise
	    (si:function-spec-default-handler op spec arg1 arg2))))))


;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "PCL:MAY-DAY-PCL;BOOT.LISP.2")
#+Genera (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
	   "-*-Mode: LISP; Package:(PCL LISP 1000); Base:10; Syntax:Common-lisp -*-")


#+Genera
(defun expand-defmethod (proto-method name qualifiers lambda-list body env)
  (when (listp name) (do-standard-defsetf-1 (cadr name)))
  (multiple-value-bind (fn-form specializers doc plist)
      (expand-defmethod-internal name qualifiers lambda-list body env)
    (let ((fn-args (cadadr fn-form))
	  (fn-body (cddadr fn-form))
	  (method-name `(method ,name ,@qualifiers ,specializers)))
      `(progn
	 (proclaim '(function ,name))
	 (defun ,method-name ,fn-args
	   ,@fn-body)
	 (load-defmethod
	   ',(if proto-method
		 (class-name (class-of proto-method))
		 'standard-method)
	   ',name
	   ',qualifiers
	   (list ,@(mapcar #'(lambda (specializer)
			       (if (and (consp specializer)
					(eq (car specializer) 'eql))
				   ``(eql ,,(cadr specializer))
				   `',specializer))
			   specializers))
	   ',(specialized-lambda-list-lambda-list lambda-list)
	   ',doc
	   ',(getf plist :isl-cache-symbol)	;Paper over a bug in KCL by
						;passing the cache-symbol
						;here in addition to in the
						;plist.
	   ',plist
	   #',method-name)))))

(defun add-lexical-functions-to-method-lambda (walked-declarations
					       walked-lambda-body
					       walked-lambda
					       original-args
					       lambda-list
					       save-original-args
					       applyp
					       aux-bindings
					       call-next-method-p
					       next-method-p-p
					       closurep)
  (cond ((and (null closurep)
	      (null applyp)
	      (null save-original-args))
	 ;; OK to use MACROLET, CALL-NEXT-METHOD is always passed some args, and
	 ;; all args are mandatory (else APPLYP would be true).
	 `(lambda ,lambda-list
	    ,@walked-declarations
	    (let ((.next-method. (car *next-methods*))
		  (.next-methods. (cdr *next-methods*)))
	      (macrolet ((call-next-method ,lambda-list
			   '(if .next-method.
				(let ((*next-methods* .next-methods.))
				  (funcall .next-method. ,@lambda-list))
				(error "No next method.")))
			 (next-method-p () `(not (null .next-method.))))
		,@walked-lambda-body))))
	((and (null closurep)
	      (null applyp)
	      save-original-args)
	 ;; OK to use MACROLET.  CALL-NEXT-METHOD is sometimes called in the
	 ;; body with zero args, so we have to save the original args.
	 (if save-original-args
	     ;; CALL-NEXT-METHOD is sometimes called with no args
	     `(lambda ,original-args
		(let ((.next-method. (car *next-methods*))
		      (.next-methods. (cdr *next-methods*)))
		  (macrolet ((call-next-method (&rest cnm-args)
			       `(if .next-method.
				    (let ((*next-methods* .next-methods.))
				      (funcall .next-method.
					       ,@(if cnm-args cnm-args ',original-args)))
				    (error "No next method.")))
			     (next-method-p () `(not (null .next-method.))))
		    (let* (,@(mapcar #'list lambda-list original-args)
			     ,@aux-bindings)
		      ,@walked-declarations
		      ,@walked-lambda-body))))))
	((and (null closurep) applyp save-original-args)
	 ;; OK to use MACROLET, since we never make a closure of
	 ;; CALL-NEXT-METHOD or NEXT-METHOD-P.  However, we do have to
	 ;; APPLY both the method body and the next method to the
	 ;; "original" arguments...
	 `(lambda ,original-args
	    (let ((.next-method. (car *next-methods*))
		  (.next-methods. (cdr *next-methods*)))
	      (macrolet ((call-next-method (&rest cnm-args)
			   `(if .next-method.
				(let ((*next-methods* .next-methods.))
				  (apply .next-method.
					   ,@(if cnm-args cnm-args
						 ',(remove '&rest original-args))))
				(error "No next method.")))
			 (next-method-p () `(not (null .next-method.))))
		(apply (function ,walked-lambda)
		       ,@(remove '&rest original-args))))))
				  
	((and (null save-original-args) (null applyp) (null closurep))
	 ;;
	 ;; more optimizations from the wimp
	 ;;
	 `(lambda ,lambda-list ,@walked-declarations
		  (let ((.next-method. (car *next-methods*))
			(.next-methods. (cdr *next-methods*)))
		    (macrolet ((call-next-method (&rest cnm-args)
				 `(if .next-method.
				      (let ((*next-methods* .next-methods.))
					(apply .next-method. ,cnm-args))
				      (error "No next method.")))
			       (next-method-p ()
				 `(not (null .next-method.))))
		      ,@walked-lambda-body))))
	
	((and (null save-original-args)
	      (null applyp))
	 ;;
	 ;; We don't have to save the original arguments.  In addition,
	 ;; this method doesn't take any &mumble arguments (this means
	 ;; that there is no way the lexical functions can be used inside
	 ;; of the default value form for an &mumble argument).
	 ;;
	 ;; We can expand this into a simple lambda expression with an
	 ;; FLET to define the lexical functions.
	 ;; 
	 `(lambda ,lambda-list
	    ,@walked-declarations
	    (let ((.next-method. (car *next-methods*))
		  (.next-methods. (cdr *next-methods*)))
	      (flet (,@(and call-next-method-p
			    '((call-next-method (&rest cnm-args)
				#+Genera
				(declare (dbg:invisible-frame :pcl-internals))
				(if .next-method.
				    (let ((*next-methods* .next-methods.))
				      (apply .next-method. cnm-args))
				    (error "No next method.")))))
		     ,@(and next-method-p-p
			    '((next-method-p ()
				(not (null .next-method.))))))
		,@walked-lambda-body))))
	
	((and (null applyp) (null closurep))
	 ;; This is the same as the case below, except we can MACROLET
	 ;; the CALL-NEXT-METHOD function(s).
	 `(lambda ,original-args
	    (let ((.next-method. (car *next-methods*))
		  (.next-methods. (cdr *next-methods*)))
	      (macrolet ((call-next-method (&rest cnm-args)
			   `(if .next-method.
				(let ((*next-methods* .next-methods.))
				  ,(if cnm-args
				       `(apply .next-method. ,@cnm-args)
				       `(funcall .next-method.
						 ,@original-args)))
				(error "No next method.")))
			 (next-method-p ()
			   `(not (null .next-method.))))
		(let* (,@(mapcar #'list
				 (remtail lambda-list (memq '&aux lambda-list))
				 original-args)
		       ,@aux-bindings)
		  ,@walked-declarations
		  ,@walked-lambda-body)))))
	
	((null applyp)
	 ;;
	 ;; This method doesn't accept any &mumble arguments.  But we
	 ;; do have to save the original arguments (this is because
	 ;; call-next-method is being called with no arguments).
	 ;; Have to be careful though, there may be multiple calls to
	 ;; call-next-method, all we know is that at least one of them
	 ;; is with no arguments.
	 ;; 
	 `(lambda ,original-args
	    (let ((.next-method. (car *next-methods*))
		  (.next-methods. (cdr *next-methods*)))
	      (flet (,@(and call-next-method-p
                            `((call-next-method (&rest cnm-args)
				#+Genera (declare (dbg:invisible-frame :pcl-internals))
				(if .next-method.
				    (let ((*next-methods* .next-methods.))
				      (if cnm-args
					  (apply .next-method. cnm-args)
					  (funcall .next-method.
						   ,@original-args)))
				    (error "No next method.")))))
		     ,@(and next-method-p-p
			    '((next-method-p ()
				(not (null .next-method.))))))
		(let* (,@(mapcar #'list
				 (remtail lambda-list (memq '&aux lambda-list))
				 original-args)
		       ,@aux-bindings)
		  ,@walked-declarations
		  ,@walked-lambda-body)))))
	(t
	 ;;
	 ;; This is the fully general case.
	 ;; We must allow for the lexical functions being used inside
	 ;; the default value forms of &mumble arguments, and if must
	 ;; allow for call-next-method being called with no arguments.
	 ;; 
	 `(lambda ,original-args
	    (let ((.next-method. (car *next-methods*))
		  (.next-methods. (cdr *next-methods*)))
	      (flet (,@(and call-next-method-p
			    `((call-next-method (&rest cnm-args)
				#+Genera (declare (dbg:invisible-frame :pcl-internals))
				(if .next-method.
				    (let ((*next-methods* .next-methods.))
				      (if cnm-args
					  (apply .next-method. cnm-args)
					  (apply .next-method. 
						 ,@(remove '&rest
							   original-args))))
				    (error "No next method.")))))
		     ,@(and next-method-p-p
			    '((next-method-p ()
				(not (null .next-method.))))))
		(apply (function ,walked-lambda)
		       ,@(remove '&rest original-args))))))))

(defun expand-defmethod-internal
       (generic-function-name qualifiers specialized-lambda-list body env)
  (declare (values fn-form specializers doc)
	   (ignore qualifiers))
  (when (listp generic-function-name)
    (do-standard-defsetf-1 (cadr generic-function-name)))
  (multiple-value-bind (documentation declarations real-body)
      (extract-declarations body)
    (multiple-value-bind (parameters lambda-list specializers)
	(parse-specialized-lambda-list specialized-lambda-list)

      
      (let* ((required-parameters
	       (mapcar #'(lambda (r s) (declare (ignore s)) r)
		       parameters
		       specializers))
	     (parameters-to-reference
	       (make-parameter-references specialized-lambda-list
					  required-parameters
					  declarations
					  generic-function-name
					  specializers))
	     (class-declarations 
	       `(declare
		  ,@(remove nil
			    (mapcar #'(lambda (a s) (and (symbolp s)
							 (neq s 't)
							 `(class ,a ,s)))
				    parameters
				    specializers))))
	     (method-lambda 
	       ;; Remove the documentation string and insert the
	       ;; appropriate class declarations.  The documentation
	       ;; string is removed to make it easy for us to insert
	       ;; new declarations later, they will just go after the
	       ;; cadr of the method lambda.  The class declarations
	       ;; are inserted to communicate the class of the method's
	       ;; arguments to the code walk.
	       (let ()
		 `(lambda ,lambda-list
		    ,class-declarations
		    ,@declarations
		    (progn ,@parameters-to-reference)
		    (block ,(if (listp generic-function-name)
				(cadr generic-function-name)
				generic-function-name)
		      ,@real-body))))

	     (call-next-method-p nil)   ;flag indicating that call-next-method
	                                ;should be in the method definition
 	     (closurep nil)		;flag indicating that #'call-next-method
 					;was seen in the body of a method
	     (next-method-p-p nil)      ;flag indicating that next-method-p
                                        ;should be in the method definition
	     (save-original-args nil)   ;flag indicating whether or not the
				        ;original arguments to the method
					;must be preserved.  This happens
					;for two reasons:
	                                ; - the method takes &mumble args,
					;   so one of the lexical functions
					;   might be used in a default value
	                                ;   form
					; - call-next-method is used without
					;   arguments at least once in the
					;   body of the method
	     (original-args ())
	     (applyp nil)		;flag indicating whether or not the
					;method takes &mumble arguments. If
					;it does, it means call-next-method
					;without arguments must be APPLY'd
					;to original-args.  If this gets set
					;true, save-original-args is set so
					;as well
	     (aux-bindings ())		;Suffice to say that &aux is one of
					;damndest things to have put in a
					;language.
	     (slots (mapcar #'list required-parameters))
	     (plist ())
	     (walked-lambda nil))
	(flet ((walk-function (form context env)
		 (cond ((not (eq context ':eval)) form)
		       ((not (listp form)) form)
		       ((eq (car form) 'call-next-method)
			(setq call-next-method-p 't)
 			(unless (cdr form)
 			  (setq save-original-args t))
			form)
		       ((eq (car form) 'next-method-p)
			(setq next-method-p-p 't)
			form)
		       ((and (eq (car form) 'function)
			     (cond ((eq (cadr form) 'call-next-method)
				    (setq call-next-method-p 't)
				    (setq save-original-args 't)
				    (setq closurep t)
				    form)
				   ((eq (cadr form) 'next-method-p)
				    (setq next-method-p-p 't)
				    (setq closurep t)
				    form)
				   (t nil))))
		       ((and (or (eq (car form) 'slot-value)
				 (eq (car form) 'set-slot-value))
			     (symbolp (cadr form))
			     (constantp (caddr form)))
			(let ((parameter
				(can-optimize-access (cadr form) required-parameters env)))
			  (if (null parameter)
			      form
			      (ecase (car form)
				(slot-value
				  (optimize-slot-value     slots parameter form))
				(set-slot-value
				  (optimize-set-slot-value slots parameter form))))))
		       (t form))))
	  
	  (setq walked-lambda (walk-form method-lambda env #'walk-function))

	  ;;
	  ;; Add &allow-other-keys to the lambda list as an interim
	  ;; way of implementing lambda list congruence rules.
	  ;;
	  (when (and (memq '&key lambda-list)
		     (not (memq '&allow-other-keys lambda-list)))
	    (let* ((rll (reverse lambda-list))
		   (aux (memq '&aux rll)))
	      (setq lambda-list
		    (if aux
			(progn (setf (cdr aux)
				     (cons '&allow-other-keys (cdr aux)))
			       (nreverse rll))
		        (nconc (nreverse rll) (list '&allow-other-keys))))))
	  ;; Scan the lambda list to determine whether this method
	  ;; takes &mumble arguments.  If it does, we set applyp and
	  ;; save-original-args true.
	  ;; 
	  ;; This is also the place where we construct the original
	  ;; arguments lambda list if there has to be one.
	  (dolist (p lambda-list)
	    (if (memq p lambda-list-keywords)
		(if (eq p '&aux)
		    (progn
		      (setq aux-bindings (cdr (memq '&aux lambda-list)))
		      (return nil))
		    (progn
		      (setq applyp t
			    save-original-args t)
		      (push '&rest original-args)
		      (push (make-symbol "AMPERSAND-ARGS") original-args)
		      (return nil)))
		(push (make-symbol (symbol-name p)) original-args)))
	  (setq original-args (if save-original-args
				  (nreverse original-args)
				  ()))
	  
	  (multiple-value-bind (ignore walked-declarations walked-lambda-body)
	      (extract-declarations (cddr walked-lambda))
	    (declare (ignore ignore))

	    
	    (when (some #'cdr slots)
	      (setq slots (slot-name-lists-from-slots slots))
	      (setq plist (list* :isl slots plist))
	      (setq walked-lambda-body (add-pv-binding walked-lambda-body
						       plist
						       required-parameters))
	      (dolist (dcl-stm walked-declarations)
		(dolist (dcl (cdr dcl-stm))
		  (when (eql (car dcl) 'ignore)
		    (setf (cdr dcl) (set-difference (cdr dcl) required-parameters))))))
	    (when (or next-method-p-p call-next-method-p)
	      (setq plist (list* :needs-next-methods-p 't plist)))

	    ;;; changes are here... (mt)
	    (let ((fn-body (if (or call-next-method-p next-method-p-p)
			      (add-lexical-functions-to-method-lambda
				walked-declarations
				walked-lambda-body
				`(lambda ,lambda-list
				   ,@walked-declarations
				   ,.walked-lambda-body)
				original-args
				lambda-list
				save-original-args
				applyp
				aux-bindings
				call-next-method-p
				next-method-p-p
				closurep)
			      `(lambda ,lambda-list
				 ,@walked-declarations
				 ,.walked-lambda-body))))
	      (values
		`(function ,fn-body)
		specializers
		documentation
		plist))))))))

;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "PCL:MAY-DAY-PCL;DEFS.LISP.2")
#+Genera (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
	   "-*-Mode: LISP; Package:(PCL LISP 1000); Base:10; Syntax:Common-lisp -*-")

(defun setfboundp (symbol)
  #+Genera (not (null (get-properties (symbol-plist symbol)
				      'lt::(derived-setf-function trivial-setf-method
					    setf-equivalence setf-method))))
  #+Lucid  (locally
	     (declare (special lucid::*setf-inverse-table*
			       lucid::*simple-setf-method-table*
			       lucid::*setf-method-expander-table*))
	     (or (gethash symbol lucid::*setf-inverse-table*)
		 (gethash symbol lucid::*simple-setf-method-table*)
		 (gethash symbol lucid::*setf-method-expander-table*)))
  #+kcl    (or (get symbol 'si::setf-method)
	       (get symbol 'si::setf-update-fn)
	       (get symbol 'si::setf-lambda))
  #+Xerox  (or (get symbol :setf-inverse)
	       (get symbol 'il:setf-inverse)
	       (get symbol 'il:setfn)
	       (get symbol :shared-setf-inverse)
	       (get symbol :setf-method-expander)
	       (get symbol 'il:setf-method-expander))

  #+:coral (or (get symbol 'ccl::setf-inverse)
	       (get symbol 'ccl::setf-method-expander))
  
  #+excl (or (get symbol 'excl::setf-inverse)
	     (get symbol 'excl::setf-method-expander))

  #-(or Genera Lucid KCL Xerox :coral excl) nil)



;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "PCL:MAY-DAY-PCL;DEFCLASS.LISP.2")
#+Genera (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
	   "-*-Mode: LISP; Package:(PCL LISP 1000); Base:10; Syntax:Common-lisp -*-")

;;;
;;; MAKE-TOP-LEVEL-FORM is used by all PCL macros that appear `at top-level'.
;;;
;;; The original motiviation for this function was to deal with the bug in
;;; the Genera compiler that prevents lambda expressions in top-level forms
;;; other than DEFUN from being compiled.
;;;
;;; Now this function is used to grab other functionality as well.  This
;;; includes:
;;;   - Preventing the grouping of top-level forms.  For example, a
;;;     DEFCLASS followed by a DEFMETHOD may not want to be grouped
;;;     into the same top-level form.
;;;   - Telling the programming environment what the pretty version
;;;     of the name of this form is.  This is used by WARN.
;;; 
(defun make-top-level-form (name times form)
  (flet ((definition-name ()
	   (if (and (listp name)
		    (memq (car name) '(class method method-combination)))
	       (format nil "~A~{ ~S~}"
		       (capitalize-words (car name) ()) (cdr name))
	       (format nil "~S" name))))
    (definition-name)
    #+Genera
    (let ((thunk-name (make-symbol "TOP-LEVEL-FORM"))
	  (true-name (if (listp name) (second name) name))
	  (true-type (if (listp name) (first name) 'defun)))
      `(eval-when ,times
	 (sys:multiple-definition  ,true-name ,true-type
	    (defun ,thunk-name () ,form)
	    (,thunk-name))))
    #+LCL3.0
    `(compiler-let ((lucid::*compiler-message-string*
		      (or lucid::*compiler-message-string*
			  ,(definition-name))))
       (eval-when ,times ,form))
    #-(or Genera LCL3.0)
    (make-progn `',name `(eval-when ,times ,form))))



;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "CLIM:MAY-DAY-PCL;WALK.LISP.2")
#+Genera (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
  "-*- Mode:LISP; Package:(WALKER LISP 1000); Base:10; Syntax:Common-lisp -*-")

#-Genera (in-package "WALKER")

#+Genera ;; Not required for any other PCL. --- rsl
(defun walk-prog/prog* (form context old-env sequentialp)
  (walker-environment-bind (new-env old-env)
    (let* ((possible-block-name (second form))
	   (blocked-prog (and (symbolp possible-block-name)
			      (not (eq possible-block-name 'nil)))))
      (multiple-value-bind (let/let* block-name bindings body)
	  (if blocked-prog
	      (values (car form) (cadr form) (caddr form) (cdddr form))
	      (values (car form) nil	     (cadr  form) (cddr  form)))
	(let* ((walked-bindings 
		 (walk-bindings-1 bindings
				  old-env
				  new-env
				  context
				  sequentialp))
	       (walked-body
		 (walk-declarations 
		   body
		   #'(lambda (real-body real-env)
		       (walk-tagbody-1 real-body context real-env))
		   new-env)))
	  (if block-name
	      (relist*
		form let/let* block-name walked-bindings walked-body)
	      (relist*
		form let/let* walked-bindings walked-body)))))))

#-Genera (in-package "PCL")

;=====================================
#+Genera (SYSTEM-INTERNALS:BEGIN-PATCH-SECTION)
#+Ignore (SYSTEM-INTERNALS:PATCH-SECTION-SOURCE-FILE "CLIM:MAY-DAY-PCL;CONSTRUCT.LISP.2")
#+Genera  (SYSTEM-INTERNALS:PATCH-SECTION-ATTRIBUTES
	   "-*-Mode:LISP; Package: PCL; Base:10; Syntax:Common-lisp -*-")


(defun expand-defconstructor (class-name name lambda-list supplied-initargs)
  (let ((class (find-class class-name nil))
	(supplied-initarg-names
	 (do ((elts supplied-initargs (cddr elts))
	      (res))
	     ((null elts) (nreverse res))
	   (push (car elts) res))))
    (when (null class)
      (error "defconstructor form being compiled (or evaluated) before~@
              class ~S is defined."
	     class-name))
    `(progn
       ;; In order to avoid undefined function warnings, we want to tell
       ;; the compile time environment that a function with this name and
       ;; this argument list has been defined.  The portable way to do this
       ;; is with defun.
       (proclaim '(notinline ,name))
       #+Ignore
       (defun ,name ,lambda-list
	 (declare (ignore ,@(specialized-lambda-list-parameters lambda-list)))
	 (error "Constructor ~S not loaded." ',name))
       (proclaim '(function ,name))

       ,(make-top-level-form `(defconstructor ,name)
			     '(load eval)
	  `(load-constructor
	     ',class-name
	     ',(class-name (class-of class))
	     ',name
	     ',supplied-initarg-names
	     ;; make-constructor-code-generators is called to return a list
	     ;; of constructor code generators.  The actual interpretation
	     ;; of this list is left to compute-constructor-code, but the
	     ;; general idea is that it should be an plist where the keys
	     ;; name a kind of constructor code and the values are generator
	     ;; functions which return the actual constructor code.  The
	     ;; constructor code is usually a closures over the arguments
	     ;; to the generator.
	     ,(make-constructor-code-generators class
						name
						lambda-list
						supplied-initarg-names
						supplied-initargs))))))
;;; Goes in BOOT.LISP:
(defmacro symbol-macrolet (macros &body body &environment env)
  (let ((specs (mapcar
		 #'(lambda (macro)
		     (list (car macro) (variable-lexical-p (car macro) env)
			   `(progn ,@(cdr macro))))
		 macros))
	;; Sort of a kludge; we don't really care about the rebinding of
	;; this gensym, but it's required by expand-with-slots.
	(gensym (gensym)))
    (expand-with-slots specs body
		       env gensym 'nil
		       #'identity)))
