;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-UTILS; Base: 10; Lowercase: Yes -*-

;; $fiHeader: lisp-utilities.lisp,v 1.10 91/04/15 15:43:01 cer Exp $

(in-package "CLIM-UTILS")

"Copyright (c) 1990, 1991 Symbolics, Inc.  All rights reserved.
 Portions copyright (c) 1988, 1989, 1990, 1991 International Lisp Associates."

;;; Define useful tools that don't exist in vanilla CL.

(defvar *keyword-package* (find-package "KEYWORD"))

;;; functionp doesn't do what we want and there isn't any other CL function
;;; that does. 
(defun funcallable-p (thing)
  (etypecase thing
    (symbol (fboundp thing))
    (function T)))

;;; Keep constant for compatability, also as the universal place
;;; to decide which type to use in which environment.
(defconstant +string-array-element-type+ 
	     #{Genera		'scl::string-char
	       excl		'cltl1::string-char
	       Otherwise	'character}
	     "Used for :ELEMENT-TYPE of arrays intended to hold characters.")

;;; Define a type that will be the right type to use whenever we make
;;; a string in CLIM.
#-excl					; already has EXTENDED-CHAR type
(deftype extended-char ()
  +string-array-element-type+)

;;; Make sure we don't get screwed by environments like Coral's that
;;; have *print-case* set to :downcase by default.
#-ansi-90
(defvar *standard-io-environment-val-cache* nil)

#-ansi-90
(defun standard-io-environment-vars-and-vals ()
  (unless *standard-io-environment-val-cache*
    (setq *standard-io-environment-val-cache*
	  (list 10				;*read-base*
		(copy-readtable nil)		;*readtable*
		(find-package "user")		;*package*
		t				;*print-escape*
		nil				;*print-pretty*
		nil				;*print-radix*
		10				;*print-base*
		nil				;*print-circle*
		nil				;*print-level*
		nil				;*print-length*
		:upcase				;*print-case*
		t				;*print-gensym*
		t				;*print-array*
		nil)))				;*read-suppress*
  (values
    '(*read-base* *readtable* *package* *print-escape* *print-pretty*
      *print-radix* *print-base* *print-circle* *print-level* *print-length*
      *print-case* *print-gensym* *print-array* *read-suppress*)
    *standard-io-environment-val-cache*))

(defmacro with-standard-io-environment (&body body)
  #-ansi-90
  `(multiple-value-bind (vars vals)
       (standard-io-environment-vars-and-vals)
     (progv vars vals
       ,@body))
  #+ansi-90
  `(with-standard-io-syntax ,@body))

#-(or ansi-90 Genera)
(defmacro with-standard-io-syntax (&body body)
  `(with-standard-io-environment ,@body))


;;; Have to provide CLIM-LISP:WITH-OPEN-STREAM, because it needs to use a different
;;; version of CLOSE.  We need this for printer streams, at least.

;;; This is a little more complicated than it absolutely needs to be.  The idea is
;;; that there should be as short a timing window during which the stream to be closed
;;; is open but we don't have our hands on it to pass to CLOSE.  We therefore don't
;;; want to bind the user's variable to the stream outside of the unwind-protect, but
;;; rather want to bind it inside.  The reason we need the temporary variable is
;;; because the user might declare the stream variable to be of type STREAM, which
;;; would not be true during the brief interval after the binding and before the
;;; setting of that variable.  I believe this implementation of WITH-OPEN-STREAM to
;;; have as small a timing window, and to be as semantically correct, as possible. 
;;; --- rsl 5 February 1991.

#+(or (not CLIM-uses-Lisp-stream-functions)	;Do this if we provide CLOSE function
      Genera					; Sigh.  CLOSE also shadowed for Genera.
      CCL-2)					; Sigh.  CLOSE also shadowed for CCL-2.
(defmacro with-open-stream ((stream-variable construction-form) &body body &environment env)
  (let ((aborted-variable (gensymbol 'aborted-p))
	(temporary-stream-variable (gensymbol 'stream)))
    (multiple-value-bind (documentation declarations actual-body)
	(extract-declarations body env)
      (declare (ignore documentation))
      `(let (,temporary-stream-variable
	     (,aborted-variable t))
       (unwind-protect
	   (multiple-value-prog1
	     (progn (setq ,temporary-stream-variable ,construction-form)
		    (let ((,stream-variable ,temporary-stream-variable))
		      ,@declarations
		      ,@actual-body))
	     (setf ,aborted-variable nil))
	 (when ,temporary-stream-variable
	   (close ,temporary-stream-variable :abort ,aborted-variable)))))))

(defun follow-synonym-stream (stream)
  #{
  Genera (si:follow-syn-stream stream)
  (and allegro (not ansi-90)) (if (eql (excl::sm_type stream) :synonym)
				    (symbol-value (excl::sm_indir-slist stream))
				    stream)
  (and ansi-90 (not Genera)) (typecase stream
				 (synonym-stream
				   (symbol-value (synonym-stream-symbol stream)))
				 (t stream))
  otherwise stream
  })

#-(or genera allegro ansi-90) 
(eval-when (compile)
  (warn "You haven't defined ~S for this implementation.  A stub has been provided."
	'follow-synonym-stream))

(defmacro default-output-stream (stream &optional must-be-variable-macro-name)
  `(cond ((member ,stream '(t nil))
	  (setq ,stream '*standard-output*))
	 ,@(when must-be-variable-macro-name
	     `(((not (and (symbolp ,stream)
			  (not (keywordp ,stream))))
		(warn "The stream argument to ~S, ~S, is invalid.~@
		       This argument must be a variable that can be bound to a new stream."
		      ',must-be-variable-macro-name ,stream)
		(setq ,stream '*standard-output*))))))

(defmacro default-input-stream (stream &optional must-be-variable-macro-name)
  `(cond ((member ,stream '(t nil))
	  (setq ,stream '*standard-input*))
	 ,@(when must-be-variable-macro-name
	     `(((not (and (symbolp ,stream)
			  (not (keywordp ,stream))))
		(warn "The stream argument to ~S, ~S, is invalid.~@
		       This argument must be a variable that can be bound to a new stream."
		      ',must-be-variable-macro-name ,stream)
		(setq ,stream '*standard-input*))))))

(defmacro default-query-stream (stream &optional must-be-variable-macro-name)
  `(cond ((member ,stream '(t nil))
	  (setq ,stream '*query-io*))
	 ,@(when must-be-variable-macro-name
	     `(((not (and (symbolp ,stream)
			  (not (keywordp ,stream))))
		(warn "The stream argument to ~S, ~S, is invalid.~@
		       This argument must be a variable that can be bound to a new stream."
		      ',must-be-variable-macro-name ,stream)
		(setq ,stream '*query-io*))))))

(defun fintern (format-string &rest format-args)
  (declare (dynamic-extent format-args))
  (intern (let ((package *package*))
	    (with-standard-io-environment
	      (let ((*package* package))
		(apply #'lisp:format () format-string format-args))))))

(defvar *gensymbol* 0)
(defun gensymbol (&rest parts)
  (declare (dynamic-extent parts))
  (when (null parts) (setf parts '(gensymbol)))
  (make-symbol (lisp:format nil "~{~A-~}~D" parts (incf *gensymbol*))))

;;; For macro writers; you can have your GENSYMBOLs start at 1.  Use
;;; this in the macro, not in its expansion...
(defmacro with-related-gensymbols (&body body)
  `(let ((*gensymbol* 0)) ,@body))

;; Used in generating internal function and method names.
;; (remove-word-from-string "com-" 'com-show-file) => "SHOW-FILE"
;; Always returns a new string that can be bashed to your heart's content
(defun remove-word-from-string (word string-or-symbol &optional only-from-beginning-p)
  (let ((string (etypecase string-or-symbol
		  (string string-or-symbol)
		  (symbol (string string-or-symbol)))))
    (let ((word-position (search word string :test #'char-equal)))
      (cond ((null word-position)
	     (concatenate 'string string))
	    ((zerop word-position)
	     (subseq string (length word)))
	    (only-from-beginning-p
	     (concatenate 'string string))
	    (t
	     (concatenate 'string
			  (subseq string 0 word-position)
			  (subseq string (+ word-position (length word)))))))))

;;; Why PUSHNEW doesn't do this is beyond me.
(defmacro push-unique (item reference &rest args &key test test-not key)
  (declare (ignore test test-not))
  `(let* ((evaled-item ,item)
	  (evaled-reference ,reference)
	  (element (find ,(if key `(funcall ,key evaled-item) 'evaled-item)
			 evaled-reference ,@args)))
     (setf ,reference
	   (if element
	       (substitute evaled-item element evaled-reference)
	       (cons evaled-item evaled-reference)))))

(defmacro catch-if (condition tag &body body)
  `(catch (if ,condition ,tag '|tag for CATCH-IF false condition|)
     ,@body))

;#+Genera
;(defmacro letf-globally (places-and-vals &body body)
;  `(sys:letf* ,places-and-vals ,@body))
;
;#-Genera
(defmacro letf-globally (places-and-vals &body body)
  ;; I don't want to use LETF-globally, mind you, but I can't easily implement
  ;; LETF{-not-globally} without something like sys:%bind-location.
  ;; Of course, this one is really LETF*-GLOBALLY, but don't tell anyone.
  ;; A minor optimization: when you bind something to itself or don't
  ;;  say what to bind it to, it doesn't get SETF'd, since it isn't
  ;;  being changed.
  (when (null places-and-vals)
    (return-from letf-globally `(progn ,@body)))
  (let ((let-forms nil)
	(set-forms nil)
	(unwind-forms nil))
    ;; remember that we can't use SCL:LOOP
    (map nil #'(lambda (place-and-val)
		 (let* ((place (pop place-and-val))
			(val-p (not (null place-and-val)))
			(val (and val-p (pop place-and-val)))
			(temp-var (gensymbol 'letf-globally-temp)))
		   (when (and val-p (equal place val)) (setf val-p nil))   ;bind to itself?
		   (push `(,temp-var ,place) let-forms)
		   (when val-p (push place set-forms) (push val set-forms))
		   (push temp-var unwind-forms) (push place unwind-forms)))
	 places-and-vals)
    `(let ,(nreverse let-forms)
       (unwind-protect
	   (progn (setf ,@(nreverse set-forms)) ,@body)
	 (setf ,@unwind-forms)))))		;Undo backwards.

(defmacro letf-globally-if (condition places-and-vals &body body)
  #+Genera (declare (zwei:indentation 1 4 2 1))
  (when (null places-and-vals)
    (return-from letf-globally-if `(progn ,@body)))
  (let ((let-forms nil)
	(set-forms nil)
	(unwind-forms nil)
	(condition-value (gensymbol 'condition)))
    (map nil #'(lambda (place-and-val)
		 (let* ((place (pop place-and-val))
			(val-p (not (null place-and-val)))
			(val (and val-p (pop place-and-val)))
			(temp-var (gensymbol 'letf-globally-temp)))
		   (when (and val-p (equal place val)) (setf val-p nil))
		   (push `(,temp-var (and ,condition-value ,place)) let-forms)
		   (when val-p (push place set-forms) (push val set-forms))
		   (push temp-var unwind-forms) (push place unwind-forms)))
	 places-and-vals)
    `(let ((,condition-value ,condition))
       (let ,(nreverse let-forms)
	 (unwind-protect
	     (progn (when ,condition-value (setf ,@(nreverse set-forms)))
		    ,@body)
	   (when ,condition-value (setf ,@unwind-forms)))))))


#-(and ansi-90 (not excl) (not Symbolics))
(eval-when (compile load eval)
  (proclaim '(declaration non-dynamic-extent)))

#+(and Ansi-90 (not excl) (not Symbolics))
(define-declaration non-dynamic-extent (spec env)
  (let ((vars (rest spec))
        (result nil))
    (dolist (v vars)
      (block process-var
        (multiple-value-bind (type local info)
                             (variable-information v env)
          (declare (ignore local))
          (case type
            (:lexical
             (when (cdr (assoc 'dynamic-extent info))
               (warn "The variable ~S has been declared ~S,~%it cannot be now declared ~S"
                     v 'dynamic-extent 'non-dynamic-extent)
               (return-from process-var))
             (when (cdr (assoc 'ignore info))
               (warn "The variable ~S has been declared ~S,~%it cannot be now declared ~S"
                     v 'ignore 'non-dynamic-extent)
               (return-from process-var))
             (push `(,v dynamic-extent nil) result))
            (otherwise
             (warn "~S is not a lexical variable, it cannot be declared ~S."
                   v 'non-dynamic-extent))))))
    (values :variable (nreverse result))))


;;;
;;; Moved DEFINE-GROUP and DEFUN-INLINE to defun-utilities.lisp, 1/8/92, York


;;; Bindings on the stack.  A work in progress.
;;; Which Lisps support this?

;; I suppose this could be done through "IMPORT"
#+(or Genera Cloe-Runtime)
(progn
(defmacro with-stack-list ((var &rest elements) &body body)
  `(sys::with-stack-list (,var ,@elements) ,@body))

(defmacro with-stack-list* ((var &rest elements) &body body)
  `(sys::with-stack-list* (,var ,@elements) ,@body))

#+Genera
(defun-inline evacuate-list (list)
  (if (and (sys:%pointerp list)
	   (not (or (sys:%pointer-lessp list sys:%control-stack-low)
		    (sys:%pointer-lessp (progn #+3600  sys:%control-stack-limit
					       #+imach (sys:%read-internal-register
							 sys:%register-control-stack-limit))
					list))))
      (copy-list list)
      list))

#+Cloe-Runtime
(defun-inline evacuate-list (list)
  (if (= (the fixnum (sys::%tr-type list)) sys::tr$k-cons-s)
      (copy-list list)
      list))
)	;#+(or Genera Cloe-Runtime)

#+excl
(progn
(defmacro with-stack-list ((var &rest elements) &body body)
  `(let ((,var (stack-list ,@elements)))
     ,@body))

(defun stack-list (&rest x)
  (declare (dynamic-extent x))
  (apply #'list x))

(defmacro with-stack-list* ((var &rest elements) &body body)
  `(let ((,var (stack-list* ,@elements)))
     ,@body))

(defun stack-list* (&rest x)
  (declare (dynamic-extent x))
  (apply #'list* x))

(defun evacuate-list (list)
  list)
)	;#+excl

#+ccl-2 ; perhaps ansi-90?
(progn
(defmacro with-stack-list ((var &rest elements) &body body)
  `(let ((,var (list ,@elements)))
     (declare (dynamic-extent ,var))
     ,@body))

(defmacro with-stack-list* ((var &rest elements) &body body)
  `(let ((,var (list* ,@elements)))
     (declare (dynamic-extent ,var))
     ,@body))

(defun evacuate-list (list)
  ;; --- Dunno if this is the right function to be calling
  ;; --- but it seems to give the right answers.
  (cond ((and (ccl::stack-area-endptr list)
              (listp list))
         (copy-list list))
        (t list)))
)	;#+ccl-2

#-(or Genera Cloe-Runtime excl ccl-2)
(progn
(defmacro with-stack-list ((var &rest elements) &body body)
  `(let ((,var (list ,@elements)))
     ,@body))

(defmacro with-stack-list* ((var &rest elements) &body body)
  `(let ((,var (list* ,@elements)))
     ,@body))

;; Since with-stack-list does nothing, this doesn't either.
;; When stack-consing works for non-Genera/CLOE, make this do something.
(defmacro evacuate-list (list)
  `,list)
)	;#-(or Genera Cloe-Runtime excl)

#+Genera
(defmacro with-stack-array ((name size &rest options) &body body)
  `(sys:with-stack-array (,name ,size ,@options) ,@body))

#-Genera
(defmacro with-stack-array ((name size &rest options) &body body)
  `(let ((,name (make-stack-array ,size ,@options)))
     ,@body))

#-Genera	;in case anybody wants to implement this...
(defun make-stack-array (size &rest options)
  (declare (dynamic-extent options))
  (apply #'make-array size options))

#+(or Genera CLOE-Runtime)
(defmacro with-rem-keywords ((new-list list keywords-to-remove) &body body)
  `(si::with-rem-keywords (,new-list ,list ,keywords-to-remove)
     ,@body))

#+(or Genera CLOE-Runtime)
(defun rem-keywords (list keywords-to-remove)
  (si::rem-keywords list keywords-to-remove))

#-(or Genera Cloe-Runtime)
(progn 
(defmacro with-rem-keywords ((new-list list keywords-to-remove) &body body)
  `(let ((,new-list (rem-keywords ,list ,keywords-to-remove)))
     ,@body))

(defun rem-keywords (list keywords)
  (macrolet ((kernel (predicate)
	       `(let ((head nil)
		      (tail nil))
		  (do ()
		      ((null list))
		    (let ((name (pop list))
			  (value (pop list)))
		      (when (not (,predicate name))
			(setq tail (setq head (list name value)))
			(return))))
		  (do ()
		      ((null list) head)
		    (let ((name (pop list))
			  (value (pop list)))
		      (when (not (,predicate name))
			(setq tail (setf (cddr tail) (list name value)))))))))
    (cond ((null list) nil)
	  ((null keywords) list)
	  ;; Special case: use EQ instead of MEMBER when only one keyword is supplied.
	  ((null (cdr keywords))
	   (let ((keyword (car keywords)))
	     (flet ((eq-predicate (name) (eq name keyword)))
	       (declare (dynamic-extent #'eq-predicate))
	       (kernel eq-predicate))))
	  (t
	   (flet ((member-predicate (name) (member name keywords)))
	     (declare (dynamic-extent #'member-predicate))
	     (kernel member-predicate))))))

;;; Another version, from 0.9 fork.
#+ignore
(defun rem-keywords (keyword-list keywords-to-remove)
  ;; remove initial ones
  (loop (cond ((null keyword-list) (return))
	      ((member (first keyword-list) keywords-to-remove)
	       (setf keyword-list (cddr keyword-list)))
	      (t (return))))
  (when keyword-list
    ;; the first key/val pair needs to be kept (because it wasn't
    ;; removed by the above loop).  Save the key/val pair, then recurse
    ;; on the subsequent tail and cons the key/val pair back on the
    ;; front.  An optimization is to remember the full list (with
    ;; key/val) and the sublist (without the key/val) so that if the
    ;; removed-from-sublist is EQ to the sublist, we don't need to put
    ;; key/val back on front but just use the original.
    (let* ((original keyword-list)
	   (key (pop keyword-list))
	   (val (pop keyword-list))
	   (nkeyword-list (rem-keywords keyword-list keywords-to-remove)))
      (if (eq keyword-list nkeyword-list)
	  original
	  (list* key val nkeyword-list)))))

)	;#-(or Genera Cloe-Runtime)


;; Why is there DOLIST in CL but no DOVECTOR or DOSEQ{uence}
(defmacro dovector ((var vector &key (start 0) end from-end simple-p) &body body)
  (unless (constantp simple-p)
    (setq simple-p nil)
    (warn "SIMPLE-P should be a constant, ignoring it"))
  (when (and simple-p (null end))
    (warn "When SIMPLE-P is T, you must supply :END"))
  (let ((fvector '#:vector)
	(startd  '#:start)
	(endd    '#:end)
	(limit   '#:limit)
	(variable (if (atom var) var (first var)))
	(count    (if (atom var) '#:count (second var)))
	(aref (if simple-p 'svref 'aref)))
    `(block nil
       (let* ((,fvector ,vector)
	      (,startd ,start)
	      (,endd ,(if simple-p `,end `(or ,end (length ,fvector))))
	      (,count (if ,from-end (1- ,endd) ,startd))
	      (,limit (if ,from-end (1- ,startd) ,endd)))
	 (declare (fixnum ,endd ,count ,limit)
		  ;; Turn on the afterburners...
		  (optimize (speed 3) (safety 0))
		  ,@(and simple-p `((type simple-vector ,fvector)))
		  #+genera (sys:array-register ,fvector))
	 (loop
	   (when (= ,count ,limit) (return))
	   (let ((,variable (,aref ,fvector ,count)))
	     ,@body)
	   (,(if from-end 'decf 'incf) ,count))))))

(defmacro doseq ((var sequence) &body body)
  (let ((fcn (gensymbol 'doseq)))
    `(flet ((,fcn (,var) ,@body))
       (etypecase ,sequence
	 (list (dolist (thing ,sequence) (,fcn thing)))
	 (vector (dovector (thing ,sequence) (,fcn thing)))))))

;;; Like (LOOP FOR X ON LIST ...)
(defmacro dorest ((var list &optional (by 'cdr)) &body body)
  `(do ((,var ,list (,by ,var)))
       ((null ,var) nil)
     ,@body))


;;; Arglist tools

;; (flatten-arglist '(foo bar &optional baz (quux)
;; 		          &rest mumble
;; 		          &key frotz (trouble) ((:izzy the-cat))))
;; (FOO BAR &OPTIONAL BAZ QUUX &REST MUMBLE &KEY FROTZ TROUBLE IZZY)
;; make-pass-on-arglist is the only caller
(defun flatten-arglist (arglist)
  (let ((new-arglist nil)
	(mode :positional))
    (dolist (arg-spec arglist)
      (cond ((listp arg-spec)
	     (case mode
	       (&optional
		 (push (first arg-spec) new-arglist))
	       (&key
		 ;; deal with "(... &key ((:keyword var) default))" syntax
		 (let ((thing (first arg-spec)))
		   (push (if (listp thing)
			     (let ((name (first thing)))
			       (if (eq (symbol-package name) *keyword-package*)
				   (intern (symbol-name name))
				   name))
			     thing)
			 new-arglist)))))
	    ((member arg-spec '(&key &rest &optional))
	     (setq mode arg-spec)
	     (push arg-spec new-arglist))
	    (t (push arg-spec new-arglist))))
    (nreverse new-arglist)))

;; (make-pass-on-arglist '(foo bar &optional baz (quux)
;; 		          &rest mumble
;; 		          &key frotz (trouble) ((:izzy the-cat))))
;; (FOO BAR BAZ QUUX MUMBLE :FROTZ FROTZ :TROUBLE TROUBLE ':IZZY THE-CAT)

;; --- It looks like &rest and &key don't get along well here.  A big
;; question in such a circumstane is who should be doing the defaulting?  
;; I.e., should this do the defaulting, by making the above arglist be
;; 	:FROTZ FROTZ :TROUBLE TROUBLE ':IZZY THE-CAT MUMBLE
;; for apply, or should it let the eventual caller do the defaulting by
;; punting the defaulting here and just doing
;;	MUMBLE
;; for apply?  There are also interactions with &allow-other-keys that
;; aren't done here.  If this is for pass-on, then &allow-other-keys
;; implies an &rest beforehand, otherwise it is impossible to get all
;; the keys that are being passed on.  There's also the question of
;; whether &allow-other-keys here should put :allow-other-keys in the
;; passed on calllist.
;;;
;;; The PROTOCOLS stuff is the only caller.
(defun make-pass-on-arglist (arglist)
  (let ((new-arglist nil)
	(fa (flatten-arglist arglist))
	(mode :positional)
	(apply-p nil))
    (do ((args fa (cdr args))
	 (original-args arglist (cdr original-args)))
	((null args) nil)
      (let ((arg (first args))
	    (arg-spec (first original-args)))
	(cond ((member arg '(&key &optional &rest))
	       (setq mode arg))
	      ((eq arg '&allow-other-keys)
	       (unless (eq mode '&key)
		 (error "~&&ALLOW-OTHER-KEYS must follow &KEY")))
	      (t (case mode
		   (&key
		     (let ((arg-name arg) (arg-var arg))
		       (cond ((and (listp arg-spec)
				   (listp (first arg-spec)))
			      (setq arg-name `',(first (first arg-spec)))
			      (setq arg-var (second (first arg-spec))))
			     (t (setq arg-name (intern (symbol-name arg) *keyword-package*))))
		       (push arg-name new-arglist)
		       (push arg-var new-arglist)))
		   (&rest (setq apply-p t)
			  (push arg new-arglist))
		   (t (push arg new-arglist)))))))
    (values
      (nreverse new-arglist)
      apply-p)))

(defun ignore-arglist (arglist)
  (flet ((lambda-list-element-compare (element and-option)
	   (and (atom element)
		(string= element and-option))))
    `(progn ,@(let ((args nil))
		(dolist (arg arglist)
		  ;; These various &keys may be in some other package.
		  (cond ((member arg '(&rest &downward-rest &key &allow-other-keys &optional)
				 :test #'lambda-list-element-compare)
			 nil)
			(t (push (cond ((atom arg) arg)
				       ((atom (car arg)) (car arg))
				       (t (cadar arg)))
				 args)
			   (when (and (consp arg) (consp (cdr arg)) (consp (cddr arg)))
			     (push (third arg) args)))))
		(nreverse args))
	    nil)))

(defun canonicalize-and-match-lambda-lists (canonical-order user-specified)
  (declare (values lambda-list ignores))
  ;; --- left out compatability code, which called canonicalize-and-match-lambda-lists-old,
  ;; from the 0.9 fork. -York 9/17/91
  (let ((new-lambda-list nil)
	(ignores nil))
    (flet ((user-var-symbol (entry)
	     ;; FOO | (FOO NIL) | ((:FOO BAR) NIL) | (FOO NIL FOO-P) | ((:FOO BAR) FOO-P) 
	     ;;--- We don't support the FOO-P syntax yet.
	     (cond ((atom entry)
		    entry)
		   ((atom (setq entry (first entry)))
		    entry)
		   (t (second entry))))
	   (user-var-name (entry)
	     ;; FOO | (FOO NIL) | ((:FOO BAR) NIL) | (FOO NIL FOO-P) | ((:FOO BAR) FOO-P) 
	     ;;--- We don't support the FOO-P syntax yet.
	     (cond ((atom entry)
		    entry)
		   ((atom (setq entry (first entry)))
		    entry)
		   (t (first entry)))))
      (declare (dynamic-extent #'user-var-symbol #'user-var-name))
      (dolist (canonical-var canonical-order)
	(let ((user-entry (first (member canonical-var user-specified
					 :test #'string-equal
					 :key #'user-var-name))))
	(cond (user-entry
	       (push (user-var-symbol user-entry) new-lambda-list)
	       (setq user-specified (remove user-entry user-specified)))
	      (t (let ((canonical-gensym (get canonical-var 'canonical-gensym)))
		   (unless canonical-gensym
		     (setq canonical-gensym (make-symbol (symbol-name canonical-var)))
		     (setf (get canonical-var 'canonical-gensym) canonical-gensym))
		   (push canonical-gensym new-lambda-list)
		   (push canonical-gensym ignores))))))
      (when (set-difference user-specified '(&key &allow-other-keys))
	(error "The arguments ~S aren't valid for this lambda list."
	       user-specified))
      (values (nreverse new-lambda-list)
	      (nreverse ignores)))))

#+Genera
(defmacro defun-property ((property symbol indicator) lambda-list &body body)
  (unless (eql property :property)
    (warn "Using ~S to define a function named ~S, which is not a property"
	  'defun-property (list property symbol indicator)))
  `(defun (:property ,symbol ,indicator) ,lambda-list ,@body))

#-Genera
(defmacro defun-property ((property symbol indicator) lambda-list &body body)
  (unless (eql property :property)
    (warn "Using ~S to define a function named ~S, which is not a property"
	  'defun-property (list property symbol indicator)))
  (let ((function-name (make-symbol (lisp:format nil "~A-~A-PROPERTY" symbol indicator))))
    `(progn (defun ,function-name ,lambda-list ,@body)
	    (eval-when (load eval) (setf (get ',symbol ',indicator) #',function-name)))))


;;; COMPILER-LET replacement from Dave Moon.

(defvar *compile-time-property-table* (make-hash-table))

;;; Retrieve information from a database that only lasts through COMPILE-FILE
;;; Symbol doesn't have to be a symbol, it can be a class object
(defun compile-time-property (symbol indicator &optional default)
  #+(or Genera Cloe-Runtime)
  (when (progn #+Cloe-runtime system::*file-declaration-environment*
	       #+Genera t)
    (multiple-value-bind (value flag)
	(#+Genera compiler:file-declaration #+Cloe-Runtime clos-internals::file-declaration
	 symbol indicator)
      (if flag value default)))
  #-(or Genera Cloe-Runtime)
  ;; For anything else, do it the dumb way that doesn't reset after compilation
  (let ((table (gethash indicator *compile-time-property-table*)))
    (unless table
      (setf (gethash indicator *compile-time-property-table*)
	    (setq table (make-hash-table))))
    (values (gethash symbol table default))))

(defsetf compile-time-property #+Genera compiler:file-declare
			       #-Genera set-compile-time-property)

#-Genera
(defun set-compile-time-property (symbol indicator value)
  #+Cloe-Runtime
  (when system::*file-declaration-environment*
    (setf (clos-internals::file-declaration symbol indicator) value))
  #-Cloe-Runtime
  (let ((table (gethash indicator *compile-time-property-table*)))
    (unless table
      (setf (gethash indicator *compile-time-property-table*)
	    (setq table (make-hash-table))))
    (setf (gethash symbol table) value)))

;;; Allegro 4.0 doesn't comply with this, but 4.1 does
#-(or Genera (and Ansi-90 (not (and Allegro (not (version>= 4.0))))))
(defmacro define-compiler-macro (name lambda-list &body body &environment env)
  env
  #+excl `(excl::defcmacro ,name ,lambda-list ,@body)
  #-(or Genera excl) (progn name lambda-list body env nil))	;Suppress compiler warnings.

#+Genera
;;; Support (proclaim '(function ...)) and (proclaim '(ftype ...)).
;;; This is part of deleting spurious multiple-definition warnings about constructors.
;;; Of course, who knows if this will work in other lisps.
(defun (:property ftype lisp:proclaim) (decl compile-time)
  (declare (ignore compile-time))		;Do it at load time as well.
  (mapc #'compiler:function-defined (cdr decl)))



#-(or Genera ansi-90)
(defmacro print-unreadable-object ((object stream &key type identity) &body body)
  `(flet ((print-unreadable-object-body () ,@body))
     (declare (dynamic-extent #'print-unreadable-object-body))
     (print-unreadable-object-1 ,object ,stream ,type ,identity
				#'print-unreadable-object-body
				',(not (null body)))))

#-(or Genera ansi-90)
;;; EXTRA-SPACE-REQUIRED is optional because old compiled code didn't always supply it.
(defun print-unreadable-object-1 (object stream type identity continuation
					 &optional (extra-space-required t))
  (write-string "#<" stream)
  ;; wish TYPE-OF worked in PCL
  (when type (lisp:format stream "~S " (class-name (class-of object))))
  (funcall continuation)
  (when identity
    (when extra-space-required (write-char #\space stream))
    (print-unreadable-object-identity
     object stream))
  (write-string ">" stream))

#-(or Genera ansi-90)
(defun print-unreadable-object-identity (object stream)
  #{Genera (format stream "~O" (sys:%pointer object))
  excl (format stream "@~X" (excl::pointer-to-fixnum object))
  ;; Lucid prints its addresses out in Hex.
  Lucid (format stream "~X" (sys:%pointer object))
  ;; Probably aren't any #+(and (not Genera) (not excl) (not PCL) (not ansi-90))
  ;; implementations (actually, this is false: Lispworks).
  otherwise (format stream "??? ~S ???" object)
  }
  )

#-(or Genera ansi-90)
(defvar *print-readably* nil)

#-(or Genera ansi-90)
(deftype real (&optional (min '*) (max '*))
  (labels ((convert (limit floatp)
	     (typecase limit
	       (number (if floatp (float limit 0f0) (rational limit)))
	       (list (map 'list #'convert limit))
	       (otherwise limit))))
    `(or (float ,(convert min t) ,(convert max t))
	 (rational ,(convert min nil) ,(convert max nil)))))

;;; ONCE-ONLY does the same thing as it does in zetalisp.  I should have just
;;; lifted it from there but I am honest.  Not only that but this one is
;;; written in Common Lisp.  I feel a lot like bootstrapping, or maybe more
;;; like rebuilding Rome.
(defmacro once-only (vars &body body)
  (let ((gensym-var (gensym))
        (run-time-vars (gensym))
        (run-time-vals (gensym))
        (expand-time-val-forms ()))
    (dolist (var vars)
      (push `(if (or (symbolp ,var)
                     (numberp ,var)
                     (and (listp ,var)
			  (member (car ,var) '(quote function))))
                 ,var
                 (let ((,gensym-var (gensym)))
                   (push ,gensym-var ,run-time-vars)
                   (push ,var ,run-time-vals)
                   ,gensym-var))
            expand-time-val-forms))    
    `(let* (,run-time-vars
            ,run-time-vals
            (wrapped-body
	      (let ,(mapcar #'list vars (reverse expand-time-val-forms))
		,@body)))
       `(let ,(mapcar #'list (reverse ,run-time-vars)
			     (reverse ,run-time-vals))
	  ,wrapped-body))))


;;; Use a lambda-list to extract arguments from a list and bind variables.
;;; This "should" signal an error if the list doesn't match the lambda-list.
;;; In implementations that have DESTRUCTURING-BIND, the easiest thing is to use it.
;;; We're actually not using the destructuring part, just the ability to match
;;; a lambda-list to a list of arguments without making a closure and using APPLY.
;;; ANSI CL has DESTRUCTURING-BIND.  Cloe and Lucid have it.
;;; Genera has it in both SYMBOLICS-COMMON-LISP and FUTURE-COMMON-LISP,
;;; but the CLIM-LISP package doesn't use either of those.
;;; Last time I checked Franz did not have it
;;;--- Maybe there is a more efficient way to do this in Allegro
(defmacro bind-to-list (lambda-list list &body body)
  (cond ((not (constantp list))
	 #+Genera `(scl:destructuring-bind ,lambda-list ,list
		     ,(ignore-arglist lambda-list)
		     ,@body)
	 #+Cloe-Runtime
                  `(cloe:destructuring-bind ,lambda-list ,list
		     ,(ignore-arglist lambda-list)
		     ,@body)
	 #+Lucid `(lucid-common-lisp:destructuring-bind ,lambda-list ,list
		    ,(ignore-arglist lambda-list)
		    ,@body)
	 ;; For the other systems, I guess we'll just give up and do it the slow way
	 #-(or Genera Lucid Cloe-Runtime)
	 `(flet ((bind-to-list-body ,lambda-list
		   ,(ignore-arglist lambda-list)
		   ,@body))
	    (declare (dynamic-extent #'bind-to-list-body))
	    (apply #'bind-to-list-body ,list)))
	(t
	 ;; This special case supposedly comes up a lot, but I think it never comes up
	 ;; This optimization plays fast and loose with order of evaluation issues
	 ;; for the default value forms in the lambda-list
	 (setq list (eval list))
	 `(symbol-macrolet
	            ,(do ((item)
			  (result nil)
			  (mode nil))
			 ((null lambda-list) (nreverse result))
		       (setq item (pop lambda-list))
		       (cond ((member item '(&optional &rest &key &aux))
			      (setq mode item))
			     ((member item lambda-list-keywords))
			     ((eq mode '&rest)
			      (push `(,item ',list) result))
			     ((eq mode '&key)
			      (multiple-value-bind (variable default supplied-p)
				  (if (atom item) (values item nil nil)
				      (values (if (atom (car item)) (car item) (cadar item))
					      (second item) (third item)))
				(do ((l list (cddr l))
				     (k (parameter-specifier-keyword item)))
				    ((null l)
				     (push `(,variable ,default) result)
				     (when supplied-p
				       (push `(,supplied-p 'nil) result)))
				  (when (eq (first l) k)
				    (push `(,variable ',(second l)) result)
				    (when supplied-p
				       (push `(,supplied-p 't) result))
				    (return)))))
			     (t
			      (multiple-value-bind (variable default supplied-p)
				  (if (atom item) (values item nil nil)
				      (values (first item) (second item) (third item)))
				(cond ((null list)
				       (push `(,variable ,default) result)
				       (when supplied-p
					 (push `(,supplied-p 'nil) result)))
				      (t
				       (push `(,variable ',(pop list)) result)
				       (when supplied-p
					 (push `(,supplied-p 't) result))))))))
	    ,@body))))

;;; Optimization to not bother with destructuring bind if none of the variables
;;; will be used
(defun lambda-list-variables-used-in-body (lambda-list body)
  ;; First collect the variables bound by lambda-list
  (let ((variables nil))
    (do* ((lambda-list lambda-list (cdr lambda-list))
	  (item (first lambda-list) (first lambda-list)))
	 ((null lambda-list))
      (cond ((member item lambda-list-keywords))
	    ((atom item)
	     (push item variables))
	    (t
	     (push (if (atom (car item)) (car item) (cadar item)) variables)
	     (when (cddr item) (push (caddr item) variables)))))
    (when variables
      #+Genera
      (lt:mapforms #'(lambda (subform kind usage state)
		       (declare (ignore usage state))
		       (when (and (member subform variables)
				  (not (member subform lt:*mapforms-bound-variables*))
				  (member kind 'lt:(set symeval)))
			 (return-from lambda-list-variables-used-in-body t)))
		   `(progn ,@body)
		   :bound-variables nil)
      #+Cloe-Runtime
      (labels ((mapper (subform context)
		 ;; It's not worth worrying about being fooled by shadowing bindings
		 ;; with the same name
		 (when (and (member subform variables)
			    (member context '(:access :assign)))
		   (return-from lambda-list-variables-used-in-body t))
		 (clos-internals::map-forms-recurse #'mapper subform context)))
	(clos-internals::map-forms-toplevel #'mapper `(progn ,@body)))
      #+(or Genera Cloe-Runtime) nil
      #-(or Genera Cloe-Runtime)
      ;; We don't know how to do this correctly in other Lisps, since there isn't any
      ;; standardized code-walker, but as a first approximation we can assume that
      ;; if the symbols appear textually they are used as variables, and if they don't,
      ;; they aren't.  Tricky use of macros could defeat this, but it should work well enough.
      ;; Of course this can be fooled by a quoted constant with the same name as a variable
      ;; into producing an unnecessary binding, but that's only an efficiency issue.
      (labels ((analyze (tree)
		 (if (atom tree)
		     (member tree variables)
		     (some #'analyze tree))))
	(analyze body)))))

;;; Get the keyword argument name from an &KEY parameter specifier
(defun parameter-specifier-keyword (spec)
  (cond ((atom spec)
	 (intern (symbol-name spec) "KEYWORD"))
	((atom (car spec))
	 (intern (symbol-name (car spec)) "KEYWORD"))
	(t (caar spec))))

;;; This is needed because FIND-CLASS in the compile-file environment doesn't look
;;; also in the run-time environment, at least in Symbolics CLOS, which is pretty
;;; embarrassing when we can't find the class T.
;;; In Lucid 4.0 this produces spurious wrong number of arguments warnings for the calls
;;; to FIND-CLASS.  There is no run-time error, it really does accept three arguments.
(defun find-class-that-works (name &optional (errorp t) environment)
  #+ccl-2 (when (eql environment 'compile-file)
            (setq environment ccl::*fcomp-compilation-environment*))
  #+excl (find-class name errorp)
  #-excl (if (compile-file-environment-p environment)
	     (or (find-class name nil environment)
		 (find-class name errorp nil))
	     (find-class name errorp environment)))


;;; F-ers

(define-modify-macro minf (&rest other-values) min)
(define-modify-macro maxf (&rest other-values) max)

(defmacro minf-or (place &rest things)
  `(if (null ,place) (setf ,place (min ,@things)) (minf ,place ,@things)))

(defmacro maxf-or (place &rest things)
  `(if (null ,place) (setf ,place (max ,@things)) (maxf ,place ,@things)))

(define-modify-macro roundf (&optional (divisor 1)) round)


;;; Simple vector support

(defun-inline copy-vector-portion (from-vector from-start to-vector to-start length)
  (declare (fixnum from-start to-start length)
	   (type simple-vector from-vector to-vector))
  (declare (optimize (speed 3) (safety 0)))
  (cond (#+Genera (< length 8) #-Genera t
	 (let (#+Genera (from-vector from-vector)
	       #+Genera (to-vector to-vector))
	   (declare #+Genera (sys:array-register from-vector to-vector))
	   (dotimes (i length)
	     #-excl (declare (ignore i))
	     (setf (svref to-vector to-start) (svref from-vector from-start))
	     (incf from-start)
	     (incf to-start))))
	#+Genera
	(t
	 (si:copy-array-portion from-vector from-start (+ from-start length)
				to-vector to-start (+ to-start length)))))

;; VECTOR must be a simple vector, FILL-POINTER serves as its fill pointer.
;; The returned values are a (possibly new) vector and the new fill pointer.
;; The idiom for using this is
;; (MULTIPLE-VALUE-SETQ (VECTOR FP) (SIMPLE-VECTOR-PUSH-EXTEND ELEMENT VECTOR FP)).
(defun simple-vector-push-extend (element vector fill-pointer &optional extension)
  (declare (values vector fill-pointer))
  (declare (fixnum fill-pointer) (type simple-vector vector))
  (let ((length (array-dimension vector 0)))
    (declare (fixnum length))
    (when (= fill-pointer length)
      ;; Grow the vector
      (let ((new-vector (make-array (+ length (max (ash length -1) (or extension 20)))
				    :element-type (array-element-type vector))))
	(copy-vector-portion vector 0 new-vector 0 length)
	(setq vector new-vector)))
    ;; Insert the new element and return the right values
    (setf (svref vector fill-pointer) element)
    (incf fill-pointer)
    (values vector fill-pointer)))

(defun simple-vector-insert-element (element index vector fill-pointer &optional extension)
  (declare (values vector fill-pointer))
  (declare (fixnum index fill-pointer) (type simple-vector vector))
  (let ((length (array-dimension vector 0)))
    (declare (fixnum length))
    (cond ((= fill-pointer length)
	   ;; Grow the vector, leaving a hole for the new element
	   (let ((new-vector (make-array (+ length (max (ash length -1) (or extension 20)))
					 :element-type (array-element-type vector))))
	     (copy-vector-portion vector 0 new-vector 0 index) 
	     (copy-vector-portion vector index new-vector (1+ index) (- length index))
	     (setq vector new-vector)))
	  (t
	   ;; Leave a hole for the new element
	   (let (#+Genera (vector vector))
	     #+Genera (declare (sys:array-register vector))
	     (do ((i fill-pointer (1- i)))
		 ((= i index))
	       (declare (fixnum i)
			(optimize (speed 3) (safety 0)))
	       (setf (svref vector i) (svref vector (1- i)))))))
    ;; Plug in the new element and return the right values
    (setf (svref vector index) element)
    (incf fill-pointer)
    (values vector fill-pointer)))


;;; Debugging support
(defmacro compiler-warn (format-string &rest format-args)
  `(macrolet ((warn-it ()
		(warn ,format-string ,@format-args)))
     (warn-it)))


;;; Some environments (like Genera 8.0.1) don't yet support DECLAIM.
(eval-when (compile eval)
  (unless (fboundp 'declaim)
    (defmacro declaim (spec) `(proclaim ',spec))))
