;;; -*- Mode: Lisp; Package: CLIM-UTILS; Syntax: Common-Lisp; Lowercase: Yes -*-

;; $fiHeader: clos.lisp,v 1.4 91/03/26 12:03:07 cer Exp $

;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;; 
;;; SILICA CLOS Extensions - adaptations of CLOS to meet SILICA's needs.
;;; 

(in-package "CLIM-UTILS")

;;; ----------------
;;; Constructors
;;;
;;; PCL supports a much more efficient mechanism for initializing
;;; instances than simply calling MAKE-INSTANCE.  We need a portable
;;; hook into that mechanism.

#+PCL
(defmacro define-constructor (name class lambda-list &rest initialization-arguments)
  `(pcl::defconstructor ,name ,class ,lambda-list ,@initialization-arguments))

#+(and PCL (not VDPCL))
(pushnew 'compile pcl::*defclass-times*)

#+Allegro-v4.0-constructors
(defmacro define-constructor (name class lambda-list &rest initialization-arguments)
  `(clos::defconstructor ,name ,class ,lambda-list ,@initialization-arguments))

#-(or PCL Allegro-v4.0-constructors)
(defmacro define-constructor (name class lambda-list &rest initialization-arguments)
  `(defun-inline ,name ,lambda-list
     (make-instance ',class ,@initialization-arguments)))

#+(and Genera PCL)
(scl:defmethod (:fasd-form #-VDPCL pcl::std-instance #+VDPCL pcl::iwmc-class) ()
  (make-load-form scl:self))


;; DEFINE-CONSTRUCTOR-USING-PROTOTYPE-INSTANCE can be used when creating an
;; instance can be copied from another instance of the same type, the init
;; args exist solely to initialize slots, and any possible
;; INITIALIZE-INSTANCE or SHARED-INITIALIZE methods don't do anything that
;; would prevent a valid create by copying and setting.  This is true of
;; graphics output records and text output records.

;; The syntax is a little bizzare in order to avoid having to do FIND-CLASS
;; at compile time and depend on that working.  Each tuple consists of three
;; required components: the slot name is the first element, the initarg is
;; the second element (in a form that is evaluated, in the event it isn't a
;; keyword), and the value is the third element, again, in a form that is
;; evaluated.  The "optional" fourth argument is the value that should be used
;; in the slot in the prototype instance.  For example,
;;   (define-constructor-using-prototype-instance
;;     make-foo foo (up over)
;;     (slot-1 :slot-1 up)
;;     (slot-2 'slot-2 (1+ over)))
;; will initialize SLOT-1 with the value of UP, initializing it either using
;; SLOT-VALUE with 'SLOT-1 or MAKE-INSTANCE with :SLOT-1, and will
;; initialize SLOT-2 with value of (1+ OVER) using SLOT-VALUE with 'SLOT-2
;; or MAKE-INSTANCE with 'SLOT-2.

#+CCL-2
(defmacro define-constructor-using-prototype-instance
	  (name class args &body tuples)
  (let ((proto-var (make-symbol (format nil "*a ~A*" class)))
	(proto '#:prototype-instance)
	(inst  '#:instance))
    `(define-group ,name define-constructor-using-prototype-instance
       (defvar ,proto-var nil)
       (defun ,name (,@args)
	 (let* ((,proto (or ,proto-var
			     (setq ,proto-var (make-instance ',class))))
		(,inst (without-scheduling
			  (ccl::copy-uvector
			    (ccl::%maybe-forwarded-instance ,proto)))))
	   ,@(mapcar #'(lambda (tuple)
			 `(setf (slot-value ,inst ',(first tuple)) ,(third tuple)))
		     tuples)
	   ,inst)))))

#-CCL-2
(defmacro define-constructor-using-prototype-instance
	  (name class args &body tuples)
  `(define-group ,name define-constructor-using-prototype-instance
     (define-constructor ,name ,class ,args
			 ,@(mapcan #'(lambda (tuple)
				       (list (second tuple) (third tuple)))
				   tuples))))



;;;
;;; Dynamic Class Creation
;;;

(defvar *dynamic-classes* (make-hash-table :test #'equal))

(defun-inline %make-standard-class (name supers)
  
  #+Lucid
  ;; Jonl thinks this is okay, but I personally find it pretty gross. -- RR
  (eval `(defclass ,name ,supers ()))
  
  #-Lucid
  ;; by which we mean PCL and Genera CLOS, at this point
  (let ((class (make-instance 'standard-class :direct-superclasses supers)))
    ;; Note that this does NOT make it so that you can find this
    ;; class with (find-class name)
    (setf (class-name class) name)
    class))

(defun find-dynamic-class (name-fn &rest supers)
  (declare (dynamic-extent supers))
  (when supers
    (do ((tail supers (cdr tail)))
	((null tail))
      (when (not #-PCL (typep (car tail) 'standard-class)
		 #+PCL (classp (car tail)))
	(setf (car tail) (find-class (car tail))))))
      
  (or (gethash supers *dynamic-classes*)
      ;;
      ;;  If there is no entry for a dynamic class with these supers
      ;;  then we have to create one.  This involves creating the class,
      ;;  setting its supers and adding the entry to *dynamic-classes*.
      ;;  
      (let ((supers (copy-list supers)))
	(setf (gethash supers *dynamic-classes*)
	      (%make-standard-class
		(intern (funcall name-fn) (find-package 'silica))
		supers)))))

(defun add-mixin (object mixin-class)
  (let ((class (class-of object)))
    (if (member mixin-class (class-precedence-list class) :test #'eq)
	(error "The class of ~S already includes ~S." object mixin-class)
	(change-class object
		      (find-dynamic-class #'(lambda () "???")
					  mixin-class class)))))

;;;
;;; DEFGENERIC ... because it isn't there.
;;;

#+ignore
(defmacro defgeneric (function-specifier lambda-list &rest options)
  (declare (ignore lambda-list))
  (let ((expansion nil)
	(setfp (and (consp function-specifier)
		    (eq (car function-specifier) 'setf)))
	(methods nil)
	(docstring nil))
    (when setfp
      (push `(pcl::do-standard-defsetf ,(cadr function-specifier))
	    expansion))
    (dolist (option options)
      (case (car option)
	(:documentation (setq docstring (second option)))
	(:method
	 (push `(defmethod ,function-specifier ,@(cdr option))
	       methods))))
    (when docstring
      (if setfp
	  (push `(setf (documentation ',(cadr function-specifier) 'setf)
		       ',docstring)
		expansion)
	  (push `(setf (documentation ',function-specifier 'function)
		       ',docstring)
		expansion)))
    `(progn ,@expansion ,@methods)))


#||

;;;
;;; Sundries..
;;;

(defun classes-in-package (package &optional map-on-package)
  (let ((classes nil))
    (unless (packagep package)
      (setq package (find-package package)))
    (if map-on-package
	(do-symbols (sym package)
	  (if (and (eq (symbol-package sym) package)
		   (find-class sym nil))
	      (push sym classes)))
	(maphash #'(lambda (key val)
		     (declare (ignore val))
		     (if (eq (symbol-package key)
			     package)
			 (push key classes)))
		 pcl::*find-class*))
    classes))

(defun collect-root-classes (&optional package)
  (w::with-collection
    (let ((classes (mapcar #'find-class (classes-in-package 
					  (or package 'silica) t))))
      (dolist (class classes)
	(if (not (intersection (pcl::class-local-supers class)
			       classes))
	    (w::collect (class-name class)))))))


||#


