;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-LISP; Base: 10; Lowercase: Yes -*-

(in-package "CLIM-LISP")

"Copyright (c) 1989, 1990 International Lisp Associates.  All rights reserved."

;;; All of this is taken from the STREAM-DEFINITION-BY-USER proposal to
;;; the X3J13 committee, made by David Gray of TI on 22 March 1989.  No
;;; Lisp implementation yet supports this proposal, so we implement it
;;; here in this separate package.  This way we will be ready when some
;;; Lisp implementation adopts it (or something like it).



;;; We shadow this (and the other functions that already exist in the
;;; Lisp package) and fake the "genericness" of the operation in order
;;; to fit into the existing implementation-dependent stream mechanisms.

#-PCL
(defgeneric streamp (stream))

(defmethod STREAMP (stream)
  (lisp:streamp stream))

;;;

#+Genera-Release-8
(defgeneric open-stream-p (stream))

#+Genera-Release-8
(defmethod OPEN-STREAM-P (stream)
  (future-common-lisp:open-stream-p stream))

;;;

#-PCL
(defgeneric input-stream-p (stream))

(defmethod INPUT-STREAM-P (stream)
  (lisp:input-stream-p stream))

;;;

#-PCL
(defgeneric output-stream-p (stream))

(defmethod OUTPUT-STREAM-P (stream)
  (lisp:output-stream-p stream))

;;;

#-PCL
(defgeneric stream-element-type (stream))

(defmethod STREAM-ELEMENT-TYPE (stream)
  (lisp:stream-element-type stream))

;;;

#-PCL
(defgeneric close (stream &key abort))

(defmethod CLOSE (stream &key abort)
  (lisp:close stream :abort abort))

;;;

#-PCL
(defgeneric pathname (stream))

(defmethod PATHNAME (stream)
  (lisp:pathname stream))

(deftype pathname () 'lisp:pathname)

;;;

#-PCL
(defgeneric truename (stream))

(defmethod TRUENAME (stream)
  (lisp:truename stream))

;;;

(defmacro write-forwarding-cl-output-stream-function (name &optional (args nil) (rest nil))
  (let ((cl-name (intern (symbol-name name) (find-package 'lisp)))
	(method-name (intern (lisp:format nil "STREAM-~A" (symbol-name name)))))
    (flet ((fix-call (form)
	     (if rest
		 `(apply #',(first form) ,@(rest form) ,rest)
		 form)))
      `(progn
	 (proclaim '(inline ,name))
	 (clim-utils::defun ,name (,@args &optional stream ,@(and rest `(&rest ,rest)))
	   ,@(and rest
		  `((declare (clim-utils::dynamic-extent ,rest))))
	   (case stream
	     ((nil) ,(fix-call `(,method-name *standard-output* ,@args)))
	     ((t) ,(fix-call `(,method-name *terminal-io* ,@args)))
	     (otherwise ,(fix-call `(,method-name stream ,@args)))))
	 (clim-utils::defmethod ,method-name (stream ,@args  ,@(and rest `(&rest ,rest)))
	   ,@(and rest `((declare (clim-utils::dynamic-extent ,rest))))
	   ,(fix-call `(,cl-name ,@args stream)))))))

(write-forwarding-cl-output-stream-function write-byte (integer))
(write-forwarding-cl-output-stream-function write-char (character))
;; (write-forwarding-cl-output-stream-function write-string (string) keys)

(defun write-string (string &optional (stream *standard-output*) &key (start 0) end)
  (case stream
    ((nil) (setf stream *standard-output*))
    ((t) (setf stream *terminal-io*)))
  (stream-write-string stream string start end))

;;; --- Can't use WRITE-FORWARDING-CL-OUTPUT-STREAM-FUNCTION because this takes &OPTIONAL
;;; instead of &KEY arguments, according to Gray stream proposal as implemented by Franz.
(defmethod stream-write-string ((stream t) string &optional (start 0) end)
  (lisp:write-string string stream :start start :end end))

(write-forwarding-cl-output-stream-function terpri)
(write-forwarding-cl-output-stream-function fresh-line)
(write-forwarding-cl-output-stream-function force-output)
(write-forwarding-cl-output-stream-function finish-output)
(write-forwarding-cl-output-stream-function clear-output)

;;;

(defmacro write-forwarding-cl-input-stream-function (name lambda-list &key eof)
  (let* ((cl-name (intern (symbol-name name) (find-package 'lisp)))
	 (method-name (intern (lisp:format nil "STREAM-~A" (symbol-name name))))
	 (args (mapcar #'(lambda (var) (if (atom var) var (first var)))
		       (set-difference lambda-list lambda-list-keywords)))
         (method-args (remove 'stream lambda-list))
	 (stream-args (remove 'stream args))
	 (call-method `(case stream
			 ((nil) (,method-name *standard-input* ,@stream-args))
			 ((t) (,method-name *terminal-io* ,@stream-args))
			 (otherwise (,method-name stream ,@stream-args)))))
    (case (first (last method-args))
      (&optional
       (setq method-args (butlast method-args))))
    `(progn
       (proclaim '(inline ,name))
       ,(if eof
	    (let ((args `(eof-error-p eof-value ,@(and (not (eq eof :no-recursive))
						       '(recursive-p)))))
	      `(defun ,name (,@lambda-list ,@args)
		 (let ((result ,call-method))
		   (cond ((not (eq result :eof))
			  result)
			 (eof-error-p
			  (signal-stream-eof stream ,@(and (not (eq eof :no-recursive))
							   '(recursive-p))))
			 (t
			  eof-value)))))
	    `(defun ,name ,lambda-list
	       ,call-method))
       (defmethod ,method-name ((stream t) ,@method-args)
	 (,cl-name ,@args ,@(and eof `(nil :eof)))))))

(write-forwarding-cl-input-stream-function read-byte (&optional stream) :eof :no-recursive)
(write-forwarding-cl-input-stream-function read-char (&optional stream) :eof t)
(write-forwarding-cl-input-stream-function unread-char (character &optional stream))
(write-forwarding-cl-input-stream-function read-char-no-hang (&optional stream) :eof t)
;; (write-forwarding-cl-input-stream-function peek-char (&optional peek-type stream) :eof t)

;;; Rewritten for true Gray stream proposal.
(defun peek-char (&optional peek-type input-stream (eof-error-p t)
			    eof-value recursive-p)
  (declare (ignore recursive-p))
  (case input-stream
    ((nil) (setf input-stream *standard-input*))
    ((t)   (setf input-stream *standard-output*)))
  (assert (or (null peek-type) (eq peek-type t) (characterp peek-type))
      (peek-type)
    "Illegal peek type ~S" peek-type)
  (loop
    (let ((ch (stream-peek-char input-stream)))
      (cond ((eq ch :eof)
	     (if eof-error-p
		 (error "eof encountered on input stream ~s"
			input-stream)
	       (return-from peek-char eof-value)))
	    ((or (null peek-type)
		 (and (eq peek-type t) (not (whitespace-char-p ch)))
		 (eq peek-type ch))
	     (return-from peek-char ch))))))

(defmethod stream-peek-char ((stream t))
  (lisp:peek-char nil stream nil :eof))

(write-forwarding-cl-input-stream-function listen (&optional stream))
(write-forwarding-cl-input-stream-function read-line (&optional stream) :eof t)
(write-forwarding-cl-input-stream-function clear-input (&optional stream))

(defun signal-stream-eof (stream &optional recursive-p)
  (declare (ignore stream recursive-p))
  (error "EOF"))      

;;; Make FORMAT do something useful on CLIM windows.  (At least CLIM:FORMAT, that is.)

(defun format (stream format-control &rest format-args)
  (when (null stream)
    (return-from format
      (apply #'lisp:format nil format-control format-args)))
  (when (eq stream 't)
    (setq stream *standard-output*))
  (cond ((streamp stream)
	 ;; this isn't going to quite work for ~&,
	 ;; but it's better than nothing.
	 (write-string (apply #'lisp:format nil format-control format-args) stream)
	 nil)
	(t
	 (apply #'lisp:format stream format-control format-args))))



(defclass FUNDAMENTAL-STREAM 
	  ()
     ()
  )

(defmethod STREAMP ((stream fundamental-stream)) t)

;;;

(defclass FUNDAMENTAL-INPUT-STREAM
	  (fundamental-stream)
     ()
  )

(defmethod INPUT-STREAM-P ((stream fundamental-input-stream)) t)

;;;

(defclass FUNDAMENTAL-OUTPUT-STREAM
	  (fundamental-stream)
     ()
  )

(defmethod OUTPUT-STREAM-P ((stream fundamental-output-stream)) t)

;;;

(defclass FUNDAMENTAL-CHARACTER-STREAM
	  (fundamental-stream)
     ()
  )

(defmethod STREAM-ELEMENT-TYPE ((stream fundamental-character-stream)) 'character)

;;;

(defclass FUNDAMENTAL-BINARY-STREAM
	  (fundamental-stream)
     ()
  )

(defclass FUNDAMENTAL-CHARACTER-INPUT-STREAM
	  (fundamental-input-stream fundamental-character-stream)
     ()
  )

(defclass FUNDAMENTAL-CHARACTER-OUTPUT-STREAM
	  (fundamental-output-stream fundamental-character-stream)
     ()
  )

(defclass FUNDAMENTAL-BINARY-INPUT-STREAM
	  (fundamental-input-stream fundamental-binary-stream)
     ()
  )

(defclass FUNDAMENTAL-BINARY-OUTPUT-STREAM
	  (fundamental-output-stream fundamental-binary-stream)
     ()
  )
