;;; -*- Mode: Lisp; Package: (CLIM-TEST :use (CLIM-LISP)); Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989 by Xerox Corporations.  All rights reserved. 
;;;

(defpackage "CLIM-TEST"
    (:use clim-lisp clim-utils clim silica ws) 
  (:nicknames ct)
  (:export ws
	   ws-text-style
	   grid
	   menu
	   focus
	   ;;mirror
	   bboard
	   insdel
	   layouts
	   ))

(in-package clim-test)


;;;
;;;  Pane Tests
;;;

(defmacro def-ws-test (name (&optional locals) pane &rest commands)
  `(defun ,name ()
     (let ((framem (ws:find-frame-manager))
	   ,@locals
	   (frame (ws:make-frame 'ws:frame)))
       (ws:with-look-and-feel-realization (framem frame)
	 (setf (ws:frame-pane frame) ,pane)
	 ,@(if commands
	       `((setf (ws:frame-commands frame) 
		       (list ,@commands))))
	 (ws:adopt-frame framem frame)
	 (ws:enable-frame frame)
	 frame))))

(def-ws-test ws ()
  (let* ((radio (ws::make-valued-state-wire 1))
	 (check (ws::make-valued-state-wire t))
	 (button-area
	  (ws:vertically ()
	   (ws:spacing (:space 2)
	       (ws:bordering ()
		(ws:spacing (:space 10)
			    (ws:horizontally ()
			     (ws:raising ()
			      (ws:spacing (:space 2)
				  (ws:make-pane 'ws:label-pane
						:text "Samples of WS Gadgets")))
			     (ws:make-pane 'ws:filler-pane :hs 20 :vs 10)
			     (ws:bordering ()
			      (ws:spacing (:space 2)
				(ws:make-pane 
				 'ws:label-button-pane 
				 :text "Done"
				 :trigger 
				 #'(lambda (&key zapper 
						 &allow-other-keys)
				     (ws:disable-frame 
				      (ws:pane-frame zapper))))))))))
				    
	   :fill 10
	   (ws:horizontally ()
	    :fill (ws:bordering ()
		   (ws:spacing (:space 10)
			       (ws:vertically ()
				(ws:make-pane 'ws:radio-button-pane
					      :wire radio
					      :text "Choice 1"
					      :value 1)
				(ws:make-pane 'ws:radio-button-pane
					      :wire radio
					      :text "Choice 2"
					      :value 2)
				(ws:make-pane 'ws:radio-button-pane
					      :wire radio
					      :text "Choice 3"
					      :value 3))))
	    :fill 10 (ws:bordering ()
		      (ws:spacing (:space 10)
				  (ws:vertically ()
				   (ws:bordering ()
				    (ws:make-pane 'ws:label-button-pane 
						  :text "Push to Test"))
				   (ws:make-pane 'ws:choicer-pane
						 :choices '("First" "Second"))
				   (ws:make-pane 'ws:checkbox-pane
						 :text "Enable Warp Drive" 
						 :wire check))))
	    :fill)))
	 (b1 (ws:raising ()
	      (ws:spacing (:space 10)
		  (ws:vertically ()
		   :fill (ws:horizontally () :fill button-area :fill)
		   :fill 20 (ws:bordering ()
			     (ws:spacing (:space 2)
				 (ws:horizontally ()
				  :fill (ws:make-pane 'ws:label-pane 
						      :text "Any String:  ")
				  (ws:make-pane 'ws:ledit-pane :nchars 15)
				  :fill)))
		   :fill)))))
    b1))

(def-ws-test ws-text-style ()
  (let* ((message (ws:make-pane 'ws:message-pane
				:text "The quick brown fox jumped..."
				:text-style (silica:intern-text-style
					     :fixed :bold :very-large)))
	 (text-styleg (ws::make-valued-state-wire :fixed))
	 (faceg (ws::make-valued-state-wire :bold))
	 (sizeg (ws::make-valued-state-wire :very-large))
	 (b1 (ws:bordering ()
	      (ws:spacing (:space 10)
			  (ws:vertically ()
			   (ws:raising () message)
			   (ws:vertically ()
			    (ws:make-pane 'ws::filler-pane :hs 10 :vs 20) 
			    (ws:horizontally ()
			     (ws:make-pane 'ws:label-pane :text "Text-Style:  ")
			     (ws:make-pane 'ws:radio-button-pane
					   :wire text-styleg
					   :text "Serif" :value :serif)
			     (ws:make-pane 'ws:radio-button-pane
					   :wire text-styleg
					   :text "Sans-Serif" :value :sans-serif)
			     (ws:make-pane 'ws:radio-button-pane
					   :wire text-styleg
					   :text "Fixed Pitch" :value :fixed))
			    (ws:horizontally ()
			     (ws:make-pane 'ws:label-pane :text "Size:  ")
			     (ws:make-pane 'ws:choicer-pane
					   :wire sizeg
					   :choices '(("8" . 8)
						      ("10" . 10)
						      ("12" . 12)
						      ("18" . 18)
						      ("24" . 24))))
			    (ws:horizontally ()
			     (ws:make-pane 'ws:label-pane :text "Face:  ") 
			     (ws:make-pane 'ws:radio-button-pane
					   :wire faceg
					   :text "Roman" :value :roman)
			     (ws:make-pane 'ws:radio-button-pane
					   :wire faceg
					   :text "Bold" :value :bold)
			     (ws:make-pane 'ws:radio-button-pane
					   :wire faceg
					   :text "Italic" :value :italic)
			     (ws:make-pane 'ws::filler-pane :hs 20 :vs 10)
			     (ws:make-pane 'ws:label-button-pane
					   :text "Done"))))))))
    (flet ((change-text-style (&key &allow-other-keys)
	     (setf (slot-value message 'silica:text-style) 
		   (silica:intern-text-style
		    (ws:value text-styleg)
		    (ws:value faceg)
		    (cdr (ws:value sizeg))))
	     (when (silica:graft message)
	       (silica:repaint-sheet message silica:+everywhere+))))

      (let ((trigger-tap 
	     (make-instance 'ws::trigger-tap :trigger #'change-text-style)))
	(ws::add-tap text-styleg trigger-tap)
	(ws::add-tap faceg trigger-tap)
	(ws::add-tap sizeg trigger-tap)))
    b1))


;;;
;;; Grid
;;;

(def-ws-test grid ()
  (ws:make-pane 'ws::grid-pane :space 2 :cell-h 60 :cell-w 60
		:nrows 3 :ncolumns 3
		:contents 
		(with-collection
		    (dotimes (i 3)
		      (collect
		       (with-collection
			   (dotimes (j 3)
			     (clim-utils:collect 
			      (ws:make-pane
			       'ws:label-button-pane
			       :hs+ ws:+fill+ :vs+ ws:+fill+
			       :halign :center :valign :center
			       :trigger (let ((i i)
					      (j j))
					  #'(lambda (&key &allow-other-keys)
					      (lisp:format t "~%~d, ~d selected"
							   i j)))
			       :text (format nil "~d, ~d" i j))))))))))


;;;
;;;
;;;

(def-ws-test ws-stream ()
  (ws:spacing (:space 2)
      (ws:make-pane 'ws::stream-pane :hs 300 :vs 300)))

(def-ws-test ws-scroll ()
  (ws:scrolling (:hs+ ws:+fill+ :vs+ ws:+fill+)
    (ws:make-pane 'ws::stream-pane :hs 300 :vs 300)))

(def-ws-test labelled-border ()
  (ws:make-pane 'ws::labelled-border-pane
		:text "Labelled Border"
		:contents (ws:make-pane 'ws::clim-pane :hs 300 :vs 300)))
;;;
;;; Menu
;;;

(defparameter *test-menu* 
  (ws:making-menu (:title "Tools"
			  :trigger #'(lambda (&key menu item &allow-other-keys)
				       (format t "Zapped: ~s, ~s: ~d~%" 
					       menu item (ws:value item))
				       (force-output t)))
		  (ws::make-item :text "Emacs" :value 1)
		  (ws::make-item :text "Marquis" :value 2)
		  (ws::make-item :text "Andrew" :value 3)
		  (ws::make-item :text "Others" :value 4)))

(defun menu ()
  (ws:do-menu *test-menu*))

;;;
;;; Focus
;;;

(ws:define-application-frame focus-frame ()
    (manager ledit-1 ledit-2)
  (:pane 
   (with-frame-slots (manager ledit-1 ledit-2)
     (setf manager
	   (ws:managing-focus ()
	     (ws:vertically ()
	       (ws:horizontally ()
		 (ws:make-pane 'ws:label-button-pane :text "Focus on -->"
			       :trigger
			       #'(lambda (&key &allow-other-keys)
				   (setf (ws::pane-keyboard-focus manager)
					 ledit-1)))
		 (ws:bordering ()
		   (setf ledit-1 
			 (ws::realize-pane 'ws:line-editor-pane :nchars 40))))
	       (ws:horizontally ()
		 (ws:make-pane 'ws:label-button-pane :text "Focus on -->"
			       :trigger
			       #'(lambda (&key &allow-other-keys)
				   (setf (ws::pane-keyboard-focus manager)
					 ledit-2)))
		 (ws:bordering ()
		   (setf ledit-2 (ws::realize-pane 'ws:line-editor-pane
						   :nchars 40)))))))))
  (:menu-group (("DeFocus" :command '(com-focus-frame-defocus)))))

(define-focus-frame-command com-focus-frame-defocus ()
  (with-frame-slots (manager) 
    (setf (ws:pane-keyboard-focus manager)
	  nil)))

(defun focus (&rest settings)
  (ws:make-frame 'focus-frame :frame-manager (ws:find-frame-manager) 
		 :title "Focus Test"
		 :user-settings settings
		 :enable t))

;;;
;;; Mirror
;;;

(ws:define-application-frame mirror-demo-frame ()
    (mirror-pane)
  (:pane 
   (ws:with-frame-slots (mirror-pane)
     (setf mirror-pane (ws:make-pane 'ws::mirrored-pane 
				     :hs 400 :hs+ ws:+fill+ :hs- ws:+fill+
				     :vs 400 :vs+ ws:+fill+ :vs- ws:+fill+)))))

(defun mirror (&rest settings)
  (ws:make-frame 'mirror-demo-frame :frame-manager (ws:find-frame-manager) 
		:title "Mirror Pane"
		:user-settings settings
		:enable t))

(defmethod ws::repaint-frame ((frame mirror-demo-frame) pane region)
  (declare (ignore region))
  (unless (silica:sheet-mirror pane)
    (silica:realize-mirror (silica:port pane) pane)
    ;; ??? This is a bug that I have to do this
    (silica:enable-mirror (silica:port pane) pane))
  (let ((medium (silica:sheet-medium pane))
	(w (silica:bounding-rectangle-width pane))
	(h (silica:bounding-rectangle-height pane)))
    
    #+test
    (draw-rectangle* medium 0 0 w h :ink silica:*black* :filled t)
    
    (silica:draw-rectangle* medium 0 0 w h :ink silica:*white* :filled t)
    (silica:draw-rectangle* medium (* .33 w) (* .33 h) (* .33 w) (* .33 h)
			    :ink silica:*black* :filled t)
    (silica:draw-rectangle* medium 0 0 (1- w) (1- h) :ink silica:*black*)
    (silica:medium-finish-output medium)))


;;;
;;; Bulletin Board
;;;

(define-application-frame bboard-frame ()
    (mirror-pane)
  (:pane 
   (with-frame-slots (mirror-pane)
     (setf mirror-pane (ws:make-pane 'ws:bboard-pane
			     :hs 400 :hs+ ws:+fill+ :hs- ws:+fill+
			     :vs 400 :vs+ ws:+fill+ :vs- ws:+fill+))))
  (:menu-group (("Insert Pane" :command '(com-bboard-frame-insert-pane))
		("Remove Pane" :command '(com-bboard-frame-remove-pane)))))

(define-bboard-frame-command com-bboard-frame-insert-pane () 
  (let ((left (random 400))
	(bottom (random 400)))
    (with-frame-slots (mirror-pane) 
      (with-look-and-feel-realization ()
	(adopt-child 
	 mirror-pane
	 (bordering ()
	  (make-pane 'ws:ledit-pane 
	     :nchars 20
	     :text "Another ..."))
	 :left left :bottom bottom))
      (repaint-pane mirror-pane))))

(define-bboard-frame-command com-bboard-frame-remove-pane () 
  (with-frame-slots (mirror-pane) 
    (let ((child (sheet-child mirror-pane)))
      (when child
	(disown-child mirror-pane child)))
    (repaint-pane mirror-pane)))

(defun bboard ()
  (let ((frame (make-frame 'bboard-frame 
			   :frame-manager (find-frame-manager)
			   :enable t)))
    frame))

;;;
;;; Inserting and Removing from a Box
;;;

(ws:define-application-frame insdel-frame ()
    (vbox 
     (added-panes :initform nil))
  (:pane
   (ws:with-frame-slots (vbox)
     (ws:vertically ()
       (ws::realize-pane 'ws:line-editor-pane :nchars 30
		     :text "This is a line editor")
       (ws::realize-pane 'ws:horizontal-divider :size :large)
       (ws:restraining ()
	 (setf vbox
	       (ws:vertically ()
		 (ws:make-pane 'ws:label-pane :text "This is a label pane")
		 (ws::realize-pane 'ws:horizontal-divider :size :medium)
		 :fill))))))
  (:menu-group (("Insert Pane" :command '(com-insdel-frame-insert-pane))
		("Remove Pane" :command '(com-insdel-frame-remove-pane)))))

(define-insdel-frame-command com-insdel-frame-insert-pane () 
  (with-look-and-feel-realization ()
    (with-frame-slots (added-panes vbox)
      ;; Inserting after :fill.
      (push (ws:make-pane 'ws:hline-pane :size :small) added-panes)
      (push (ws:make-pane 'ws:label-pane :text "Another ...") added-panes)
      (ws:insert-panes vbox 
		       (list
			(cadr added-panes)
			(car added-panes)))
      (repaint-pane vbox)
      (ws:notify-user (pane-frame vbox)
		      "Pane Inserted"))))

(define-insdel-frame-command com-insdel-frame-remove-pane () 
  (declare (ignore button))
  (with-frame-slots (added-panes vbox)
    (if added-panes
	(progn
	  (ws:remove-pane vbox (pop added-panes))
	  (ws:remove-pane vbox (pop added-panes))
	  (ws:repaint-pane vbox)
	  (ws:notify-user (pane-frame vbox)
			  "Pane removed"))
	(ws:notify-user
	 (pane-frame vbox)
	 "Trying to delete my panes, eh?"))))

(defun insdel (&rest settings)
  (ws:make-frame 'insdel-frame
		 :title "Insdel" :width 400 :height 200
		 :user-settings settings
		 :frame-manager (ws:find-frame-manager)
		 :enable t))


;;;
;;; Layouts
;;;

(ws:define-application-frame layouts-frame ()
    ()
  (:panes
   (layout-1
    (ws:vertically ()
      (ws:horizontally () button-1 (ws:bordering () ledit-1))
      (ws:horizontally () button-2 (ws:bordering () ledit-2))))
   (layout-2
    (ws:vertically ()
      (ws:horizontally () button-2 (ws:bordering () ledit-2))
      (ws:horizontally () button-1 (ws:bordering () ledit-1))))
   ;; should be renamed "line-editor-pane"
   (ledit-1 (ws:make-pane 'ws:ledit-pane :nchars 40))
   (ledit-2 (ws:make-pane 'ws:ledit-pane :nchars 40))
   (button-1 (ws:make-pane 'ws:label-button-pane :text "Smancy -->"
			   :trigger
			   #'(lambda (&key &allow-other-keys) nil)))
   (button-2 (ws:make-pane 'ws:label-button-pane :text "Fancy  -->"
			   :trigger
			   #'(lambda (&key &allow-other-keys) nil)))))

(defun layouts (&rest settings)
  (ws:make-frame 'layouts-frame
		 :title "Layout"
		 :user-settings settings
		 :frame-manager (ws:find-frame-manager)
		 :enable t))



