;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-TEST; Base: 10 -*-

(in-package :clim-test)

;; Override any existing transformation
(defmacro with-absolute-transformation ((window transformation) &body body)
  `(with-drawing-options (,window :transformation +identity-transformation+)
     (setf (medium-transformation ,window) ,transformation)
     NIL
     ,@body))

;; Make sure we have the identity transformation:
(defmacro with-identity-transform ((window) &body body)
  `(with-absolute-transformation (,window +identity-transformation+)
     ,@body))

#||
;;; Make a transformation which translates units to millimeters, with center as (0,0)
(defun window-mm-transformation (window)
  (let ((graft (silica:graft window)))
    (with-bounding-rectangle* (wl wt wr wb) graft
       (let* ((ww-in-mm (silica:graft-width-mm graft))
	      (wh-in-mm (silica:graft-height-mm graft))
	      (mm-to-width-ratio (/ (- wr wl) ww-in-mm))
	      (mm-to-height-ratio (/ (- wb wt) wh-in-mm))
	      (mm-to-pixel-transformation (make-scaling-transformation
					    mm-to-width-ratio mm-to-height-ratio))
	      (translation-transformation (make-translation-transformation
					    (floor (- wr wl) 2)  (floor (- wb wt) 2) )))
	 (compose-transformations mm-to-pixel-transformation translation-transformation)))))
||#
;;; Try this for now.
(defun window-mm-transformation (window)
  (with-bounding-rectangle* (wl wt wr wb) window
    (make-transformation 3.4 0 0 -3.4 (floor (- wr wl) 2) (floor (- wb wt) 2))))

(defmacro with-mm-transformation ((window -x -y +x +y &key absolute-p) &body body)
  `(let ((transform (window-mm-transformation ,window)))
     (,@(if absolute-p				;Override any existing transformation!
	    `(with-absolute-transformation (,window transform))
	    `(with-drawing-options (,window :transformation transform)))	    
      (with-drawing-options (,window :line-scale-lines nil)
	(with-bounding-rectangle* (,-x ,-y ,+x ,+y)
				  (untransform-region transform (sheet-region ,window))
	  ,@body)))))

(defun draw-grid (window &optional (ink (medium-ink window)))
  (with-drawing-options (window :ink ink)
    (with-mm-transformation (window -x -y +x +y :absolute-p t)
      (draw-line* window -x 0 +x 0 :line-thickness 3)
      (draw-line* window 0 -y 0 +y :line-thickness 3)
      (do ((x (ceiling -x) (1+ x)))
	  ((> x +x))
	(unless (zerop x)
	  (if (zerop (mod x 5))
	      (draw-line* window x -y x +y
			  :line-thickness (if (zerop (mod x 10)) 2 1))
	      (draw-line* window x -1 x +1))))
      (do ((y (ceiling -y) (1+  y)))
	  ((> y +y))
	(unless (zerop y)
	  (if (zerop (mod y 5))
	      (draw-line* window -x y +x y
			  :line-thickness (if (zerop (mod y 10)) 2 1))
	      (draw-line* window -1 y +1 y))))))
  (w::medium-force-output window))

(defun draw-some-rectangles (window &optional inks)
  (with-mm-transformation (window -x -y +x +y)
    (draw-some-rectangles-internal window inks -x -y +x +y)))

(defun draw-some-rectangles-internal (window inks -x -y +x +y)
  (do ((x (* 10 (ceiling -x 10)) (+ x 10))
       (end-x (floor (- +x 10)))
       (ink (medium-ink window))
       (filled-p t (not filled-p)))
      ((> x end-x))
    (do ((y (* 10 (ceiling -y 10)) (+ y 10))
	 (end-y (floor (- +y 10)))
	 (filled-p (not filled-p) (not filled-p)))
	((> y end-y))
      (when inks (setf ink (pop inks)))
      (draw-rectangle* window (+ x 2) (+ y 2) (+ x 8) (+ y 8)
		       :filled filled-p
		       :ink ink
		       :line-thickness 2))))

#|| Old version
(defun draw-some-rectangles-internal (window inks -x -y +x +y)
  (do* ((increment 1 (1+ increment))
	(x -x (+ x increment 1))
	(y -y (+ y increment 1))
	(x2 0 (+ x2 (* increment (signum x))))
	(y2 0 (+ y2 (* increment (signum y))))
	(ink (medium-ink window)))
       ((or (> x +x) (> y +y)))
    (when inks (setf ink (pop inks)))
    (draw-rectangle* window x y (+ x increment) (+ y increment)
		     :filled (evenp increment) :ink ink
		     :line-thickness (/ increment 2))
    (draw-rectangle* window x2 y2 (- x2 increment) (+ y2 increment)
		     :ink ink
		     :line-thickness (/ increment 2))))
||#

(defun draw-some-circles (window &optional inks)
  (with-mm-transformation (window -x -y +x +y)
    (draw-some-circles-internal window inks -x -y +x +y)))

(defun draw-some-circles-internal (window inks -x -y +x +y)
  (do* ((radius 1 (1+ radius))
	(x -x (+ x radius -1))
	(y +y (- y radius -1))
	(ink (medium-ink window)))
       ((or (> x +x) (< y -y)))
    (when inks (setf ink (pop inks)))
    (draw-circle* window x y radius :filled (evenp radius)
		  :line-thickness (/ radius 2) :ink ink)))

(defun draw-some-texts (window &optional inks)
  (with-mm-transformation (window -x -y +x +y)
    (draw-some-texts-internal window inks -x -y +x +y)))

(defun draw-some-texts-internal (window inks -x -y +x +y)
  (do* ((increment 0 (1+ increment))
	(x -x (+ x increment increment))
	(y -y (+ y increment increment))
	(ink (medium-ink window)))
       ((or (> x +x) (> y +y)))
    (when inks (setf ink (pop inks)))
    (draw-text* window (code-char (+ (char-code #\A) increment)) x y :ink ink
		:text-face (nth (mod increment 4)
				'(:roman :italic :bold (:bold :italic)))
		:ink ink)))

(defun draw-some-things (window &optional inks)
  (macrolet ((draw-something (what-to-draw)
	       (let ((function (fintern "~A-~A~A" 'draw-some what-to-draw 's)))
		 `(progn #+ignore ; no, it doesn
			 (window-clear window)	;--- ??? Does this work on generic media?
			 (ci::draw-rectangle-with-rectangle window  window :ink +background+)
			 (draw-grid window)
			 (,function window ,@(if (eql what-to-draw 'text) () '(inks)))
			 (sleep 10)))))
    (draw-something rectangle)
    (draw-something circle)
    (draw-something text)))

(defun clear-window (window)
  (with-mm-transformation (window -x -y +x +y :absolute-p t)
    (draw-rectangle* window -x -y +x +y :ink +background+)))

(defun show-window (window)
  #-Silica (progn (window-expose window) t)	;Always deexpose when done in #-Silica
  #+Silica (enable-frame (pane-frame window)))

(defun hide-window (window)
  #-Silica (window-deexpose window)
  #+Silica (disable-frame (pane-frame window)))

(defmacro with-window-showing ((window) &body body)
  `(with-window-showing-internal
     ,window
     (named-continuation window-window-showing () ,@body)))

(defun with-window-showing-internal (window continuation)
  (declare (dynamic-extent continuation))
  (let ((deexpose? nil))
    (unwind-protect
	(progn (setf deexpose? (show-window window))
	       (funcall continuation))
      (when deexpose? (hide-window window)))))

(defun draw-some-things-internal (window inks -x -y +x +y)
  (with-window-showing (window)
    (macrolet ((draw-something (what-to-draw)
		 (let ((function (fintern "~A-~A~A" 'draw-some what-to-draw 's-internal)))
		   `(progn (clear-window window)
			   (draw-grid window)
			   (,function window ,(if (eql what-to-draw 'text) () 'inks)
			    -x -y +x +y)
			   (w::medium-force-output window)
			   (sleep 1)))))
      (draw-something rectangle)
      (draw-something circle)
      (draw-something text))))

(defvar *color-wheel*
	(let ((colors (mapcar #'symbol-value
			      '(+red+ +yellow+ +green+ +cyan+ +blue+ +magenta+))))
	  (nconc colors colors)
	  colors))

(defvar *gray-wheel*
	(let ((grays (with-collection (dotimes (i 8)
					(let ((x (/ (mod (* i 4) 7) 7.0)))
					  (collect (make-color-rgb x x x)))))))
	  (nconc grays grays)
	  grays))

(defun draw-some-colors (window function)
  (funcall function window *color-wheel*))

(defun draw-some-colors-internal (window function &rest args)
  (declare (dynamic-extent args))
  (apply function window *color-wheel* args))

(defun draw-some-grays (window function)
  (funcall function window *gray-wheel*))

(defun draw-some-grays-internal (window function &rest args)
  (declare (dynamic-extent args))
  (apply function window *gray-wheel* args))

;;; E.g., (draw-some-colors window #'draw-some-things)


(defun draw-some-rotated-things (window function inks &optional (rotation))
  (unless rotation
    (let* ((denominator #+Ignore  (+ (random 10) 3)               #-Ignore 6)
	   (numerator #+Ignore	  (+ (random (1- denominator)) 1) #-Ignore 1))
      (setf rotation (* 2pi (/ numerator denominator)))))
  (with-mm-transformation (window -x -y +x +y)
    (with-rotation (window rotation)
      (funcall function window inks -x -y +x +y))))

(defun draw-some-scaled-things (window function inks &optional (scale-x) (scale-y scale-x))
  (unless scale-y
    (let* ((denominator #+Ignore (+ (random 10) 3)               #-Ignore 3)
	   (numerator-x #+Ignore (+ (random (1- denominator)) 1) #-Ignore 1)
	   (numerator-y #+Ignore (+ (random (1- denominator)) 1) #-Ignore 2))
      (setf scale-x (/ numerator-x denominator)
	    scale-y (/ numerator-y denominator))))
  (with-mm-transformation (window -x -y +x +y)
    (with-scaling (window scale-x scale-y)
      (funcall function window inks -x -y +x +y))))




(defun clim-pane-frame (&key (server-path *default-server-path*)
			   (hs 600)
			   (vs #+Imach 200 #-Imach 400))
  (let ((framem (find-frame-manager :server-path server-path))
	(frame (make-frame 'frame)))
    (with-look-and-feel-realization (framem frame)
      (setf (frame-pane frame)
	    (make-pane 'ws::clim-pane :hs+ +fill+ :vs+ +fill+ :hs hs :vs vs))
      (adopt-frame framem frame)
      (enable-frame frame))
    (frame-pane frame)))

(defun definitive-drawing-test (window &optional (color-p))
  (flet ((do-something (window ignore -x -y +x +y)
	   (if color-p
	       (draw-some-colors-internal window #'draw-some-things-internal
					  -x -y +x +y)
	       (draw-some-grays-internal window #'draw-some-things-internal
					 -x -y +x +y))))
    (with-window-showing (window)
      (draw-some-scaled-things window #'do-something nil)
      (draw-some-rotated-things window #'do-something nil))))
