;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-TEST; Base: 10 -*-

(in-package "CLIM-TEST")

(defvar *table-graphics-tests* (make-array 10 :fill-pointer 0))

(defmacro define-table-cell-test (name description &body body)
  (let ((fcn-name
	  (or (get name 'table-cell-test)
	      (setf (get name 'table-cell-test)
		    (make-symbol (format nil "~S-TABLE-TEST" name))))))
    `(progn 
       (defun ,fcn-name (stream)
	 ,@body)
       (setf (get ',fcn-name 'documentation) ',description)
       (add-table-cell-test ',fcn-name))))

(defun add-table-cell-test (function-name)
  (unless (find function-name *table-graphics-tests*)
    (vector-push-extend function-name *table-graphics-tests*)))

(defun describe-table-cell-output (stream case-number)
  (let* ((function (aref *table-graphics-tests* case-number))
	 (documentation (get function 'documentation)))
    (format stream "Test ~D:  " case-number)
    (write-string documentation stream)))

(defun do-table-cell-output (stream case-number)
  (funcall (aref *table-graphics-tests* case-number) stream))

(defun table-graphics-test (stream &optional (documentation-window stream) (case 0))
  (window-clear stream)
  (window-clear documentation-window)
  (with-text-face (:bold documentation-window)
    (write-string "Table tests" documentation-window))
  (terpri documentation-window)
  (describe-table-cell-output documentation-window case)
  (force-output documentation-window)
  (stream-set-cursor-position* stream 30 30)
  (formatting-table (stream)
    (formatting-row (stream)
      (formatting-cell (stream)
	(write-string "cell one" stream))
      (formatting-cell (stream)
	(write-string "cell two" stream))
      (formatting-cell (stream)
	(write-string "cell three" stream))
      )
    (formatting-row (stream)
      (formatting-cell (stream :align-y :bottom)
	(write-string "graphics -->" stream))
      (formatting-cell (stream)
	(do-table-cell-output stream case))
      (formatting-cell (stream :align-y :top)
	(write-string "<-- graphics" stream)))
    (formatting-row (stream)
      (formatting-cell (stream)
	(write-string "cell one" stream))
      (formatting-cell (stream)
	(write-string "cell two" stream))
      (formatting-cell (stream)
	(write-string "cell three" stream))))
  (draw-line* stream 30 15 30 45 :ink +green+)
  (draw-line* stream 15 30 45 30 :ink +green+)
  (terpri documentation-window)
  (write-string "Click on display to replay contents" documentation-window)
  (force-output documentation-window)
  (read-gesture :stream stream)
  (ci::erase-viewport stream)
  (repaint-sheet stream (sheet-region stream))
  )

(defun table-graphics-test-column (stream &optional (documentation-window stream) (case 0))
  (window-clear stream)
  (window-clear documentation-window)
  (with-text-face (:bold documentation-window)
    (write-string "Table tests" documentation-window))
  (terpri documentation-window)
  (describe-table-cell-output documentation-window case)
  (write-string " [column]" documentation-window)
  (force-output documentation-window)
  (stream-set-cursor-position* stream 30 30)
  (formatting-table (stream)
    (formatting-column (stream)
      (formatting-cell (stream)
	(write-string "cell one" stream))
      (formatting-cell (stream :align-y :bottom)
	(write-string "graphics -->" stream))
      (formatting-cell (stream)
	(write-string "cell one" stream))
      )
    (formatting-column (stream)
      (formatting-cell (stream)
	(write-string "cell two" stream))
      (formatting-cell (stream)
	(do-table-cell-output stream case))
      (formatting-cell (stream)
	(write-string "cell two" stream)))
    (formatting-column (stream)
      (formatting-cell (stream)
	(write-string "cell three" stream))
      (formatting-cell (stream :align-y :top)
	(write-string "<-- graphics" stream))
      (formatting-cell (stream)
	(write-string "cell three" stream))))
  (draw-line* stream 30 15 30 45 :ink +green+)
  (draw-line* stream 15 30 45 30 :ink +green+)
  (terpri documentation-window)
  (write-string "Click on display to replay contents" documentation-window)
  (force-output documentation-window)
  (read-gesture :stream stream)
  (ci::erase-viewport stream)
  (repaint-sheet stream (sheet-region stream))
  )

(define-table-cell-test rectangle "Normal, filled rectangle (0 0 20 20)"
  (draw-rectangle* stream 0 0 20 20))

(define-table-cell-test rect-and-circle "Rectangle (0 0 20 20), Red circle (20 20 10)"
  (draw-rectangle* stream 0 0 20 20)
  (draw-circle* stream 20 20 10 :ink +red+))

(define-table-cell-test minus-rectangle "Blue rectangle (-10 -10 10 10)"
  (draw-rectangle* stream -10 -10 10 10 :ink +blue+))

(define-table-cell-test text-rectangle "\"foo\" and rectangle (20 20 30 30)"
  (write-string "foo" stream)
  (draw-rectangle* stream 20 20 30 30))

(define-table-cell-test surrounding-output "ci::surrounding-output-with-border"
  (ci::surrounding-output-with-border (stream)
    (write-string "Foobar" stream)))

(define-table-cell-test stream-cursor-position "multiple-value-bind, draw-rectangle"
  (multiple-value-bind (x y)
      (stream-cursor-position* stream)
    (draw-rectangle* stream x y (+ x 10) (+ y 10) :ink +yellow+)))

(define-table-cell-test simple-surrounding "create record, query size, surround with rect"
  (let ((record (with-new-output-record (stream)
		  (write-string "Foobar" stream))))
    (multiple-value-bind (left top right bottom)
	(bounding-rectangle* record)
      (multiple-value-bind (xoff yoff)
	  ;; damn, we have to convert because we want to draw in table-cell relative
	  ;; coordinates.
	  (ci::convert-from-absolute-to-relative-coordinates 
	    stream (output-record-parent record))
	(decf left xoff)
	(decf right xoff)
	(decf top yoff)
	(decf bottom yoff))
      (draw-rectangle* stream left top right bottom :filled nil :ink +cyan+))))

(define-table-cell-test simple-graphics-surrounding
			"create graphics-record, query size, surround with rect"
  (let ((record (with-new-output-record (stream)
		  (draw-circle* stream 10 10 10 :ink +blue+))))
    (multiple-value-bind (left top right bottom)
	(bounding-rectangle* record)
      (multiple-value-bind (xoff yoff)
	  ;; damn, we have to convert because we want to draw in table-cell relative
	  ;; coordinates.
	  (ci::convert-from-absolute-to-relative-coordinates 
	    stream (output-record-parent record))
	(decf left xoff)
	(decf right xoff)
	(decf top yoff)
	(decf bottom yoff))
      (draw-rectangle* stream left top right bottom :filled nil :ink +cyan+))))

(define-table-cell-test set-cursor-position* "move the cursor, output some text"
  (stream-set-cursor-position* stream 50 50)
  (write-string ">Wally<" stream))

(define-clim-test table-tests (window &optional documentation-window)
  (let ((initially t))
    (dotimes (i (length *table-graphics-tests*))
      (unless initially
	(fresh-line documentation-window)
	(write-string "Click on display for next test" documentation-window)
	(force-output documentation-window)
	(read-gesture :stream window :timeout 60))
      (setq initially nil)
      (window-clear documentation-window)
      (table-graphics-test window documentation-window i)
      (fresh-line documentation-window)
      (write-string "Click on display to run the test as a column" documentation-window)
      (force-output documentation-window)
      (read-gesture :stream window)
      (table-graphics-test-column window documentation-window i))))
