;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; TEST Suites for SILICA
;;; Copyright (c) 1988, 1989 by Xerox Corporation.  All rights reserved.
;;;

(in-package "SILICA")

;;; Many of the tests are automatically invoked by calling test-silica.
;;; However there are a number of other tests that aren't.  These include:
;;;
;;; (test-standard-sheet)
;;; --> <sheet>
;;; (test-sheet-events *)
;;;
;;;

;;;
;;; (test-pixels)
;;;

(defvar *silica-tests* nil)

(defun test-silica ()
  (format t "----- Silica Test Suite -----~%")
  (dolist (test *silica-tests*)
    (format t "~%----- ~a -----~%" test)
    (funcall test))
  (format t "~%----- End Test Suite -----~%"))

(defmacro def-silica-test (name args &body body)
  (with-gensyms (made-sheets)
    `(progn
       (unless (member ',name *silica-tests*)
	 (setq *silica-tests* (nconc *silica-tests* (list ',name))))
       (defun ,name ,args
	 (let (,made-sheets)
	   (macrolet ((ms (&rest args &key keep &allow-other-keys)
			(remf args keep)
			`(let ((s
				(make-sheet 
				 :supers
				 '(mute-repainter
				   ,@(getf args :supers))
				 ,@args
				 :parent (find-graft)
				 :transformation
				 (make-translation-transformation 100 100)
				 :region (make-rectangle* 0 0 200 200)
				 :medium t
				 :input-contract-class
				 'mute-input-contract)))
			   ,@(unless keep
			       `((push s ,',made-sheets)))
			   (install-settings s :title "A Silica Test"
					     :left 100
					     :bottom 100)
			   s))
		      (enable (&rest sheets)
			`(progn ,@(with-collection
				      (dolist (sheet sheets)
					(collect `(enable-sheet ,sheet))))
				(sleep 1))))

	     (time ,@body)
	     (sleep 3)
	     (dolist (sheet ,made-sheets)
	       (disown-child (sheet-parent sheet) sheet))))))))

(defun draw-frob (medium x1 y1 x2 y2)
  (draw-line* medium x1 y1 x1 y2 :line-thickness 3 :ink *black*)
  (draw-line* medium x1 y1 x2 y1 :line-thickness 3)
  (draw-line medium (make-point x1 y2) (make-point x2 y2) :line-thickness 3)
  (draw-line medium (make-point x2 y1) (make-point x2 y2) :line-thickness 3)
  (draw-line medium (make-point x1 y1) (make-point x2 y2))
  (draw-line medium (make-point x1 y2) (make-point x2 y1))
  (medium-finish-output medium))


;;;
;;; Basic Test
;;;

(def-silica-test basic-test ()
  (let* ((sheet (ms)) w h)
    (enable sheet)
    (setq w (bounding-rectangle-width sheet))
    (setq h (bounding-rectangle-height sheet))
    (using-clim-medium (medium sheet)
      (draw-frob medium (/ w 4) (/ h 4) (* 3 (/ w 4)) (* 3 (/ h 4))))))

(def-silica-test test-using-dm ()
  (let* ((sheet (ms :output-contract-class 'providing-output-contract)) w h)
    (enable sheet)
    (setq w (bounding-rectangle-width sheet))
    (setq h (bounding-rectangle-height sheet))
    (dotimes (i 1000)
      (using-display-medium (medium :clim sheet)
	(medium-transformation medium)))))

(def-silica-test text-test ()
  (let* ((sheet (ms :output-contract-class 'providing-output-contract))
	 (ts0 (make-text-style :fix :roman :normal))
	 (ts1 (make-text-style :fix :bold :normal))
	 (ts2 (make-text-style :fix :italic :normal))
	 (ts3 (make-text-style :serif :italic :large))
	 (ts4 (make-text-style :sans-serif :bold :small))
	 w h)
    (enable sheet)
    (setq w (bounding-rectangle-width sheet))
    (setq h (bounding-rectangle-height sheet))
    (dotimes (i 1)
      (using-display-medium (medium :clim sheet)
	(draw-text* medium "normal" 20 20 :text-style ts0)
	(draw-text* medium "bold" 20 50 :text-style ts1)
	(draw-text* medium "italic" 20 80 :text-style ts2)
	(draw-text* medium "italic" 20 110 :text-style ts3)
	(draw-text* medium "bold" 20 140 :text-style ts4)
	(medium-finish-output medium)))))

(def-silica-test rect-test ()
  (let* ((sheet (ms)))
    (enable sheet)
    (setq *sheet* sheet)
    (with-bounding-rectangle* (minx miny maxx maxy) sheet
      (using-clim-medium (medium sheet)
	(draw-rectangle* medium minx miny (1- maxx) (1- maxy))
	(medium-finish-output medium)))))

;;;
;;; Pixmap Tests
;;;

(def-silica-test pixmap-test ()
  (let* ((sheet (ms))
	 (medium (sheet-medium sheet))
	 w h pm)
    
    (enable sheet)
    (setq w (bounding-rectangle-width sheet))
    (setq h (bounding-rectangle-height sheet))
    (draw-frob medium (/ w 8) (/ h 8) (* 3 (/ w 8)) (* 3 (/ h 8)))
    (sleep 1)
    (setq pm (read-pixmap* sheet (/ w 8) (/ h 8) 
			   (+ (* 2 (/ w 8)) 3) (+ (* 2 (/ h 8)) 3)))
    
    (copy-area medium (* 5 (/ w 8)) (/ h 8)
	       pm 0 0 (pixmap-width pm) (pixmap-height pm))
    (copy-area medium (/ w 8) (* 5 (/ h 8))
	       pm 0 0 (pixmap-width pm) (pixmap-height pm))
    (copy-area medium (* 5 (/ w 8)) (* 5 (/ h 8))
	       pm 0 0 (pixmap-width pm) (pixmap-height pm))
    (medium-finish-output medium)))

;;;
;;; Standard Sheet Test
;;;

(defun test-standard-sheet ()
  (let* ((sheet
          (make-standard-sheet :parent (find-graft) 
                               :transformation (make-translation-transformation
						100 100)
                               :region (make-rectangle* 0 0 200 200)))
         (w (bounding-rectangle-width sheet))
         (h (bounding-rectangle-height sheet))
         (x1 (/ w 4))
         (y1 (/ h 4))
         (x2 (* 3 (/ w 4)))
         (y2 (* 3 (/ h 4))))
    (enable-sheet sheet)
    (sleep 3)
    (draw-line sheet (make-point x1 y1) (make-point x1 y2))
    (draw-line sheet (make-point x1 y2) (make-point x2 y2))
    (draw-line sheet (make-point x2 y2) (make-point x2 y1))
    (draw-line sheet (make-point x2 y1) (make-point x1 y1))
    (draw-line* sheet x1 y1 x2 y2)
    (draw-line* sheet x1 y2 x2 y1)
    (medium-finish-output sheet)
    sheet))

(defun test-sheet-events (sheet)
  (let (event)
    (loop (setq event (get-next-event sheet :last-event event
				      :timeout 10))
	  (when event 
	    (typecase event
	      (motion-event
	       (format t "Event: ~s ~s (~d,~d) (~d,~d) ~%" (event-type event)
		       (event-direction event)
		       (event-x event) (event-y event)
		       (event-native-x event) (event-native-y event)))
	      (otherwise
	       (format t "Recieved: ~s~%" (event-type event))))
	    (force-output)))))

;;;
;;; Test Mirror
;;;

(defun test-mirrors ()
  (process-mirror-test-events (create-test-sheet)))

(defun create-test-sheet ()
  (let* ((sheet
          (make-standard-sheet :parent (find-graft) 
                               :transformation 
			       (make-translation-transformation 100 100)
                               :region (make-rectangle* 0 0 500 500)))
         (w (bounding-rectangle-width sheet))
         (h (bounding-rectangle-height sheet))
         (x1 (/ w 4))
         (y1 (/ h 4))
         (x2 (* 3 (/ w 4)))
         (y2 (* 3 (/ h 4))))
    (enable-sheet sheet)
    (sleep 3)
    (draw-line sheet (make-point x1 y1) (make-point x1 y2))
    (draw-line sheet (make-point x1 y2) (make-point x2 y2))
    (draw-line sheet (make-point x2 y2) (make-point x2 y1))
    (draw-line sheet (make-point x2 y1) (make-point x1 y1))
    (draw-line* sheet x1 y1 x2 y2)
    (draw-line* sheet x1 y2 x2 y1)
    (medium-finish-output sheet)

    sheet))

(defun process-mirror-test-events (sheet)
  (make-process 
   #'(lambda ()
       (let (event)
	 (loop (setq event (get-next-event sheet :last-event event
					   :timeout 10))
	       (when event 
		 (typecase event
		   (repaint-event
		    (let ((sheet (event-sheet event)))
		      (fill-sheet 
		       sheet
		       (cond ((graftp (sheet-parent sheet)) +green+)
			     ((sheet-mirror sheet) +red+)
			     (t +blue+)))))
		   (button-press-event
		    (let ((sheet (event-sheet event)))
		      (if (eq (event-button event) :middle)
			  (dolist (child (sheet-children sheet))
			    (disown-child sheet child))
			  (add-child-sheet 
			   sheet (event-x event) (event-y event)
			   (eq (event-button event) :left))))))))))
   :name "Mirror Test Events")
  sheet)

(defun add-child-sheet (sheet x y &optional (mirror-p nil))
  (let ((child (make-standard-sheet 
		:parent sheet
		:event-queue (sheet-event-queue sheet)
		:transformation (make-translation-transformation x y)
		:region (if mirror-p (make-rectangle* 0 0 50 50)
			    (make-rectangle* 0 0 200 200)))))
      
    (when mirror-p (w::realize-mirror (port child) child))
    (enable-sheet child)
    (sleep 3)
    (w::medium-finish-output (sheet-medium child))
    (unless mirror-p (queue-repaint child +everywhere+))
    child))

(defun fill-sheet (s &optional (ink +red+))
  (draw-rectangle* s 0 0 1000 1000 :filled t :ink ink)
  (medium-finish-output s))

(defun show-sheets (sheet &key from-root)
  (when from-root
    (setq sheet (do ((sheet sheet (sheet-parent sheet)))
		    ((or (null (sheet-parent sheet))
			 (typep (sheet-parent sheet) 
				'graft))
		     sheet))))
  (labels ((show-one (sheet depth)
	     (lisp:format t "~&~vT~A ~A ~Dx~D" 
			  (* depth 2)
			  sheet
			  (sheet-transformation sheet)
			  (bounding-rectangle-width (sheet-region sheet))
			  (bounding-rectangle-height (sheet-region sheet)))
	     (dolist (c (sheet-children sheet))
	       (show-one c (1+ depth)))))
    (show-one sheet 0)))

(defun describe-mirror (sheet)
  (let ((mirror (sheet-mirror sheet)))
    (if mirror
	(lisp:format t "~&Mirror ~S: x: ~D, y: ~D, width: ~D, height: ~D"
		     mirror
		     (xlib:drawable-x mirror)
		     (xlib:drawable-y mirror)
		     (xlib:drawable-width mirror)
		     (xlib:drawable-height mirror))
	(write-string "No mirror"))))
		     

;;;
;;; Text Pixels
;;;
;;; Tests whether rectangles and filled rectangles get painted correctly in
;;; both of the basic orientations. 
;;; 
	  
(defun test-pixels ()
  (let* ((p1 
	  (make-standard-sheet :parent (find-graft) 
                               :transformation (make-translation-transformation
						100 100)
                               :region (make-rectangle* 0 0 100 100)))
	 (s1
	  (make-standard-sheet :parent p1
                               :transformation (make-translation-transformation
						10 10)
                               :region (make-rectangle* 0 0 4 4)
			       ))
	 (s2
          (make-standard-sheet :parent p1
                               :transformation (make-translation-transformation
						20 20)
                               :region (make-rectangle* 0 0 4 4)))
	 (p2 
	  (make-standard-sheet :parent (find-graft :origin :nw) 
                               :transformation (make-translation-transformation
						100 100)
                               :region (make-rectangle* 0 0 100 100)))
	 
	 (s3
          (make-standard-sheet :parent p2 
                               :transformation (make-translation-transformation
						10 10)
                               :region (make-rectangle* 0 0 4 4)))
	 (s4
          (make-standard-sheet :parent p2 
                               :transformation (make-translation-transformation
						20 20)
                               :region (make-rectangle* 0 0 4 4))))

    (dolist (p  (list p1 p2))
      (install-settings p :plain t)
      (enable-sheet p))
    (enable-sheet s1)
    (enable-sheet s2)
    (enable-sheet s3)
    (enable-sheet s4)
    (sleep 3)
    (draw-rectangle* s1 0 0 3 3 :filled nil)
    (draw-rectangle* s2 0 0 3 3 :filled t)
    (draw-rectangle* s3 0 0 3 3 :filled nil)
    (draw-rectangle* s4 0 0 3 3 :filled t)
    
    (draw-rectangle* p1 8 8 15 15 :filled nil)
    (draw-rectangle* p1 18 18 25 25 :filled nil)
    (draw-rectangle* p2 8 8 15 15 :filled nil)
    (draw-rectangle* p2 18 18 25 25 :filled nil)
    
    (medium-finish-output p1)
    (list p1 p2)))     


#||
(defvar *pm*)
  
(def-silica-test pixmap-track-test ()
  (let* ((sheet (ms :input-contract-class 'invoking-input-contract
		    :supers '(my-input-handler)
		    :keep t))
	 (medium (sheet-medium sheet))
	 w h)
    
    (enable sheet)
    (setq w (bounding-rectangle-width sheet))
    (setq h (bounding-rectangle-height sheet))
    (draw-frob medium (/ w 8) (/ h 8) (* 3 (/ w 8)) (* 3 (/ h 8)))
    (sleep 1)
    (setq *pm* (read-pixmap* sheet (/ w 8) (/ h 8) 
			     (+ (* 2 (/ w 8)) 3) (+ (* 2 (/ h 8)) 3)))
    (medium-finish-output medium)))

(defclass my-input-handler (input-handler)
    ())

(defmethod button-press ((pane my-input-handler) button-name 
			 &key &allow-other-keys)
  (if (eq :middle button-name)
      (track-position pane)
      (call-next-method)))

(defmethod track-position ((sheet my-input-handler))
  (let* ((width (bounding-rectangle-width sheet))
	 (height (bounding-rectangle-height sheet))
	 (lastx nil)
	 (lasty nil)
	 (save-pixmap (make-pixmap :width width :height height)))
    (using-display-medium (medium :clim sheet)
      (loop
       (multiple-value-bind (x y state) (poll-pointer sheet)
	 (when (or (null lastx) (not (= x lastx)) (not (= y lasty)))
	   (when lastx
	     ;; Restore graphics at previous position
	     (copy-area medium lastx lasty save-pixmap 0 0 width height))
	   ;; Save graphics at new position
	   (setq save-pixmap 
		 (read-pixmap* sheet x y width height save-pixmap))
	   ;; Paint pixmap being tracked
	   (copy-area medium x y pixmap 0 0 width height)
	   (setq lastx x lasty y))
	 (when (state-match-p state (not :middle))
	   ;; Exit tracking when the mouse button is released.
	   (move-sheet* sheet lastx lasty)
	   (return t)))))))

||#
#||
;;;
;;; Testing Color Pixmap
(defun foo ()
  (let ((tpm (make-pixmap :width 16 :height 16)))
    (setf (slot-value tpm 'data) (make-array 256
					     :element-type '(unsigned-byte 8)
					     :initial-element 2)
	  (slot-value tpm 'format) :dws
	  (slot-value tpm 'color-table)
	  (list +white+ +aquamarine+ +goldenrod+ +black+)
	  (slot-value tpm 'color-count) 4)
    tpm))

(defun bar (&key (host "holmes") (screen-id 0))
  (let* ((p (make-port :host host :display-id 0 :screen-id screen-id))
	 (s (make-sheet :parent (find-graft :port p)))
	 (pm (foo)))
    (enable s)
    (port-force-output p)
    (using-clim-medium (medium s)
      (copy-area medium 0 0 pm 0 0 16 16)
      (medium-finish-output medium))))

(defun bar (pm)
  (let* ((p (find-port))
	 (s (make-sheet :parent (find-graft :port p))))
    (enable-sheet s)
    (port-force-output p)
    (using-display-medium (dm :clim s)
      (copy-area dm 0 0 pm 0 0 16 16)
      (medium-finish-output dm))))

||#

(def-silica-test origin-test ()
  (let* ((r (make-rectangle* 0 0 200 200))
	 (xf (make-translation-transformation 100 100))
	 (xf-flip +identity-transformation+)
	 (sheet-sw (ms :region r :transformation xf))
	 (sheet-nw (ms :parent (find-graft :port (find-port) :origin :nw)
		       :region r :transformation xf))
	 w h)
    (enable sheet-sw sheet-nw)
    (setq w (bounding-rectangle-width sheet-sw))
    (setq h (bounding-rectangle-height sheet-sw))
    (sleep 1)
    (draw-frob (sheet-medium sheet-sw) (/ w 8) (/ h 8)
	       (* 3 (/ w 8)) (* 3 (/ h 8)))
    (draw-frob (sheet-medium sheet-nw) (/ w 8) (/ h 8)
	       (* 3 (/ w 8)) (* 3 (/ h 8)))))

(def-silica-test line-test ()
  (let* ((sheet (ms))
	 (medium (sheet-medium sheet))
	 w h x1 x2 y1 y2 p1 p2)
    
    (enable sheet)
    (setq w (bounding-rectangle-width sheet))
    (setq h (bounding-rectangle-height sheet))
    (setq x1 (/ w 4))
    (setq y1 (/ h 4))
    (setq x2 (* 3 (/ w 4)))
    (setq y2 (* 3 (/ h 4)))
    (setq p1 (make-point x1 y1))
    (setq p2 (make-point x2 y2))

    (format t "Timing CLIM draw-line~%")
    (time
     (progn
       (dotimes (n 1000)
	 (draw-line medium p1 p2))
       (medium-finish-output medium)))
    (format t "Timing CLIM draw-line*~%")
    (time
     (progn
       (dotimes (n 1000)
	 (draw-line* medium x1 y1 x2 y2))
       (medium-finish-output medium)))
    
    ))



#|| OLD STUFF to convert

(defun windows ()
  (let ((sheet (make-sheet :parent (find-graft)
			   :region (make-rectangle* 0 0 500 500)))
	(rect (make-rectangle* 0 0 50 50)))
    
    #+wait
    (dotimes (j 10)
      (dotimes (i 10)
	(make-sheet
	 :parent sheet
	 :region rect
	 :transformation (make-translation-transformation (* i 50) (* j 50)))))
    (enable-sheet sheet)))


||#

