;;; -*- Syntax: Common-lisp; Base: 10; Mode: LISP; Package: CLIM-DEMO -*-

(in-package "CLIM-DEMO")

"Copyright (c) 1990, International Lisp Associates, Inc.  All rights reserved."

;;;
;;; Function to make and return a frame with a single scrollable pane
;;;

(defun clim-scroller (&key (server-path *default-server-path*)
			   (hs 600) (vs #+Imach 200 #-Imach 400))
  (let ((framem (find-frame-manager :server-path server-path))
	(frame (make-frame 'frame))
	stream)
    ;; Enter the context in which to define the contents of this frame
    (with-look-and-feel-realization (framem frame)
      ;; declare what the frame's child will be
      (setf (frame-pane frame)			;; Frames have only one child
	    (scrolling				;; Make a scrolling pane
		(:subtransformationp t		;; Required for scrolling panes
	      :hs+ +fill+			;; If the frame gets bigger in the
	      :vs+ +fill+			;;   horizontal or vertical dimensions
						;;   expand to fill all available space
	      :hs hs :vs vs)			;; Initial h. and v. sizes
	      ;; A form to generate the contents
	      (setq stream (make-pane 'extended-stream-pane))))
      (adopt-frame framem frame)
      ;; Make the frame visible
      (enable-frame frame)
      (values stream frame))))

;;;***TEXT STYLES***
;;;***FORMATTED OUTPUT***
;;;
;;; Text styles
;;;

;;; Run this function to see the various text styles displayed
(defun show-text-styles (stream)
  (fresh-line stream)
  (let ((sizes '(:small :normal :large))
	(families '(:fix :serif :sans-serif))
	(faces '(:roman :bold :italic (:bold :italic))))
    ;; The enclosed code will do output to be rendered in tabular form
    (formatting-table (stream)
      ;; The enclosed code will constitute a row of my table
      (formatting-row (stream)
	(formatting-cell (stream)
	  (declare (ignore stream))
	  ;; Leave a blank in the upper-left-hand corner
	  )
	;; This first row will contain column headings
	(dolist (face faces)
	  (formatting-cell (stream)
	    (format stream "~S" face))))
      (dolist (family families)
	;; Make one row for each family
	(formatting-row (stream)
	  ;; The first element of the row is the family itself
	  (formatting-cell (stream)
	    (format stream "~S" family))
	  (dolist (face faces)
	    ;; The then there is one additional element per face
	    (formatting-cell (stream)
	      ;; Each cell will be several lines high, showing all the sizes
	      (dolist (size sizes)
		(with-text-style ((list family face size) stream)
		  (fresh-line stream)
		  (format stream "~S"  size)))))))))
  (force-output stream))

;;; ***TRANSFORMATIONS***
;;;
;;; Translation, rotation
;;;

;;;
;;; Currently, rotation can only be performed about the origin.  This macro allows
;;; you to rotate about any arbitrary point by first translating the point you choose
;;; to the origin, then doing the rotation, then translating it back.

(defmacro with-rotation-about ((medium angle x y) &body body)
  `(with-translation (,medium ,x ,y)
     (with-rotation (,medium ,angle)
       (with-translation (,medium (- ,x) (- ,y))
	 ,@body))))


;;;
;;; The following pair of functions demonstrate the significance of the order
;;; in which transformations are composed.  The funtion ROLL below first rotates 
;;; and then translates to create the impression of rolling.  The function SWIRL is
;;; identical except for the order in which the transformations are applied; the result
;;; is very different.

(defun roll (stream)
  (window-clear stream)
  (labels ((draw ()
	     ;; Avoid do output recording in an attempt to get better performance
	     (with-output-recording-options (stream :draw-p t :record-p nil)
	       ;; Draw
	       (draw-rectangle* stream 0 0 40 40 :ink +flipping-ink+)
	       (force-output stream)	;Let the user see something.
	       ;; and immediately erase
	       (draw-rectangle* stream 0 0 40 40 :ink +flipping-ink+))))
    (let ((hdist 200)
	  (rotation (* 4 pi))
	  (nsteps 40))
      (dotimes (i nsteps)
	(with-translation (stream (* hdist (/ i nsteps)) 0)
	  (with-rotation-about (stream (* -1 rotation (/ i nsteps)) 20 20)
	    (draw)))))))

(defun swirl (stream)
  (window-clear stream)
  (labels ((draw ()
	     ;; Avoid do output recording in an attempt to get better performance
	     (with-output-recording-options (stream :draw-p t :record-p nil)
	       ;; Draw
	       (draw-rectangle* stream 0 0 40 40 :ink +flipping-ink+)
	       (force-output stream)	;Let the user see something.
	       ;; and immediately erase
	       (draw-rectangle* stream 0 0 40 40 :ink +flipping-ink+))))
    (let ((hdist 200)
	  (rotation (* 4 pi))
	  (nsteps 40))
      (dotimes (i nsteps)
	(with-rotation-about (stream (* -1 rotation (/ i nsteps)) 20 20)
	  (with-translation (stream (* hdist (/ i nsteps)) 0)
	    (draw)))))))

;;;
;;; Transform example
;;;

;;; This macro establishes a context in which the coordinate system is the standard
;;; Cartesian one, with (0,0) in the lower left hand corner.
(defmacro with-cartesian-coords ((&optional stream height)
				  &body body)
  (clim-utils:default-output-stream stream)
  `(with-cartesian-coords-1 ,stream ,height
			     (clim-utils:named-continuation with-room-for-graphics () ,@body)))

(defun with-cartesian-coords-1 (stream height continuation)
  (unless height
    (setq height (rectangle-height (sheet-region stream))))
  (let* ((reflection-height (/ height 2.0))
	 (transform
	   ;; Make reflection about a horizontal line at the middle of the desired
	   ;; region
	   (make-reflection-transformation* 0 reflection-height
					    1 reflection-height)))
    (with-drawing-options (stream :transformation transform)
      (funcall continuation))))

(defun enellipse (entity stream)
  (multiple-value-bind (left top right bottom)
      (entity-edges entity)
    (let* ((fudge 5)
	   (x-radius (/ (- right left) 2.0))
	   (y-radius (/ (- bottom top) 2.0)))
      (draw-ellipse* 
       stream
       (+ left x-radius) (+ top y-radius) 
       (+ fudge x-radius) 0 0 (+ fudge y-radius)
       :filled nil)))
  (force-output stream))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;***INCREMENTAL REDISPLAY***

(defun non-incremental-time-drawer (stream)
  (let (second minute hour date month year day daylight-savings-p time-zone)
    (multiple-value-setq
      (second minute hour date month year day daylight-savings-p time-zone)
      (get-decoded-time))
    (write-string (nth day
		       '("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun"))
		  stream)
    (write-string " " stream)
    (write-string (nth (1- month)
		       '("Jan" "Feb" "Mar" "Apr" "May" "June"
			 "July" "Aug" "Sept" "Oct" "Nov" "Dec"))
		  stream)
    (format stream " ~2D" date)
    (format stream " ~2D:" hour)
    (format stream "~2,'0D:" minute)
    (format stream "~2,'0D " second)))

;;; Run this function to see the non-incremental redisplay of the time
(defun non-incremental-time-loop (stream)
  (loop 
    (window-clear stream)
    (non-incremental-time-drawer stream)
    (force-output stream)
    (sleep 1)))

;;; Example of UPDATING-OUTPUT.
(defun time-drawer (stream)
  (let (second minute hour date month year day daylight-savings-p time-zone)
    ;; Return an output record that can be REDISPLAYED
    (updating-output (stream)
      (multiple-value-setq
	  (second minute hour date month year day daylight-savings-p time-zone)
	(get-decoded-time))
      (updating-output (stream :unique-id 'day :cache-value day)
	(write-string (nth day
			   '("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun"))
		      stream)
	  (write-string " " stream))
      (updating-output (stream :unique-id 'month :cache-value month)
	(write-string (nth (1- month)
			   '("Jan" "Feb" "Mar" "Apr" "May" "June"
			     "July" "Aug" "Sept" "Oct" "Nov" "Dec"))
		      stream))
      (updating-output (stream :unique-id 'date :cache-value date)
	(format stream " ~2D" date))
      (updating-output (stream :unique-id 'hour :cache-value hour)
	(format stream " ~2D:" hour))
      (updating-output (stream :unique-id 'minute :cache-value minute)
	(format stream "~2,'0D:" minute))
      (updating-output (stream :unique-id 'second :cache-value second)
	(format stream "~2,'0D " second)))))

;;;
;;; Run this function to see the incremental redisplay of the time
;;;
(defun time-loop (stream)
  (window-clear stream)
    (with-text-size (:large stream)
      (let ((output-record (time-drawer stream)))
	(loop (redisplay output-record stream)
	  (force-output stream)
	  (sleep 1)))))


;;;
;;; Use of ACCEPTING-VALUES and UPDATING-OUTPUT
;;;
(defun shape-chooser (stream)
  (let (shape size filled-p)
    (fresh-line stream)
    (setq shape (accept '(alist-member :alist (("Square" :value Square)
					       ("Circle" :value Circle)
					       ("Triangle" :value Triangle)))
			:stream stream :Prompt "Shape" :default 'Square))
    (stream-clear-input stream)
    (fresh-line stream)
    (setq size (accept '(alist-member :alist (("Large" :value 15)
					      ("Medium" :value 10)
					      ("Small" :value 5)))
		       :default 10 :prompt "Size" :stream stream))
    (stream-clear-input stream)
    (fresh-line stream)
    (setq filled-p (accept 'boolean  :default t :prompt "Filled" :stream stream))
    (multiple-value-bind (x y)
	(stream-cursor-position* stream)
      (let ((center-x (+ x 50))
	    (center-y (+ y 50)))
	;; A change in any of the parameters that would result in a different drawing
	;; cause this output record to be noticed as changed.
	(updating-output (stream
			   ;; Use a :UNIQUE-ID that's unique within the UPDATING-OUTPUT
			   ;; provided by ACCEPTING-VALUES.  That will improve performance.
			   :unique-id 'shape
			   ;; A :CACHE-VALUE is chosen that will have a different value
			   ;; whenever the the output would change.
			   :cache-value (list shape size filled-p) :cache-test #'tree-equal)
	  (case shape
	    (circle
	      (draw-circle* stream center-x center-y size :filled filled-p))
	    (square
	      (draw-rectangle* stream (- center-x size) (- center-y size)
			       (+ center-x size) (+ center-y size)
			       :filled filled-p))
	    (triangle
	      (draw-polygon*  stream
			      (list (- center-x size) (+ center-y size)
				    center-x (- center-y size)
				    (+ center-x size) (+ center-y size))
			       :filled filled-p))))))
    (values shape size filled-p)))

;;;
;;; The same code has a dynamic behavior when enclosed in a call to ACCEPTING-VALUES
;;;

(defun avv-shape-chooser (stream)
  (window-clear stream)
  (accepting-values (stream)
    (shape-chooser stream)))


;;;***PRESENTATION TYPES***
;;;
;;; Presentation types
;;; 

;;;
;;; Uses of :ABBREVIATION-FOR
;;;

;;; From the system:
;(define-presentation-type boolean ()
;  :abbreviation-for '(alist-member :alist (("Yes" :value t) ("No" :value nil))))

(define-presentation-type teenage ()
  ;; Let the new type "teenage" inherit the attributes of the type (integer 13 19)
  :abbreviation-for '(integer 13 19)
  ;; the describer is run when the user is prompted for input
  :describer ((stream)
	      ;; Without this describer the default prompt would ask me for
	      ;; an integer between 13 and 19.  This way I express myself at 
	      ;; a higher level
	      (write-string "a teenager's age" stream)))

;;; A presentation type's parser defines how a datum of that type is parsed, 
;;; typically by reading from a stream of characters, most often that the user is
;;; typing.  The printer of a presentation type defines how a datum of that type should 
;;; be textually presented to the user

;;; Here's a presentation type with a simple parser and printer
;;;

(define-presentation-type palindrome ()
  :parser ((stream &key default &aux token)
	   (loop
	     (setq token (read-token stream))
	     (when (and (string-equal token "") default)
	       (setq token default))
	     (let ((length (length token)))
	       (when
		 (string-equal
		   (subseq token 0 (floor length 2))
		   (reverse 
		     (subseq token (ceiling length 2) length)))
		 (return token))
	       (input-not-of-required-type stream token 'palindrome))))
  :printer ((object stream &key acceptably) acceptably
	    (write-string object stream)))

;;; Here's an example of a presentation type with a non-trivial parser
;;; Use of manual input parsing, invalid input handling, and 
;;; PRESENTATION-REPLACE-INPUT
;;;
(define-presentation-type monogram ()
  :parser ((stream &key default &aux token)
	   (loop
	     ;; Remember where the input cursor is
	     (let ((ip (input-position stream)))
	       (setq token (read-token stream))
	       (when (and (string-equal token "") default)
		 (setq token default))
	       (when (and (<= (length token) 3)
			  (every #'alpha-char-p token))
		 ;; Input is valid as a monogram
		 ;; Invoke our printer to replace what the user has typed,
		 ;; giving him the feedback that his input was acceptable
		 ;; This is optional
		 (setf token (string-upcase token))
		 (presentation-replace-input stream token 'monogram
					     :buffer-start ip)

		 ;; return from loop with a valid input
		 (return token))
	       ;; Input is not valid.  Call standard function to indicate this
	       ;; and loop to try again
	       (input-not-of-required-type stream token 'monogram))))
  :printer ((ob stream &key acceptably) acceptably
	    (write-string (string-upcase ob) stream)))


;;;
;;; Use of COMPLETING-FROM-SUGGESTIONS and presentation arguments
;;;
(defvar *months*
	'("January" "February" "March" "April"
	  "May" "June" "July" "August" "September"
	  "October" "November" "December"))

;;; Only months with the letter R in their name are good for eating oysters.
(define-presentation-type month (&optional only-oyster-months)
  :parser ((stream &key default &aux (month-n 1))
	   ;; This macro provides completion behavior.  When prompted for a month,
	   ;; if the user types Control-Shift-? or the Help key, he'll be presented
	   ;; with a list of possibilities from which to choose
	   (completing-from-suggestions (stream)
	     (dolist (month *months*)
	       (when (or (not only-oyster-months)
			 (find #\r month))
		 (suggest month month-n))
	       (incf month-n))))

  ;; How to determine if one presentation type based on MONTH is a subtype
  ;; of another.  This is to ensure that:
  ;; (presentation-subtypep 'month 'month) => T
  ;; (presentation-subtypep 'month '(month t)) => NIL
  ;; (presentation-subtypep '(month t) 'month) => T
  :data-args-comparator ((subargs superargs)
			 (or (null (first superargs))
			     (first subargs)))

  ;; How to determine if a given month is valid, based on the presentation type
  ;; This is to avoid making "May" mouse-senstive, for example, when the input
  ;; context is (MONTH T)
  :object-validator ((object)
		     (or (not only-oyster-months)
			 (not (null (find #\r (elt *months* object))))))
  :printer ((object stream &key acceptably)
	    (write-string (elt *months* (1- object)) stream))
  :describer ((stream)
	      (if only-oyster-months
		  (write-string "a month good for eating oyters" stream)
		  (write-string "a month" stream))))


***PRESENTATION TRANSLATORS***

;;; --- shifted gestures don't work yet.
;;; Define the logical gesture :ROUND that corresponds to the physical gesture
;;; shift-Left.
(define-gesture :round :left '(:shift))

(define-presentation-translator rounded-float
     ;; Translates FLOATs to INTEGERs
     (float					; from-type
       integer					; to-type
       :documentation ((stream)
		       (write-string "Round up the floating point number" stream))
       :gesture :left)
   ;; This translator takes a single argument: the object
   ;; under the pointer
   (object)
  (values (round object)))

;;;
;;; To see how the above translator works, PRESENT a floating point number to a
;;; window, e.g. (PRESENT 98.6 'FLOAT :STREAM window) and then ACCEPT an integer.
;;; With the translator defined the float will be pointer sensitive.  Without the
;;; translator it won't be.

;;;
;;; To remove this translator you could do:
;;;
;;;
;;;(ci::remove-presentation-translator
;;;  (ci::find-presentation-translator-named 'rounded-float))

;;;***OUTPUT RECORDING***
;;;
;;; Simple output recording
;;;

(defun accept-baby-name (stream)
   (let ((*standard-output* stream)
	 (*query-io* stream))
     (format t "~&Some baby names:~%~%")
     (dolist (name '("John" "Mary" "Vaclav" "Ottakar"))
       (present name 'string)
       (fresh-line))
     (accept 'string :prompt "Choose the baby's name")))

;;; --- strange mouse senstivity
(defun accept-baby-name-as-rectangle (stream)
   (let ((*standard-output* stream)
	 (*query-io* stream)
	 (x 0))
     (format t "~&Some baby names:~%~%")
     (dolist (name '("John" "Mary" "Vaclav" "Ottakar"))
       (with-output-as-presentation (:object name :type 'string :stream *standard-output*)
	 (draw-rectangle* *standard-output* (+ x 0) 20 (+ x 20) 40 :filled nil))
       (incf x 30))
     (stream-set-cursor-position* stream 0 50)
     (accept 'string :prompt "Choose the baby's name")))


;;; ***APPLICATION FRAME***
;;; Example of defining an application.  This application has three commands, one of which
;;; is available via the command processor/interactor pane, the other two of which are
;;; available as menu items in the frame's standard command menu
(define-application-frame tester
			  ;; No super classes
			  ()
  ;; A slot to give us easy access later to the display pane
  ((display-pane :initform nil))
  ;; Endow the frame with two vertically stacked panes, one of them an interactor
  (:pane
    ;; No call with WITH-LOOK-AND-FEEL-REALIZATION is needed within DEFINE-APPLICATION-FRAME
    ;; Get a handle on the display-pane slot so that we can pass it to MAKE-CLIM-PANE,
    ;; which will set it to the pane created below
    (with-frame-slots (display-pane)		;frame
      (vertically ()
	;; Make a pane and set the slot display-pane to it
	(make-clim-pane (display-pane :hs 700 :vs 300))
	;; Make an interactor pane
	(make-clim-interactor (nil :hs 700 :vs 45)))))
  ;; Give us a macro for defining commands for this application, and created
  ;; a command table with the same name as this application
  (:command-definer t)
  ;; Give us a proper command loop (currently the the only valid value for
  ;; this option)
  (:top-level (clim-top-level))
  ;; Find the specified menu group and make its commands available in the standard
  ;; place in this frame
  (:menu-group tester-group))

;;; A command with four arguments
(define-tester-command com-draw-circle ((center-x 'number)
					(center-y 'number)
					(radius 'number)
					(filled 'boolean :default nil))
   (with-frame-slots (display-pane)
     (draw-circle* display-pane center-x center-y radius :filled filled)))

;;; A command with no arguments
(define-tester-command com-clear ()
   (with-frame-slots (display-pane)
     (window-clear display-pane)))

(define-tester-command com-quit ()
   ()
   (with-frame (frame)
     ;; Call to exit the application neatly
     (frame-exit frame)))

;;;
;;; Now that the commands are defined, how should they be accessible to the user?
;;;

;;; Make com-draw-circle available by typing "Draw Circle" to the command processor
(add-command-to-command-table "Draw Circle" 'com-draw-circle 'tester)

;;; Make the other two available as menu items
(define-menu-group tester-group
  (("Clear" :command '(com-clear)
    ;; Accelerators aren't yet supported in Release 0.9
    :keyboard-accelerator #\Control-L)
   ("Quit" :command '(com-quit))))


;;; Standard cliches for starting up and managing an application
(defvar *testers* nil)

(defun do-tester (&key (server-path *default-server-path*))
  (let ((framem (find-frame-manager :server-path server-path))
	frame)
    (setq frame (cdr (assoc framem *testers*)))
    (unless frame
      ;; Make a frame of the class we defined above
      (setq frame (make-frame 'tester))
      (adopt-frame framem frame)
      (push (cons framem frame) *testers*))
    (unwind-protect
	(progn
	  ;; Expose the frame
	  (enable-frame frame)
	  ;; Start the top-level loop
	  (run-frame-top-level frame))
      ;; Deexpose the frame
      (disable-frame frame))
    frame))

#|
;;;
;;; The source to DRAGGING-OUTPUT demonstrates the manipulation of output records,
;;; REPLAY, and TRACKING-MOUSE.

;;; Use the macro by enclosing it around some drawing code.  The coordinate system used
;;; will be relative to the position of the pointer.  The output done moves with the
;;; pointer until the user clicks left.  The ink used should be +FLIPPING-INK+
;;; E.g. (dragging-output (s) (draw-circle* s 0 0 20 :filled t :ink +flipping-ink+))

(defmacro dragging-output ((&optional stream) &body body)
  (default-output-stream stream)
  `(dragging-output-1 ,stream
		      (named-continuation dragging-output (,stream)
			,@body)))

(defun entity-center (entity)
  (multiple-value-bind (left top right bottom)
      (entity-edges entity)
    (values (floor (+ left right) 2)
	    (floor (+ top bottom) 2))))

(defun dragging-output-1 (stream continuation)
  (declare (dynamic-extent continuation))
  (let ((output-record
	  ;; Run the body collecting in an output record instead of actually drawing 
	  ;; it on the stream
	  (with-output-to-output-record (stream)
	    (funcall continuation stream))))
    (multiple-value-bind (x-off y-off)
	(entity-center output-record)
      (multiple-value-bind (x-start y-start)
	  (entity-position output-record)
	;; Let the x-off and y-off be the x and y distance from the center of the
	;; drawing to the upper-left-hand corner of the drawing
	(setq x-off (- x-off x-start) y-off (- y-off y-start)))
      ;; entity-center might reasonably return a rational, but we have to pass
      ;; integers as the offsets to REPLAY ('cause they will eventually figure into
      ;; CLX coordinate calculations)
      (setq x-off (round x-off)
	    y-off (round y-off))
      ;; Inhibit output recording, since we're going to be drawing this over and over
      ;; and we're not going to do anything with the output except erase it in an instant
      (with-output-recording-options (stream :record-p nil :draw-p t)
	(let (old-x old-y)
	  (block track
	    (tracking-pointer (stream)
	      (:pointer-motion (sheet x y)
	       ;; When the pointer moved
	       (when old-x
		 ;; and the drawing had been drawn before, draw it again, to erase it
		 ;; (at least it will be erased if the caller used +flipping-ink+
		 (replay output-record stream))
	       (when (eql sheet stream)
		 (setq old-x x old-y y)
		 ;; move the center of the drawing to the current position of the
		 ;; pointer
		 (set-output-record-position output-record (- x x-off) (- y y-off))
		 ;; and draw it again
		 (replay output-record stream)))
	      (:button-press (x y)
	       ;; When the user pushes a button . . .
	       ;;  . . . if the drawing had been drawn before, erase it
	       (when old-x (replay output-record stream))
	       ;; then return
	       (return-from track (values x y))))))))))
|#
