;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-TEST; Base: 10 -*-

(in-package :clim-test)

;;; This file is supposed to run some of the tutorial examples.

;;; First, let's copy some tutorial examples:

;;;***TEXT STYLES***
;;;***FORMATTED OUTPUT***
;;;
;;; Text styles
;;;

;;; Run this function to see the various text styles displayed
(defun show-text-styles (stream)
  (fresh-line stream)
  (let ((sizes '(:small :normal :large))
	(families '(:fix :serif :sans-serif))
	(faces '(:roman :bold :italic (:bold :italic))))
    ;; The enclosed code will do output to be rendered in tabular form
    (formatting-table (stream)
      ;; The enclosed code will constitute a row of my table
      (formatting-row (stream)
	(formatting-cell (stream)
	  (declare (ignore stream))
	  ;; Leave a blank in the upper-left-hand corner
	  )
	;; This first row will contain column headings
	(dolist (face faces)
	  (formatting-cell (stream)
	    (format stream "~S" face))))
      (dolist (family families)
	;; Make one row for each family
	(formatting-row (stream)
	  ;; The first element of the row is the family itself
	  (formatting-cell (stream)
	    (format stream "~S" family))
	  (dolist (face faces)
	    ;; The then there is one additional element per face
	    (formatting-cell (stream)
	      ;; Each cell will be several lines high, showing all the sizes
	      (dolist (size sizes)
		(with-text-style ((list family face size) stream)
		  (fresh-line stream)
		  (format stream "~S"  size)))))))))
  (force-output stream))

;;; ***TRANSFORMATIONS***
;;;
;;; Translation, rotation
;;;

;;;
;;; Currently, rotation can only be performed about the origin.  This macro allows
;;; you to rotate about any arbitrary point by first translating the point you choose
;;; to the origin, then doing the rotation, then translating it back.

;;; --- The above is no longer true; MAKE-ROTATION-TRANSFORMATION[*] now implements :ORIGIN
;;; correctly.  However, the macro is still useful.

(defmacro with-rotation-about ((medium angle x y) &body body)
  `(with-drawing-options (,medium :transformation
				  (make-rotation-transformation*
				   ,angle ,x ,y))
     ,@body))

;;;
;;; The following pair of functions demonstrate the significance of the order
;;; in which transformations are composed.  The funtion ROLL below first rotates 
;;; and then translates to create the impression of rolling.  The function SWIRL is
;;; identical except for the order in which the transformations are applied; the result
;;; is very different.

(defun roll (stream)
  (window-clear stream)
  (labels ((draw ()
	     ;; Avoid do output recording in an attempt to get better performance
	     (with-output-recording-options (stream :draw-p t :record-p nil)
	       ;; Draw
	       (draw-rectangle* stream 0 0 40 40 :ink +flipping-ink+)
	       (force-output stream)	;Let the user see something.
	       ;; and immediately erase
	       (draw-rectangle* stream 0 0 40 40 :ink +flipping-ink+))))
    (let ((hdist 200)
	  (rotation (* 4 pi))
	  (nsteps 40))
      (dotimes (i nsteps)
	(with-translation (stream (* hdist (/ i nsteps)) 0)
	  (with-rotation-about (stream (* -1 rotation (/ i nsteps)) 20 20)
	    (draw)))))))

(defun swirl (stream)
  (window-clear stream)
  (labels ((draw ()
	     ;; Avoid do output recording in an attempt to get better performance
	     (with-output-recording-options (stream :draw-p t :record-p nil)
	       ;; Draw
	       (draw-rectangle* stream 0 0 40 40 :ink +flipping-ink+)
	       (force-output stream)	;Let the user see something.
	       ;; and immediately erase
	       (draw-rectangle* stream 0 0 40 40 :ink +flipping-ink+))))
    (let ((hdist 200)
	  (rotation (* 4 pi))
	  (nsteps 40))
      (dotimes (i nsteps)
	(with-rotation-about (stream (* -1 rotation (/ i nsteps)) 20 20)
	  (with-translation (stream (* hdist (/ i nsteps)) 0)
	    (draw)))))))

;;;***INCREMENTAL REDISPLAY***

(defun non-incremental-time-drawer (stream)
  (let (second minute hour date month year day daylight-savings-p time-zone)
    (multiple-value-setq
      (second minute hour date month year day daylight-savings-p time-zone)
      (get-decoded-time))
    (write-string (nth day
		       '("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun"))
		  stream)
    (write-string " " stream)
    (write-string (nth (1- month)
		       '("Jan" "Feb" "Mar" "Apr" "May" "June"
			 "July" "Aug" "Sept" "Oct" "Nov" "Dec"))
		  stream)
    (format stream " ~2D" date)
    (format stream " ~2D:" hour)
    (format stream "~2,'0D:" minute)
    (format stream "~2,'0D " second)))

;;; Run this function to see the non-incremental redisplay of the time
(defun non-incremental-time-loop (stream &optional (ntimes most-positive-fixnum))
  (dotimes (n ntimes)
    #+Genera-release-8 (declare (ignore n))
    (window-clear stream)
    (non-incremental-time-drawer stream)
    (force-output stream)
    (sleep 1)))

;;; Example of UPDATING-OUTPUT.
(defun time-drawer (stream)
  (let (second minute hour date month year day daylight-savings-p time-zone)
    ;; Return an output record that can be REDISPLAYED
    (updating-output (stream)
      (multiple-value-setq
	  (second minute hour date month year day daylight-savings-p time-zone)
	(get-decoded-time))
      (updating-output (stream :unique-id 'day :cache-value day)
	(write-string (nth day
			   '("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun"))
		      stream)
	  (write-string " " stream))
      (updating-output (stream :unique-id 'month :cache-value month)
	(write-string (nth (1- month)
			   '("Jan" "Feb" "Mar" "Apr" "May" "June"
			     "July" "Aug" "Sept" "Oct" "Nov" "Dec"))
		      stream))
      (updating-output (stream :unique-id 'date :cache-value date)
	(format stream " ~2D" date))
      (updating-output (stream :unique-id 'hour :cache-value hour)
	(format stream " ~2D:" hour))
      (updating-output (stream :unique-id 'minute :cache-value minute)
	(format stream "~2,'0D:" minute))
      (updating-output (stream :unique-id 'second :cache-value second)
	(format stream "~2,'0D " second)))))

;;;
;;; Run this function to see the incremental redisplay of the time
;;;
(defun time-loop (stream &optional (ntimes most-positive-fixnum))
  (window-clear stream)
    (with-text-size (:large stream)
      (let ((output-record (time-drawer stream)))
	(dotimes (n ntimes)
	  #+Genera-release-8 (declare (ignore n))
	  (redisplay output-record stream)
	  (force-output stream)
	  (sleep 1)))))

;;;
;;; Use of ACCEPTING-VALUES and UPDATING-OUTPUT
;;;
(defun shape-chooser (stream)
  (let (shape size filled-p)
    (fresh-line stream)
    (setq shape (accept '(alist-member :alist (("Square" :value Square)
					       ("Circle" :value Circle)
					       ("Triangle" :value Triangle)))
			:stream stream :Prompt "Shape" :default 'Square))
    (stream-clear-input stream)
    (fresh-line stream)
    (setq size (accept '(alist-member :alist (("Large" :value 15)
					      ("Medium" :value 10)
					      ("Small" :value 5)))
		       :default 10 :prompt "Size" :stream stream))
    (stream-clear-input stream)
    (fresh-line stream)
    (setq filled-p (accept 'clim::boolean  :default t :prompt "Filled" :stream stream))
    (multiple-value-bind (x y)
	(stream-cursor-position* stream)
      (let ((center-x (+ x 50))
	    (center-y (+ y 50)))
	;; A change in any of the parameters that would result in a different drawing
	;; cause this output record to be noticed as changed.
	(updating-output (stream
			   ;; Use a :UNIQUE-ID that's unique within the UPDATING-OUTPUT
			   ;; provided by ACCEPTING-VALUES.  That will improve performance.
			   :unique-id 'shape
			   ;; A :CACHE-VALUE is chosen that will have a different value
			   ;; whenever the the output would change.
			   :cache-value (list shape size filled-p) :cache-test #'tree-equal)
	  (case shape
	    (circle
	      (draw-circle* stream center-x center-y size :filled filled-p))
	    (square
	      (draw-rectangle* stream (- center-x size) (- center-y size)
			       (+ center-x size) (+ center-y size)
			       :filled filled-p))
	    (triangle
	      (draw-polygon*  stream
			      (list (- center-x size) (+ center-y size)
				    center-x (- center-y size)
				    (+ center-x size) (+ center-y size))
			       :filled filled-p))))))
    (values shape size filled-p)))

;;;
;;; The same code has a dynamic behavior when enclosed in a call to ACCEPTING-VALUES
;;;

;;; --- drawing functions blow out on AVV streams at the moment
(defun avv-shape-chooser (stream)
  (window-clear stream)
  (accepting-values (stream)
    (shape-chooser stream)))





;;;***PRESENTATION TYPES***
;;;
;;; Presentation types
;;; 

;;;
;;; Uses of :ABBREVIATION-FOR
;;;

;;; From the system:
;(define-presentation-type boolean ()
;  :abbreviation-for '(alist-member :alist (("Yes" :value t) ("No" :value nil))))

(define-presentation-type teenage ()
  ;; Let the new type "teenage" inherit the attributes of the type (integer 13 19)
  :abbreviation-for '(integer 13 19)
  ;; the describer is run when the user is prompted for input
  :describer ((stream)
	      ;; Without this describer the default prompt would ask me for
	      ;; an integer between 13 and 19.  This way I express myself at 
	      ;; a higher level
	      (write-string "a teenager's age" stream)))

;;; A presentation type's parser defines how a datum of that type is parsed, 
;;; typically by reading from a stream of characters, most often that the user is
;;; typing.  The printer of a presentation type defines how a datum of that type should 
;;; be textually presented to the user

;;; Here's a presentation type with a simple parser and printer
;;;

(define-presentation-type palindrome ()
  :parser ((stream &key default &aux token)
	   (loop
	     (setq token (read-token stream))
	     (when (and (string-equal token "") default)
	       (setq token default))
	     (let ((length (length token)))
	       (when
		 (string-equal
		   (subseq token 0 (floor length 2))
		   (reverse 
		     (subseq token (ceiling length 2) length)))
		 (return token))
	       (ci::input-not-of-required-type token 'palindrome))))
  :printer ((object stream &key acceptably) acceptably
	    (write-string object stream)))

;;; Here's an example of a presentation type with a non-trivial parser
;;; Use of manual input parsing, invalid input handling, and 
;;; PRESENTATION-REPLACE-INPUT
;;;
(define-presentation-type monogram ()
  :parser ((stream &key default &aux token)
	   (loop
	     ;; Remember where the input cursor is
	     (let ((ip (input-position stream)))
	       (setq token (read-token stream))
	       (when (and (string-equal token "") default)
		 (setq token default))
	       (when (and (<= (length token) 3)
			  (every #'alpha-char-p token))
		 ;; Input is valid as a monogram
		 ;; Invoke our printer to replace what the user has typed,
		 ;; giving him the feedback that his input was acceptable
		 ;; This is optional
		 (setf token (string-upcase token))
		 (presentation-replace-input stream token 'monogram
					     :buffer-start ip)

		 ;; return from loop with a valid input
		 (return token))
	       ;; Input is not valid.  Call standard function to indicate this
	       ;; and loop to try again
	       (ci::input-not-of-required-type token 'monogram))))
  :printer ((ob stream &key acceptably) acceptably
	    (write-string (string-upcase ob) stream)))


;;;
;;; Use of COMPLETING-FROM-SUGGESTIONS and presentation arguments
;;;
(defvar *months*
	'("January" "February" "March" "April"
	  "May" "June" "July" "August" "September"
	  "October" "November" "December"))

;;; Only months with the letter R in their name are good for eating oysters.
(define-presentation-type month (&optional only-oyster-months)
  :parser ((stream &key default &aux month-n)
	   ;; This macro provides completion behavior.  When prompted for a month,
	   ;; if the user types Control-Shift-? or the Help key, he'll be presented
	   ;; with a list of possibilities from which to choose
	   (completing-from-suggestions (stream)
	     (setq month-n 1)
	     (dolist (month *months*)
	       (when (or (not only-oyster-months)
			 (find #\r month))
		 (suggest month month-n))
	       (incf month-n))))

  ;; How to determine if one presentation type based on MONTH is a subtype
  ;; of another.  This is to ensure that:
  ;; (presentation-subtypep 'month 'month) => T
  ;; (presentation-subtypep 'month '(month t)) => NIL
  ;; (presentation-subtypep '(month t) 'month) => T
  :data-args-comparator ((subargs superargs)
			 (or (null (first superargs))
			     (first subargs)))

  ;; How to determine if a given month is valid, based on the presentation type
  ;; This is to avoid making "May" mouse-senstive, for example, when the input
  ;; context is (MONTH T)
  :object-validator ((object)
		     (or (not only-oyster-months)
			 (not (null (find #\r (elt *months* (1- object)))))))
  :printer ((object stream &key acceptably)
	    (declare (ignore acceptably)) ; for now
	    (write-string (elt *months* (1- object)) stream))
  :describer ((stream)
	      (if only-oyster-months
		  (write-string "a month good for eating oyters" stream)
		  (write-string "a month" stream))))


;;; ***PRESENTATION TRANSLATORS***

;;; --- shifted gestures don't work yet.
;;; Define the logical gesture :ROUND that corresponds to the physical gesture
;;; shift-Left.
(ci::define-logical-gesture :round :left '(:shift))

(define-presentation-translator rounded-float
     ;; Translates FLOATs to INTEGERs
     (float					; from-type
       integer					; to-type
       :documentation ((stream)
		       (write-string "Round up the floating point number" stream))
       :gesture :left)
   ;; This translator takes a single argument: the object
   ;; under the pointer
   (object)
  (values (round object)))

;;;
;;; To see how the above translator works, PRESENT a floating point number to a
;;; window, e.g. (PRESENT 98.6 'FLOAT :STREAM window) and then ACCEPT an integer.
;;; With the translator defined the float will be pointer sensitive.  Without the
;;; translator it won't be.

;;;
;;; To remove this translator you could do:
;;;
;;;
;;;(ci::remove-presentation-translator
;;;  (ci::find-presentation-translator-named 'rounded-float))



;;;***OUTPUT RECORDING***
;;;
;;; Simple output recording
;;;

(defun accept-baby-name (stream)
   (let ((*standard-output* stream)
	 (*query-io* stream))
     (format t "~&Some baby names:~%~%")
     (dolist (name '("John" "Mary" "Vaclav" "Ottakar"))
       (present name 'string)
       (fresh-line))
     (accept 'string :prompt "Choose the baby's name")))

;;; --- strange mouse senstivity
(defun accept-baby-name-as-rectangle (stream)
   (let ((*standard-output* stream)
	 (*query-io* stream)
	 (x 0))
     (format t "~&Some baby names:~%~%")
     (dolist (name '("John" "Mary" "Vaclav" "Ottakar"))
       (with-output-as-presentation (:object name :type 'string :stream *standard-output*)
	 (draw-rectangle* *standard-output* (+ x 0) 20 (+ x 20) 40 :filled nil))
       (incf x 30))
     (stream-set-cursor-position* stream 0 50)
     (accept 'string :prompt "Choose the baby's name")))


;;;
(define-clim-test show-show-text-styles (window)
  "Show various text styles, plus table formatting."
  (show-text-styles window))

(define-clim-test show-roll (window)
  "Roll example from tutorial"
  (roll window))

(define-clim-test show-swirl (window)
  "Swirl example from tutorial"
  (swirl window))

(define-clim-test (show-non-incremental-time-loop :self-terminating t) (window)
  "Non-incremental time loop from tutorial"
  (non-incremental-time-loop window (max *test-suite-timeout* 10.)))

(define-clim-test (show-incremental-time-loop :self-terminating t :output-recording t) (window)
  "Incremental time loop from tutorial"
  (time-loop window (max *test-suite-timeout* 10.)))

(define-clim-test (show-shape-chooser :self-terminating t :output-recording t)
		  (window test documentation-window)
  "Accept test from tutorial"
  (format window "Please enter the shape, size, and filled-p attributes of a shape:~2%")
  (shape-chooser window)
  (read-a-gesture-to-terminate test documentation-window window))

(define-clim-test (show-avv-shape-chooser :self-terminating t :output-recording t) (window)
  "AVV test of shape-chooser"
  (avv-shape-chooser window))

(define-clim-test (tutorial-presentation-types :self-terminating t :output-recording t)
		  (window test documentation-window)
  "Show ACCEPT of various presentation types"
  (accept 'teenage :stream window :prompt "How old is Judy Jetson? ")
  #+clim-0.9 (read-gesture :stream window)	;Get rid of terminating character
  (terpri window)
  (accept 'monogram :stream window)
  #+clim-0.9 (read-gesture :stream window)
  (terpri window)
  (format window "~2%The following will offer help if you press the ~A key.~%"
	  #+Genera #\Help #-Genera "Control-?")
  (accept 'month :stream window)
  #+clim-0.9 (read-gesture :stream window)
  (terpri window)
  (accept '(month t) :stream window)
  #+clim-0.9 (read-gesture :stream window)
  (terpri window)
  (read-a-gesture-to-terminate test documentation-window window))

(define-clim-test (use-rounded-float :self-terminating t :output-recording t)
		  (window test documentation-window)
  "Show presentation type translators.
Click shift-left on a float to translate it to an integer."
  (write-string "PI: " window)
  (present pi 'float :stream window)
  (write-string ", E: " window)
  (present (exp 1) 'float :stream window)
  (terpri window)
  (accept 'integer :stream window :prompt "try clicking on an integer.")
  (read-a-gesture-to-terminate test documentation-window window))
