;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-DEMO; Base: 10 -*-

(in-package "CLIM-DEMO")


;;; Basic dimensions
(defparameter *outside-width* 640)
(defparameter *outside-height* 390)
(defparameter *test-width* (- *outside-width* 40))
(defparameter *test-height* (- *outside-height* 20))
(defparameter *center-x* (round *test-width* 2))
(defparameter *center-y* (round *test-height* 2))

;;; Windows
(defvar *test-stream*)

(eval-when (load)
  (setq *default-server-path* (car *server-path-defaults*)))

;; from CLIM:0-9;DEMO;EXAMPLES.LISP
(defun clim-scroller (&key (server-path *default-server-path*) (hs 600) (vs #+Imach 200 #-Imach 400))
  (let ((framem (find-frame-manager :server-path server-path))
	(frame (make-frame 'frame))
	stream)
    ;; Enter the context in which to define the contents of this frame
    (with-look-and-feel-realization (framem frame)
      ;; declare what the frame's child will be
      (setf (frame-pane frame)			;; Frames have only one child
	    (scrolling				;; Make a scrolling pane
	      (:subtransformationp t		;; Required for scrolling panes
	       :hs+ +fill+			;; If the frame gets bigger in the
	       :vs+ +fill+			;;   horizontal or vertical dimensions
						;;   expand to fill all available space
	       :hs hs :vs vs)			;; Initial h. and v. sizes
	      ;; A form to generate the contents
	      (setq stream (make-pane 'extended-stream-pane))))
      (adopt-frame framem frame)
      ;; Make the frame visible
      (enable-frame frame)
      (values stream frame))))

(defparameter *clim-stream* (clim-scroller :hs *test-width* :vs *test-height*))

(window-clear *clim-stream*)

;;;--- We know that X and Y are just offsets from wherever the frame would be put otherwise
(defun make-lightest-weight-frame (&key (server-path *default-server-path*) (x 0) (y 0)
				   (width 100) (height 100))
  (let ((framem (find-frame-manager :server-path server-path))
	(frame (make-frame 'frame :plain t :left x :bottom y))
	stream)
    ;; Enter the context in which to define the contents of this frame
    (with-look-and-feel-realization (framem frame)
      ;; declare what the frame's child will be
      (setf (frame-pane frame)			;; Frames have only one child
	    (make-clim-pane (stream :vs height :hs width :scroll-bars nil)))
      (adopt-frame framem frame)
      ;; Make the frame visible
      (enable-frame frame)
      (values stream frame))))

;;; Other special variables
(defparameter *record?* t)				;Dynamically control output-recording
(defparameter *draw?* t)				;Dynamically control output drawing

(defparameter *equilateral-triangle-top-ratio* (/ 1 (cos (/ pi 6))))
(defparameter *equilateral-triangle-bottom-ratio* (tan (/ pi 6)))

(defparameter *line-o-text* "The quick brown fox jumpedoverthelazydog.")
(defparameter *long-line-o-text*
  "The quick brown fox jumped over the lazy dog.The quick brown fox jumped over it.")

(defparameter *metering-count-limit* 1)
(defparameter *metering-values?* t)

(defpackage "CLIM-OUTPUT-BENCHMARKS"
  (:nicknames "CLOB")
  (:use))

(defvar *tests-to-run*)

(defun benchmark-driver (&optional (stream *clim-stream*)
			 (report-stream *trace-output*))
  (with-output-recording-options (stream :record-p *record?* :draw-p *draw?*)
    (window-clear stream)
    (let ((frame (pane-frame stream)))
      (dolist (item *tests-to-run*)
	; DESTRUCTURING-BIND isn't CLtL I Common Lisp; it's ANSI Lisp only.  --RWK
	;(destructuring-bind (test &optional no-clear-and-enable) item
	(let ((test (first item))
	      (no-clear-and-enable (second item)))
          (if no-clear-and-enable
            #+ccl
	    ;; To avoid redrawing the borders of the selected Listener
            (ccl:window-select (w::sheet-mirror! stream))
            #-ccl
            ()
            (progn 
              (window-clear stream)
              #-Genera
              (enable-frame frame)
              #+Genera
              (scl:send (w::sheet-mirror! stream) :select)
              #+coral (ccl:window-select (w:sheet-mirror! stream))
              (process-next-event (port frame) :timeout 0)
              ))
          (let ((*trace-output* report-stream))
            (format *trace-output* "~%Test Start~%~A" test)
            (time (funcall test stream))
            (format *trace-output* "~%Test End")))))))







;;; Tests

(defun clob::vertical-lines (&optional (stream *clim-stream*))
  (do ((x 10 (+ x 2)))
      ((> x *test-width*))
    (draw-line* stream x 10 x *test-height* :line-thickness 1)))

(defun clob::vertical-gradation (&optional (stream *clim-stream*))
  (do ((width 1 (1+ width))
       (x 10 (+ x width 3)))
      ((> x *test-width*))
    (draw-line* stream x 10 x *test-height* :line-thickness width)))

(defun clob::horizontal-lines (&optional (stream *clim-stream*))
  (do ((y 10 (+ y 2)))
      ((> y *test-height*))
    (draw-line* stream 10 y *test-width* y :line-thickness 1)))

(defun clob::horizontal-gradation (&optional (stream *clim-stream*))
  (do ((width 1 (1+ width))
       (y 10 (+ y width 3)))
      ((> y *test-height*))
    (draw-line* stream 10 y *test-width* y :line-thickness width)))


;;; Polygons

(defun clob::concentric-squares (&optional (stream *clim-stream*))
  (do ((radius 3 (+ radius 3)))
      ((> radius *center-y*))
    (let ((top (- *center-y* radius))
          (bottom (+ *center-y* radius)))
      (draw-rectangle* stream top top bottom bottom :ink +flipping-ink+))))

(defun clob::concentric-triangles (&optional (stream *clim-stream*))
  (do ((radius 3 (+ radius 3)))
      ((> radius *center-y*))
    (let ((top-y (round (* *equilateral-triangle-top-ratio* radius)))
          (bottom-y (+ *center-y* (round (* *equilateral-triangle-bottom-ratio* radius)))))
      (draw-polygon* stream (list *center-y* (- *center-y* top-y)
                                  (- *center-y* radius) bottom-y
                                  (+ *center-y* radius) bottom-y)
			  :ink +flipping-ink+))))



;;; Curves
(defun clob::concentric-circles (&optional (stream *clim-stream*))
  (do ((i 3 (+ i 3)))
      ((> i *center-y*))
    (draw-circle* stream *center-x* *center-y* i :ink +flipping-ink+)))

(defun clob::concentric-ellipses (&optional (stream *clim-stream*))
  (do ((x-radius 3 (+ x-radius 3))
       (y-radius 2 (+ y-radius 2)))
      ((> x-radius *center-y*))
    (draw-ellipse* stream *center-x* *center-y* x-radius 0 0 y-radius
			  :ink +flipping-ink+)))


;;; Transforms
(defun clob::spin (&optional (stream *clim-stream*))
  (multiple-value-bind (nil1 nil2 height width) (bounding-rectangle* (sheet-region stream))
    (declare (ignore nil1 nil2))
    (with-translation (stream (round width 2) (- (round height 2) 100))
      (flet ((draw (stream)
	       (draw-rectangle* stream 0 0 50 50 :ink +blue+)
	       (draw-circle* stream 70 30 20 :ink +magenta+)))
	(do ((angle 0 (+ angle (/ pi 4))))
	    ((> angle 2pi) nil)
	  (with-rotation (stream angle)
	    (with-translation (stream 100 0)
	      (draw stream))))))))

(defun clob::spin-ila-logo (&optional (stream *clim-stream*))
  (multiple-value-bind (nil1 nil2 height width)
      (bounding-rectangle* (sheet-region stream))
    (declare (ignore nil1 nil2))
    (with-translation (stream (round width 2) (- (round height 2) 100))
      (with-translation (stream 0 -25)
	(do ((angle 0 (+ angle (/ pi 4)))
	     (scale 1 (* scale 7/8)))
	    ((< scale .07) nil)
	  ;; ((> angle 2pi) nil)
	  (with-rotation (stream angle)
	    (with-scaling (stream scale)
	      (with-translation (stream 100 0)
		(with-drawing-options
		  ;;-- There's some weird extra 20 pixels
		  (stream :transformation (make-transformation 1 0 0 -1 20 120))
		  (dotimes (i 5)
		    (let* ((i (1+ i))
			   (offset (- (* i 2) 18))
			   (origin (* (1- i) 20))
			   (ii (+ origin offset))
			   (jj (- 100 offset)))
		      (draw-polygon*
			stream
			(list origin origin origin jj ii 100 ii ii 100 ii jj origin)
			:ink +red+))))))))))))


;;; Text

(defun clob::simple-l-to-r-text (&optional (stream *clim-stream*))
  (dotimes (i 22)
    (declare (ignore i))
    (write-string *line-o-text* stream)
    (write-string *line-o-text* stream)
    (terpri stream)))

(defun clob::simple-long-l-to-r-text (&optional (stream *clim-stream*))
  (dotimes (i 22)
    (declare (ignore i))
    (write-string *long-line-o-text* stream)
    (terpri stream)))

(defun clob::vertical-plain-text (&optional (stream *clim-stream*))
  (let ((glyph-width 15))
    (dotimes (i (round *test-width* glyph-width))
      (let ((x (* i glyph-width)))
	(draw-text* stream *line-o-text* x 10 :toward-point (make-point x *test-height*))))))

;;; Temporary windows
(defparameter *temporary-minimal-windows*
	      (let (w)
		(do ((i 50 (+ i 5)))
		    ((> i 250))
		  (push (make-lightest-weight-frame :x i :y i
						    :width 80
						    :height 80)
			w))
		(nreverse w)))

(defparameter *temp-text-window*
	      (let* ((w (make-lightest-weight-frame :x 300 :y 300 :width 100 :height 100)))
		(clim::format w " Remove ~% Add ~% File ~% Move ~% Copy ~% Expand")
		w))

(defun clob::empty-temp-windows (&optional ignore)
  (declare (ignore ignore))
  (dolist (stream *temporary-minimal-windows*)
    (let ((frame (pane-frame stream)))
      (enable-frame frame)
      (disable-frame frame))))

(defun clob::text-window (&optional ignore)
  (declare (ignore ignore))
  (let ((frame (pane-frame *temp-text-window*)))				
    (dotimes (i 10)
      (declare (ignore i))
      (enable-frame frame)
      #+ccl
      (frame-repaint-pane frame *temp-text-window*)
      (disable-frame frame))))


;;; Symbolics's Rel. 0 Beta tests
(defun clob::smbx-test-1 (&optional ignore)
  (declare (ignore ignore))
  (dotimes (i 10)
    (declare (ignore i))
    (let* ((stream (make-lightest-weight-frame :width 550 :height 350))
	   (frame (pane-frame stream)))
      (enable-frame frame)
      (disable-frame frame)
      (stop-frame frame t t))))

(defun clob::smbx-test-2 (&optional (stream *clim-stream*))
  (let ((frame (pane-frame stream)))
    (dotimes (i 10)
      (declare (ignore i))
      (enable-frame frame)
      (disable-frame frame))))

(defparameter *smbx-test-3-string*
	"The very quick brown foxes jumped over the stupid lazy dogs.")
(defparameter *smbx-test-3a-count* 20)
(defparameter *smbx-test-3b-count* 7)
(defparameter *smbx-test-3c-count* 200)
(defparameter *smbx-test-3b-appearances*
	`((:sans-serif :roman :small)
	  (:sans-serif :roman :normal)
	  (:sans-serif :roman :large)))

(defun clob::smbx-test-3a (&optional (stream *clim-stream*))
  (dotimes (i 10)
    (declare (ignore i))
    (stream-set-cursor-position* stream 0 0)
    (dotimes (i *smbx-test-3a-count*)
      (declare (ignore i))
      (write-string *smbx-test-3-string* stream)
      (terpri stream))))

(defun clob::smbx-test-3b (&optional (stream *clim-stream*))
  (dotimes (i 10)
    (declare (ignore i))
    (stream-set-cursor-position* stream 0 0)
    (dotimes (i *smbx-test-3b-count*)
      (declare (ignore i))
      (dolist (appearance *smbx-test-3b-appearances*)
	(with-text-style (appearance stream)
	  (write-string *smbx-test-3-string* stream)
          (terpri stream))))))

(defparameter 
  *smbx-test-3c-strings*
  (let (strings)
    (dotimes (i *smbx-test-3c-count*)
      (declare (ignore i))
      (push (format nil "~%~A ~3,D" *smbx-test-3-string* i) strings))
    (nreverse strings)))

(defun clob::smbx-test-3c (&optional (stream *clim-stream*))	;Only run with output recording on
  (stream-set-cursor-position* stream 0 0)
  (dolist (string *smbx-test-3c-strings*)
    (write-string string stream))
  (multiple-value-bind (viewport-min-x viewport-min-y viewport-max-x viewport-max-y)
      (bounding-rectangle* (pane-viewport stream))
    (declare (ignore viewport-min-x viewport-max-x))
    (let ((viewport-height (- viewport-max-y viewport-min-y)))
      (multiple-value-bind (ignore1 ignore2 ignore3 bottom-y)
	  (bounding-rectangle* stream)
        (declare (ignore ignore1 ignore2 ignore3))
	(let ((line-height (stream-line-height stream)))
	  (do ((y bottom-y (- y line-height)))
	      ((< y viewport-height))
	    (scroll-extent stream :x 0 :y y)))
	(scroll-extent stream :x 0 :y 0)))))

(defun clob::smbx-test-3d (&optional (stream *clim-stream*))	;Assumes follows immediately after 3c
  (let ((line-height (stream-line-height stream)))
    (multiple-value-bind (nil1 nil2 nil3 max-y)
	(bounding-rectangle* stream)
      (declare (ignore nil1 nil2 nil3))
      (scroll-extent stream :x 0 :y 0)
      (do ((y 0 (+ y line-height)))
	  ((> y max-y))
      (scroll-extent stream :x 0 :y y)))))


;;; ... do AVV later
(defparameter *smbx-test-5a-n-cols* #+Genera  #+ccl 16)

(defun clob::smbx-test-5a (&optional (stream *clim-stream*))
  (dotimes (i 10)
    (let ((x (+ 20 (* i 20))))
      (dotimes (j *smbx-test-5a-n-cols*)
	(let ((y (+ 20 (* j 20))))
	  (stream-set-cursor-position* stream x y)
	  (present (aref "CLIM" (ldb (byte 2 0) j)) 'character :stream stream)))))
  (accept 'character :stream stream :prompt "Pick one"))

;;;

(defparameter *clim-random-state* (make-random-state))

(defun clob::smbx-test-6a (&optional (stream *clim-stream*))
  (multiple-value-bind (nil1 nil2 inside-width inside-height)
      (bounding-rectangle* (sheet-region stream))
    (declare (ignore nil1 nil2))
    (let ((ran (make-random-state *clim-random-state*))
	  (x-delta (round inside-width 32))
	  (y-delta (round inside-height 32)))
      (dotimes (i 32)
	(let ((x (+ x-delta (* i x-delta))))
	  (dotimes (y 32)
	    (let ((y (+ y-delta (* y y-delta)))
		  (value (random 1024 ran)))
	      (case (ldb (byte 2 0) value)
		(0 (draw-circle* stream x y
				 (min x (abs (- inside-width x))
				      y (abs (- inside-height y))
				      (ldb (byte 6 0) value))
				 :filled nil))
		(1 (draw-rectangle* stream x y
				    (min inside-width
					 (+ x (ldb (byte 6 0) value)))
				    (min inside-height
					 (+ y (ldb (byte 6 0) (logxor value (ash value 3)))))
				    :filled nil))
		(2 (draw-line* stream x y
			       (min inside-width (+ x (ldb (byte 6 0) value)))
			       (min inside-height (+ y (ldb (byte 6 0)
							    (logxor value (ash value 3)))))))
		(3 (draw-ellipse* stream x y
				  (min x (abs (- inside-width x))
				       y (abs (- inside-height y))
				       (ldb (byte 6 0) value))
				  0 0
				  (min x (abs (- inside-width x))
				       y (abs (- inside-height y))
				       (ldb (byte 6 0) (logxor value (ash value 3))))
				  :filled nil))
;	    (4 (draw-triangle ))
;	    (5 (draw-polygon))
;	    (6 (draw-oval  ))
;	    (7 (draw-spline ))
		))))))))

(defun clob::smbx-test-6b (&optional (stream *clim-stream*))	;Assumes history will not be cleared, window exposed
  (repaint-sheet stream (sheet-region stream)))

(defparameter *base* 0)
(defparameter *max-base* 2)
(defparameter *smbx-test-7-nrows* 15)
(defvar *smbx-test-7b-i*)

(defun monthly-mortgage-payment (principal interest-rate nyears n-yearly-payments)
  (setq interest-rate (/ interest-rate 100))
  (let* ((il (/ interest-rate n-yearly-payments))
	 (nl (* nyears n-yearly-payments))
	 (1+il (1+ il))
	 (temp (expt 1+il nl)))
    (values (float (/ (* principal il temp)
		      (1- temp))))))

(defvar *interest-rate*)
(defparameter *interest-rates* '(9.0 9.5 10.0 10.5 11.0 11.5 12.0 12.5 13.0 13.5))
(defparameter *principal-amounts*
	'(100000 125000 150000 200000 250000 300000 350000 400000 450000 500000))
(defparameter *mortgage-nyears* '(5 10 15 30))

(defun clob::smbx-test-7a (&optional (stream *clim-stream*))
  (let ((output-record
	  (updating-output (stream)
	    (formatting-table (stream)
	      ;; The Rows
	      (dotimes (row *smbx-test-7-nrows*)
		(formatting-row (stream)
		  (dotimes (col 16)
		    (let* ((cell-address (+ (* row 16) col))
			   (value (+ *base* cell-address)))
		      (formatting-cell (stream)
			(updating-output (stream :unique-id cell-address :cache-value value)
			  (format stream "~A" value)))))))))))
    (stream-set-cursor-position* stream 0 0)
    (dotimes (*base* *max-base*)
      (redisplay output-record stream))))

(defun clob::smbx-test-7b (&optional (stream *clim-stream*))
  (let* ((*smbx-test-7b-i* 1)			;This should be inutile - but, apparently CLIM needs it
	 (output-record
	   (updating-output (stream)
	     (formatting-table (stream)
	       ;; The Rows
	       (do ((row *smbx-test-7b-i* (1- row)))
		   ((= row 0))
		 (formatting-row (stream)
		   (updating-output (stream :unique-id (* 1000 (1+ row)))
		     (dotimes (col 16)
		       (let* ((cell-address (+ (* row 16) col))
			      (value (+ *base* cell-address)))
			 (formatting-cell (stream)
			   (updating-output (stream :unique-id cell-address :cache-value value)
			     (format stream "~A" value)))))))))))
	 (*base* 1))
    (stream-set-cursor-position* stream 0 0)
    (do ((*smbx-test-7b-i* 1 (1+ *smbx-test-7b-i*)))
	((= *smbx-test-7b-i* *smbx-test-7-nrows*))
      (redisplay output-record stream))))

(defun clob::smbx-test-7c (&optional (stream *clim-stream*))
  (let (output-record)
    (stream-set-cursor-position* stream 0 0)
    (dolist (*interest-rate* *interest-rates*)
      (redisplay
	(or output-record
	    (setq output-record
		  (updating-output(stream)
		    (let ((amounts *principal-amounts*)
			  (nyears *mortgage-nyears*))
		      (formatting-table (stream)
			(formatting-row (stream)
			  (formatting-cell (stream)
			    (format stream "Interest Rate: ~A% " *interest-rate*))
			  (let ((id 0))
			    (dolist (nyear nyears)
			      (formatting-cell (stream)
				(updating-output (stream :unique-id id :cache-value nyears)
				  (format stream "   ~D yr  " nyear)))
			      (incf id))))
			;; The Rows
			(dolist (principal amounts)
			  (let ((base-cell-address (* principal 100)))
			    (formatting-row (stream)
			      (formatting-cell (stream)
				(updating-output
				  (stream :unique-id base-cell-address :cache-value principal)
				  (multiple-value-bind (thousands rem)
				      (floor principal 1000.)
				    (format stream "$~3D,~3,48D" thousands rem))))
			      (let ((id 0))
				(dolist (nyear nyears)
				  (let* ((cell-address (+ base-cell-address (incf id)))
					 (value (monthly-mortgage-payment
						  principal *interest-rate* nyear 12.)))
				    (formatting-cell (stream)
				      (updating-output (stream :unique-id cell-address
							       :cache-value value)
					(multiple-value-bind (thousands rem)
					    (floor value 1000.)
					  (format stream "$~:[~*   ~;~2D,~]~2,1,6,v$ "
						  (plusp thousands) thousands
                                                  #\0 rem)))))))))))))))
	stream))))

;;;


(defparameter *smbx-test-12-list*
	'(0 1 2 3 4 5 6 7 8 9
	    10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29
	    300 301 302 303 304 305 306 307 308 309 310 311 312 313 314 315 316 317 318 319 320 321 322 323 324 325 326 327 328 329
	    4000 4001 4002 4003 4004 4005 4006 4007 4008 4009))

(defun clob::smbx-test-12 (&optional (stream *clim-stream*))
  (stream-set-cursor-position* stream 0 0)
  (format-items *smbx-test-12-list* :stream stream :presentation-type t :max-height 70)
  (format-items *smbx-test-12-list* :stream stream :presentation-type t :max-width 250))


(eval-when (load)
  (setq *tests-to-run*
        '((clob::vertical-lines) 
          (clob::vertical-gradation) 
          (clob::horizontal-lines) 
          (clob::horizontal-gradation) 
          (clob::concentric-squares) 
          (clob::concentric-triangles) 
          (clob::concentric-circles) 
          (clob::concentric-ellipses) 
          ;;;--- broken in CLIM 0.9
          (clob::spin) 
          (clob::spin-ila-logo) 
          (clob::simple-l-to-r-text) 
          (clob::simple-long-l-to-r-text) 
          (clob::vertical-plain-text) 
          (clob::empty-temp-windows t)
          (clob::text-window t)
          (clob::smbx-test-1 t)
          (clob::smbx-test-2)
          (clob::smbx-test-3a) 
          (clob::smbx-test-3b) 
          (clob::smbx-test-3c) 
          (clob::smbx-test-3d) 
          (clob::smbx-test-5a) 
          (clob::smbx-test-6a) 
          (clob::smbx-test-6b) 
          (clob::smbx-test-7a) 
          (clob::smbx-test-7b) 
          (clob::smbx-test-7c) 
          (clob::smbx-test-12))))


#||
(with-open-file (rs "clim:qa;performance;reports;3650-Genera7-CLIM0-9.text" :direction :output)
  (benchmark-driver *clim-stream* rs))

(with-open-file (rs "f:>mas-b>3640-Genera7-CLIMdevo.text" :direction :output)
  (benchmark-driver *clim-stream* rs))


(let ((*record?* nil)
      (*tests-to-run*
       '((clob::text-window t))))
  (benchmark-driver *clim-stream*))

||#
