;;; -*- Mode: LISP; Syntax: Common-lisp; Package: USER; Base: 10; Lowercase: Yes -*-

(in-package #+ansi-90 "CL-USER" #-ansi-90 "USER")

;;; To get CLIM working at your site you should:
;;; 
;;;  - Get all the CLIM source files from ILA.  The complete list of source
;;;    file names can be found in the defsystems for CLIM which appears in 
;;;    this file.
;;; 
;;;  - Edit the variable *clim-directory* below to specify the directory at
;;;    your site where the CLIM sources and binaries will be.  This variable
;;;    can be found by searching from this point for the string "***" in
;;;    this file.
;;; 
;;;  - Use the function (compile-clim) to compile CLIM for your site.
;;; 
;;;  - Once CLIM has been compiled it can be loaded with (load-clim).

;;; ***                SITE SPECIFIC CLIM DIRECTORY                        ***
;;;
;;; *clim-directory* is a variable which specifies the directory Y
;;; is stored in at your site.  If the value of the variable is a single
;;; pathname, the sources and binaries should be stored in that
;;; directory.  If the value of that directory is a cons, the CAR should
;;; be the source directory and the CDR should be the binary directory.
;;; 


#+Genera
(eval-when (compile load eval)
  (when (eql (sct:get-release-version) 8)
    (pushnew :Genera-Release-8 *features*)
    (multiple-value-bind (major minor) (sct:get-system-version)
      (declare (ignore minor))
      (cond ((= major 425)
	     (pushnew :Genera-Release-8-0 *features*))
	    ((= major 436)
	     (pushnew :Genera-Release-8-1 *features*))
	    ((>= major 437)
	     (pushnew :Genera-Release-8-2 *features*))))))

;;; CLIM is implemented using the "Gray Stream Proposal" (STREAM-DEFINITION-BY-USER)
;;; a proposal to X3J13 in March, 1989 by David Gray of Texas Instruments.  In that
;;; proposal, stream objects are built on certain CLOS classes, and stream functions
;;; (e.g., WRITE-CHAR) are non-generic interfaces to generic functions (e.g.,
;;; STREAM-WRITE-CHAR).  These "trampoline" functions are required because their
;;; STREAM argument is often optional, which means it cannot be used to dispatch to
;;; different methods.

;;; Various Lisp vendors have their own stream implementations, some of which are
;;; identical to the Gray proposal, some of which implement just the trampoline
;;; functions and not the classes, etc.  If the Lisp vendor has not implemented the
;;; classes, we will shadow those class names (and the predicate functions for them)
;;; in the CLIM-LISP package, and define the classes ourselves.  If the vendor has
;;; not implemented the trampoline functions, we will shadow their names, and write
;;; our own trampolines which will call our generic function, and then write default
;;; methods which will invoke the COMMON-LISP package equivalents.

#+(or Allegro 
      Minima)
(pushnew :clim-uses-lisp-stream-classes *features*)

#+(or Allegro
      Genera				;Except for STREAM-ELEMENT-TYPE
      Minima
      CLOE-Runtime
      CCL-2)				;Except for CLOSE (and WITH-OPEN-STREAM)
(pushnew :clim-uses-lisp-stream-functions *features*)

#+excl
(pushnew :allegro-v4.0-constructors *features*)

;; This version of the SYSDCL file is for #+SILICA version of the system:
(pushnew :silica *features*)
 
;;; CLIM-ANSI-Conditions means this lisp truly supports the ANSI CL condition system
;;; CLIM-Conditions      means that it has a macro called DEFINE-CONDITION but that it works
;;;                      like Allegro 3.1.13 or Lucid.
(eval-when (eval compile load)
  (pushnew #+Symbolics :CLIM-ANSI-Conditions
	   #+(and Allegro (not ansi-90))   :CLIM-Conditions
	   #+Lucid     :CLIM-Conditions
           #+ansi-90   :CLIM-ANSI-Conditions
	   #-(or Symbolics Allegro Lucid ansi-90)
	   (error "Figure out what condition system for this Lisp")
	   *features*)
  )


(setq defsys:*load-all-before-compile* t)

(defun load-truename (&optional (errorp nil))
  (declare #+(or excl (and dec vax common) ccl) (ignore errorp))
  (flet (#-(or excl (and dec vax common) ccl)
	 (bad-time ()
	   (when errorp
	     (error "LOAD-TRUENAME called but a file isn't being loaded."))))
    #+Lispm  (or sys:fdefine-file-pathname (bad-time))
    #+excl   excl::*source-pathname*
    #+Xerox  (pathname (or (il:fullname *standard-input*) (bad-time)))
    #+(and dec vax common) (truename (sys::source-file #'load-truename))
    ;;
    ;; The following use of  `lucid::' is a kludge for 2.1 and 3.0
    ;; compatibility.  In 2.1 it was in the SYSTEM package, and i
    ;; 3.0 it's in the LUCID-COMMON-LISP package.
    ;;
    #+LUCID (or lucid::*source-pathname* (bad-time))
    #+CCL ccl:*loading-file-source-file*)) ; slh

(defparameter *sysdcl-directory*
  (or (load-truename t)
      (error "Because load-truename is not implemented in this port~%~
                    of CLIM, you must manually edit the definition of the~%~
                    variable *clim-directory*.")))

#-(and ansi-90 (not Allegro))
(defparameter *clim-directory* (defsys:make-pathname 
				:directory '(:relative :back)
				:defaults *sysdcl-directory*))

;;; ANSI lisps have logical pathnames, need less kludging.
#+(and ansi-90 (not Allegro))
(defparameter *clim-directory* (logical-pathname "clim:"))

(defun make-clim-pathname (subdir)
  ;; Use this unexported defsys utility to deal with pathname case
  (setq subdir (defsys::pretty-pathname-component subdir))
  (defsys:make-pathname 
   :directory `(#-(and ansi-90 (not Allegro)) :relative
		#+(and ansi-90 (not Allegro)) :absolute
                ,subdir)
   :defaults *clim-directory*))
	
(defparameter *clim-binary-directory*
  #+(and Genera (not PCL)) (make-clim-pathname "clos-bins")
  #+(and Genera PCL) (make-clim-pathname "pcl-bins")
  #+Cloe-Runtime '("d:\\clim\\src\\" . "d:\\clim\\bin\\")
  #+(and ccl (not ansi-90)) ":ccl-fasl:"
  #+(and ansi-90 (not Allegro))
     (make-pathname :directory `(:absolute "binary")
		    :defaults (logical-pathname "CLIM:"))
  #+:lucid *clim-fasl-tree*
  #+:excl nil
  )

(defsys:defsystem clim (:default-pathname (make-clim-pathname "sys")
			:default-binary-pathname *clim-binary-directory* 
			:needed-systems (clim-utils silica clim-stream ws ptk)
			:load-before-compile (clim-utils silica clim-stream ws ptk))
  ("precom")
  )
    
#+excl
(defsys:defsystem clim-excl-patches (:default-pathname (make-clim-pathname :back)
				     :default-binary-pathname *clim-binary-directory*)
  ("patch26" :binary-only t :features (or allegro-v3.1))
  ("patch28" :binary-only t :features (or allegro-v3.1)))

(defsys:defsystem clim-utils (:default-pathname (make-clim-pathname "utils")
			      :default-binary-pathname *clim-binary-directory*
			      :patch-file-pattern "patch-utils*"
			      :needed-systems (#+ignore clim-excl-patches)
			      :load-before-compile (#+ignore clim-excl-patches))
  
  ("excl-verification" :features (or excl))
  ("pcl-patches" :features (or pcl))  ;; Improves PCL's method-body function naming

  ;; get this before the packages, since it defines the CONDITIONS package
  ("condition-support-for-7-2" :features (and Genera (not Genera-Release-8)))

  ("lisp-package-fixups" :features (not excl))
  ("defpackage" :features (not ansi-90))

  ;("packages")	      ;; define CLIM-LISP and CLIM-UTILS packages.
  ("clim-lisp-package")
  ("clim-utils-package")
  ; --- temporarily replaced by the following ("clim-package")
  ("clim2-package") ; --- for merge activity of 9/91.  Doughty & York
  ("clim-internals-package") ; --- for merge activity of 9/91.  Doughty & York
;  ("clim-packages")   ;; define CLIM and CLIM-SHARED packages.
  ("defun-utilities") ;; EXTRACT-DECLARATIONS and friends
  ("defun" :features (not ansi-90))
  ("reader")
  ("clos-patches")
  ;; ("conditions") -- needs to be moved to other levels
  ("utilities")
  ("multi-value-setf")
  ("lisp-utilities")
  ("condpat" :features (or clim-conditions))  ;get the define-condition macro
  ("defresource")
  ("clos")
  ("queue")
  ("protocols")
  ("autoconstructor")
  ("cl-stream-classes" :features (not clim-uses-lisp-stream-classes))
  ("cl-stream-functions" :features (not clim-uses-lisp-stream-functions))
  ("genera-streams" :features Genera)
  ;; since we no longer shadow TRUENAME and PATHNAME, don't try
  ;; to redefine them.
  ;;("excl-streams" :features excl)
  ("ccl-streams" :features CCL-2)
  ("temp-strings") ;--- moved from CLIM
  )

(defsys:defsystem silica (:default-pathname (make-clim-pathname "silica")
			  :default-binary-pathname *clim-binary-directory* 
			  :patch-file-pattern "patch-silica*"
			  :needed-systems (silica-low silica-core
					   silica-x silica-genera
					   silica-mcl)
			  :load-before-compile (silica-low silica-core
						silica-x silica-genera
						silica-mcl)))

(defsys:defsystem silica-low (:default-pathname (make-clim-pathname "silica")
			      :default-binary-pathname *clim-binary-directory* 
			      :needed-systems (clim-utils)
			      :load-before-compile (clim-utils))

;  ("silica-pkg")

  ;("transformations")
  ("1-transformations")  
  ;("regions")
  ("1-regions")
  ("1-region-arithmetic")
  ("xform-region-aux")
  ("paints")
  ("pixmaps")
  ("cursors")
  ("graphics-protocol")
  ("text-style" :load-before-compile ("graphics-protocol"))
  
  ("basic-port")
  
  ("graphics-definitions" :load-before-compile ("graphics-protocol"))
  ("graphics-functions" :load-before-compile ("graphics-protocol" "graphics-definitions"))
  )

(defsys:defsystem silica-core (:default-pathname (make-clim-pathname "silica")
			       :default-binary-pathname *clim-binary-directory* 
			       :needed-systems (clim-utils silica-low)
			       :load-before-compile (clim-utils silica-low))

  ("silica-sheet")
  ("silica-event")
  ("silica-window")
  ("silica-port")
  ("silica-output")
  ("resources")
  ("silica-input")
  ("silica-track")
  ("silica-audio")
  ;("silica-env") ;; --- contains the PRINT-OBJECT methods we've despised for so long.
  ;; ("silica-multi")
  ("silica-standard")
  ("silica-clg")
  )

(defsys:defsystem silica-x (:default-pathname (make-clim-pathname "silica")
			    :default-binary-pathname *clim-binary-directory* 
			    :needed-systems (silica-core)
			    :load-before-compile (silica-core))
;;; --- temporarily removed
;  ("x-package"	  :features (or xlib))
  ("clx-rbanding" :features (or xlib))
  ("x-keysyms"    :features (or xlib))
  ("x-port"       :features (or xlib))
  ("x-wm"         :features (or xlib))
  ("x-medium"     :features (or xlib))
  ("x-types"      :features (or xlib))
  ("x-clg"        :features (or xlib))
  )

(defsys:defsystem silica-genera (:default-pathname (make-clim-pathname "silica")
			         :default-binary-pathname *clim-binary-directory* 
				 :needed-systems (silica-core)
				 :load-before-compile (silica-core))
  
;;; --- temporarily removed
;  ("genera-package"       :features (or genera))
  ("genera-keysyms"    :features (or genera))
  ("genera-port"       :features (or genera))
  ("genera-wm"         :features (or genera))
  ("genera-medium"     :features (or genera))
  ("genera-types"      :features (or genera))
  ("genera-clg"        :features (or genera))
  )

(defsys:defsystem silica-mcl (:default-pathname (make-clim-pathname "silica")
			      :default-binary-pathname *clim-binary-directory* 
			      :needed-systems (silica-core)
			      :load-before-compile (silica-core))
  
;;; --- temporarily removed
  ;;("mcl-package"    :features (or ccl))
  ("mcl-port"       :features (or ccl))
  ("mcl-wm"         :features (or ccl))
  ("mcl-medium"     :features (or ccl))
  ("mcl-misc"	      :features (or ccl))
  ("mcl-types"      :features (or ccl))
  ("mcl-clg"        :features (or ccl))
  )

(defsys:defsystem clim-stream (:default-pathname (make-clim-pathname  "stream")
			       :default-binary-pathname *clim-binary-directory* 
			       :needed-systems (silica)
			       :load-before-compile (silica))
  ;;("stream-pkg" :features (not clim-uses-lisp-streams))
;  ("new-stream-pkg")

  ("gray-streams" :features (not clim-uses-lisp-streams))
  
  ("define-stream-protocol")
  ("stream-defs")
  ("stream-protocols")
  ("stream-output-protocol")
  ("stream-input-protocol")
  ("pointer")
  ("text-cursor")  
  
  ;; ??? should be moved down soon --- RR
  ("x-stuff" :features (or xlib))
  ("genera-stuff" :load-before-compile () :features Genera)
  )


(defsys:defsystem ws (:default-pathname (make-clim-pathname  "ws")
		      :default-binary-pathname *clim-binary-directory* 
		      :patch-file-pattern "patch-ws*"
		      :needed-systems (silica clim-stream ws-core ws-panes)
		      :load-before-compile (silica clim-stream ws-core ws-panes))
  )

(defsys:defsystem ws-core (:default-pathname (make-clim-pathname  "ws")
			   :default-binary-pathname *clim-binary-directory* 
			   :needed-systems (silica clim-stream)
			   :load-before-compile (silica clim-stream))
  
;  ("ws-pkg")
  ;; Moved here from UTILS, since it only defines conditions used at this level
  ;; or above.
  ("conditions")
  ("command")
  ("menu-groups")	;--- new on 12/23/91 doughty
  ("pane")
  ("basic-pane-definitions")
  ("frame")
  ("wire")
  ("dashboard")

  ("xx-framem")

  ("db-parts")  
  ("db-layout")
  ("top-level-sheet")
  ("genera-top-level-sheet" :features (or genera))
  
  ;; Basic Application Panes
  ("db-basic-stream")
  ("db-application")

  ;; Basic Composite Panes
  ("db-box")
  ("db-table")  
  ("db-border")
  
  ;; Basic Gadget Implementations
  ("db-text")
  ("db-buttons")
  ("db-scroll")  
  ("db-menu")
  ("db-misc")
  ("adaptive-gadget-abstracts")
  ("adaptive-gadget-impl")
  ("adaptive-scroll-bar")
  ("adaptive-radio-box")
  ("db-new-scroll")
  )

(defsys:defsystem ws-panes (:default-pathname (make-clim-pathname  "ws")
			    :default-binary-pathname *clim-binary-directory* 
			    :needed-systems (silica)
			    :load-before-compile (silica ws-core))

  ("db-rack")
  ("db-bboard")
  ("db-grid")
  ("db-graphics")
  ("db-choicer")
  ("db-list")
  ("db-prompt")
  ("db-constructors"))


(defsys:defsystem ws-demo (:default-pathname (make-clim-pathname "ws")
			   :default-binary-pathname *clim-binary-directory*
			   ;; remove this dependency so we can load the
			   ;; demos even if CLIM wasn't built with the
			   ;; defsystem tool
			   ;;:needed-systems (ws)
			   ;;:load-before-compile (ws)
			   )
  
  ("demo-hello")
  ;; ("demo-graphics")
  ("demo-ico")
  ("demo-hanoi")
  ("demo-panes")
  ("demo-draw")
  ("demo-grid")
  )


(defsys:defsystem clim-test (:default-pathname (make-clim-pathname "test")
			     :default-binary-pathname *clim-binary-directory* )
  ("silica-test")
  ("ws-test")
  ("clim-test")
  )

(defsys:defsystem ptk (:default-pathname (make-clim-pathname "clim")
		       :default-binary-pathname *clim-binary-directory*
		       :patch-file-pattern "patch-clim*"
		       :needed-systems (silica clim-stream ws-core)
		       :load-before-compile (silica clim-stream ws-core))
;  ("pkgdcl")
;  ("shift-mask")
  ("logical-gestures")

;;  ("defprotocol") --- to stream/define-stream-protocol
;;  ("stream-defprotocols") --- to stream/strem-protocols
  ("recording-protocols")
  
;;  ("temp-strings") --- moved to utils

  ("clim-defs")
  ("stipples")

; ("clim-sheet")
  ("clim-silica-glue")

;;;  ("stream-class-defs")  --- to stream/stream-defs
  ("recording-defs")
  
  ("interactive-defs")

  ("view-defs") ; from 1.0

;;;  ("cursor") --- to stream/cursor

;;;  ("input-defs") --- some to stream/pointer
;;;  ("input-protocol") --- to stream/stream-input-protocol
;;;  ("output-protocol") --- to stream/stream-output-protocol

;;;  all to stream/<port>-stuff
;;;  ("clim-x-stuff" :features (or xlib))
;;;  ("clim-genera-stuff" :load-before-compile () :features Genera)
  ;; ??? Some of this needs to be moved to stream/coral-stuff.lisp
  ("clim-coral-stuff" :features (or ccl))

  ("output-recording-protocol" :load-before-compile ("clim-defs"))
  ("graphics-output-recording" :load-before-compile ("clim-defs"))
   
  ("interactive-protocol" :load-before-compile ("clim-defs"))
  ("input-editor-commands")			; :load-before-compile ("lisp-utilities"))

  ("formatted-output-defs")
  ("incremental-redisplay" :load-before-compile ("clim-defs" "output-recording-protocol"))

  ("coordinate-sorted-set")
;;  ("window-stream")
  ("completer")

  ("ptypes1"
   :load-before-compile ("clim-defs"))
  ("presentations"
   :load-before-compile ("clim-defs" "ptypes1"))
  ("translators"
   :load-before-compile ("presentations"))
  ("histories"
   :load-before-compile ("presentations"))
  ("ptypes2"
   :load-before-compile ("translators"))
  (#-CLOE-Runtime "standard-types" #+CLOE-Runtime "std-typs"
   :load-before-compile ("ptypes2"))

  ("presented-margin-comp" :features (not Silica))

  ;; because table-formatting defines methods on PRESENTATION
  ("table-formatting" :load-before-compile ("clim-defs"
					    "incremental-redisplay"))

  ("graph-formatting" :load-before-compile ("clim-defs"
					    "incremental-redisplay"))
  ("surround-output" 
   :load-before-compile ("clim-defs" "incremental-redisplay"))
  ("text-formatting"
   :load-before-compile ("clim-defs" "incremental-redisplay"))

  ("vertical-string" :features (not Silica)) ; well, I merged it anyway, so it's there

  ("menus" :load-before-compile ("clim-defs"))
  ("x-menus" :features (or xlib))
  
; ("presentation-types" :load-before-compile ("presentations"))
  ("accept" :load-before-compile ("clim-defs" "ptypes2"))
  ("present"
   :load-before-compile ("clim-defs" "ptypes2"))
  ("commands-ii" :load-before-compile ("clim-defs"))
  ("command-processor" :load-before-compile ("clim-defs" "commands-ii"))
  ("basic-translators"
   :load-before-compile ("ptypes2" "commands-ii"))

; commented out COMA for now.  Will use it as a test, though
;  ("coma" :load-before-compile ("command-processor"))
;  ("define-program" :load-before-compile ("clim-defs"))

  ;; moved before frame-command-loop because frame-command-loop
  ;; references the class EXTENDED-STREAM-PANE
  ("db-extended-stream");

  ("frame-command-loop" :load-before-compile ("clim-defs"))
  ("display-menu-group" :load-before-compile ("clim-defs"))
;  ("default-application" :load-before-compile ("define-program"))
  ("silica-tracking-pointer")
						;("tracking-pointer")
  ("dragging-output" :load-before-compile ("silica-tracking-pointer"))

  ("accept-values" :load-before-compile ("clim-defs" "incremental-redisplay"))
  ("item-list-manager")


  ("old-clim-compatibility")

  ("stream-trampolines" :load-before-compile (;; "defprotocol"
					      "graphics-output-recording"
					      ;; "stream-defprotocols"
					      ))
    
  )

(defsys:defsystem clim-demo (:default-pathname (make-clim-pathname "demo")
			     :needed-systems (silica))
  ("aaai-demo-driver")
  ("graphics-demos" :load-before-compile ("aaai-demo-driver"))
  ("new-cad-demo"   :load-before-compile ("aaai-demo-driver"))
  ("navdata"	    :load-before-compile ())
  ("navfun"         :load-before-compile ("aaai-demo-driver" "navdata"))
  ("puzzle"         :load-before-compile ("aaai-demo-driver"))
  ("listener"       :load-before-compile ("aaai-demo-driver"))
  )

#+Genera
(defvar *turn-declares-into-asserts* nil)

#+Genera
(defvar *error-on-invalid-declares* nil)

#+Genera
(defun make-assert-forms-from-var-dcls (var-dcls)
  (let ((assert-forms nil))
    (when var-dcls
      (dolist (var var-dcls)
	(destructuring-bind (var-name . vards) var
	  (let ((type-dcl (assoc 'cl:type vards))
		(ignore-dcl (assoc 'ignore vards)))
	    ;; if this variable is ignored we can't
	    ;; do this.  Anyway, the erroneous condition
	    ;; will almost certainly be caught by someone else.
	    (unless (and ignore-dcl
			 (cdr ignore-dcl))
	      (when (and type-dcl
			 (subtypep (cdr type-dcl) 'number)
			 #+Ignore
			 (clos:find-class (cdr type-dcl) nil))
		(push 
		  `(unless (or (null *error-on-invalid-declares*)
			       (typep ,var-name ',(cdr type-dcl)))
		     (cerror "Go ahead anyway."
			     "The variable ~S is not of type ~S."
			     ',var-name ',(cdr type-dcl)))
		  #+Ignore `(assert (typep ,var-name ',(cdr type-dcl)))
		  assert-forms)))))))
    (nreverse assert-forms)))

#+Genera
(advise compiler:parse-declares :after declare->assert nil
  (when *turn-declares-into-asserts*
    (let* ((body (second values))
	   (var-dcls (third values))
	   (assert-forms (make-assert-forms-from-var-dcls var-dcls)))
      (when assert-forms
	(setf (second values)
	      (append assert-forms body))))))

(defun compile-and-load (system &rest keys &key (propagate t) trace
						speed safety
						#+Genera declares
			 &allow-other-keys)
  (setf keys (copy-list keys))
  (dolist (key '(:propagate :trace :speed :safety #+Genera :declares))
    (remf keys key))
  (defsys:with-compiler-options (:speed speed :safety safety)
    (defsys:with-delayed-compiler-warnings
      ;; Avoid annoying confirmations for REGION-EDGES*, and frame vs. command-processor stuff
     (let (#+genera (si:inhibit-fdefine-warnings ':just-warn)
	   #+genera (*turn-declares-into-asserts* declares)
	   )
       #+Genera
       (when declares
	 (format t "*** Note:  Generating ASSERTs for DECLAREs. ***"))
       (apply #'defsys:compile-system system :propagate propagate :trace trace 
	      keys)
       (defsys:load-system system :trace trace)))))

(defun compile-clim (&rest keys)
  (apply #'compile-and-load 'clim keys))

(defun load-clim (&rest load-system-args)
  (apply #'defsys:load-system 'clim load-system-args))

#+ignore
(defun clim-files-with-subdir ()
  (let ((files '(nil)))
    (dolist (sys '(clim-utils silica ptk ws))
      (let ((subdir (concatenate 'string (string-downcase (string sys)) "/")))
	(nconc files
	       (mapcar #'(lambda (mod)
			   (concatenate 'string
					subdir (defsys::module-name mod)))
		       (defsys::system-module-list
			(defsys::lookup-system sys))))))
    (cdr files)))

(defun clim-files (&key (sys '(clim-utils silica ptk ws)) subdir-p)
  (declare (ignore subdir-p))
  (let ((files '(nil)))
    (dolist (sys (if (listp sys) sys (list sys)))
      (nconc files
	     (mapcar #'(lambda (mod)
			 (defsys::module-name mod))
		     (defsys::system-module-list
		      (defsys::lookup-system sys)))))
    (cdr files)))

(defun module-feature-p (module)
  (let ((features (defsys::module-features module)))
    (labels ((feature-p (feature)
	       (cond ((eql feature t) t)
		     ((symbolp feature)
		      (member feature *features* :test #'string-equal))
		     (t (let ((operator (first feature))
			      (features (rest feature)))
			  (ecase operator
			    (and (every #'feature-p features))
			    (or (some #'feature-p features))
			    (not (not (feature-p (first features))))))))))
      (feature-p features))))

;;; Add modules specified in the DEFSYSTME (that aren't commented
;;; out at read-time via #+)
(defun all-clim-modules ()
  (let ((systems-already-listed nil))
    (labels ((modules-of-system (sys)
	       (unless (find sys systems-already-listed)
		 (push sys systems-already-listed)
		 (let ((modules-list nil))
		   (let* ((system (defsys::lookup-system sys))
			  (subsystems (defsys::system-needed-systems system))
			  (modules (defsys::system-module-list system)))
		     (dolist (subsys subsystems)
		       (setq modules-list (append modules-list (modules-of-system subsys))))
		     (setq modules-list (append modules-list modules)))
		   modules-list))))
      (modules-of-system 'clim))))

;;; Only those modules that would be loaded/compiled in the current
;;; environment.  That is, respect the :FEATURES specs.
(defun current-clim-modules ()
  (let ((modules (all-clim-modules)))
    (remove-if-not #'module-feature-p modules)))

(defun all-clim-files ()
  (let ((modules (all-clim-modules)))
    (mapcar #'defsys::module-src-path modules)))

(defun current-clim-files ()
  (let ((modules (current-clim-modules)))
    (mapcar #'defsys::module-src-path modules)))

#+genera
(defun select-clim-as-tag-table ()
  ;; Use "current" files, since others will be in non-existant
  ;; packages and so forth.
  (let ((files (current-clim-files)))
    (zwei:select-file-list-as-tag-table files "CLIM Silica")))

;;; Minimize size of compiled code for CCL

#+ccl
(setq ccl:*save-doc-strings* nil
      ccl:*save-definitions* nil
      ccl:*record-source-file* nil
      ccl:*save-local-symbols* nil
      ccl:*fasl-save-local-symbols* nil)

 
#+ccl-2
(progn

(defmethod rewrite-condition ((condition condition) id)
  (declare (ignore id))
  ;; Default to not rewriting.
  nil)

(defvar *rewrite-ids* nil)

(defmacro define-condition-rewrite (name superclasses &optional slots
                                            &rest options)
  (let* ((bucket (assoc :rewrite options))
         (rewrite (rest bucket)))
    (setq options (remove bucket options))
    (destructuring-bind (rewrite-class rewrite-fun &rest rewrite-args) rewrite
      (let ((cond (make-symbol "CONDITION")))
        `(progn (define-condition ,name ,superclasses ,slots ,@options)
                (defmethod rewrite-condition ((,cond ,rewrite-class)
                                              (id (eql ',name)))
                  (,rewrite-fun ,cond ,@rewrite-args))
                (pushnew ',name *rewrite-ids*))))))

(defun invoke-rewrite-condition (condition)
  (dolist (id *rewrite-ids*)
    (rewrite-condition condition id)))

(defun rewrite-simple (error format-string signal-function &rest arg-tests)
  (declare (dynamic-extent arg-tests))
  (when (and (string= (simple-condition-format-string error)
                      format-string)
             (every #'funcall
                    arg-tests
                    (simple-condition-format-arguments error)))
    (funcall signal-function error)))

(defun signal-file-not-found (error)
  (let ((pathname (first (simple-condition-format-arguments error))))
    (error 'file-not-found :pathname pathname)))

(defun path-or-string-p (path)
  (or (pathnamep path)
      (stringp path)))

(define-condition-rewrite file-not-found (file-error)
  ()
  (:report (lambda (condition stream)
             (format stream "File ~S does not exist."
                     (file-error-pathname condition))))
  (:rewrite simple-error rewrite-simple "File ~S does not exist."
            #'signal-file-not-found #'path-or-string-p))

(defmacro with-error-rewrites (&body body)
  `(handler-bind ((error #'invoke-rewrite-condition))
     ,@body))
                                
) ;; End #+CCL-2

#-CCL-2
(defmacro with-error-rewrites (&body body)
  `(progn ,@body))

#+(and ansi-90 (not Allegro))
(defun reload-clim-translations ()
  (with-simple-restart (error "Skip loading CLIM translations.")
    (block OK
      (loop
        (with-simple-restart (error "Retry loading CLIM translations.")
          (return-from OK
            (handler-case (with-error-rewrites
                            (load-logical-pathname-translations "CLIM"))
              (file-error (error)
                 (warn "~&You should place the suitable translations in the following file:~@
                          ~A"
                       (file-error-pathname error))))))))))

#+ccl-2
(unless (member 'reload-clim-translations ccl:*restore-lisp-functions*
                :key #'ccl:function-name)
  (setq ccl:*restore-lisp-functions*
        (append ccl:*restore-lisp-functions*
                (list #'reload-clim-translations))))

#+(and Ansi-90 (not Allegro))
(defvar *patch-major-version* 0)

#+(and Ansi-90 (not Allegro))
(defun load-clim-patches ()
  (with-simple-restart (error "Skip loading CLIM patches.")
    (block OK
      (loop
        (with-simple-restart (error "Retry loading CLIM patches.")
          (return-from OK
            (let ((error-path nil))
              (handler-case
                (with-error-rewrites
                  (loop with default = "CLIM:SYS;PATCH;pat.lisp"
                        for minor from 1
                        for patname = (format nil "CLIMPat-~D-~D"
                                              *patch-major-version*
                                              minor)
                        for lpath = (make-pathname :name patname
                                                   :type "lisp"
                                                   :defaults default)
                        for fpath = (make-pathname :name patname
                                                   :type "fasl"
                                                   :defaults default)
                        for path = (or (probe-file fpath)
                                       (probe-file lpath))
                        while path
                        do
                        (setq error-path path)
                        (load path)))
                (file-error (error)
                   (warn "~&An error was encountered referencing patch file ~S."
                       (file-error-pathname error)))
                (error (error)
                   (warn "~&An error was encountered loading patch file ~S.~%~A"
                         error-path
                         error))))))))))


#+ccl-2
(unless (member 'load-clim-patches ccl:*restore-lisp-functions*
                :key #'ccl:function-name)
  (setq ccl:*restore-lisp-functions*
        (append ccl:*restore-lisp-functions*
                (list #'load-clim-patches))))

#+ccl-2
(defun dump-clim (&key (version *patch-major-version*)
                       (name (format nil "CLIM 0.9.1d~D" version)))
  (setq *patch-major-version* version)
  (format t "~&~%== Dumping CLIM version ~D.0~%"
          *patch-major-version*)
  (save-application (merge-pathnames name "CLIM:")
                    :init-file "ccl;init-clim"))

#+genera
(defun find-useless-externals (&optional (package "CLIM2"))
  (let ((useless nil))
    (do-external-symbols (sym package)
      (unless (or (boundp sym)
		  (fboundp sym)
		  ;; How do you check for a type?
		  #+genera
		  (get sym 'deftype)
		  (clos:find-class sym nil))
	(push sym useless)))
    useless))
