;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-STREAM; Base: 10; Lowercase: Yes -*-

;; $fiHeader: input-protocol.lisp,v 1.5 91/03/29 18:00:57 cer Exp $

(in-package "CLIM-STREAM")

"Copyright (c) 1990, 1991 Symbolics, Inc.  All rights reserved.
 Portions copyright (c) 1989, 1990 International Lisp Associates."

;;; This file implements our extended input protocol on top of the
;;; proposed "standard" protocol defined in CL-STREAMS.LISP.


;;; This is the class that you mix in to any extended input stream
;;; implementation that you define.  It exists only to provide the
;;; extended-input-stream-p method and to hang
;;; implementation-independent code (see the STREAM-READ-GESTURE :AROUND
;;; method below).
(defclass basic-extended-input-protocol
	  (fundamental-character-input-stream)
     ())

(define-protocol-p-method extended-input-stream-p basic-extended-input-protocol)

(defmethod stream-read-gesture :around
	   ((stream basic-extended-input-protocol)
	    &key timeout peek-p
	    (input-wait-test *input-wait-test*)
	    (input-wait-handler *input-wait-handler*)
	    (pointer-button-press-handler *pointer-button-press-handler*))
  ;; All output should be visible before you do any input.
  ;;--- Should this call FINISH-OUTPUT instead?
  (when (output-stream-p stream)
    (stream-force-output stream))
  (call-next-method stream 
		    :timeout timeout :peek-p peek-p
		    :input-wait-test input-wait-test
		    :input-wait-handler input-wait-handler
		    :pointer-button-press-handler pointer-button-press-handler)
  )

;;; Our implementation of the extended-input protocol.
(defclass input-protocol-mixin
	 (basic-extended-input-protocol)
    ((input-buffer :accessor stream-input-buffer
		   :initarg :input-buffer)
     ;; Initialized by the per-implementation initialization code
     (pointers :accessor stream-pointers :initarg :pointers)
     (pointer-motion-pending :initform nil) ;--- "primary" pointer
     ;; --- commented out 12/22/91 doughty
     #+Ignore
     (primary-pointer :accessor stream-primary-pointer
		      :initarg primary-pointer)
     (text-cursor :accessor stream-text-cursor
		  :initarg :text-cursor))
  (:default-initargs
    :input-buffer (make-queue)
    :pointers nil
    :text-cursor (make-instance 'text-cursor)))

;;; Copied from stream-input-protocol 12/22/91 --- doughty
(defmethod stream-primary-pointer ((stream input-protocol-mixin))
  (let ((port (port stream)))
    (when port
      (or (port-pointer port)
	  (setf (port-pointer port) (make-instance 'pointer))))))

(defmethod initialize-instance :after ((stream input-protocol-mixin)
				       &key (initial-cursor-active t))
  (with-slots (text-cursor) stream
    (when text-cursor
      (setf (cursor-stream text-cursor) stream)
      (setf (cursor-active text-cursor) initial-cursor-active)))
  ;; --- This code should be per-port, since only the port can know how many
  ;; pointing devices there are.  We also want to share pointer objects among
  ;; all the windows on a port.
  #-Silica
  (with-slots (pointers primary-pointer) stream
    (setf primary-pointer
	  (first (or pointers
		     (setf pointers (list (make-instance 'pointer))))))))

#+Ignore ; --- slot no longer exists 12/22/91 doughty
(defmethod (setf stream-pointers) :after (new-pointers (stream input-protocol-mixin))
  (with-slots (primary-pointer) stream
    (unless (member primary-pointer new-pointers)
      (setf primary-pointer (first new-pointers)))))

;;; --- Cross-protocol violation here because the cursor-position* is a method on
;;; --- extended-OUTPUT-protocol right now.  Secondly, this should probably be a method
;;; --- on the abstract class, anyway.
#+Silica
(defmethod stream-set-cursor-position* :after ((stream input-protocol-mixin) x y)
  (let ((cursor (stream-text-cursor stream)))
    (when cursor
      (cursor-set-position* cursor x y))))

;;; --- Ditto for this method.  Added 1/4/92 by doughty to cause the cursor to always
;;; --- be properly updated.
#+Silica
(defmethod stream-set-cursor-position*-internal :after ((stream input-protocol-mixin) x y)
  (let ((cursor (stream-text-cursor stream)))
    (when cursor
      (cursor-set-position* cursor x y))))

(defmethod initialize-menu :before (port (menu input-protocol-mixin) associated-window)
  (declare (ignore port associated-window))
  (let ((cursor (stream-text-cursor menu)))
    (when cursor
      (setf (cursor-active cursor) nil))))

(defmethod handle-repaint :after 
	   ((stream input-protocol-mixin) (region nowhere) &key &allow-other-keys)
  ;; Repainting nowhere, don't repaint the cursor
  )

(defvar *cursor-repaint-rectangle* (make-bounding-rectangle 0 0 0 0))

(defmethod handle-repaint :after
	   ((stream input-protocol-mixin) region &key &allow-other-keys)
  (let ((cursor (stream-text-cursor stream))
	(viewport (pane-viewport stream)))
    (when (and cursor viewport)
      (when (and (cursor-active cursor)
		 (cursor-state cursor)
		 (cursor-focus cursor))
	(multiple-value-bind (x y)
	    (bounding-rectangle* cursor)
	  (multiple-value-bind (width height)
	      (cursor-width-and-height-pending-protocol cursor)
	    (let ((cursor-rect 
		    (bounding-rectangle-set-edges *cursor-repaint-rectangle*
						  0 0 (+ x width) (+ y height))))
	      (when (region-intersects-region-p region cursor-rect)
		(note-cursor-change cursor 'cursor-active t t)))))))))

(defmethod pointer-motion-pending ((stream input-protocol-mixin)
				   &optional
				   (pointer #+Ignore (stream-primary-pointer stream)))
  (declare (ignore pointer))					;---
  (with-slots (pointer-motion-pending) stream
    (prog1 pointer-motion-pending
	   (setf pointer-motion-pending nil))))

(defmethod (setf pointer-motion-pending) (new-value
					   (stream input-protocol-mixin)
					   &optional (pointer 
						       #+Ignore (stream-primary-pointer stream)))
  (declare (ignore pointer))					;---
  (with-slots (pointer-motion-pending) stream
    (setf pointer-motion-pending new-value)))

#-Silica
(progn
(defmethod (setf window-visibility) :after (visibility (stream input-protocol-mixin))
  (declare (ignore visibility))
  (ensure-pointer-window stream))

(defmethod window-stack-on-top :after ((stream input-protocol-mixin))
  (ensure-pointer-window stream))

(defmethod window-stack-on-bottom :after ((stream input-protocol-mixin))
  (ensure-pointer-window stream))

(defun ensure-pointer-window (window)
  (dolist (pointer (stream-pointers window))
    (set-pointer-window-and-location window pointer)))
)
#-Silica
(defmethod window-set-viewport-position* :around ((stream input-protocol-mixin) new-x new-y)
  (declare (ignore new-x new-y))
  (let ((cursor (stream-text-cursor stream)))
    (if cursor
	;; Shifting the viewport may need to redraw the cursor
	(multiple-value-bind (x y)
	    (cursor-position* cursor)
	  (call-next-method)
	  (cursor-set-position* cursor x y))
	(call-next-method))))

#+Silica
(progn 

;;; --- assumes that input-protocol-mixin will be mixed in with a sheet that
;;; has standard-input-contract
(defmethod queue-event ((stream INPUT-PROTOCOL-MIXIN)
			(event key-press-event)
			&key &allow-other-keys)
  (let ((char (event-char event))
	(keysym (event-keysym event)))
    ;; this probably wants to be STRING-CHAR-P, but that will still require some thought.
    (cond ((and char (standard-char-p char)) (queue-put (stream-input-buffer stream) char))
	  ((and keysym
		(not (typep keysym 'modifier-keysym))
	   (queue-put (stream-input-buffer stream) event)))
	  (keysym
	   ;; must be a shift keysym
	   ;; must update the pointer shifts.
	   (let ((pointer (stream-primary-pointer stream)))
	     (when pointer
	       (setf (pointer-button-state pointer) (event-input-state event)))
	     )))))


(defmethod queue-event ((stream INPUT-PROTOCOL-MIXIN) 
			(event key-release-event)
			&key &allow-other-keys)
  ;;--- key state table?  Not unless all sheets are helping maintain it.
  (let ((keysym (event-keysym event)))
    (when (and keysym (typep keysym 'modifier-keysym))
      ;; update the pointer shifts.
      (let ((pointer (stream-primary-pointer stream)))
	(when pointer
	  (setf (pointer-button-state pointer) (event-input-state event)))
	)))
  nil)

;;;--- this should be in CLOS, or maybe is already under some other name?
;;; --- Just replace it with copy-event...
#+PCL
(defun copy-instance (instance)
  (cond ((string-equal pcl::*pcl-system-date* "5/22/89  Victoria Day PCL")
	 (unless (pcl::iwmc-class-p instance)
	   (error "Can't copy the instance ~S" instance))
	 (let* ((class (class-of instance))
		(copy (pcl::allocate-instance class))
		(wrapper (pcl::class-wrapper class))
		(layout (pcl::wrapper-instance-slots-layout wrapper))
		(old-slots (pcl::iwmc-class-static-slots instance))
		(new-slots (pcl::iwmc-class-static-slots copy)))
	   (dotimes (i (length layout))
	     (setf (svref new-slots i) (svref old-slots i)))
	   copy))
	;; assume "pcl for the 90's"
	(t
	 (unless (pcl::standard-class-p (class-of instance))
	   (error "Can't copy the instance ~S" instance))
	 (let* ((class (class-of instance))
		(copy (pcl::allocate-instance class))
		(wrapper (pcl::class-wrapper class))
		(layout (pcl::wrapper-instance-slots-layout wrapper))
		(old-slots (pcl::std-instance-slots instance))
		(new-slots (pcl::std-instance-slots copy)))
	   (dotimes (i (length layout))
	     (setf (svref new-slots i) (svref old-slots i)))
	   copy))))

#-(or PCL ccl-2)
(defun copy-instance (instance)
  (let* ((class (class-of instance))
	 (copy (allocate-instance class)))
    (dolist (slot (clos:class-slots class))
      (let ((name (clos:slot-definition-name slot)))
	(setf (slot-value copy name) (slot-value instance name))))
    copy))

#+ccl-2
(defun copy-instance (instance)
  (let* ((class (class-of instance))
	 (copy (allocate-instance class)))
    ;; We don't have to copy the CLASS slots
    (dolist (slot (ccl:class-instance-slots class))
      (let ((name (ccl:slot-definition-name slot)))
	(setf (slot-value copy name) (slot-value instance name))))
    copy))

;;; --- need to modularize stream implementation into fundamental and extended
;;; layers
;;; so that we can tell when to queue up non-characters into the stream.
;;; These don't need to set pointer-motion-pending because they synchronize through the io buffer.
(defmethod queue-event ((stream input-protocol-mixin) 
			(event button-press-event)
			&key &allow-other-keys)
  (queue-put (stream-input-buffer stream) (copy-instance event)))

(defmethod queue-event ((stream input-protocol-mixin) 
			(event button-release-event)
			&key &allow-other-keys)
  ;; --- What to do such that tracking-pointer can really
  ;; --- see these.
  )

;;; --- Handle "clicks" differently than press and release?
;;; so that we can tell when to queue up non-characters into the stream.
(defmethod queue-event ((stream input-protocol-mixin) 
			(event button-click-event)
			&key &allow-other-keys)
  (queue-put (stream-input-buffer stream) (copy-instance event)))

(defmethod queue-event ((stream input-protocol-mixin) 
			(event pointer-motion-event)
			&key &allow-other-keys)
  (let ((pointer (stream-primary-pointer stream)))
    (pointer-set-position* pointer (event-x event) (event-y event))
    (pointer-set-native-position pointer (event-native-x event) (event-native-y event))
    (setf (pointer-button-state pointer) (event-input-state event))
    (setf (pointer-window pointer) stream)
    (setf (pointer-motion-pending stream pointer) T)))

(defmethod queue-event :after ((stream input-protocol-mixin)
			       (event pointer-enter-event)
			       &key)
  (let ((text-cursor (stream-text-cursor stream)))
    (when text-cursor (setf (cursor-focus text-cursor) t))))

(defmethod queue-event :before ((stream input-protocol-mixin)
				(event pointer-exit-event)
				&key)
  ;; what about unhighlighting highlighted presentations?
  (let ((text-cursor (stream-text-cursor stream)))
    (when text-cursor (setf (cursor-focus text-cursor) nil))))

(defmethod sheet-transformation-changed :after ((stream input-protocol-mixin) &key)
  (let ((pointer (stream-primary-pointer stream)))
    (when pointer (pointer-decache pointer))))

) ; End of #+Silica progn

;;; What file should this go in?
(defparameter *abort-gestures* '(:abort))
(defparameter *end-gestures* '(:end))

;;; maybe this should really be characters as in 1.0?
(defvar *accelerator-gestures* nil)
(defvar *accelerator-numeric-argument* nil)


;;--- Inline this?  It's called for every call to STREAM-READ-GESTURE
;;; Internal, no user will ever use this.
(defmacro with-cursor-state ((state stream) &body body)
  (let ((text-cursor (gensymbol 'text-cursor))
	(old-state (gensymbol 'old-state))
	(abort-p (gensymbol 'abort-p))
	(new-state (gensymbol 'new-state)))
    `(let* ((,text-cursor (stream-text-cursor ,stream))
	    (,old-state (cursor-state ,text-cursor))
	    (,abort-p t)
	    (,new-state ,state))
       (unwind-protect
	   (progn (when ,text-cursor
		    (cond ((eql ,old-state ,new-state))
			  (t (setf (cursor-state ,text-cursor) ,new-state)
			     (setf ,abort-p nil))))
		  ,@body)
	 (when ,text-cursor
	   (unless ,abort-p
	     (setf (cursor-state ,text-cursor) ,old-state)))))))

;;; This is what you use as a user/programmer to "turn the cursor off"
(defmacro with-cursor-visibility ((active stream) &body body)
  (let ((text-cursor (gensymbol 'text-cursor))
	(old-active (gensymbol 'old-active))
	(abort-p (gensymbol 'abort-p))
	(new-active (gensymbol 'new-active)))
    `(let* ((,text-cursor (stream-text-cursor ,stream))
	    (,old-active (cursor-active ,text-cursor))
	    (,abort-p t)
	    (,new-active ,active))
       (unwind-protect
	   (progn (when ,text-cursor
		    (cond ((eql ,old-active ,new-active))
			  (t (setf (cursor-active ,text-cursor) ,new-active)
			     (setf ,abort-p nil))))
		  ,@body)
	 (when ,text-cursor
	   (unless ,abort-p
	     (setf (cursor-active ,text-cursor) ,old-active)))))))

(defmethod stream-read-gesture ((stream input-protocol-mixin)
				&key timeout peek-p
				     (input-wait-test *input-wait-test*)
				     (input-wait-handler *input-wait-handler*)
				     pointer-button-press-handler)
  (declare (ignore pointer-button-press-handler))
  (with-cursor-state (t stream)
    (loop
      (multiple-value-bind (input-happened flag)
	  (stream-input-wait
	    stream
	    :timeout timeout :input-wait-test input-wait-test)
	(case flag
	  (:timeout
	    (return-from stream-read-gesture (values nil :timeout)))
	  (:input-wait-test
	    ;; only call the input-wait-handler if we didn't get a first-rate
	    ;; gesture back from stream-input-wait.
	    (when input-wait-handler
	      (funcall input-wait-handler stream)))
	  (otherwise 
	    (when input-happened
	      (with-slots (input-buffer) stream
		(let ((gesture (queue-get input-buffer)))
		  (when gesture
		    (let ((new-gesture (receive-gesture (or *outer-self* stream) gesture)))
		      (when new-gesture
			(when peek-p (queue-unget input-buffer gesture))
			(return-from stream-read-gesture new-gesture)))))))))))))

;;; --- Think clearly about what the class hierarchy should be here.
;;; --- Why didn't it work to specialize on basic-extended-input-protocol-mixin??
(defmethod receive-gesture ((stream input-protocol-mixin)
			    ;; any button events that make it into the stream...
			    (gesture button-event))
  ;; --- the around method may have to bind the specials back to the args
  ;; --- to be on the safe side...
  (when *pointer-button-press-handler*
    ;; This may throw or something, but otherwise we will return the gesture
    (funcall *pointer-button-press-handler* stream gesture))
  gesture)

(defmethod receive-gesture ((stream input-protocol-mixin) (gesture (eql ':resynchronize)))
  ;; signal resynchronize, naturally
  (throw 'resynchronize t))

(defmethod receive-gesture ((stream input-protocol-mixin) (gesture list))
  ;; --- We do this here rather than by throwing, because
  ;; --- we don't want you to lose the accumulated "Show Directory ..."
  ;; --- you've already typed just because a damage event came in.
  (let ((*outer-self* nil))
    (receive-list-gesture stream (first gesture) (rest gesture)))
  ;; don't return this gesture to the higher level
  nil)

(defmethod receive-list-gesture ((stream input-protocol-mixin) (type (eql 'redisplay-pane)) 
				 args)
  (redisplay-frame-pane (first args)))

(defmethod receive-list-gesture ((stream input-protocol-mixin)
				 (type (eql 'execute-frame-command)) 
				 command)
  ;; instead of executing here, we throw to the command catch tag if there is one
  (let ((command-function (first (second command))))
    ;; --- Other side of horrible kludge (intern because this is loaded before
    ;; ws) 
    (cond ((eql command-function (intern "XX-DO-MENU-FOR" 'ws))
	   (apply #'execute-frame-command command))
	  (t
	   (dolist (this-context *input-context*)
	     (let* ((context (first this-context))
		    (tag (second this-context)))
	       (when (presentation-subtypep 'command context)
		 (throw tag (values (second command) context)))))
	   ;; if no command context applies, then we can do no better than execute here and
	   ;; resynchronize
	   (apply #'execute-frame-command command)
	   ;; probably wants to resynchronize, right?
	   ;; should signal, though
	   (throw 'command-executed T)))))

(defmethod receive-list-gesture ((stream input-protocol-mixin) (type (eql 'stop-frame)) args)
  (apply #'stop-frame args))

(defmethod receive-gesture ((stream input-protocol-mixin) (gesture repaint-event))
  ;; Handle synchronous repaint request
  (repaint-sheet (event-sheet gesture) (repaint-region gesture))
  (free-event gesture)
  ;; don't return.
  nil)

;;; default method
(defmethod receive-gesture ((stream input-protocol-mixin) gesture)
  ;; don't translate it
  gesture
  )

;;; Ok, there's still an issue.  Just how do we want to handle this, given
;;; that it's non-trivial to turn a character back into a key-press event.
;;; This issue is still pending.
#+Ignore
(defmethod receive-gesture :around ((stream input-protocol-mixin) (gesture character))
  (when (member gesture *abort-characters*)
    (abort))
  (call-next-method))

(defmethod receive-gesture :around ((stream input-protocol-mixin) (gesture key-press-event))
  (when (keysym-and-shift-mask-member
	  (event-keysym gesture)
	  (event-input-state gesture)
	  *accelerator-gestures*
	  (port stream))
    (signal 'accelerator-gesture
	    :event gesture
	    :numeric-argument (or *accelerator-numeric-argument* 1)))
  (when (keysym-and-shift-mask-member
	  (event-keysym gesture)
	  (event-input-state gesture)
	  *abort-gestures*
	  (port stream))
    ;; only echo "[Abort]" when the cursor is visible.
    ;; Maybe this predicate should be done by the above
    (let ((cursor (slot-value stream 'text-cursor)))
      (when (and (cursor-active cursor)
		 (cursor-state cursor)
		 (cursor-focus cursor))
	(write-string "[Abort]" stream)
	(force-output stream)))
    (abort))
  (call-next-method))

;;; This allows us to write RECEIVE-GESTURE methods for our own new gesture types.
;;; Perhaps we should punt the list form of gesture, in favor of defining new classes
;;; and specializing that way...

;;; Now, how does the input editor work in this scheme?
;;; Can BUTTON-PRESS-EVENTS be handled using this mechanism, as well, eliminating
;;; the need for the :AROUND method on STREAM-READ-GESTURE?

;;; This function is just a convenience for the programmer, defaulting the
;;; keyword :STREAM argument to *standard-input*.  The application can call
;;; stream-read-gesture directly.
(defun read-gesture (&rest args &key (stream *standard-input*) &allow-other-keys)
  (declare (arglist &key (stream *standard-input*)
			 timeout peek-p input-wait-test input-wait-handler
			 pointer-button-press-handler))
  (declare (dynamic-extent args))
  (with-rem-keywords (keywords args '(:stream))
    (apply #'stream-read-gesture stream keywords)))

(defmethod stream-unread-gesture ((stream input-protocol-mixin) gesture)
  (with-slots (input-buffer) stream
    (queue-unget input-buffer gesture)))

(defun unread-gesture (gesture &key (stream *standard-output*))
  (stream-unread-gesture stream gesture))

;;; Our extended input protocol replaces this method and others below
;;; (from FUNDAMENTAL-INPUT-STREAM) so that the input-wait and other
;;; behavior works even when the application calls standard CL stream
;;; operations.
(defmethod stream-read-char ((stream input-protocol-mixin))
  (let ((gesture nil))
    (loop
      ;; Don't pass off a pointer-button-press-handler that ignores clicks,
      ;; we want this to be runnable inside a WITH-INPUT-CONTEXT.
      (setq gesture (stream-read-gesture (or *outer-self* stream)))
      (when (characterp gesture)
	(return-from stream-read-char gesture))
      (beep stream))))				;??

(defmethod stream-unread-char ((stream input-protocol-mixin) character)
  (stream-unread-gesture (or *outer-self* stream) character))

;;; Again, we only need this function (as opposed to using :timeout)
;;; because the X3J13 proposal includes it explicitly.
(defmethod stream-read-char-no-hang ((stream input-protocol-mixin))
  (let ((gesture nil))
    (loop
      ;; Don't pass off a pointer-button-press-handler that ignores clicks,
      ;; we want this to be runnable inside a WITH-INPUT-CONTEXT.
      (setq gesture (stream-read-gesture (or *outer-self* stream) :timeout 0))
      (when (or (null gesture) (characterp gesture))
	(return-from stream-read-char-no-hang gesture)))))

(defmethod stream-peek-char ((stream input-protocol-mixin))
  ;; stream-listen used to return the char, but Hornig says it has to return T
  (or nil ;(stream-listen stream)
      (let ((char (stream-read-char (or *outer-self* stream))))
	(prog1 char (stream-unread-gesture (or *outer-self* stream) char)))))

;;; We think that the "standard" demands that this only see characters.
;;; However, it does not want to flush any pending action elements that
;;; might precede the character, 'cause LISTEN should have no side effects.
(defmethod stream-listen ((stream input-protocol-mixin))
  #-Silica (stream-event-handler stream :timeout 0)	;Process pending keyboard input events
  (with-slots (input-buffer) stream
    (when (queue-empty-p input-buffer)
      (return-from stream-listen nil))
    ;; map over the input buffer looking for characters.  
    ;; If we find one, return true
    (flet ((find-char (gesture)
	     (when (characterp gesture)
	       (return-from stream-listen T))))
      (declare (dynamic-extent #'find-char))
      (map-over-queue #'find-char input-buffer))
    nil))

(defmethod stream-read-line ((stream input-protocol-mixin))
  (with-temporary-string (result :length 100 :adjustable t)
    ;; Read the first char separately, since we are supposed to react differently
    ;; to EOF on the first char than we do when in the middle of a line.
    (let ((ch (stream-read-char stream)))
      ;;--- Reconcile various error cases.  When CH is NIL then that probably means
      ;; that the caller supplied error-p nil, and we may have to return the eof-val.
      ;; Of course, we aren't dealing with EOF on our window streams at all.
      (unless (eql ch :eof)
	(loop
	  ;; Process the character
	  (cond ((or (eql ch #\newline)
		     (eql ch :eof))
		 (return-from stream-read-line
		   (evacuate-temporary-string result)))
		(t
		 (vector-push-extend ch result)))
	  (setq ch (stream-read-char stream)))))))

(defmethod stream-clear-input ((stream input-protocol-mixin))
  (with-slots (input-buffer) stream
    (queue-flush input-buffer)))

#+Genera
;; The eof argument is as for the :tyi message
(defmethod stream-compatible-read-char ((stream input-protocol-mixin) &optional eof)
  (let ((char (stream-read-char stream)))
    (cond ((not (eql char ':eof)) char)
	  (eof (error 'sys:end-of-file :stream stream :format-string eof))
	  (t nil))))

#+Genera
;; The eof argument is as for the :tyi-no-hang message
(defmethod stream-compatible-read-char-no-hang ((stream input-protocol-mixin) &optional eof)
  (let ((char (stream-read-char-no-hang stream)))
    (cond ((not (eql char ':eof)) char)
	  (eof (error 'sys:end-of-file :stream stream :format-string eof))
	  (t nil))))

#+Genera
;; The eof argument is as for the :tyipeek message
(defmethod stream-compatible-peek-char ((stream input-protocol-mixin) &optional eof)
  (let ((char (stream-peek-char stream)))
    (cond ((not (eql char ':eof)) char)
	  (eof (error 'sys:end-of-file :stream stream :format-string eof))
	  (t nil))))

;;; Extended Input
(defmethod stream-input-wait ((stream input-protocol-mixin)
			      &key timeout input-wait-test)
  (with-slots (input-buffer) stream
    ;; The assumption is that the input-wait-test function can only change
    ;; its value when an event is received and processed.  The only
    ;; commonly-desired exception is a timeout, which we provide directly.
    (cond ((not (queue-empty-p input-buffer))
	   (return-from stream-input-wait t))
	  ((and input-wait-test (funcall input-wait-test stream))
	   (return-from stream-input-wait (values nil :input-wait-test))))
    ;; Non-silica version is always one-process
    #-Silica
    (loop
      (let ((flag (stream-event-handler stream :timeout timeout
					       :input-wait-test input-wait-test)))
	(cond ((or (eq flag ':timeout) (eq flag ':input-wait-test))
	       (return-from stream-input-wait (values nil flag)))
	      ((not (queue-empty-p input-buffer))
	       (return-from stream-input-wait (values t :input-buffer)))
	      ((and input-wait-test (funcall input-wait-test stream))
	       (return-from stream-input-wait (values nil :input-wait-test))))))
    #+Silica
    (let* ((flag nil)
	   (start-time (get-internal-real-time))
	   (end-time (and timeout
			  (+ start-time
			     (* timeout internal-time-units-per-second)))))
      (flet ((waiter ()
		     (when (not (queue-empty-p input-buffer))
		       (setq flag :input-buffer))
		     (when (and input-wait-test (funcall input-wait-test stream))
		       (setq flag :input-wait-test))
		     (when (and end-time
				(> (get-internal-real-time) end-time))
		       (setq flag :timeout))
		     flag))
	(declare (dynamic-extent #'waiter))
	(port-event-wait (port stream) #'waiter :timeout timeout)
	(when flag
	  (return-from stream-input-wait
	    (values (when (eql flag ':input-buffer) T)
		    flag)))))))

#+Genera
(defmethod stream-compatible-any-tyi
	   ((stream input-protocol-mixin)
	    &optional eof)
  (stream-compatible-any-tyi-internal stream nil eof))

#+Genera
(defmethod stream-compatible-any-tyi-no-hang
	   ((stream input-protocol-mixin)
	    &optional eof)
  (stream-compatible-any-tyi-internal stream 0 eof))

#+Genera
(defun stream-compatible-any-tyi-internal (stream timeout eof)
  (let ((character (stream-read-gesture (or *outer-self* stream) :timeout timeout)))
    (cond ((null character) nil)
	  ((eq character ':eof) (and eof (error "~a" eof)))
	  ((and (characterp character)
		(let ((activation (si:input-editor-option :activation)))
		  (and activation
		       (apply (car activation) character (cdr activation)))))
	   ;; When called from WITH-CLIM-COMPATIBLE-INPUT-EDITING, turn activation characters
	   ;; into the appropriate blips to be compatible with the Genera input editor.
	   (si:ie-make-blip :activation character nil))
	  ((and (characterp character)
		(let ((blip-character (si:input-editor-option :blip-character)))
		  (and blip-character
		       (apply (car blip-character) character (cdr blip-character)))))
	   ;; When called from WITH-CLIM-COMPATIBLE-INPUT-EDITING, turn blip characters
	   ;; into the appropriate blips to be compatible with the Genera input editor.
	   (si:ie-make-blip :blip-character character nil))
	  (t character))))

#+Genera
;;; Return T so READ-CHAR will echo.
(defmethod stream-compatible-interactive ((stream input-protocol-mixin))
  t)

#+Genera
;;; Needed for Y-OR-N-P
(defmethod stream-compatible-input-wait ((stream input-protocol-mixin)
					 whostate function &rest arguments)
  (declare (dynamic-extent arguments)
	   (ignore whostate))
  (stream-input-wait stream :input-wait-test #'(lambda (stream)
						 (declare (sys:downward-function))
						 (declare (ignore stream))
						 (apply function arguments))))

#+Genera
;;; Needed for Y-OR-N-P
(defgeneric stream-compatible-notification-cell (stream)
  (:selector :notification-cell))

#+Genera
(defmethod stream-compatible-notification-cell ((stream input-protocol-mixin))
  nil)


;;; STREAM-POINTER-POSITION* method returns x,y in history coordinates.
;;; Around methods take care of this (see protocol-intermediaries.lisp)
(defmethod stream-pointer-position* ((stream input-protocol-mixin) &key (timeout 0) pointer)
  (stream-pointer-position-in-window-coordinates stream :timeout timeout :pointer pointer))

(defmethod stream-set-pointer-position* ((stream input-protocol-mixin) x y &key pointer)
  (set-stream-pointer-position-in-window-coordinates stream x y :pointer pointer))

;;; The primitive we are interested in.  The pointer is in inside-host-window coords.
#+Silica
(defun stream-pointer-position-in-window-coordinates (stream &key (timeout 0) pointer)
  (let ((pointer (or pointer (stream-primary-pointer stream))))
    (pointer-position* pointer)))

#-Silica
(defun stream-pointer-position-in-window-coordinates (stream &key (timeout 0) pointer)
  ;; Process any pending pointer motion events.
  (stream-event-handler stream :timeout timeout)
  (let ((pointer (or pointer (stream-primary-pointer stream))))
    (multiple-value-bind (left top) (window-offset stream)
      (declare (fixnum left top))
      (if pointer
	  (multiple-value-bind (x y) (pointer-position* pointer)
	    (declare (fixnum x y))
	    (values (the fixnum (- x left)) (the fixnum (- y top))))
	  (values 0 0)))))

(defun set-stream-pointer-position-in-window-coordinates (stream x y &key pointer)
  (declare (fixnum x y))
  (unless pointer (setf pointer (stream-primary-pointer stream)))
  (setf (pointer-position-changed pointer) t)
  #+Silica
  (progn (pointer-set-position* pointer x y)
	 ;; --- maybe this should be moved directly to pointer-set-position?
	 ;; --- that requires that the WINDOW slot of the pointer be
	 ;; --- constantly maintained by the port layer.
	 (let ((port (port stream)))
	   (when port
	     (warp-cursor port stream x y))))
  #-Silica
  (multiple-value-bind (left top) (window-offset stream)
    (declare (fixnum left top))
    (set-stream-pointer-in-screen-coordinates
      stream pointer (the fixnum (+ x left)) (the fixnum (+ y top)))))

#-Silica
(defmethod stream-note-pointer-button-press ((stream input-protocol-mixin)
					     pointer button shift-mask x y)
  (declare (ignore pointer))
  (with-slots (input-buffer) stream
    (queue-put input-buffer
	       ;; X and Y had better be fixnums
	       (make-button-press-event stream x y button shift-mask))))

#-Silica
(defmethod stream-note-pointer-button-release ((stream input-protocol-mixin)
					       pointer button shift-mask x y)
  (declare (ignore pointer))
  (with-slots (input-buffer) stream
    (queue-put input-buffer
	       ;; X and Y had better be fixnums
	       (make-button-release-event stream x y button shift-mask))))

;;; required methods:
;;;   STREAM-EVENT-HANDLER
;;;   STREAM-RESTORE-INPUT-FOCUS
;;;   STREAM-SET-INPUT-FOCUS
