;;; -*- Mode:Lisp; Package: clim-stream; Syntax:COMMON-LISP; Base:10 -*-
;;;
;;; $Header$
;;;
;;; This File contains implementations for various common lisps of a 
;;; %gray-stream, a special stream that forwards implementation requiests to a
;;; real gray stream using the Gray stream protocols.
;;;
;;; If there was a portable way of specifying streams to CLs (e.g. the Gray
;;; Proposal) then this file would not be necessary, but rather we could just
;;; make CLIM streams follow the CL Stream protocol. 
;;;
;;; However, given this isn't the case, we endorse the Gray Proposal as a CL
;;; standard  and make our CLIM streams obey its protocols.
;;; 
;;; Code supporting allegro and symbolics was derived from code from the Deli
;;; Window System.  Thanx, MCC.
;;; Code supporting lucid is from rao.
;;;

(in-package "CLIM-STREAM")

#-(or excl symbolics lucid ccl)   ; slh - added ccl
  (error "CLIM Streams not supported in this Common Lisp")

;;; -------
;;; ALLEGRO

#+excl
(progn
(defun make-%gray-stream (gray-stream)
  (let ((stm (excl::make-vanilla-stream)))
    (setf (excl::sm_describe Stm) #'%gray-stream-describe
	  (excl::sm_flags Stm) 3	; i/o
	  (excl::sm_type Stm)  :%gray-stream
	  (excl::sm_write-char Stm) #'%gray-stream-write-char
	  (excl::sm_write-byte Stm) #'%gray-stream-write-char
	  (excl::sm_write-string Stm) #'%gray-stream-write-string
	  (excl::sm_read-char Stm) #'%gray-stream-read-char
	  (excl::sm_read-byte Stm) #'%gray-stream-read-char
	  (excl::sm_listen Stm) #'%gray-stream-listen
	  (excl::sm_read-char-nh Stm) #'%gray-stream-illegal
	  (excl::sm_misc Stm) #'%gray-stream-misc)
    (setf (excl::sm_fio-fn Stm) gray-stream)
    Stm))

(defun %gray-stream-describe (stream)
  (declare (ignore stream))
  (write-string "A Gray Stream"))

(defun %gray-stream-illegal (stream &rest ignore)
  (declare (ignore ignore))
  (error "illegal operation for canvas stream ~s" stream))

(defun %gray-stream-misc (stream code &optional arg)
  (case code
    (:unread (%gray-stream-unread-char stream arg))
    (:input-edit-p nil)
    (:clear-input (%gray-stream-clear-input stream))
    ((:filename :charpos) nil)
    (:close (%gray-stream-close stream))
    (:force-output (%gray-stream-force-output stream))
    (:clear-output nil)
    (:element-type :string-char)
    (t (error "illegal code to canvas stream: ~s" code))))
) ;end of #+excl progn


;;; -----
;;; LUCID

#+lucid
(progn
  
(lucid:define-stream-type %gray-stream
    (lucid::basic-stream)
  ((gray-stream nil)
   (echo nil))
  (:constructor %make-%gray-stream))

(defun make-%gray-stream (gray-stream)
  (let ((stream (%make-%gray-stream :gray-stream gray-stream)))
    (setf (lucid::basic-stream-element-type stream) 'string-char
	  (lucid::basic-stream-direction stream) :io)
    stream))
   
(lucid::define-stream-method lucid::internal-write-char
    ((stream %gray-stream) char)
  (%gray-stream-write-char stream char))

(lucid::define-stream-method lucid::internal-write-string 
    ((stream %gray-stream) string start end)
  (%gray-stream-write-string stream string start (if end (- end start))))

(lucid::define-stream-method lucid::internal-read-char 
    ((stream %gray-stream) no-hang-p)
  (declare (ignore no-hang-p))
  (%gray-stream-read-char stream))

(lucid::define-stream-method lucid::internal-unread-char
    ((stream %gray-stream) char)
  (%gray-stream-unread-char stream char))

(lucid::define-stream-method lucid::internal-listen ((stream %gray-stream))
  (%gray-stream-listen stream))

(lucid::define-stream-method lucid::internal-clear-input
    ((stream %gray-stream))
  (%gray-stream-clear-input stream))

(lucid::define-stream-method lucid::internal-close ((stream %gray-stream))
  (%gray-stream-close stream))

(lucid::define-stream-method lucid::internal-force-output
    ((stream %gray-stream) wait-p)
  (declare (ignore wait-p))
  (%gray-stream-force-output stream))

) ;; end of LUCID


;;; ---------
;;; SYMBOLICS

#+symbolics
(progn
(scl:defflavor %gray-stream
	((gray-stream nil)
	 (echo nil))
	(si:BIDIRECTIONAL-STREAM)
  :settable-instance-variables)

(defun make-%gray-stream (gray-stream)
  (scl:make-instance '%gray-stream :gray-stream gray-stream))

(scl:defmethod (:tyo %gray-stream) (arg1 &rest rest)
  (declare (ignore rest))
  (%gray-stream-write-char scl:self arg1))

(scl:defmethod (:string-out %gray-stream)
	   (string &optional start end &rest rest)
  (declare (ignore rest))
  (%gray-stream-write-string scl:self
			   string start
			   (if end (- end start))))

(scl:defmethod (:tyi %gray-stream) (&optional arg1 &rest rest)
  (declare (ignore arg1 rest))
  (%gray-stream-read-char scl:self))

(scl:defmethod (:untyi %gray-stream) (arg1 &rest rest)
  (declare (ignore rest))
  (%gray-stream-unread-char scl:self arg1))

(scl:defmethod (:listen  %gray-stream) (&optional arg1 &rest rest)
  (declare (ignore arg1 rest))
  (%gray-stream-listen scl:self))

(scl:defmethod (:clear-input %gray-stream) (&rest ignore)
  (declare (ignore IGNORE))
  (%gray-stream-clear-input scl:self))

(scl:defmethod (:close %gray-stream) (&rest ignore)
  (declare (ignore IGNORE))
  (%gray-stream-close scl:self))

(scl:defmethod (:force-output %gray-stream) (&rest ignore)
  (declare (ignore IGNORE))
  (%gray-stream-force-output scl:self))

(scl:defmethod (:element-type %gray-stream) ()
  :string-char)

(scl:defmethod (:unclaimed-message %gray-stream) (operation &rest rest)
  (sys:stream-default-handler
   scl:self operation (car rest) (cdr rest)))

) ;end of #+symbolics progn

#+ccl
(progn
(defclass %gray-stream (ccl::stream)
  ((gray-stream :initform nil
                :initarg :gray-stream
                :accessor real-gray-stream)
   (echo :initform nil
         :initarg :echo
         :accessor %gray-stream-echo)))

(defun make-%gray-stream (gray-stream)
  (make-instance '%gray-stream :gray-stream gray-stream))

(defmethod ccl:stream-tyo ((stream %gray-stream) char)
  (%gray-stream-write-char stream char))

(defmethod ccl:stream-tyi ((stream %gray-stream))
  (%gray-stream-read-char stream))

(defmethod ccl:stream-untyi ((stream %gray-stream) char)
  (%gray-stream-unread-char stream char))

(defmethod ccl:stream-listen ((stream %gray-stream))
  (%gray-stream-listen stream))

(defmethod ccl:stream-clear-input ((stream %gray-stream))
  (%gray-stream-clear-input stream))

(defmethod ccl:stream-close ((stream %gray-stream))
  (%gray-stream-close stream))

(defmethod ccl:stream-force-output ((stream %gray-stream))
  (%gray-stream-force-output stream))

(defmethod stream-element-type ((stream %gray-stream))
  'character)

(defmethod ccl:stream-eofp ((stream %gray-stream))
  nil)

(defmethod ccl:stream-fresh-line ((stream %gray-stream))
  (%gray-stream-fresh-line stream))

)

;;
;;  The following macros allow access to fields of the gray stream object
;;    in a lisp independent way.
;;    In EXCL the fields are stored in aribitrarily chosen empty fields
;;    of the stream object.  In Symbolics, there are iv's in the flavor
;;    object. 

(defmacro %gray-stream-p (stream)
  #+excl       `(eq (excl::sm_type ,stream)  :%gray-stream)
  #+(or 
     symbolics ccl
     lucid)    `(eq (type-of ,stream) '%gray-stream))

#-ccl
(defmacro real-gray-stream (stream)
  #+excl       `(excl::sm_fio-fn ,stream)
  #+symbolics  `(scl:send ,stream :gray-stream)
  #+lucid      `(%gray-stream-gray-stream ,stream))

(defmacro %gray-stream-char-to-unread (stream)
  #+excl       `(excl::sm_term-svchar ,stream)
  #+symbolics  `(scl:send ,stream :un-tyi-char)
  #+lucid      `(%gray-stream-un-tyi-char ,stream)
  #+ccl        `(w::window-stm-un-tyi-char ,stream))

#-ccl
(defsetf %gray-stream-char-to-unread (stream) (new-char)
  #+excl       `(setf (excl::sm_term-svchar ,stream) ,new-char)
  #+symbolics  `(scl:send ,stream :set-un-tyi-char ,new-char)
  #+lucid      `(setf (%gray-stream-un-tyi-char ,stream) ,new-char))
  
#-ccl
(defmacro %gray-stream-echo (stream)
  #+excl       `(excl::sm_fio-mode ,stream)
  #+symbolics  `(scl:send ,stream :echo)
  #+lucid      `(%gray-stream-echo ,stream))

#-ccl
(defsetf %gray-stream-echo (stream) (new-char)
  #+excl       `(setf (excl::sm_fio-mode ,stream) ,new-char)
  #+symbolics  `(scl:send ,stream :set-echo ,new-char)
  #+lucid      `(setf (%gray-stream-echo ,stream) ,new-char))

;;;
;;; LISP independent helpers that dispatch to GRAY proposed functions
;;;

(defun %gray-stream-write-char (stream ch)
  (stream-write-char (real-gray-stream stream) ch))

(defun %gray-stream-write-string (stream string &optional start length)
  (stream-write-string (real-gray-stream stream) string 
		       start (and length (+ start length))))

(defun %gray-stream-fresh-line (stream)
  (stream-fresh-line (real-gray-stream stream)))

(defun %gray-stream-force-output (stream)
  (stream-force-output (real-gray-stream stream)))

(defun %gray-stream-close (stream &rest keys)
  #+excl (declare (ignore keys))
  #+excl(excl::st-close-down-stream stream)

  #-excl(declare (dynamic-extent keys))
  #-excl(apply #'close (real-gray-stream stream) keys))
  
(defun %gray-stream-read-char (stream)
  (stream-read-char (real-gray-stream stream)))

(defun %gray-stream-unread-char (stream char)
  (stream-unread-char (real-gray-stream stream) char))

(defun %gray-stream-listen (stream)
  (stream-listen (real-gray-stream stream)))

(defun %gray-stream-clear-input (stream)
  (stream-clear-input (real-gray-stream stream)))
