;;; -*- Mode: Lisp; Package: ON-X; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "ON-X")

;;;
;;; Window Manager Services
;;;

(defmethod ring-bell ((port x-port))
  ;; 0 percent from normal
  (let ((display (x-display port)))
    (xlib:bell display  0)
    (xlib:display-force-output display)))

(defmethod warp-cursor ((port x-port) sheet x y)
  (let ((mirror (sheet-mirror! sheet)))
    (when mirror
      (multiple-value-setq (x y)
	(transform-point* (fetch-native-transformation sheet) x y))
      (xlib:warp-pointer mirror (integerize-coordinate x) (integerize-coordinate y)))))

(defmethod cut-buffer  ((port x-port))
  (multiple-value-bind (sequence type format bytes-after)
      (xlib:cut-buffer (x-display port) :buffer 0)
    (declare (ignore type format bytes-after))
    sequence))

(defmethod (setf cut-buffer)  (new-contents (port x-port))
  (setf (xlib:cut-buffer (x-display port) :buffer 0) 
	new-contents))

;;;
;;; Setting Installation
;;;

(defmacro with-wm-hints ((hints sheet &optional x-window) &body body)
  `(let* ((,(or x-window (setq x-window (gensym))) (sheet-mirror ,sheet))
	  (,hints (or (xlib:wm-hints ,x-window)
		      (xlib:make-wm-hints))))
     ,@body
     (setf (xlib:wm-hints ,x-window) ,hints)))

(defmacro with-wm-normal-hints ((hints sheet &optional x-window) &body body)
  `(let* ((,(or x-window (setq x-window (gensym))) (sheet-mirror ,sheet))
	  (,hints (or (xlib:wm-normal-hints ,x-window)
		      (xlib:make-wm-size-hints 
		       :user-specified-position-p nil
		       :user-specified-size-p nil))))
     ,@body
     (setf (xlib:wm-normal-hints ,x-window) ,hints)))

(defmethod install-mirror-settings 
	   ((port x-port) (sheet mirrored-sheet-mixin)
	    &key 
	    (plain      nil plain-sp)
	    (save-under nil save-under-sp)
	    state region title
	    &allow-other-keys)
  
  (let ((mirror (sheet-mirror sheet)))
    
    (with-wm-hints (hints sheet x-window)
      (setf (xlib:wm-hints-input hints) :on))	;ensure we see keystrokes
    
    (when plain-sp
      (setf (xlib::window-override-redirect mirror) 
	    (if plain :on :off)))
    
    (when save-under-sp
      (setf (xlib::window-save-under mirror) 
	    (if save-under :on :off)))
    
    (when state
      (with-wm-hints (hints sheet x-window)
	(unless (eq (xlib:wm-hints-initial-state hints) state)
	  (setf (xlib:wm-hints-initial-state hints) state)
	  (when (sheet-enabled-p sheet)
	    (disable-sheet sheet)
	    (enable-sheet sheet)))))

    (when region
      (with-wm-normal-hints (hints sheet)
	(with-new-native-region (sheet) 
	  (setf (xlib:wm-size-hints-x hints)      new-x
		(xlib:wm-size-hints-y hints)      new-y
		(xlib:wm-size-hints-width hints)  new-w
		(xlib:wm-size-hints-height hints) new-h))))
    
    (when title
      (unless (stringp title) (setq title (string title)))
      (setf (xlib:wm-name (sheet-mirror sheet)) title)
      #+ignore
      (unless (xlib:wm-icon-name (sheet-mirror sheet))
	(setf (xlib:wm-icon-name (sheet-mirror sheet)) title)))))

;;; This is an :AFTER method only so I could easily send it out and
;;; have Bruce Seely try it.  If it works, we should just uncomment out
;;; the last form of the above method (removing the UNLESS test,
;;; since we want the icon title to change every time we change
;;; the frame title.
#+ignore
(defmethod install-mirror-settings :after
	   ((port x-port) (sheet mirrored-sheet-mixin)
	    &key title
	    &allow-other-keys)
  (when title
    (unless (stringp title) (setq title (string title)))
    (setf (xlib:wm-icon-name (sheet-mirror sheet)) title)))

#||

(defmethod install-mirror-icon ((port x-port)
				(sheet mirrored-sheet-mixin)
				icon)
  (when (sheetp icon)
    ;; ??? More error checking to make sure icon is on the same desktop
    (let ((hints (xlib:wm-hints (sheet-mirror sheet)))
	  (xf (sheet-transformation icon)))
      (setf (xlib:wm-hints-icon-window hints) 
	    (sheet-mirror icon))
      (setf (xlib:wm-hints-icon-x hints)
	    (translation-x xf))
      (with-slots (height-pixel) port
	(setf (xlib:wm-hints-icon-y hints)
	      ;; ??? Not accounting for icon height
	      (- height-pixel (translation-y xf))))
      (setf (xlib:wm-hints (sheet-mirror sheet)) hints)
      icon)))

(defmethod install-mirror-icon-title ((port x-port)
				      (sheet mirrored-sheet-mixin)
				      icon-title)
  (when (stringp icon-title)
    (setf (xlib:wm-icon-name (sheet-mirror sheet)) icon-title)))

(defmethod install-mirror-icon-pixmap ((port x-port)
				       (sheet mirrored-sheet-mixin)
				       pixmap)
  (when (pixmapp pixmap)
    (setf (xlib:wm-hints-icon-pixmap (xlib:wm-hints (sheet-mirror sheet)))
	  (realize-pixmap pixmap (port sheet)))))

(defmethod install-mirror-icon-location ((port x-port)
					 (sheet mirrored-sheet-mixin)
					 location)
  (let ((hints (xlib:wm-hints (sheet-mirror sheet))))
    (setf (xlib:wm-hints-icon-x hints)
	  (point-x location))
    (with-slots (height-pixel) port
      (setf (xlib:wm-hints-icon-y hints)
	    ;; ??? Not accounting for icon height
	    (- height-pixel (point-y location))))
    (setf (xlib:wm-hints (sheet-mirror sheet)) hints)))

||#



