;;; -*- Mode: Lisp; Package: ON-X; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "ON-X")


;;; Ad Hoc functionality to convert some bitmaps. 
;;; ??? All the pixmap stuff below needs to be improved and validated.
;;; ??? 1 days work hacking this stuff and abstracting.

(defparameter *clx-byte-lsb-first-p* (or #+clx-little-endian T nil))

#+ignore
(progn
(defmethod export-pixmap ((port x-port) (pixmap pixmap))
  (let* ((width (pixmap-width pixmap))
	 (height (pixmap-height pixmap))
	 (image (generate-image 
		 port pixmap 
		 (pixmap-format pixmap) (pixmap-data pixmap)))
	 (bit-lsb-first-p (xlib::image-x-bit-lsb-first-p image))
	 (translate (if bit-lsb-first-p
			"084c2a6e195d3b7f"
			"0123456789abcdef"))
	 (array (xlib::image-x-data image))
	 (actual-bytes-per-line (xlib::image-x-bytes-per-line image))
	 (useful-bytes-per-line (ceiling width 8))
	 (nibbles-per-line (* 2 useful-bytes-per-line))
	 line
	 lines)
    (dotimes (row height)
      (setq line (make-string nibbles-per-line))
      (dotimes (pos useful-bytes-per-line)
	(let ((byte (aref array (+ (* row actual-bytes-per-line) pos))))
	  (setf (aref line (* 2 pos))
		(aref translate (ldb (if bit-lsb-first-p
					 (byte 4 0)
					 (byte 4 4))
				     byte)))
	  (setf (aref line (1+ (* 2 pos)))
		(aref translate (ldb (if bit-lsb-first-p
					 (byte 4 4)
					 (byte 4 0)) byte)))))
      (push line lines))
    `(make-pixmap :width ,width :height ,height
		  :data ',(reverse lines))))

(defmethod generate-image ((port x-port) (pixmap pixmap) format (data array))
  (ecase format
    (:dws
     (let* ((width (pixmap-width pixmap))
	    (height (pixmap-height pixmap))
	    (color-list (or (color-table pixmap)
			    (list +black+ +white+)))
	    (color-ids (mapcar #'(lambda (color) (realize-color port color))
			       color-list))
	    )
       
       (if (> (xlib:screen-root-depth (x-screen port)) 2)
	   ;; color screen
	   (make-pixmap-image
	    (xlib:screen-root (x-screen port))
	    server-pixels width height)
	   ;; monochrome screen
	   (make-bitmap-image
	    (xlib:screen-root (x-screen port))
	    (make-clx-bitmap-from-pixels
	     data
	     width height
	     1;; pad to 8-bit boundaries
	     ;; swap bytes
	     nil)
	    width height))))))
)

(defun read-x-bitmap-file (pixmap file)
  (let ((data (xlib::read-bitmap-file file)))
    (setf (pixmap-format pixmap) :clx-image)
    (setf (pixmap-data pixmap) data)
    (setf (pixmap-width pixmap) (xlib:image-width data))
    (setf (pixmap-height pixmap) (xlib:image-height data))))

;;;
;;; Pixmap Support
;;;

(defmethod realize-pixmap ((port x-port) (pixmap pixmap))
  (with-slots (pixmap-gc x-root x-screen) port
    (let ((xbm (xlib:create-pixmap 
		:width    (pixmap-width pixmap)
		:height   (pixmap-height pixmap)
		:depth    (slot-value port 'depth)
		:drawable (xlib::screen-root (x-screen port))))
	  (bitmap-p (= (pixmap-ncolors pixmap) 2)))
      (unless pixmap-gc
	(setf pixmap-gc
	      (xlib:create-gcontext 
	       :drawable   x-root
	       :foreground (xlib:screen-black-pixel x-screen)
	       :background (xlib:screen-white-pixel x-screen)
	       :fill-style :stippled)))
      (when (pixmap-data pixmap)
	(put-image 
	 xbm pixmap-gc 
	 (generate-image port pixmap (pixmap-format pixmap)
			 (pixmap-data pixmap))
	 :width (pixmap-width pixmap)
	 :height (pixmap-height pixmap)
	 :foreground (realize-color port (elt (pixmap-colors pixmap) 1))
	 :background (realize-color port (elt (pixmap-colors pixmap) 0))
	 :bitmap-p bitmap-p))
      xbm)))

(defun put-image (xbm gcontext image 
		      &key width height
		      bitmap-p foreground background)
  (when bitmap-p 
    (setf (xlib:gcontext-foreground gcontext) foreground)
    (setf (xlib:gcontext-background gcontext) background))
  (xlib:put-image xbm gcontext image 
		  :x 0 :y 0 :width width :height height
		  :bitmap-p bitmap-p)
  xbm)

(defmethod unrealize-pixmap ((port x-port) xbm)
  (xlib::free-pixmap xbm))

(defmethod generate-image ((port x-port) (pixmap pixmap) format data)
  ;;; PCL bug
  ;;; EQL METHODS WERE KILLING ME!!!!
  (ecase format
    (:clx-image data)))

(defmethod generate-image ((port x-port) (pixmap pixmap) format (data list))
  (assert (eql format :bitmap) (format) 
	  "List data should be used with :bitmap format")
  ;; format eql's :bitmap
  ;; This format only handle one bit deep bitmaps.
  ;; Not a very efficient approach
  ;; Should be optimized.
  (let* ((width (pixmap-width pixmap))
	 (height (pixmap-height pixmap))
	 (i -1)
	 (translate "0123456789abcdef")
	 c
	 (array (make-array (* height (ceiling width 8)) 
			    :element-type 'xlib::card8))
	 (image (xlib:create-image 
		 :width width :height height 
		 :data array 
		 :depth 1
		 :byte-lsb-first-p *clx-byte-lsb-first-p*)))
    
    (dolist (line data)
      (setq c -1)
      (dotimes (pos (ceiling width 8))
	#+(or ansi-90 Genera-Release-8) (declare (ignore pos))
	(setf (aref array (incf i))
	      (+ (* (position (aref line (incf c)) translate) 16)
		 (position (aref line (incf c)) translate)))))
    
    ;; (setf (pixmap-format pixmap) :clx-image)
    ;; (setf (pixmap-data pixmap) image)
    
    image))

(defmethod generate-image ((port x-port) (pixmap pixmap) format (data array))
  (ecase format
    (:clx-image (pixmap-data pixmap))
    (:bitmap
     (let ((image (xlib:create-image 
	    :width (pixmap-width pixmap)
	    :height (pixmap-height pixmap)
	    :depth 1
	    :data data
	    :byte-lsb-first-p *clx-byte-lsb-first-p*)))
       ;; (setf (pixmap-format pixmap) :clx-image)
       ;; (setf (pixmap-data pixmap) image)
       image))
    (:dws
     (let* ((width (pixmap-width pixmap))
	    (height (pixmap-height pixmap))
	    (color-list (or (color-table pixmap)
			    (list +black+ +white+)))
	    (color-ids (mapcar #'(lambda (color) (realize-color port color))
			       color-list))
	    (server-pixels (make-array (length data)
				       :element-type '(unsigned-byte 8))))
       (dotimes (i (length data))
	 (setf (aref server-pixels i) (elt color-ids (aref data i))))
       (if (> (xlib:screen-root-depth (x-screen port)) 2)
	   ;; color screen
	   (make-pixmap-image
	    (xlib:screen-root (x-screen port))
	    server-pixels width height)
	   ;; monochrome screen
	   (make-bitmap-image
	    (xlib:screen-root (x-screen port))
	    (make-clx-bitmap-from-pixels
	     server-pixels
	     width height
	     1;; pad to 8-bit boundaries
	     ;; swap bytes
	     nil)
	    width height))))))

(defun make-pixmap-image (root data width height)
  (let* ((root-depth (xlib:drawable-depth root)))
    (xlib:create-image :width  width
		       :height height
		       :format :z-pixmap
		       :depth  root-depth
		       :bits-per-pixel 8
		       :bytes-per-line width
		       :data   data
		       :byte-lsb-first-p *clx-byte-lsb-first-p*)))

(defun make-bitmap-image (root data width height &key (bit-lsb-first-p nil))
  (declare (ignore root))
  (xlib:create-image :width  width
		     :height height
		     :depth  1
		     ;; ??? bug in put-image makes copy necessary
		     ;; :data   (copy-seq data)
		     :data data
		     :bit-lsb-first-p bit-lsb-first-p
		     :byte-lsb-first-p *clx-byte-lsb-first-p*))

(defmacro increment-byte-pointer (pointer swapping hi-lo-flag)
  `(if ,swapping
       (progn
	 (if ,hi-lo-flag		
	     ;; if just filled high byte?, go to low byte
	     (setq ,pointer (- ,pointer 1))
	     ;; else go to high byte next word
	     (setq ,pointer (+ ,pointer 3)))
	 ;; flip high-byte flag
	 (setq ,hi-lo-flag (not ,hi-lo-flag)))	      
       ;; no byte-swapping, just increment pointer
       (incf ,pointer)))

(defun make-clx-bitmap-from-pixels (pixels width height padding swap) 
  ;; this function is used to create a bitmap version of a z-format pixmap
  ;; which only has two values, 0 and 1.  no data checking is performed on the
  ;; input pixels, and the low order bit of each value is used to determine the
  ;; bit value of the pixel.  so a two-valued pixmap with 4's and 6's would for
  ;; all practical purposes be a mono-color bitmap.
  
  ;; pixels are stored in bytes.  leftmost pixel of eight is stored in the most
  ;; significant bit of the byte.  padding is used to indicate the number of
  ;; bytes to which the bitmap scan-lines should be padded.  (it is assumed
  ;; that the pixmap has no padding.)  typical values would be 0, for no bitmap
  ;; padding, 1, for 8-bit padding, and 4, for 32-bit padding. if swap is true,
  ;; put the high-order half of each 16-bit chunk first. 

  (let* ((line-length (ceiling (/ width 8))) ; length of scan-line in bytes
	 ;; length of a padded bitmap scan-line
	 (wid (+ line-length (rem line-length padding)))
	 ;; number of extra bytes needed only for padding
	 (extra-bytes (- wid line-length))
	 ;; array containing bitmap data
	 (bitmap (make-array (* wid height) :element-type '(unsigned-byte 8)))
	 ;; pointer used as index through the bitmap bytes
	 (byte-pointer (if swap 1 0))
	 ;; pointer used as index to bits in byte
	 (bit-pointer 7)
	 ;; pointer used as index to pixmap
	 (pixel-pointer 0)
	 ;; flag used when byte-swapping to determine hi-lo
	 (high-byte (if swap t nil)))
    (dotimes (i1 height)
      #+(or ansi-90 Genera-Release-8) (declare (ignore i1))
      (dotimes (i2 width)
	#+(or ansi-90 Genera-Release-8) (declare (ignore i2))
	(progn
	  ;; set bit "bit-pointer" of byte "byte-pointer" of "bitmap"
	  ;; to the value of the low-order bit of value "pixel-pointer"
	  ;; of "pixmap", which is a vector of values.
	  (setf (elt bitmap byte-pointer)
		(dpb (elt pixels pixel-pointer)
		     (byte 1 bit-pointer)
		     (elt bitmap byte-pointer)))

	  ;; update pointers
	  (if (<= bit-pointer 0)
	      (progn			; this byte full, go to next byte
		(setq bit-pointer 7)
		(increment-byte-pointer byte-pointer swap high-byte))	    
	      ;; still room, keep filling this byte
	      (decf bit-pointer))
	  ;; look at next pixel
	  (incf pixel-pointer)))

      ;; one scan-line done, add padding if necessary
      (when (> padding 0)

	;; pad out current byte
	(when (< bit-pointer 7)
	  (setf (elt bitmap byte-pointer)
		(dpb 0 (byte (1+ bit-pointer) 0)
		     (elt bitmap byte-pointer)))
	  (setq bit-pointer 7)
	  (increment-byte-pointer byte-pointer swap high-byte))

	;; zero out extra padding bytes
	(dotimes (i3 extra-bytes)
	  #+(or ansi-90 Genera-Release-8) (declare (ignore i3))
	  (setf (elt bitmap byte-pointer) 0)
	  (increment-byte-pointer byte-pointer swap high-byte))))
    bitmap))


;;;
;;; Cursors 
;;;

(defvar *x-cursors*
  '(:vertical-scroll  116
    :scroll-up  114
    :scroll-down 106
    :vertical-thumb 112
    :horizontal-scroll 108
    :scroll-left 110
    :scroll-right 112
    :horizontal-thumb 114
    :default 132
    :button 132				;Or 38?
    :busy. 150
    :prompt 92
    ;; For rubber banding
    :upper-left 134
    :upper-right 136
    :lower-left 12
    :lower-right 14
    :move 52
    ;; Prompting for a position
    :position 34))

(defmethod available-cursors ((port x-port))
  (append *cursors* *x-cursors*))
			      
(defmethod realize-cursor :around ((port x-port) cursor)
  (with-slots (cursor-cache) port
    (or (getf cursor-cache cursor)
	(setf (getf cursor-cache cursor)
	      (call-next-method)))))

(defmethod realize-cursor ((port x-port) (cursor symbol))
  (realize-cursor port 
		  (or (getf *cursors* cursor)
		      (getf *x-cursors* cursor))))

(defmethod realize-cursor ((port x-port) (cursor number))
  (with-slots (cursor-font x-screen) port
    (xlib:create-glyph-cursor
      :source-font cursor-font
      :source-char cursor
      :mask-font cursor-font
      :mask-char (1+ cursor)
	;;; ??? Should query for black and white or use the b/w from screen
      :foreground (xlib:make-color :red   0.0 :green 0.0 :blue  0.0)
      :background (xlib:make-color :red   1.0 :green 1.0 :blue  1.0))))

(defmethod realize-cursor ((port x-port) (cursor cursor))
  (with-slots (x-screen) port
    (let* ((colors (xlib:query-colors 
		     (xlib:screen-default-colormap x-screen)
		     (list (xlib:screen-black-pixel x-screen)
			   (xlib:screen-white-pixel x-screen)))))
      (xlib:create-cursor :source (realize-pixmap port (w::image cursor))
			  :mask   (realize-pixmap port (w::mask cursor))
			  :x      (w::x-offset cursor)
			  :y      (- (pixmap-height (w::image cursor))
				     (w::y-offset cursor))
			  ;; black & white respectively
			  :foreground (first colors)
			  :background (second  colors)))))

(defmethod install-port-cursor ((port x-port) sheet cursor)
  (unless (eql cursor (port-cursor port))
    (with-slots (x-display) port
      (setf (port-cursor port) cursor)
      (setf (xlib:window-cursor (sheet-mirror (fetch-mirrored-sheet sheet)))
	    (realize-cursor port cursor))
      (xlib:display-force-output x-display))
    cursor))

(defmethod set-cursor-location ((port x-port) sheet location)
  (xlib:warp-pointer (sheet-mirror (fetch-mirrored-sheet sheet))
		     (round (point-x location))
		     (round (point-y location))))

;;;
;;; COLORS
;;;

(defmethod realize-color ((port x-port) (color number))
  "get a new color-id for this color and declare it to the connection "
  (warn "Some caller to ~S used an obsolete shade integer instead of a color."
	'realize-color)
  (realize-color port (shade-to-color color)))

(defmethod realize-color ((port x-port) color)
  "get a new color-id for this color and declare it to the connection "
  ;; Janssen produced this implementation
  (with-slots (x-screen) port
    (cond
      ((> (xlib:screen-root-depth x-screen) 1) 
					;otherwise no point in trying
       ;;; ??? Shouldn't be allocating from the screens default colormap.
       (xlib:alloc-color (xlib:screen-default-colormap x-screen)
			 (xlib:make-color
			  :red (color-red-value color)
			  :green (color-green-value color)
			  :blue (color-blue-value color))))
      (t				; monochrome case
       ;; Janssen says: this is just a hack now ...
       (if (almost-white-p COLOR)
	   (xlib:screen-white-pixel x-screen)
	   (xlib:screen-black-pixel x-screen))))))

(defmethod realize-color ((port x-port) (color symbol))
  ;; The only symbolic color we deal with is the invisible
  ;; ink kludge
  (assert (eql color ':invisible-ink))
  (with-slots (x-screen) port
	      (xlib:screen-white-pixel x-screen)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Are these really needed by DWS??
;;;
;;; return a list of all available color names for this canvas.
;;; for x, all colors are available.
(defmethod available-colors ((port x-port) canvas)
  (declare (ignore canvas))
  (cond
    ((> (xlib:screen-root-depth (x-screen port)) 1)  (color-names))
    (t  '(:white :black))))

(defmethod screen-color ((port x-port) canvas color)
  "return the actual color put on the screen for the given color"
  (declare (ignore canvas))
  (let ((colors
	 (xlib:query-colors (xlib:screen-default-colormap (x-screen port))
			    (list (realize-color port color))))
	red-value green-value blue-value)
    (setf red-value (xlib:color-red (car colors))
	  green-value (xlib:color-green (car colors))
	  blue-value (xlib:color-blue (car colors)))
    (if (and 
	  (eql (color-red-value color) red-value)
	  (eql (color-green-value color) green-value)
	  (eql (color-blue-value color) blue-value))
	;; if the screen colors are the same as the wsii-color, return it
	color
	;; otherwise construct a new color which is what is on the screen
	(make-color-rgb red-value green-value blue-value))))

;;; Support for obsolete SHADE integers.
;;; This is preserved mostly as an example of auto-stipple-generation
;;; for grays.
(defmethod realize-shade ((port x-port) shade)
  ;; This code was based on an implementation provided by Stan which was better
  ;; than what I was doing.
  
  ;; For the sake of efficiency, we let the server dictate the size of the
  ;; stipple pattern.
  (multiple-value-bind (width height)
      (xlib:query-best-stipple 8 8 (x-root port))
    (let* ((width (cond ((< width 4) 4)
			((evenp width) width)
			(t (+ 1 width))))
	   (height (cond ((< height 4) 4)
			 ((evenp height) height)
			 (t (+ 1 height))))
	   (stipple (xlib:create-pixmap :width width :height height 
					:depth 1
					:drawable (x-root port)))
	   ;; To draw on the stipple, we get an "image" the size of the
	   ;; stipple, draw on it, and then paint that image back on the
	   ;; stipple. 
	   (array (make-array (* height (ceiling width 8)) 
			      :element-type 'xlib::card8))
	   (image (xlib:create-image :width width :height height :depth 1
				     :data array
				     :byte-lsb-first-p *clx-byte-lsb-first-p*))
	   #+ignore
	   (image (xlib:get-image stipple :x 0 :y 0
				  :width width :height height))
	   (image-data (xlib::image-x-data image)))
      ;; The image-data is a vector that can be treated as a two-d array of
      ;; bytes, where each row of the pixmap is contained in a row of the array.
      ;; For whatever reasons, X might pad the rows with extra bytes, so the
      ;; number of columns that contain "real" data is likely smaller than the
      ;; number of columns in the array.  So... we make a two-d array displaced
      ;; to this image-data, and then treat it as if it contained just the
      ;; number of bytes that contain "real" info.
      (let ((nrows (xlib:image-height image))
	    (ncolumns (/ (xlib:image-width image) 8))
	    (array (make-array (list (/ (array-total-size image-data)
					(xlib::image-x-bytes-per-line image))
				     (xlib::image-x-bytes-per-line image))
			       :element-type (array-element-type image-data)
			       :displaced-to image-data)))
	;; Some special cases that we handle, to ensure a pleasing appearence.
	;; We only deal with shades <= 50%; darker shades are computed as the
	;; complement shade.
	(case (if (> shade 50)
		  (- 100 shade)
		  shade)
	  (50 (dotimes (row nrows)
		(let ((byte (ecase (rem row 2)
			      (0 #b10101010)
			      (1 #b01010101))))
		  (dotimes (column ncolumns)
		    (setf (aref array row column) byte)))))
	  (37.5 (dotimes (row nrows)
		  (let ((byte (ecase (rem row 8)
				(0 #b10101011)
				(1 #b01011101)
				(2 #b11101010)
				(3 #b01010111)
				(4 #b10111010)
				(5 #b11010101)
				(6 #b10101110)
				(7 #b01110101))))
		    (dotimes (column ncolumns)
		      (setf (aref array row column) byte)))))
	  (25 (dotimes (row nrows)
		(let ((byte (ecase (rem row 4)
			      (0 #b01000100)
			      (1 #b00100010)
			      (2 #b10001000)
			      (3 #b00010001))))
		  (dotimes (column ncolumns)
		    (setf (aref array row column) byte)))))
	  (12.5 (dotimes (row nrows)
		  (let ((byte (ecase (rem row 8)
				(0 #b00000001)
				(1 #b00001000)
				(2 #b01000000)
				(3 #b00000010)
				(4 #b00010000)
				(5 #b10000000)
				(6 #b00000100)
				(7 #b00100000))))
		    (dotimes (column ncolumns)
		      (setf (aref array row column) byte)))))
	  (7.25 (dotimes (row nrows)
		  (let ((byte (ecase (rem row 8)
				(0 #b00000001)
				(1 #b00000000)
				(2 #b00010000)
				(3 #b00000000)
				(4 #b01000000)
				(5 #b00000000)
				(6 #b00000100)
				(7 #b00000000))))
		    (dotimes (column ncolumns)
		      (setf (aref array row column) byte)))))
	  (0 (dotimes (row nrows)
	       (dotimes (column ncolumns)
		 (setf (aref array row column) #b00000000))))
	  (t (error "~%Can't yet handle arbitrary shades")))
	;; If the shade is > 50%, it is the inverse of the complement shade,
	;; so... 
	(when (> shade 50)
	  (dotimes (row nrows)
	    (dotimes (column ncolumns)
	      (setf (aref array row column)
		    (mask-field (byte 8 0)
				(lognot (aref array row column)))))))
	;; Save the pixmap in the server
	(with-slots (stipple-gc x-screen) port
	  (unless stipple-gc
	    (setf stipple-gc
		  (xlib:create-gcontext 
		   :drawable   stipple
		   :foreground (xlib:screen-black-pixel x-screen)
		   :background (xlib:screen-white-pixel x-screen)
		   :fill-style :stippled)))
	  (put-image stipple stipple-gc image
		     :width width :height height
		     :foreground (realize-color port +black+)
		     :background (realize-color port +white+)
		     :bitmap-p t))
	stipple))))



;;;
;;;  Text Style Mapping (ILA)
;;;

;;; COND-X-ERROR belongs in some DEFS file

(eval-when (eval compile load)
  (defvar *cond-x-error-lambda-list* '(display error-name error-args)))

(defmacro cond-x-error ((error-name display)
			error-clause
			&body body)
  (let ((arglist (pop error-clause))
	(old-error-handler (make-symbol "OLD-ERROR-HANDLER")))
    (multiple-value-bind (args ignores)
	(canonicalize-and-match-lambda-lists *cond-x-error-lambda-list*
						arglist)
      `(let ((,old-error-handler (xlib:display-error-handler ,display)))
	 (flet ((error-handler (display error-key &rest stuff)
		  (declare (dynamic-extent stuff))
		  (if (eql error-key ',error-name)
		      (funcall #'(lambda ,args
				   (declare (ignore ,@ignores))
				   ,@error-clause)
			       display error-key stuff)
		      (apply ,old-error-handler display error-key stuff))))
	   (declare (dynamic-extent #'error-handler))
	   (letf-globally (((xlib:display-error-handler ,display)
			       #'error-handler))
	     (unwind-protect
		 (progn ,@body)
	       ;; read out the error, if any
	       (xlib:display-finish-output ,display))))))))

;;; Sugar for the above.
(defmacro ignoring-x-error ((error-name display) &body body)
  (let ((block-name (gensymbol 'ignoring-x-error)))
    `(block ,block-name
       (cond-x-error (,error-name ,display)
		     (()
		      (return-from ,block-name (values)))
		     ,@body))))

;;; We want to avoid a lexical closure being consed on each call to
;;; character-text-style-mapping.
(defun open-the-font-1 (display font-name text-style character-set)
  (cond-x-error (xlib:name-error display)
		(() (error "Can't find font ~A for ~A characters ~
				        in character set ~A"
			   font-name text-style character-set))
		(xlib:open-font display font-name)))

(defmethod text-style-mapping :around ((port x-port) character-set text-style)
  (let ((font (call-next-method)))
    (when (or (stringp font) (symbolp font))
      (let* ((font-name (string font)))
	(with-slots (x-display) port
	  (setf font (open-the-font-1 x-display font-name text-style character-set)))
	(add-text-style-mapping
	  port character-set (parse-text-style text-style) font)))
    font))

(defmethod realize-text-style ((port x-port) text-style)
  (let* ((character-set *standard-character-set*)) ;; should really be a parameter
    (text-style-mapping port character-set text-style)))

(defparameter *x-logical-size-alist* 
  '((:tiny        6)
    (:very-small  8)
    (:small      10)
    (:normal     12)
    (:large      14)
    (:very-large 20)
    (:huge       24)))

(defmethod standardize-text-style ((display-device x-port) character-set style)
  (standardize-text-style-1
    display-device style character-set *x-logical-size-alist*))

#+ignore
(define-display-device *clx-device* clx-device
  :font-for-undefined-appearance sans12)

(defmethod initialize-clx-display-device ((display-device x-port) display)
  (add-text-style-mapping 
   display-device *standard-character-set* *undefined-text-style* 
   'sans12)
  (let ((vendor (xlib:display-vendor-name display)))
    (cond ((string= vendor "MIT X Consortium")
	   (ecase (xlib:display-release-number display)
	     (2 (initialize-clx-display-device-r2 display-device display))
	     ((3 4) (initialize-clx-display-device-r3 display-device display))))
	  ;; If not a known implementation, assume that it uses the new font
	  ;; naming protocol. 
	  (t
	   (initialize-clx-display-device-r3 display-device display)))))

(defvar *clx-font-families* '((:fix "*-courier-*")
			      (:sans-serif "*-helvetica-*")
			      (:serif "*-charter-*" 
			       "*-new century schoolbook-*" "*-times-*")))

(defun disassemble-x-font-name (name)
  (let ((cpos 0)
	(tokens nil))
    (loop
      (let ((dpos (position #\- name :start cpos)))
	(when (null dpos)
	  (push (subseq name cpos) tokens)
	  (return))
	(push (if (= cpos dpos)
		  nil
		  (subseq name cpos dpos))
	      tokens)
	(setf cpos (1+ dpos))))
    ;; Simple test to see if font conforms to assumed font naming convention 
    (when (>= (length tokens) 9)
      (reverse tokens))))

(defmethod initialize-clx-display-device-r3 ((display-device x-port) display)
  ;; Sort of cheating, but ...
  (add-text-style-mapping display-device *standard-character-set*
			  *undefined-text-style*
			  `(:style ,@(multiple-value-list
					 (text-style-components
					  *default-text-style*))))
  (flet ((font-name->text-style (font-name family)
	   (let* ((tokens (disassemble-x-font-name
			   font-name)))
	     (when tokens
		  (let* ((italic? (member (fifth tokens) '("i" "o") :test #'equalp))
			 (bold? (equalp (fourth tokens) "Bold"))
			 (face (if italic?
				   (if bold? '(:bold :italic) :italic)
				 (if bold? :bold :roman)))
			 (point-size (parse-integer (ninth tokens)))
			 ;; We now use numeric sizes.
			 (size (round point-size 10)
			       #+ignore
			       (case point-size
				 (60  :tiny)
				 (80  :very-small)
				 (100 :small)
				 (120 :normal)
				 (140 :large)
				 (180 :very-large)
				 (240 :huge))))
		    (when size (make-text-style family face size)))))))
    (dolist (family-stuff *clx-font-families*)
      (let ((family (car family-stuff)))
	(dolist (font-pattern (cdr family-stuff))
	  (dolist (xfont-name (xlib:list-font-names display font-pattern))
	    (let ((text-style (font-name->text-style xfont-name family)))
	      (when text-style
		(unless (text-style-mapping-exists-p
			 display-device *standard-character-set* text-style)
		  (add-text-style-mapping display-device
					  *standard-character-set* 
					  text-style xfont-name)))))))))
  ;; It's a bad idea to override the undefined mapping as above without
  ;; guaranteeing that the default text style maps to something useful.  The
  ;; following mapping will get used for the default text style unless one of
  ;; the fonts we read from the server will serve adequately for that style,
  ;; but if not, at least we have a fallback.  For example, on the Genera X
  ;; server XLIB:LIST-FONTS is broken, so no mappings get defined.
  (unless (text-style-mapping-exists-p
	   display-device *standard-character-set* *default-text-style*)
    (add-text-style-mapping display-device *standard-character-set*
			    *default-text-style*
			    '8x13)))

(defmethod initialize-clx-display-device-r2 ((display-device x-port) display)
  (declare (ignore display))		
  ;; These fonts should always work; don't ask.
  ;; Cheating a little...
  (add-text-style-mapping display-device *standard-character-set*
			     *undefined-text-style* '8x13)
  ;; This is the macro expansion of the DEFINE-CHARACTER-TEXT-STYLE-MAPPINGS
  ;; macro. 
  (define-text-style-mappings-load-time display-device
      *standard-character-set*
    '((:family :fix
       (:face :roman (:size :very-large (:style :fix :roman :large)
		      :large fg-20
		      :normal fg-13
		      :small 6x10
		      :very-small (:style :fix :roman :small)
		      :tiny (:style :fix :roman :very-small))
	:italic (:size :very-large (:style :fix :italic :large)
		 :large fgi-20
		 :normal 8x13
		 :small 6x10
		 :very-small (:style :fix :italic :small)
		 :tiny (:style :fix :italic :very-small))
	:bold (:size :very-large (:style :fix :bold :large)
	       :large fcor-20
	       :normal fgb-13
	       :small 6x10
	       :very-small (:style :fix :bold :small)
	       :tiny (:style :fix :bold :tiny))
	(:bold :italic) (:size :very-large (:style :fix (:bold :italic)
						   :large)
			 :large fcor-13
			 :normal fgb-13
			 :small 6x10
			 :very-small (:style :fix (:bold :italic) :small)
			 :tiny (:style :fix (:bold :italic) :very-small)))))))

;;;
;;; Text-Style Operations
;;;

(defmethod text-style-ascent ((text-style text-style) (medium basic-x-medium))
  (xlib:font-ascent (realize-text-style (port medium) text-style)))
					
(defmethod text-style-descent ((text-style text-style) (medium basic-x-medium))
  (xlib:font-descent (realize-text-style (port medium) text-style)))
					
(defmethod text-style-height ((text-style text-style) (medium basic-x-medium))
  (+ (text-style-ascent text-style medium)
     (text-style-descent text-style medium)))

(defmethod string-width (string (text-style text-style) 
				(medium basic-x-medium)
				&key (start 0) end)
  (xlib:text-width (realize-text-style (port medium) text-style) string
		   :start start :end end))

(defmethod string-height (string (text-style text-style) 
				 (medium basic-x-medium))
  (declare (ignore string))
  (text-style-height text-style medium))

(defmethod char-width (char (text-style text-style) (medium basic-x-medium))
  (xlib:text-width (realize-text-style (port medium) text-style) 
		   (string char)))


#||
;;;
;;; OLD Font Realization Code for old silica fonts
;;;

(defmethod realize-font ((port x-port) (font font))
  (open-server-font port font))

(defvar *font-families* '((:fix-pitch "*-courier-*")
			  (:sans-serif "*-helvetica-*")
			  (:serif "*-charter-*" "*-new century schoolbook-*" 
			   "*-times-*"))
  )

(defstruct (cached-clx-font (:type list)
			    (:constructor %make-cached-clx-font))
  (font)
  (size       0 :type number)
  (italic? nil :type boolean)
  (bold?   nil :type boolean))

(defun make-cached-clx-font (clxfont)
  (let ((font-name (xlib:font-name clxfont)))
    (%make-cached-clx-font
     :font clxfont
     :size 
     (let ((point-size (xlib:font-property clxfont :point_size)))
       (if point-size 
	   (round point-size 10)
	   ;; x/news doesn't give this font property
	   ;; Try to find it by parsing the font name.  This doesn't work for
	   ;; some font names, especially for contour fonts.
	   (or (do ((pos 0 (position #\- font-name :start (1+ pos)))
		    (count 1 (1+ count)))
		   ((or (null pos)
			(= count 7))
		    (when pos
		      (parse-integer font-name :start (1+ pos)
				     :junk-allowed t))))
	       ;; You can't get a point size by looking at font geometry.
	       ;; Period.  But I guess something has to be done, and I'm not
	       ;; sure that I can just put in NIL for point-size.
	       (+ (xlib::font-descent clxfont) (xlib::font-ascent clxfont)))))
     
     ;; Stan trys to use the XLIB:FONT-PROPERTYs called :SLANT and
     ;; :WEIGHT-NAME, but they don't seem to work all the time, so
     ;; I try looking at the name. 
     :italic? (or (search "-i-" font-name :test #'char-equal)
		  (search "-o-" font-name :test #'char-equal))
     :bold? (search "bold" font-name :test #'char-equal))))

(defmethod open-server-font :around ((port x-port) 
				     (font font))
  ;; The font-cache in the port is a list of the form
  ;; (>*font-families*< (font . clxfont) (font . clxfont) ...)
  (let ((font-cache (port-prop port :font-cache)))
    (when (or (null font-cache)
	      (not (eq (car (port-prop port :font-cache)) *font-families*)))
      (setq font-cache
	    (setf (port-prop port :font-cache) 
		  (list *font-families*))))
    (or (cdr (assoc font (cdr font-cache) :test #'font-equal))
	(let ((clxfont (call-next-method)))
	  (push (cons font clxfont) (cdr font-cache))
	  clxfont))))

(defmethod open-server-font ((port x-port) (font font))
  (with-slots (family) font
    (let ((clxfonts (mapcan #'(lambda (pattern) 
				(list-fonts port pattern))
			    (cdr (assoc family *font-families* :test #'eq)))))
      (setq clxfonts 
	    (stable-sort clxfonts 
			 #'(lambda (f1 f2) (closer-font-p font f1 f2))))
      (if clxfonts
	  (cached-clx-font-font (first clxfonts))
	  (error "No CLX fonts matching ~S." font)))))

(defmethod list-fonts ((port x-port) pattern)
  (or (copy-list (cdr (assoc pattern (port-prop port :clx-fonts)
			     :test #'string-equal)))
      (let ((fonts (mapcar #'make-cached-clx-font
			   (xlib:list-fonts (x-display port) pattern))))
	(push (cons pattern fonts) 
	      (port-prop port :clx-fonts))
	fonts)))

(defmethod closer-font-p ((font font) f1 f2)
  ;; Is the (cached-)CLX font F1 a closer match to FONT than the (cached-CLX)
  ;; font F2 is?  If one font has the correct slope and the other doesn't,
  ;; it is closer.  Otherwise, if one font is a closer size than the other,
  ;; it is "better".  In a tie, pick the font with the correct weight.
  ;; Otherwise, pick one at random.
  (with-slots (size italic? bold?) font
    (flet ((correct-slope-p (f)
	     (if italic?
		 (cached-clx-font-italic? f)
		 (not (cached-clx-font-italic? f))))
	   (correct-weight-p (f)
	     (if bold?
		 (cached-clx-font-bold? f)
		 (not (cached-clx-font-bold? f))))
	   (size-error (f)
	     (abs (- size (cached-clx-font-size f)))))
      (cond ((and (correct-slope-p f1)
		  (not (correct-slope-p f2))) t)
	    ((and (not (correct-slope-p f1))
		  (correct-slope-p f2)) nil)
	    ((< (size-error f1) (size-error f2)) t)
	    ((> (size-error f1) (size-error f2)) nil)
	    ((and (correct-weight-p f1)
		  (not (correct-weight-p f2))) t)
	    (t nil)))))

;;;
;;; Font Operations
;;;

(defmethod font-ascent ((font font) (medium basic-x-medium))
  (xlib:font-ascent (realize-font (port medium) font)))
					
(defmethod font-descent ((font font) (medium basic-x-medium))
  (xlib:font-descent (realize-font (port medium) font)))
					
(defmethod font-height ((font font) (medium basic-x-medium))
  (+ (font-ascent font medium)
     (font-descent font medium)))

(defmethod string-width (string (font font) (medium basic-x-medium) &key (start 0) end)
  (xlib:text-width (realize-font (port medium) font) string
		   :start start :end end))

(defmethod string-height (string (font font) (medium basic-x-medium))
  (declare (ignore string))
  (font-height font medium))

(defmethod char-width (char (font font) (medium basic-x-medium))
  (xlib:text-width (realize-font (port medium) font) (string char)))


||#

