;;; -*- MODE: LISP; PACKAGE: ON-X; SYNTAX: COMMON-LISP -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "ON-X")

;;;
;;; on-x:  x port implementation
;;;

;;; First one to set *default-server-path* wins.
(unless *default-server-path*
  (setq *default-server-path* '(:x11 :host "localhost" :display 0 :screen 0)))

(pushnew '(:x11 :host "localhost" :display 0 :screen 0) 
	 *server-path-defaults*
	 :key #'car :test #'eq)

;;; 
;;; x port
;;;

(defclass x-port (port)
    (
     ;; x specific stuff
     (x-display :initform nil :accessor x-display)
     (x-screen  :accessor x-screen)
     (x-root :accessor x-root)
     (visual-class)
     (depth)
     (height-pixel)
     
     (pixmap-gc :initform nil)
     (stipple-gc :initform nil)
     (rband-gc)
     
     
     ;; the slot cursor-cache is used to map between silica and x cursors.
     ;; the cursor-font holds the x cursor font.
     (cursor-font)
     (cursor-cache :initform nil :type nil)
     
     (type :allocation :class 
	   :initform :x11
	   :reader port-type)
     ))

(defmethod find-port-type ((type (eql :x11))) 
  (find-class 'x-port))

;; A hack to detect children of root level to deal with wm fights.
(defmacro top-level-p (sheet) `(graftp (sheet-parent ,sheet)))

(defun decode-x-server-path (server-path)
  ;; --- kludge.  We want to support subclasses of x-port
  ;; without building in all their names here.
  (if (member (car server-path) '(:x11 :lispview))
      (let* ((keys (cdr server-path))
	     (defaults (cdr (assoc :x11 *server-path-defaults*)))
	     (host    (or (getf keys :host)    (getf defaults :host)))
	     (display (or (getf keys :display) (getf defaults :display)))
	     (screen  (or (getf keys :screen)  (getf defaults :screen))))
	(values host display screen))
      (error "~S is not an X11 server path." server-path)))

(defmacro with-decoded-x-server-path ((host display screen) server-path &body body)
  `(multiple-value-bind (,host ,display ,screen)
       (decode-x-server-path ,server-path)
     ,@body))
    
(defmethod port-name ((port x-port))
  (with-decoded-x-server-path (h d s) (port-server-path port)
    (format nil "~a:~d.~d" h s d)))

(defmethod initialize-instance :after
	   ((port x-port) &key server-path &allow-other-keys)
  ;; this is assuming we are only porting to the root window of a screen,
  ;; which need not be the case.
  ;; and also the zeroth screen.
  (with-slots (x-display x-screen x-root rband-gc
	       height-pixel
	       visual-class depth
	       cursor-font) 
      port
    (with-decoded-x-server-path (host display-id screen-id) server-path
      (setf x-display       (xlib:open-display host :display display-id))
      (setf x-screen        (nth screen-id (xlib:display-roots x-display))
	    x-root          (xlib::screen-root x-screen)
	    height-pixel	  (xlib:screen-height x-screen)
	    cursor-font     (xlib:open-font x-display "cursor"))

      (let ((root-visual (xlib:screen-root-visual x-screen)))
	(block find-depth-info
	  ;; Iterate over depths
	  (dolist (depth-info (xlib:screen-depths x-screen))
	    ;; Iterate over visual types for the depth
	    (dolist (vis (cdr depth-info))
	      ;; When you find the root visual, set the depth
	      (when (= (xlib:visual-info-id vis)
		       root-visual)
		(setf visual-class (xlib:visual-info-class vis))
		(setf depth (first depth-info))
		(when (and (eq visual-class :static-gray) (= depth 1))
		  (setf visual-class :monochrome))
		(return-from find-depth-info))))))
    
      ;; ??? This code is here because eql specialers are broken.
      ;; Get rid of this and add back the slot-unbound method in x-stdi
      ;; pcl bug
      (with-slots (rband-gc x-root x-screen) port
	(setf rband-gc (xlib:create-gcontext
			:drawable x-root
			:function boole-xor
			:stipple (realize-color port +gray+)
			:line-width 0
			:foreground (xlib:screen-black-pixel x-screen)
			:subwindow-mode :include-inferiors)))
    
      (initialize-clx-display-device port x-display))))

(defmethod debug-port ((port x-port))
  (with-slots (x-display) port
    (setf (xlib:display-after-function x-display)
	  #'xlib::display-finish-output)))

(defmethod port-match ((port x-port) server-path)
  (let ((pkeys (cdr (port-server-path port)))
	(akeys (cdr server-path)))
    (and (eq (car server-path) :x11)
	 (string= (getf akeys :host)
		  (getf pkeys :host))
	 (=  (getf akeys :display)
	     (getf pkeys :display))
	 (=  (getf akeys :screen)
	     (getf pkeys :screen)))))

(defmethod destroy-port :after ((port x-port))
  ;; don't worry about deallocating/destroying resources, e.g.
  ;; xlib:destroy-window, since closing down will clean up resources
  (with-slots (x-display) port
    (when x-display (xlib:close-display x-display))))

(defmethod port-force-output ((port x-port))
  (xlib:display-force-output (x-display port)))

(defmethod port-finish-output ((port x-port))
  (xlib:display-finish-output (x-display port)))

(defmethod do-with-port ((port x-port) function)
  (with-slots (x-display) port
    (xlib:with-display (x-display)
      (xlib:with-event-queue (x-display)
	(funcall function)))))

(defmethod do-with-pointer ((port x-port) function)
  (with-slots (x-display x-root) port
    (let (status)
      (unwind-protect
	   (progn
	     (setq status
		   (xlib:grab-pointer x-root
				      #.(xlib:make-event-mask 
					 :button-press :button-release
					 :pointer-motion :pointer-motion-hint
					 :enter-window :leave-window)
				      :owner-p t
				      ;; :sync-keyboard-p t
				      ))
	     (case status
	       (:success (funcall function))
	       (otherwise
		(error "Don't know what to do if I can't get grab"))))
	(when (eq status :success) (xlib:ungrab-pointer x-display))
	(xlib:display-force-output x-display)))))

	   

#+(and allegro unix)
;;; support for the irwin sigio hack
(progn

;; assuming in an image that has sigio loaded
(unless (fboundp 'system:initialize-sigio-handling) 
  (require :sigio))

(system:initialize-sigio-handling)

;; closing stream should automatically of removed the handler
(defmethod destroy-input :before ((port x-port))
  (with-slots (x-display) port
    (let ((x-fd (xlib::display-input-stream x-display)))
      (system:remove-sigio-handler x-fd))))

(defmethod restart-input :after ((port x-port))
  (with-slots (event-process x-display) port
    (let ((x-fd (xlib::display-input-stream x-display)))
      (system:set-sigio-handler 
       x-fd 
       #'(lambda (fd) 
	   (declare (optimize (speed 3) (safety 0))
		    (fixnum fd)
		    (ignore fd))
	   (unless (or (null mp:*current-process*)
		       sys::*disallow-scheduling*
		       (eq mp:*current-process* event-process)
		       ;; (xlib::display-waiting-reply-p x-display)
		       (not (eq (mp:process-whostate event-process)
				xlib::*read-whostate*)))
	     (mp:process-allow-schedule event-process))
	    
	   nil)))))

)


;;;
;;; With-event-process-stopped
;;;

#+Allegro
(progn
  
(defmacro with-event-process-stopped ((port) &body body)
  (with-gensyms (in-other-process process preempting-processes)
    `(with-slots ((,process event-process)
		  (,preempting-processes w::preempting-processes))
	 ,port
       (let ((,in-other-process 
	      (and ,process (not (eq mp:*current-process* ,process)))))

	 (when ,in-other-process
	   (mp::without-scheduling-internal ; Even keeps SIGIO at bay...
	    (pushnew mp:*current-process* ,preempting-processes)
	    (stop-port-process ,port)))
      
	 (unwind-protect (progn ,@body)
	   (when ,in-other-process
	     (mp::without-scheduling-internal
	      (setf ,preempting-processes
		    (delete mp:*current-process* ,preempting-processes
			    :test #'eq :count 1))

	      (when (null ,preempting-processes)
		(if (not (eq (mp:process-whostate ,process) 
			     w::*suspended-port-state*))
		    (running-unexpectedly))))
	     (mp:process-allow-schedule ,process)))))))

;;
;; Used by other process when they need to read from the CLX event queue.
;;
;; This function is called only by with-event-handler-stopped, and is lexically
;; bound by mp::without-scheduling-internal.  It may be called when the
;; default handler is already stopped, so we need to check for that.  If
;; the default handler is already stopped, we know it will not restart, since
;; *stopping-processes* always contains at least ourselves.
;;
(defmethod stop-port-process (port)
  (declare (optimize (speed 3) (safety 0)))
  (with-slots (x-display 
	       (process w::event-process)
	       (preempting-processes w::preempting-processes)) 
      port
    (unless (eq (mp:process-whostate process) w::*suspended-port-state*)
    
      (when (> (length preempting-processes) 1)
	;; This can only happen when there is another process in the
	;; process-wait below.  Thus we want to wait for that other process to
	;; stop the handler.
	(process-wait 
	 "waiting for X11 event dispatcher to be thrown out by another process"
	 #'(lambda () (eq (mp:process-whostate process)
			  w::*suspended-port-state*)))
	(return-from stop-port-process))
    
      (loop
       (process-wait 
	"waiting for X11 event dispatcher to block on event read"
	#'(lambda ()
	    (and (eq (mp:process-whostate process) xlib::*read-whostate*)
		 #+ignore
		 (xlib:display-awaiting-event x-display))))
       ;; Now, we have to check it again, because the scheduler may have moved
       ;; both us and the default handler onto the run queue.  Also, another
       ;; process may have stopped it in the meantime.
       (when (and (eq (mp:process-whostate process) xlib::*read-whostate*)
		  #+ignore
		  (xlib:display-awaiting-event x-display))
	 (return)))
   
      (mp:process-interrupt process 
			    #'(lambda () (throw :suspend-event-process nil)))
      (mp:process-allow-schedule process)
    
      ;; Wait for it to be fully thrown out.
      (process-wait "waiting for X11 event dispatcher to be thrown out"
		    #'(lambda () (eq (mp:process-whostate process)
				     w::*suspended-port-state*))))))
  
)

#-Allegro
(defmacro with-event-process-stopped ((port) &body body)
  (declare (ignore port))
  `(progn
     (warn "With event process stopped is not implemented for this port ~
            So you will probably soon hang waiting for input.")
     ,@body))


;;;
;;; Mirror Protocol Handlers
;;;

;; same syntax for compatibility for now.  
;; --- Acknowledged poor style (from before): binding magic names
;; --- Acknowledged poor style: with capturing x2/y2 as w/h.
;; --- Acknowledged poor style: all the setqs
(defmacro with-new-native-region ((sheet) &body body)
  `(multiple-value-bind (new-x new-y new-w new-h)
       (sheet-target-native-edges* ,sheet)
     (setq new-w (round (- new-w new-x))
	   new-h (round (- new-h new-y))
	   new-x (round new-x)
	   new-y (round new-y))
     ,@body))
    
;; same syntax for compatibility for now.  
;; --- Acknowledged poor style (from before): binding magic names
;; --- Acknowledged poor style: all the setqs
(defmacro with-current-native-pos ((sheet &optional x-window) &body body)
  (declare (ignore x-window))
  `(multiple-value-bind (cur-x cur-y)
       (sheet-actual-native-edges* (port ,sheet) ,sheet)
     ,@body))

(defmethod realize-graft ((port x-port) graft)
  (with-slots (x-screen) port
    (with-slots (units 
		 width-pixel height-pixel width-mm height-mm 
		 pixels-per-point) 
	graft
      ;; screws to sun.  X can't get information correctly, so just guessing.
      (setf width-mm       360.0
	    height-mm      280.0
	    width-pixel    (xlib:screen-width x-screen)
	    height-pixel   (xlib:screen-height x-screen)
	    pixels-per-point 1)
      
      (setf (slot-value graft 'region) 
	    (ecase units
	      (:pixel (make-rectangle* 0 0 width-pixel height-pixel))
	      (:mm    (make-rectangle* 0 0 width-mm height-mm))
	      (:homogenous (make-rectangle* 0.0 0.0 1.0 1.0))))
      
      (setf (sheet-native-transformation graft) 
	    +identity-transformation+)
      (setf (sheet-mirror graft) (x-root port))
      (update-native-transformation port graft))))

(defmethod realize-mirror ((port x-port) sheet)
  (with-slots (x-display x-screen) port
    (let* ((x-parent (sheet-mirror! sheet))
	   (x-window
	    (with-new-native-region (sheet)
	      (xlib:create-window 
	       ;; ??? See enable-mirror
	       ;; :override-redirect :on 
	       :parent x-parent
	       :x new-x :y new-y
	       :width new-w :height new-h
	       :background (xlib:screen-white-pixel x-screen)
	       :event-mask 
	       #.(xlib:make-event-mask 
		  :button-press :button-release
		  :key-press :key-release 
		  :pointer-motion :pointer-motion-hint
		  :enter-window :leave-window 
		  :structure-notify
		  :exposure
		  ;; Needed for popup menus
		  :owner-grab-button
		  )))))
		  
      (setf (sheet-mirror sheet) x-window)
      (setf (sheet-native-transformation sheet) 
	    +identity-transformation+)
      (update-native-transformation port sheet)
      x-window)))

(defmethod destroy-mirror ((port x-port) sheet)
  (ignore-errors			
   ;; The only error expected is closed-display 
   ;; ?? Don't want to think about portably handling conditions.
   (xlib:destroy-window (sheet-mirror sheet))
   (port-force-output port)))

(defmethod enable-mirror ((port x-port) sheet)
  (let ((mirror  (sheet-mirror sheet)))

    ;; ??? See realize-mirror 
    ;; (setf (xlib::window-override-redirect mirror) :off)
    
    (with-slots (x-display) port
      (xlib:map-window mirror)
      
      ;; ??? Cheap hack for making sure temporary windows come up on top.
      (setf (xlib:window-priority mirror) :top-if)
      
      (xlib:display-force-output x-display))))

(defmethod disable-mirror ((port x-port) sheet)
  (let ((mirror  (sheet-mirror sheet)))
    (with-slots (x-display) port
      (xlib:unmap-window mirror)
      (xlib:display-force-output x-display))))

(defmethod raise-mirror ((port x-port) (sheet mirrored-sheet-mixin))
  (with-slots (x-display) port
    (setf (xlib:window-priority (sheet-mirror sheet) nil) :above)
    (xlib:display-force-output x-display)))
  
(defmethod bury-mirror ((port x-port) (sheet mirrored-sheet-mixin))
  (with-slots (x-display) port
    (setf (xlib:window-priority (sheet-mirror sheet) nil) :below)
    (xlib:display-force-output x-display)))

(defmethod mirror-origin ((port x-port) sheet)
  port sheet
  :NW)

(defmethod mirror-inside-region* ((port x-port) (graft graft))
  (values 0
	  0
	  (graft-width-pixel graft)
	  (graft-height-pixel graft)))

(defmethod mirror-inside-region* ((port x-port) (sheet mirrored-sheet-mixin))
  ;; --- someday look at the margins
  ;; --- I hope this is a reasonable substitute!!
  (multiple-value-bind (x1 y1 x2 y2)
      (sheet-actual-native-edges* port sheet)
    (values 0 0 (round (- x2 x1)) (round (- y2 y1)))))

(defmethod sheet-actual-native-edges* ((port x-port)
				       (sheet mirrored-sheet-mixin))
  port
  (let* ((x-window (sheet-mirror sheet))
	 (parent-mirror (sheet-mirror! (sheet-parent sheet)))
	 ;; Is this really the only way to get parent?
	 (x-parent (multiple-value-second (xlib:query-tree x-window)))
	 (x (xlib::drawable-x x-window))
	 (y (xlib::drawable-y x-window))
	 (w (xlib::drawable-width x-window))
	 (h (xlib::drawable-height x-window)))
    ;; Can deal with reparenting window managers
    (when (not (eq parent-mirror x-parent))
      (multiple-value-setq (x y)
	(xlib:translate-coordinates
	  x-parent x y parent-mirror)))
    (values x y
	    (+ x w) (+ y h))))

(defmethod set-sheet-actual-native-edges* ((port x-port) (sheet mirrored-sheet-mixin)
					   x1 y1 x2 y2)
  (let* ((x-window (sheet-mirror sheet))
	 (x-display (x-display port))
	 (w (round (- x2 x1)))
	 (h (round (- y2 y1)))
	 (x (round x1))
	 (y (round y1)))
    (xlib:with-display (x-display)
      (setf (xlib:drawable-x x-window) x
	    (xlib:drawable-y x-window) y
	    (xlib:drawable-width x-window) w
	    (xlib:drawable-height x-window) h))
    (xlib:display-force-output x-display)))

(defmethod mirror-resource-id ((port x-port) (sheet sheet))
  (let ((x-window (sheet-mirror sheet)))
    (when x-window
      (xlib:window-id x-window))))

(defmethod mirror-region* ((port x-port) (sheet sheet))
  (let* ((x-window (sheet-mirror sheet))
	 (x (xlib:drawable-x x-window))
	 (y (xlib:drawable-y x-window)))
    (when x-window
      (values x y 
	      (+ x (xlib:drawable-width x-window))
	      (+ y (xlib:drawable-height x-window))))))

;;;
;;; Event Handling
;;;

;;
;; Delivers one event at a time so that it can be invoked in a single threaded
;; environment directly by the application.
;; 
;;; Process ONE event and return
(defmethod process-next-event ((port x-port) &key (timeout nil) wait-test state)
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (with-slots (x-display distributor) port
    (multiple-value-bind (nevents outcome) 
	;; Block until timeout or event
	;;#+Lucid (values 0 nil)
	;;#-Lucid
	(xlib:event-listen x-display timeout)
      (declare (ignore nevents))
      (if (eq outcome :timeout)
	  nil
	  (progn
	    (process-wait "Pending Distribution"
	      #'(lambda ()
		  (and distributor
		       (distributor-enabled distributor))))
	    (x-invoke-distributor x-display distributor port)
	    t)))))

(defvar *debugging-x-events* nil)

;;; Process ONE event and return
(defun x-invoke-distributor (x-display distributor port)
  (declare (optimize (speed 3) (safety 0) (compilation-speed 0)))
  (macrolet ((distribute (&rest keys &key sheet &allow-other-keys)
	       `(let ((sheet ,(if sheet sheet
				  '(mirror->sheet port event-window))))
		  (distribute-device-event
		   distributor
		   port
		   sheet
		   ,@keys
		   :event-window event-window
		   :event-key event-key
		   :time time
		   :native-x x :native-y y
		   ;; ??? Portable state is same as CLX state
		   :state state))))
    ;; ANY NEW CLAUSES YOU ADD TO THIS SHOULD RETURN T, TO ENSURE
    ;; THE PROCESSING OF *ONLY ONE* EVENT!
    (xlib:event-case (x-display :force-output-p nil :discard-p t
				;;; --- Obsolete commentary: 
				;;  --- This shouldn't matter,
				;;  --- but it does in franz.
				;;; --- [This used to say :TIMEOUT 1. -- rsl]
				:timeout nil)
      ;; Device Events
      ((:motion-notify :enter-notify :leave-notify) (event-window time) 
	(multiple-value-bind (x y same-screen-p child state)
	    (xlib:query-pointer event-window)
	  (declare (ignore same-screen-p child))
	  (let ((event-key :pointer-motion)
		(sheet (mirror->sheet port event-window)))
	    (when sheet
	      (distribute :sheet sheet :moved-p t)))
	  t))
       
      ((:button-press :button-release) 
	  (event-key event-window x y state time code)
	(let ((sheet (mirror->sheet port event-window))
	      click-type)
	      
	  (setq click-type 
		(when (and (handle-clicks? distributor)
			   ;; ??? allow dws to cut off clicks
			   ;; Eventually generate a "click sequence"
			   ;; and let the dispatcher deal with the
			   ;; sequence as a sequence or as a single
			   ;; event.
			   (or (null sheet) (handle-clicks? sheet)))
		  (get-clicks x-display code time)))
	  (if click-type
	      (distribute :sheet sheet
			  :event-key click-type :code code)
	      (distribute :sheet sheet :code code))
	  t))
      ((:key-press :key-release)
	  (event-key event-window x y state time code)
	(let* ((keysym (x-keysym->keysym
			(xlib:keycode->keysym 
			 x-display code
			 (xlib:default-keysym-index x-display code state))))
	       (shift-mask (state->shift-mask state))
	       ;; Canonicalize the only interesting key right here.
	       ;; If we get a key labelled "Return", we canonicalize it
	       ;; into #\Newline.
	       ;; This may be misguided, but it'll almost certainly help us
	       ;; in the short run.
	       (char (cond ((and (eql keysym ':return)
				 (or (zerop shift-mask)
				     (= shift-mask (make-shift-mask :shift))))
			    #\Newline)
			   (t (xlib:keycode->character x-display code state)))))
	  (distribute :code code
		      :keysym keysym
		      :char (and (typep char 'standard-char)
				 char)
		      :keyboard-p t)
	  t))
			
      ;; window oriented events.
      (:exposure (event-window x y width height)
	(let ((sheet (mirror->sheet port event-window)))
	  (when sheet
	    (multiple-value-bind (min-x min-y max-x max-y)
		(careful-untransform-rectangle*
		 (sheet-native-transformation sheet) 
		 x y (+ x width) (+ y height))
	      (queue-repaint sheet
			     (make-rectangle* min-x min-y max-x max-y)))))
	t)
      (:map-notify (event-window)
	(let ((sheet (mirror->sheet port event-window)))
	  (when sheet (enable-sheet sheet :port-trigger t))
	  t))
      (:unmap-notify (event-window)
	(let ((sheet (mirror->sheet port event-window)))
	  (when sheet (disable-sheet sheet :port-trigger t))
	  t))
      (:configure-notify (event-window ;x y width height
				       #+ignore send-event-p)
	(let ((sheet (mirror->sheet port event-window)))
	  (when (and sheet
		     #+ignore
		     ;; ??? crock because now twm doesn't send event
		     (or send-event-p
			 ;; ??? crock hack for figuring out existence
			 ;; of window manager.
			 (null (top-level-p sheet))))
	    (xlib:with-state (event-window)
	      (mirror-region-updated port sheet)))
	  t))
      (:destroy-notify ()
	t)
      ((:reparent-notify :no-exposure ) () 
	t)
      (otherwise (event-key)
	(when *debugging-x-events*
	  (format *error-output* "Unknown CLX event ~S~%" event-key))
	t))))
  
(defvar *click-interval* 250)

(defun get-clicks (display first-code first-time)
  ;; ???  Assuming that press and release on the same button must be
  ;; interleaved i.e. no dropping of these events by the server is allowed.
  ;; Also not concerned about motion if these events happen fast enough.  This
  ;; may not be a reasonable assumption, espeically if the click-interval is
  ;; large.  Also not worried about changes to modifier state, though events on
  ;; other buttons will terminate clicks. 
  (let ((timeout (/ *click-interval* 1000.0)))
    (labels ((watch (&optional (count 1) 
			       (start-time first-time))
	       (if (= count 4) 
		   count
		   (or (xlib:event-case (display :timeout timeout 
						 :discard-p nil
						 :peek-p t
						 :force-output-p nil)
			 ((button-press button-release) (time code)
			   (if (or (> time (+ start-time *click-interval*))
				   (not (= code first-code)))
			       count
			       (progn (xlib:discard-current-event display)
				      (watch (1+ count) time)))))
		       ;;; Timed out
		       count))))
      (ecase (watch)
	(1 nil)
	(2 :button-click)
	(3 :button-click ;; :button-click-hold
	   ;; ???  Going to lose a down stroke, but this needs more work then I
	   ;; can do during this round of changes.
	   )
	(4 :button-click-click)))))

(defmethod get-port-canonical-gesture-spec (gesture-spec (port x-port))
  ;; here, we must take the gesture spec, turn it back into
  ;; a keycode, then see what the keysyms are for that keycode
  (let ((x-display (slot-value port 'x-display))
	(keysym (if (atom gesture-spec) gesture-spec (car gesture-spec)))
	(shifts (if (atom gesture-spec) 0 (cdr gesture-spec))))
    (let ((x-keysym (keysym->x-keysym keysym))
	  (x-keycode nil))
      (unless x-keysym (return-from get-port-canonical-gesture-spec nil))
      (setq x-keycode (xlib:keysym->keycodes x-display x-keysym))
      ;; will this ever happen?
      (when (listp x-keycode)
	(setq x-keycode (first x-keycode)))
      ;; now need to figure out necessary shifts for this character
      (when x-keycode
	;; [for now, just check shift...]
	;; This could be written to iterate over all possible shift masks seeing which
	;; of them are required to type this particular character.  That could also be
	;; cached, I suppose.  We'll just do :SHIFT for now.
	(when (= (xlib:keycode->keysym 
		  x-display x-keycode
		  (xlib:default-keysym-index x-display x-keycode (make-shift-mask :shift)))
		 x-keysym)
	  (setq shifts (logior shifts (make-shift-mask :shift))))
	;; now, SHIFTS includes any shift that was necessary to type the original keysym.
	;; Now, we backtranslate the keysym and shift-mask we now have into a new gesture.
	(cons
	 (x-keysym->keysym
	  (xlib:keycode->keysym 
	   x-display x-keycode
	   (xlib:default-keysym-index x-display x-keycode shifts)))
	 shifts)))))

;;;
;;; Input Dispatching Support
;;;

(defmacro x-button-number->standard-button-name (code)
  `(aref '#(nil :left :middle :right) ,code))

(defmethod filter-port-event-keys ((port x-port) contract delivery-thunk
				   &rest keys
				   &key event-key state code 
				   &allow-other-keys)

  (declare (ignore contract)
	   (dynamic-extent keys))

  (when (member event-key '(:motion-notify :enter-notify :leave-notify))
    (setq event-key :pointer-motion))

  (let ((button
	 (when (member event-key
		       '(:button-press 
			 :button-release
			 :button-click
			 ;; :button-click-hold  ???
			 :button-click-click))
	   (x-button-number->standard-button-name code))))
  
    (apply delivery-thunk
	   :event-key event-key :button button
	   keys)))

;;;
;;; Input Querying
;;;

#+ignore
(defmethod slot-unbound (class
			 (port x-port)
			 (slot-name (eql 'rband-gc)))
  (declare (ignore class))
  (with-slots (rband-gc x-root x-screen) port
    (setf rband-gc (xlib:create-gcontext
		    :drawable x-root
		    :function boole-xor
		    :stipple (realize-color port +gray+)
		    :line-size 0
		    :foreground (xlib:screen-black-pixel x-screen)
		    :subwindow-mode :include-inferiors))))

(defmethod do-poll-pointer ((port x-port) sheet)
  (multiple-value-bind (x y same-screen-p child mask)
      (xlib:query-pointer (sheet-mirror sheet))
    (declare (ignore same-screen-p child))
    (values x y mask)))

(defmethod prompt-for-location ((port x-port))
  (with-event-process-stopped (port)
    (with-slots (x-root x-display height-pixel) port
      (multiple-value-bind (x y)
	  (clx-utils::clx-prompt-for-screen-location 
	   x-root x-display 
	   (realize-cursor port :position))
	(values x (- height-pixel y))))))

;;
;; ???
;; The following should take args in root coodinate system, but right now they
;; assume a sw coord system
;;
;; Also they rounds for client convenience, though in the future, I'll force
;; clinet to be in the graft's native coordinate system on this call.
;;

(defmethod prompt-for-region ((port x-port) left bottom width height)
  (with-event-process-stopped (port)
    (with-slots (x-root x-display height-pixel rband-gc) port
      (multiple-value-setq 
	  (left bottom width height)
	(clx-utils::clx-prompt-for-screen-region 
	 x-root x-display rband-gc
	 (realize-cursor port :upper-left) 
	 (realize-cursor port :lower-left)
	 (realize-cursor port :upper-right)
	 (realize-cursor port :lower-right)
	 (round left) (round (- height-pixel bottom)) 
	 (round width)
	 (round height)))
      (values left (- height-pixel bottom height) width height))))

(defmethod prompt-for-region-location ((port x-port) width height)
  (with-event-process-stopped (port)
    (with-slots (x-root x-display rband-gc height-pixel) port
      (multiple-value-bind (x y)
	  (clx-utils::clx-prompt-for-screen-region-location 
	   x-root x-display 
	   (realize-cursor port :upper-left) rband-gc
	   (round width)
	   (round height))
	(values x (- height-pixel y height))))))


;;;
;;; Debugging Utilities
;;; 

#+debugging-utils
(progn

  ;; Actually don't need this since i can use xdpyinfo
(defun list-screen-data (port)
  (with-slots (x-screen) port
    (format t "Root Depth ~d -- ~s~%~%" 
	    (xlib:screen-root-depth x-screen)
	    (xlib:screen-root-visual x-screen))
    (dolist (depth (xlib:screen-depths x-screen))
      (format t "Depth: ~d -- ~s~%" (first depth) (second depth)))))
    


)

