;;; -*- Mode: Lisp; Package: ON-X; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "ON-X")

;;;
;;; BASIC-X-DISPLAY-MEDIUM
;;;   A Foundation Class for Display Mediums on X
;;;

(defclass basic-x-medium (medium)
    ((gcontext :accessor gcontext)
     (drawable :accessor drawable)))

(defclass basic-x-display-medium (basic-x-medium display-medium)
    ())

(defclass basic-x-pixmap-medium (basic-x-medium pixmap-medium)
    ((associated-medium :initarg :associated-medium)))

(defmethod initialize-instance :after 
	   ((medium basic-x-pixmap-medium) &key pixmap &allow-other-keys)
  ;; Trigger the transformation setup
  (when pixmap (setf (medium-pixmap medium) pixmap)))

(defmethod medium-force-output ((medium basic-x-medium))
  (xlib:display-force-output (x-display (port medium))))

(defmethod medium-finish-output ((medium basic-x-medium))
  (xlib:display-finish-output (x-display (port medium))))

(defmethod (setf medium-pixmap) :after (pixmap (medium basic-x-pixmap-medium))
 ;; --- No :REUSE protocol any more 9/30/91.
  ;; --- This might be a good place to invent
  ;; --- a specialized one.
  (let ((xf (scale-transformation +identity-transformation+ 1 -1 
				  ;; :reuse (device-transformation medium)
				  )))
    (setf (device-transformation medium)
	  (translate-transformation  xf 0 (1- (pixmap-height pixmap))
				     ;; :reuse xf
				     ))))

(defmethod slot-unbound (class (medium basic-x-medium) 
			       (slot-name (eql 'gcontext)))
  (declare (ignore #-PCL class))
  ;; A GCONTEXT can be used on any drawable with same root and same depth as
  ;; the drawable it is created for.  All children of port will necessarily be
  ;; of same root, but I'm also assuming that they will be of the same depth as
  ;; the root of their screen.   
  (setf (slot-value medium 'gcontext)
	(with-slots (x-screen x-root) (port medium)
	  (xlib:create-gcontext 
	   :drawable x-root
	   :foreground (xlib:screen-black-pixel x-screen)
	   :background (xlib:screen-white-pixel x-screen)
	   :fill-style :solid))))

(defmethod slot-unbound (class (medium basic-x-display-medium)
			       (slot-name (eql 'drawable)))
  (declare #-PCL (ignore class))
  (setf (slot-value medium 'drawable) 
	(sheet-mirror (fetch-mirrored-sheet (medium-sheet medium)))))

(defmethod cleanup-display-medium ((display-medium basic-x-display-medium))
  ;; gcontext should still be applicable since medium are only reused within
  ;; the scope of a port.
  (slot-makunbound display-medium 'drawable))

(defmethod slot-unbound (class (medium basic-x-pixmap-medium)
			       (slot-name (eql 'drawable)))
  (declare #-PCL (ignore class))
  (let* ((other (slot-value medium 'associated-medium))
	 (sheet (medium-sheet other))
	 (mirror (sheet-mirror! sheet)))
    (setf (slot-value medium 'drawable) 
	  (xlib:create-pixmap :drawable mirror
			      :width (bounding-rectangle-width sheet)
			      :height (bounding-rectangle-height sheet)
			      ;; --- fix this
			      :depth 1))))

;;;
;;; DM Operations.
;;;

(defmethod (setf device-clipping-region) :after
  ((region region) (medium basic-x-medium))
  (with-slots (gcontext drawable) medium
    (typecase region
      (rectangle 
	(with-bounding-rectangle* (minx miny maxx maxy) region
	  (setf (xlib:gcontext-clip-mask gcontext)
		  (list (round minx) (round miny) 
			(round (- maxx minx))
			(round (- maxy miny))))))
      (nowhere
	(setf (xlib:gcontext-clip-mask gcontext)
		(list 0 0 0 0)))
      (everywhere
	;; ???  This doesn't work for pixmap-medium
	(setf region
		(transform-region 
		  (insured-device-transformation medium)
		  (sheet-region (medium-sheet medium))))

	(with-bounding-rectangle* (minx miny maxx maxy) region
	  (setf (xlib:gcontext-clip-mask gcontext)
		  (list (round minx) (round miny) 
			(round (- maxx minx)) (round (- maxy miny))))))

      ;; For non-handled region types, a do-clipped operation will have to do
      ;; it piecewise at output time.
      (standard-region-intersection
	(setf (xlib:gcontext-clip-mask gcontext :unsorted)
		(mapcan #'(lambda (rectangle)
			    (list (round (bounding-rectangle-min-x rectangle))
				  (round (bounding-rectangle-min-y rectangle))
				  (round (rectangle-width rectangle))
				  (round (rectangle-height rectangle))))
			(region-set-regions region))))
      #+Ignore
      (rectangle-set 
	(setf (xlib:gcontext-clip-mask gcontext :unsorted)
		(mapcan #'(lambda (rectangle)
			    (list (round (bounding-rectangle-min-x rectangle))
				  (round (bounding-rectangle-min-y rectangle))
				  (round (rectangle-width rectangle))
				  (round (rectangle-height rectangle))))
			(rectangle-set-rectangles region))))
      (otherwise (unimplemented "clipping on weird region?")))))

(defmethod (setf medium-pixmap) :after 
	   ((pixmap pixmap) (medium basic-x-pixmap-medium))
  (setf (drawable medium) (realize-pixmap (port medium) pixmap)))

