;;; -*- Mode: Lisp; Package: ON-X; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "ON-X")

;;;
;;; CLG Media
;;;

(defclass clg-medium (basic-clg-medium basic-x-medium 
				       #+Ignore clg-imaging-context)
    ())

(defclass clg-display-medium (clg-medium basic-clg-display-medium 
					 basic-x-display-medium)
    ())

(defclass clg-pixmap-medium (clg-medium basic-clg-pixmap-medium
					basic-x-pixmap-medium)
    ())

(defmethod display-medium-type ((port x-port) (type (eql :clim)))
  'clg-display-medium)

(defmethod pixmap-medium-type ((port x-port) (type (eql :clim)))
  'clg-pixmap-medium)

;; medium-force/finish-output inherit from basic-x-medium

;;;
;;;
;;;

(eval-when (compile load eval)
  
(defun generate-update-gcontext-for-option (option gcontext value)
  ;; should be an ecase eventually
  (case option
    (line-thickness
     `(setf (xlib:gcontext-line-width ,gcontext)
	    (round (or ,value 0))))
    (line-dashed
      `(setf (xlib:gcontext-line-style ,gcontext)
	     (if ,value :dash :solid)))
    (ink   
     `(update-gcontext-ink ,value ,gcontext port medium))
    (text-style     
     `(setf (xlib:gcontext-font ,gcontext)
	    (realize-text-style port ,value)))))
  
(defmacro with-updated-gcontext ((gcontext medium &key using) &body body)
  `(let ((gcontext ,gcontext))
     
     ,@(mapcar #'(lambda (option)
		   (generate-update-gcontext-for-option
		    option
		    gcontext
		    (case option
		      (line-thickness
		       `(line-style-thickness  
			 (medium-line-style ,medium)))
		      (line-dashed
			`(line-style-dashed
			   (medium-line-style ,medium)))
		      (otherwise
		       `(,(fintern "~A-~A" 'medium option) ,medium)))))
	       using)
     ,@body))

)



#|| For Texting

(with-updated-gcontext (gcontext medium
				 :using (operation foreground ink))
  (xlib:draw-point drawable gcontext x y))

||#

;;;
;;; Generation of Graphics Operation Code
;;;

(defun devicize-point (xf point-x point-y)
  (multiple-value-setq (point-x point-y)
    (transform-point* xf point-x point-y))
  (values (integerize-coordinate point-x) (integerize-coordinate point-y)))

(defun devicize-rectangle (xf minx miny maxx maxy)
  (multiple-value-setq (minx miny maxx maxy)
    (transform-rectangle* xf minx miny maxx maxy))
  (values (integerize-coordinate minx) (integerize-coordinate miny)
	  (integerize-coordinate maxx) (integerize-coordinate maxy)))

#+Ignore
(defmacro devicize-points (transformation points)
  ;;; consing here and also in the list of points case.
  `(do ((point-tail ,points (cddr point-tail)))
       ((null point-tail) ,points)
     (multiple-value-bind (x y)
	 (transform-point* ,transformation (first point-tail) 
			    (second point-tail))
       (setf (first point-tail) (integerize-coordinate x)
	     (second point-tail) (integerize-coordinate y)))))

(defun devicize-points-into (transformation points into-vector)
  (etypecase points
    (list
     (loop
	 (when (endp points) (return-from devicize-points-into into-vector))
       (let ((x (pop points))
	     (y (pop points)))
	 (multiple-value-setq (x y) (transform-point* transformation x y))
	 (vector-push (integerize-coordinate x) into-vector)
	 (vector-push (integerize-coordinate y) into-vector))))
    (vector
     (do ((i 0 (+ i 2))
	  (length (length points)))
	 ((>= i length) into-vector)
       (multiple-value-bind (x y) 
	   (transform-point* transformation (aref points i)(aref points (1+ i)))
	 (vector-push (integerize-coordinate x) into-vector)
	 (vector-push (integerize-coordinate y) into-vector))))))
	   
(eval-when (load eval compile)

(defmacro do-clgop-spreads (((type arg args*) (clgop)) &body body)
  `(dolist (spread-arg (gf-spread-arguments ,clgop))
     (let ((,arg (pop spread-arg))
	   (,type (pop spread-arg))
	   (,args* spread-arg))
       ,@body)))

)

;;; Make a vector of the right type, length = next-higher power of 2.
(defresource coordinate-sequence (length)
  :constructor (make-array (ash 1 (integer-length length))
			   :element-type 'fixnum
			   :fill-pointer 0)
  :matcher (>= (array-dimension coordinate-sequence 0) length)
  :initializer (setf (fill-pointer coordinate-sequence) 0))

(defmacro def-x-clg-methods ((name &key using (sequence-length-increment 0))
			     &body body)
  (let* ((clgop (find-graphics-function name))
	 (name* (gf-spread-function-name clgop))
	 (random-gensym nil))
    (flet
	((generate-clgop-body (clgop)
	   (let ((body `((with-updated-gcontext (gcontext medium :using ,using)
					       ,@body))))
	     (do-clgop-spreads ((type arg args*) (clgop))
	       (declare (ignore arg))
	       (ecase type
		 (point (push 
			 `(multiple-value-setq ,args*
			    (devicize-point device-transformation ,@args*))
			 body))
		 (rectangle
		  (push
		   `(multiple-value-setq ,args*
		      (devicize-rectangle device-transformation
					  ,@args*))
		   body))
		 (point-sequence
		  (setf body
		    `((using-resource (,(setf random-gensym (gensymbol 'coords))
				       coordinate-sequence
				       (+ (length ,(first args*)) 
					  ,sequence-length-increment))
		        (devicize-points-into device-transformation
					      ,(first args*) ,random-gensym)
				      (let ((,(first args*) ,random-gensym))
					,@body)))))))
	     `(with-slots (drawable gcontext port device-transformation) medium
		,@body))))
      `(progn
	 ;; Only Implementation method
	 (define-graphics-function-method
	     ,name* ((medium clg-medium) ,@(gf-method-lambda-list clgop))
	   ,(generate-clgop-body clgop))
	   
	 ;; With output protection for Display Media
	 (define-graphics-function-method
	     ,name* :around ((medium clg-display-medium) ,@(gf-method-lambda-list clgop))
	   ;; GDMF Genera CLOS in 8.2 can't deal with IGNORE declarations
	   ;; for keyword args to methods.
	   #-Genera (declare (ignore ,@(gf-method-argument-list clgop)))
	   #+Genera (progn ,@(gf-method-argument-list clgop))
	   (with-output-protection medium
	     (call-next-method)))))))


;;;
;;; Graphics Operations
;;;

(def-x-clg-methods (draw-point :using (operation foreground ink))
    (xlib:draw-point drawable gcontext point-x point-y))

(def-x-clg-methods (draw-points :using (operation foreground ink))
    (xlib:draw-points drawable gcontext coordinate-sequence))

(def-x-clg-methods (draw-line
		    :using (operation foreground ink line-thickness line-dashed))
    (xlib:draw-line drawable gcontext from-x from-y to-x to-y nil))

(def-x-clg-methods (draw-lines :using (operation foreground ink line-thickness line-dashed))
    (xlib:draw-segments drawable gcontext coordinate-sequence))

(def-x-clg-methods (draw-rectangle :using (operation foreground ink 
						     line-thickness line-dashed))
  (when (typep medium 'display-medium)
    (validate-medium medium))
  (if (rectilinear-transformation-p device-transformation)
      (progn
	(when (and filled
		   (reflection-transformation-p device-transformation))
	  ;; --- This is a compensation measure.  The definition of
	  ;; filled rectangles being areas and expecting them to be
	  ;; half-open is quite well and good for a screen layout
	  ;; algorithm, but runs into mud very quickly as soon as there
	  ;; is a reflection.  In this case, the reflection is assumed
	  ;; to be about the X axis, but reflections about the Y axis
	  ;; and Y=X and Y=-X are also rectilinear.  There is no good
	  ;; solution to this problem that DCPL can think of.  Indeed,
	  ;; CLIM says rendering is complex, yet it depends on filled
	  ;; and unfilled rectangles to have a particular contract.
	  ;; Genera is very careful about drawing of areas, so an
	  ;; arbitrary region can be arbitrarily subdivided, and usage
	  ;; of :FLIPping ink will cause exactly the original region to
	  ;; be drawn.  X may or may not be as careful.  Anyway... This
	  ;; compensation makes draw-rectangle be different from
	  ;; draw-(filled)-polygon of the equivalent lines.  [Don't
	  ;; remove this comment lightly -- the Genera method refers to
	  ;; this comment as the 'source'.  Fixed in Supra?].
	  (incf y1) (incf y2))

	;; CLX requires that rectangles have non-negative widths and heights.
	;; We permit the user to pass in coordinates which are not necessarily
	;; canonical, either before or after transformation.  The following two
	;; forms take care of the bug in CLX.
	(when (< x2 x1) (rotatef x1 x2))
	(when (< y2 y1) (rotatef y1 y2))

	(xlib:draw-rectangle drawable gcontext
			     x1 y1 (- x2 x1) (- y2 y1) filled))
      (call-next-method)))

(def-x-clg-methods (draw-polygon :using (operation foreground ink line-thickness line-dashed)
				 :sequence-length-increment (if closed 2 0))
  (when (and closed (> (length coordinate-sequence) 2))
    (let ((x (elt coordinate-sequence 0))
	  (y (elt coordinate-sequence 1))
	  (length (length coordinate-sequence)))
      (unless (and (= x (elt coordinate-sequence (- length 2)))
		   (= y (elt coordinate-sequence (- length 1))))
	(vector-push x coordinate-sequence)
	(vector-push y coordinate-sequence))))
  (xlib:draw-lines drawable gcontext coordinate-sequence 
		   :fill-p filled :shape :non-convex))

(def-x-clg-methods (draw-ellipse :using (operation foreground ink line-thickness line-dashed))
    ;; can x draw rotated ellipses?
    (assert (= radius-1-dy radius-2-dx 0))
  (let ((delta-angle (- end-angle start-angle)))
    (unless (<= 0 delta-angle 2pi)
      (setq delta-angle (mod delta-angle 2pi)))
    (xlib:draw-arc drawable gcontext
 		   (integerize-coordinate (- center-x radius-1-dx))
		   (integerize-coordinate (- center-y radius-2-dy))
 		   (integerize-coordinate (* radius-1-dx 2))
		   (integerize-coordinate (* radius-2-dy 2))
 		   ;; CLX measures the second angle relative to the first
 		   start-angle delta-angle
 		   filled)))


(def-x-clg-methods (draw-text :using (operation foreground ink text-style line-dashed))
    transform-glyphs
  toward-point
  (let ((string (if (stringp string-or-char)
		    string-or-char
		    (string string-or-char)))
	(text-style (medium-text-style medium)))
    (ecase align-x
      (:left nil)
      (:center (decf point-x (floor (string-width string text-style medium
						  :start start :end end)
				    2)))
      (:right (decf point-x (string-width string text-style medium
					  :start start :end end))))
    (ecase align-y
      (:baseline nil)
      (:bottom (decf point-y (text-style-descent text-style medium)))
      (:center (decf point-y (- (text-style-descent text-style medium)
				(floor (text-style-height text-style medium)
				       2))))
      (:top (incf point-y (text-style-ascent text-style medium))))
    (xlib:draw-glyphs drawable gcontext point-x point-y string
		      :start start :end end)))


;;;
;;; Copy Area handled specially
;;;

(defmethod copy-area :around ((medium clg-display-medium) x y 
			      source left bottom width height
			      &optional (boole boole-1))
  (declare (ignore x y source left bottom width height boole))
  (with-output-protection medium
    (call-next-method)))

(defmethod copy-area :around (medium x y 
			      (source clg-display-medium) left bottom width height
			      &optional (boole boole-1))
  (declare (ignore medium x y left bottom width height boole))
  (with-output-protection source
    (call-next-method)))

(defmethod copy-area
	   ((medium clg-medium) x y 
	    (source pixmap) left bottom width height
	    &optional (boole boole-1))
  (with-slots (drawable gcontext port device-transformation)
	      medium
    (multiple-value-setq (x y) 
      (careful-transform-rectangle* device-transformation 
				    x y (+ x width) (+ y height)))
    (xlib:with-gcontext (gcontext :function boole)
      (xlib:copy-area (realize-pixmap port source) gcontext
		      left 
		      (- (pixmap-height source) (+ bottom height))
		      width height
		      drawable (integerize-coordinate x) (integerize-coordinate y)))))

(defmethod copy-area
	   ((medium clg-display-medium) x y 
	    (source clg-medium) left bottom width height
	    &optional (boole boole-1))
  (with-slots (drawable gcontext port device-transformation)
	      medium
    (multiple-value-setq (x y)
      (careful-transform-rectangle* device-transformation
				    x y (+ x width) (+ y height)))
    ;; Adjust the offset into the source by the source's transformation
    (with-slots ((source-device-transformation device-transformation)) source
      (multiple-value-setq (left bottom)
	(careful-transform-rectangle* source-device-transformation
				      left bottom
				      (+ left width) (+ bottom height))))
    (xlib:with-gcontext (gcontext :function boole)
      (xlib:copy-area (slot-value source 'drawable) gcontext
		      (integerize-coordinate left)
		      (integerize-coordinate bottom)
		      (round width) (round height)	;--- round???
		      drawable (integerize-coordinate x) (integerize-coordinate y)))))

(defmethod copy-area ((pixmap pixmap) dst-x dst-y 
		      (medium clg-display-medium) src-x src-y src-w src-h
		      &optional (boole boole-1))
  ;;; This all only works for pixels coordinate systems.
  (with-slots (drawable gcontext port device-transformation) medium
    (multiple-value-setq (src-x src-y)
      (careful-transform-rectangle* device-transformation
				    src-x src-y (+ src-x src-w)
				    (+ src-y src-h)))
    (let ((xbm (realize-pixmap port pixmap)))
      (xlib:with-gcontext (gcontext :function boole)
	(xlib:copy-area drawable gcontext
			(integerize-coordinate src-x) (integerize-coordinate src-y)
			src-w src-h
			xbm dst-x (- (pixmap-height pixmap) dst-y src-h))))))
    

;;;
;;; GCONTEXT UPDATING Helpers
;;; 

(defmethod update-gcontext-ink ((ink number) gcontext port medium)
  (declare (ignore medium))
  (warn "Some caller to ~S used an obsolete shade integer instead of a color."
	'update-gcontext-ink)
  ;; Maybe should use eql specializers
  (case ink
    (0
     (setf (xlib:gcontext-fill-style gcontext) :solid
	   (xlib:gcontext-foreground gcontext) 
	   (xlib:screen-white-pixel (x-screen port))))
    (100
     (setf (xlib:gcontext-fill-style gcontext) :solid
	   (xlib:gcontext-foreground gcontext) 
	   (xlib:screen-black-pixel (x-screen port))))
    (otherwise
     (case (slot-value port 'visual-class)
       (:monochrome
	(setf (xlib:gcontext-fill-style gcontext) :opaque-stippled  
	      (xlib:gcontext-foreground gcontext) 
	      (xlib:screen-black-pixel (x-screen port))
	      (xlib:gcontext-background gcontext) 
	      (xlib:screen-white-pixel (x-screen port))
	      (xlib:gcontext-stipple gcontext) (realize-shade port ink)))
       (otherwise
	(setf (xlib:gcontext-fill-style gcontext) :solid
	      (xlib:gcontext-foreground gcontext) 
	      (realize-color port (shade-to-color ink)))))))
  (setf (xlib:gcontext-function gcontext) boole-1))

(defmethod update-gcontext-ink ((ink symbol) gcontext port medium)
  ;; Maybe should use eql specializers
  
  (case ink
    (:foreground
      (update-gcontext-ink (medium-foreground medium) gcontext port medium))
    (:background
      (update-gcontext-ink (medium-background medium) gcontext port medium))
    (:flipping-ink 
      ;; For flipping ink, find the color map entry number that, when XOR'd
      ;; with the foreground color number, produces the background color and
      ;; vice versa.  It will produce a random "inverse" color for other
      ;; colors, but it will still be reversable.
      (let ((fg (realize-color port (medium-foreground medium)))
	    (bg (realize-color port (medium-background medium))))
	(setf (xlib:gcontext-fill-style gcontext) :solid
	      (xlib:gcontext-foreground gcontext) (boole boole-xor fg bg)))))
  
  ;; Set the gcontext
  (case ink
    ((:foreground :background)
     (setf (xlib:gcontext-function gcontext) boole-1))
    (:flipping-ink
     (setf (xlib:gcontext-function gcontext) boole-xor))
    ;; kludge for "transparent ink"
    (:invisible-ink
     (setf (xlib:gcontext-function gcontext) boole-2))))

(defmethod update-gcontext-ink ((ink pixmap) gcontext port medium)
  (declare (ignore medium))
  (setf (xlib:gcontext-function gcontext) boole-1
	(xlib:gcontext-fill-style gcontext) :tiled
	(xlib:gcontext-tile gcontext) (realize-pixmap port ink)))

(defmethod update-gcontext-ink (ink gcontext port medium)
  (declare (ignore medium))
  (setf (xlib:gcontext-function gcontext) boole-1
	(xlib:gcontext-fill-style gcontext) :solid
	(xlib:gcontext-foreground gcontext) (realize-color port ink)))
