;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

(defgeneric transformationp (object))
(defgeneric translation-x (transformation))
(defgeneric translation-y (transformation))
(defgeneric copy-transformation (transformation &optional reuse))
(defgeneric reset-to-identity (transformation))
(defgeneric reset-to-translation (dx dy reuse))
(defgeneric reset-to-st-transformation (m00 m11 m20 m21 reuse))
(defgeneric reset-to-srt-transformation (m00 m01 m10 m11 m20 m21 reuse))

(defgeneric reflection-transformation-p (transformation))
(defgeneric rectilinear-transformation-p (transformation))

(defgeneric compose-transformations (transformation1 transformation2 &key reuse))
(defgeneric invert-transformation (transformation &key reuse))

(defgeneric compose-with-translation (transformation dx dy &key reuse))
(defgeneric compose-with-scaling (transformation scaling-x scaling-y &key reuse))
(defgeneric compose-with-rotation (transformation rotation &key reuse))

;;; This is pretty bad.  We have one set of names in the last spec (and the
;;; 0.9 document), one in the old implementation, and a new set from Rao
;;; in the current implementation.  Add the old/never names as a compatibility
;;; boost

;;; The names from the old implementation.
(defun compose-translation (transformation dx dy &key reuse)
  (compose-with-translation transformation dx dy :reuse reuse))
(defun compose-scaling (transformation scaling-x scaling-y &key reuse)
  (compose-with-scaling transformation scaling-x scaling-y :reuse reuse))
(defun compose-rotation (transformation rotation &key reuse)
  (compose-with-rotation transformation rotation :reuse reuse))

;;; The names from the spec.
(defun compose-translation-transformation (transformation dx dy &key reuse)
  (compose-with-translation transformation dx dy :reuse reuse))
(defun compose-scaling-transformation (transformation scaling-x scaling-y &key reuse)
  (compose-with-scaling transformation scaling-x scaling-y :reuse reuse))
(defun compose-rotation-transformation (transformation rotation &key reuse)
  (compose-with-rotation transformation rotation :reuse reuse))

(defgeneric transform-region (transformation region &key reuse))
(defgeneric untransform-region (transformation region &key reuse))
(defgeneric transform-rectangle* (transformation min-x min-y max-x max-y))
(defgeneric untransform-rectangle* (transformation min-x min-y max-x max-y))
(defgeneric transform-point* (transformation point-x point-y))
(defgeneric transform-dimensions (transformation width height))

;;; Similar naming problem for dimensions vs. dimensions* vs. distance*
;;; Add synonyms for spec'd names.
(defun transform-distance* (transformation width height)
  (transform-dimensions transformation width height))
(defun untransform-distance* (transformation width height)
  (untransform-dimensions transformation width height))

;;;
;;; TRANSFORMATION Types
;;;

(defclass transformation ()
    ()
  (:documentation "Describes a coordinate system transformation."))

(defmethod transformationp (object) (declare (ignore object)) nil)
(defmethod transformationp ((xf transformation)) t)

(defun transformation-type-error () 
  (error "Bad transformation type passed in somewhere"))

					; IDENTITY

(defclass identity-transformation (transformation)
    ())

(defvar +identity-transformation+ (make-instance 'identity-transformation))

(defmethod translation-x ((transformation identity-transformation)) 0)
(defmethod translation-y ((transformation identity-transformation)) 0)

(defclass translation-capacity ()
    ((m20 :initarg :m20 :initform 0 :initarg :translation-x 
	  :accessor translation-x)
     (m21 :initarg :m21 :initform 0 :initarg :translation-y 
	  :accessor translation-y)))

	 
					; TRANSLATION

(defclass translation (translation-capacity transformation)
    ())

(define-constructor %make-translation translation
  (translation-x translation-y)
  :translation-x translation-x :translation-y translation-y)

					; ST-TRANSFORMATION
(defclass st-transformation (translation-capacity transformation)
    ((m00 :initarg :m00 :initform 1)
     (m11 :initarg :m11 :initform 1)))
  
(define-constructor %make-st-transformation st-transformation (m00 m11 m20 m21)
		    :m00 m00 :m11 m11 :m20 m20 :m21 m21)

					; SRT-TRANSFORMATION

(defclass srt-transformation (translation-capacity transformation)
    ((m00 :initarg :m00 :initform 1)
     (m01 :initarg :m01 :initform 0)
     (m10 :initarg :m10 :initform 0)
     (m11 :initarg :m11 :initform 1)))
  
(define-constructor %make-srt-transformation srt-transformation 
  (m00 m01 m10 m11 m20 m21)
  :m00 m00 :m01 m01 :m10 m10 :m11 m11 :m20 m20 :m21 m21)


;;;
;;; Constructors
;;;

(defun make-translation-transformation (translation-x translation-y &key reuse)
  (reset-to-translation translation-x translation-y reuse))

;;;--- There may be a low level numerics function that computes sin and cos more
;;;--- efficiently than two separate calls?
;;; --- Yes, it's called CIS, but that conses a complex number.  Sigh.  --- rsl

(defun make-rotation-transformation* (angle origin-x origin-y &key reuse)
  (let ((angle (mod (float angle 0s0) (float (* 2 pi) 0s0))))
    (declare (type short-float angle))
    (if (zerop angle)
	+identity-transformation+
	(let* ((c (cos angle))
	       (s (sin angle))
	       (1-c (- 1s0 c))
	       (origin-x (float origin-x 0s0))
	       (origin-y (float origin-y 0s0)))
	  (declare (type short-float c s 1-c origin-x origin-y))
	  (reset-to-srt-transformation
	    c s (- s) c
	    (+ (* 1-c origin-x) (* s origin-y))
	    (- (* 1-c origin-y) (* s origin-x))
	    reuse)))))

;;; --- This should be &KEY ORIGIN so we can have a REUSE keyword -- rsl
(defun-inline make-rotation-transformation 
    (angle &optional (origin nil origin-p))
  (if origin-p
      (make-rotation-transformation* angle (point-x origin) (point-y origin))
      (make-rotation-transformation* angle 0 0)))

(defun make-scaling-transformation* (mx my origin-x origin-y &key reuse)
  (if (and (= mx 1) (= my 1)) 
      +identity-transformation+
      (reset-to-st-transformation mx my
				  (* (- 1 mx) origin-x)
				  (* (- 1 my) origin-y)
				  reuse)))

;;; See comments above about &KEY REUSE
(defun-inline make-scaling-transformation
    (mx my &optional (origin nil origin-p))
  (if origin-p
      (make-scaling-transformation* mx my (point-x origin) (point-y origin))
      (make-scaling-transformation* mx my 0 0)))

(defun make-reflection-transformation* (x1 y1 x2 y2 &key reuse)
  (let* ((nx (- y1 y2))
	 (ny (- x2 x1))
	 (nxx (* nx nx))
	 (nxy (* nx ny))
	 (nyy (* ny ny))
	 (denom (+ nxx nyy)))
    (when (zerop denom)
      (error "Can't construct reflection for two identical points"))
    (let* ((norm (/ 2 denom))
	   (nxx (* nxx norm))
	   (nxy (* nxy norm))
	   (nyy (* nyy norm)))
      (reset-to-srt-transformation (- 1 nxx) (- nxy) (- nxy) (- 1 nyy)
				   (+ (* nxx x1) (* nxy y1))
				   (+ (* nxy x1) (* nyy y1))
				   reuse))))

(defun-inline make-reflection-transformation (point-1 point-2)
  (make-reflection-transformation* (point-x point-1) (point-y point-1)
				   (point-x point-2) (point-y point-2)))

(defun-inline make-transformation-1 (mxx mxy myx myy tx ty)
  (if (and (= mxy 0) (= myx 0))				;Non-skewing
      (if (and (= mxx 1) (= myy 1))			;Non-scaling
	  (if (and (= tx 0) (= ty 0))			;Non-translating
	      +identity-transformation+			;So, it's the identity
	      (%make-translation tx ty))		;Translating
	  (%make-st-transformation mxx myy tx ty))	;Scaling and translating
      (%make-srt-transformation mxx mxy myx myy tx ty)));The whole shebang

(defun-inline reset-transformation-1 (mxx mxy myx myy tx ty reuse)
  (if (and (= mxy 0) (= myx 0))
      (if (and (= mxx 1) (= myy 1))
	  (if (and (= tx 0) (= ty 0))
	      (reset-to-identity reuse)
	      (reset-to-translation tx ty reuse))
	  (reset-to-st-transformation mxx myy tx ty reuse))
      (reset-to-srt-transformation mxx mxy myx myy tx ty reuse)))

;;; --- Should this take REUSE?  Maybe there should be different
;;; --- functions which do this. -- rsl
(defun-inline make-transformation (mxx mxy myx myy tx ty &key reuse)
  (if reuse
      (reset-transformation-1 mxx mxy myx myy tx ty reuse)
      (make-transformation-1 mxx mxy myx myy tx ty)))

(defun make-3-point-transformation* (x1 y1 x1-image y1-image
					x2 y2 x2-image y2-image
					x3 y3 x3-image y3-image
					&key reuse)
  (let* ((x1y2 (* x1 y2)) (x2y1 (* x2 y1))
	 (x2y3 (* x2 y3)) (x3y2 (* x3 y2))
	 (x3y1 (* x3 y1)) (x1y3 (* x1 y3))
	 (1/det (+ x1y2 (- x2y1) x2y3 (- x3y2) x3y1 (- x1y3))))
    (when (zerop 1/det)
      (error "Can't construct 3-point transformation for colinear points"))
    (setq 1/det (/ 1/det))
    (let ((x2-x1 (- x2 x1)) (y1-y2 (- y1 y2))
	  (x3-x2 (- x3 x2)) (y2-y3 (- y2 y3))
	  (x1-x3 (- x1 x3)) (y3-y1 (- y3 y1))
	  (x1y2-x2y1 (- x1y2 x2y1))
	  (x2y3-x3y2 (- x2y3 x3y2))
	  (x3y1-x1y3 (- x3y1 x1y3)))
      (make-transformation
       (* (+ (* x1-image y2-y3) (* x2-image y3-y1) (* x3-image y1-y2)) 1/det)
       (* (+ (* y1-image y2-y3) (* y2-image y3-y1) (* y3-image y1-y2)) 1/det)
       (* (+ (* x1-image x3-x2) (* x2-image x1-x3) (* x3-image x2-x1)) 1/det)
       (* (+ (* y1-image x3-x2) (* y2-image x1-x3) (* y3-image x2-x1)) 1/det)
       (* (+ (* x1-image x2y3-x3y2) 
	     (* x2-image x3y1-x1y3) 
	     (* x3-image x1y2-x2y1))
	  1/det) 
       (* (+ (* y1-image x2y3-x3y2) 
	     (* y2-image x3y1-x1y3)
	     (* y3-image x1y2-x2y1))
	  1/det)
       :reuse reuse))))

(defun make-3-point-transformation (point-1 point-1-image
					    point-2 point-2-image
					    point-3 point-3-image
					    &key reuse)
  (make-3-point-transformation*
   (point-x point-1) (point-y point-1)
   (point-x point-1-image) (point-y point-1-image)
   (point-x point-2) (point-y point-2)
   (point-x point-2-image) (point-y point-2-image)
   (point-x point-3) (point-y point-3)
   (point-x point-3-image) (point-y point-3-image)
   :reuse reuse))

;;;
;;; The TRANSFORMATION Protocols
;;;

;;;
;;; Copying
;;;

(defmethod copy-transformation ((transformation identity-transformation) 
				&optional out)
  (if (null out)
      +identity-transformation+
      (reset-to-identity out)))

(defmethod copy-transformation ((transformation translation) &optional out)
  (with-slots (m20 m21) transformation
    (if (null out)
	(%make-translation m20 m21)
	(reset-to-translation m20 m21 out))))

(defmethod copy-transformation
	   ((transformation st-transformation) &optional out)
  (with-slots (m00 m11 m20 m21) transformation
    (if (null out)
	(%make-st-transformation m00 m11 m20 m21)
	(reset-to-st-transformation m00 m11 m20 m21 out))))

(defmethod copy-transformation ((transformation srt-transformation) 
				&optional out)
  (with-slots (m00 m01 m10 m11 m20 m21) transformation
    (if (null out)
	(%make-srt-transformation m00 m01 m10 m11 m20 m21)
	(reset-to-srt-transformation m00 m01 m10 m11 m20 m21 out))))

;;;
;;; Reset to Identity
;;;

(defmethod reset-to-identity ((xf null))
  +identity-transformation+)

(defmethod reset-to-identity ((xf identity-transformation))
  +identity-transformation+)

(defmethod reset-to-identity ((xf translation))
  (with-slots (m20 m21) xf
    (setf m20 0
	  m21 0))
  xf)

(defmethod reset-to-identity ((xf st-transformation))
  (with-slots (m00 m11 m20 m21) xf
    (setf m00 1 m11 1 m20 0 m21 0))
  xf)

(defmethod reset-to-identity ((xf srt-transformation))
  (with-slots (m00 m01 m10 m11 m20 m21) xf
    (setf m00 1 m01 0 m10 0 m11 1 m20 0 m21 0))
  xf)

;;;
;;; Internal Transformation Type Maintenance (Coercion/Conversion, etc)
;;;

(defmacro %release-xf (xf) xf)
  
(defmethod reset-to-translation (dx dy (xf null))
  (%make-translation dx dy))

(defmethod reset-to-translation (dx dy xf)
  (%release-xf xf)
  (%make-translation dx dy))

(defmethod reset-to-translation (dx dy (xf translation))
  (with-slots (m20 m21) xf
    (setf m20 dx
	  m21 dy)
    xf))

(defmethod reset-to-translation (dx dy (xf st-transformation))
  (with-slots (m00 m11 m20 m21) xf
    (setf m00 1 m11 1 m20 dx m21 dy)
    xf))

(defmethod reset-to-translation (dx dy (xf srt-transformation))
  (with-slots (m00 m01 m10 m11 m20 m21) xf
    (setf m00 1 m01 0 m10 0 m11 1 m20 dx m21 dy)
    xf))

(defmethod reset-to-st-transformation (m00 m11 m20 m21 (xf null))
  (%make-st-transformation m00 m11 m20 m21))

(defmethod reset-to-st-transformation (m00 m11 m20 m21 xf)
  (%release-xf xf)
  (%make-st-transformation m00 m11 m20 m21))

(defmethod reset-to-st-transformation (new-m00 new-m11 new-m20 new-m21 
					       (xf st-transformation))
  (with-slots (m00 m11 m20 m21) xf
    (setf m00 new-m00 m11 new-m11 m20 new-m20 m21 new-m21)
    xf))

(defmethod reset-to-st-transformation (new-m00 new-m11 new-m20 new-m21 
					       (xf srt-transformation))
  (with-slots (m00 m01 m11 m10 m20 m21) xf
    (setf m00 new-m00 m11 new-m11 m20 new-m20 m21 new-m21
	  m01 0 m10 0)
    xf))

(defmethod reset-to-srt-transformation (m00 m01 m10 m11 m20 m21 (xf null))
  (%make-srt-transformation m00 m01 m10 m11 m20 m21))

(defmethod reset-to-srt-transformation (m00 m01 m10 m11 m20 m21 xf)
  (%release-xf xf)
  (%make-srt-transformation m00 m01 m10 m11 m20 m21))

(defmethod reset-to-srt-transformation (new-m00 new-m01 new-m10 
						new-m11 new-m20 new-m21 
						(xf srt-transformation))
  (with-slots (m00 m01 m10 m11 m20 m21) xf
    (setf m00 new-m00 m01 new-m01
	  m10 new-m10 m11 new-m11
	  m20 new-m20 m21 new-m21)
    xf))

;;;
;;; Predicates
;;;

;;; --- What do we do about floating-point fuzz?  ZEROP and = are
;;; --- probably not good enough predicates for this kind of comparison.
;;; --- -- rsl & York

(defmethod identity-transformation-p ((transformation identity-transformation))
  t)

(defmethod identity-transformation-p ((transformation translation))
  (with-slots (m20 m21) transformation
    (and (zerop m20) (zerop m21))))

(defmethod identity-transformation-p ((transformation st-transformation))
  (with-slots (m00 m11 m20 m21) transformation
    (and (zerop m20) (zerop m21)
	 (= m00 1) (= m11 1))))

(defmethod identity-transformation-p ((transformation srt-transformation))
  (with-slots (m00 m01 m10 m11 m20 m21) transformation
    (and (zerop m20) (zerop m21)
	 (= m00 1) (= m11 1)
	 (zerop m10) (zerop m01))))

;;;

(defmethod translation-transformation-p ((transformation identity-transformation))
  t)

(defmethod translation-transformation-p ((transformation translation))
  t)

(defmethod translation-transformation-p ((transformation st-transformation))
  (with-slots (m00 m11) transformation
    (and (= m00 1) (= m11 1))))

(defmethod translation-transformation-p ((transformation srt-transformation))
  (with-slots (m00 m01 m10 m11) transformation
    (and (= m00 1) (= m11 1)
	 (zerop m10) (zerop m01))))

;;;

(defmethod invertible-transformation-p ((transformation identity-transformation))
  t)

(defmethod invertible-transformation-p ((transformation translation))
  t)

(defmethod invertible-transformation-p ((transformationation st-transformation))
  (with-slots (m00 m11) transformationation
    (not (or (zerop m00) (zerop m11)))))

(defmethod invertible-transformation-p ((transformation srt-transformation))
  (with-slots (m00 m01 m10 m11) transformation
    (not (zerop (- (* m00 m11) (* m01 m10))))))

;;;

(defmethod reflection-transformation-p ((transformation identity-transformation))
  nil)

(defmethod reflection-transformation-p ((transformation translation))
  nil)

(defmethod reflection-transformation-p ((transformation st-transformation))
  (with-slots (m00 m11) transformation
    (minusp (* m00 m11))))

(defmethod reflection-transformation-p ((transformation srt-transformation))
  (with-slots (m00 m01 m10 m11) transformation
    (minusp (- (* m00 m11) (* m01 m10)))))

(define-unimplemented-protocol-method reflection-transformation-p transformation
  ((transformation transformation)))

;;;

(defmethod rigid-transformation-p ((transform identity-transformation))
  t)

(defmethod rigid-transformation-p ((transform translation))
  t)

(defmethod rigid-transformation-p ((transform srt-transformation))
  ;; ??? I don't understand this ??? -- RR
  (with-slots (m00 m01 m10 m11) transform
    (and (= (- (* m00 m11) (* m01 m10)) 1)
	 (= (+ (* m11 m01) (* m10 m11)) 1)
	 (= (+ (expt m00 2) (expt m10 2)) 1))))

(defmethod rigid-transformation-p ((transform st-transformation))
  (with-slots (m00 m11) transform
    (= (abs m00) 1)
    (= (abs m11) 1)))


;;;

(defmethod even-scaling-transformation-p ((transform identity-transformation))
  t)

(defmethod even-scaling-transformation-p ((transform translation))
  t)

(defmethod even-scaling-transformation-p ((transform srt-transformation))
  (with-slots (m00 m01 m10 m11) transform
    (and (zerop m01) (zerop m10) (= m00 m11))))

(defmethod even-scaling-transformation-p ((transform st-transformation))
  (with-slots (m00 m11) transform
    (= m00 m11)))

;;;

(defmethod scaling-transformation-p ((transform identity-transformation))
  t)

(defmethod scaling-transformation-p ((transform translation))
  t)

(defmethod scaling-transformation-p ((transform st-transformation))
  t)

(defmethod scaling-transformation-p ((transform srt-transformation))
  (with-slots (m01 m10) transform
    (and (zerop m01) (zerop m10))))

;;;

(defmethod rectilinear-transformation-p ((transform identity-transformation))
  t)

(defmethod rectilinear-transformation-p ((transformation st-transformation))
  t)

(defmethod rectilinear-transformation-p ((transformation translation))
  t)

(defmethod rectilinear-transformation-p ((transformation srt-transformation))
  (with-slots (m00 m01 m10 m11) transformation
    (or (and (= m01 0) (= m10 0))
	(and (= m00 0) (= m11 0)))))

(define-unimplemented-protocol-method rectilinear-transformation-p transformation
  ((transformation transformation)))


;;;
;;; ???
;;;  These are here to handle reflecting rectangles used to define areas
;;;  The problem is that reflecting makes the "open" end flip.
;;;  Of course this is still assuming discrete coordinates.
;;;

(defun careful-transform-region (xf r)
  (typecase r
    (nowhere r)
    (standard-rectangle
     (let ((r (transform-region xf r)))
       (when (reflection-transformation-p xf)
	 (multiple-value-bind (old-x old-y) (rectangle-position* r) 
	   (setf* (rectangle-position* r) (values old-x (1+ old-y)))))
       r))
    (otherwise
     (error "Can't handle this type of region"))))

(defun careful-untransform-region (xf r)
  (let ((r (untransform-region xf r)))
    (assert (typep r 'standard-rectangle) ()
	    "Can't handle this type of region")
    (when (reflection-transformation-p xf)
      (multiple-value-bind (old-x old-y) (rectangle-position* r) 
	(setf* (rectangle-position* r) (values old-x (1+ old-y)))))
    r))

(defun careful-transform-rectangle* (xf minx miny maxx maxy)
  (multiple-value-bind (minx miny maxx maxy)
      (transform-rectangle* xf minx miny maxx maxy)
    (when (reflection-transformation-p xf)
      (incf miny)
      (incf maxy))
    (values minx miny maxx maxy)))

(defun careful-untransform-rectangle* (xf minx miny maxx maxy)
  (multiple-value-bind (minx miny maxx maxy)
      (untransform-rectangle* xf minx miny maxx maxy)
    (when (reflection-transformation-p xf)
      (incf miny)
      (incf maxy))
    (values minx miny maxx maxy)))
  
		     
;;;
;;;  Composition and Inversion
;;;

#+Ignore
(defmethod compose-transformations :around (xf1 xf2 &key reuse)
;  (break)
  (call-next-method xf1 xf2 :key reuse))

(defmethod compose-transformations ((xf1 identity-transformation)
				    (xf2 transformation)
				    &key reuse)
  (copy-transformation xf2 reuse))

(defmethod compose-transformations ((xf1 transformation)
				    (xf2 identity-transformation)
				    &key reuse)
  (copy-transformation xf1 reuse))

(defmethod compose-transformations 
	   ((xf1 translation) (xf2 translation) &key reuse)
  (with-slots ((x1 m20)
	       (y1 m21)) xf1
    (with-slots ((x2 m20)
		 (y2 m21)) xf2
      (reset-to-translation (+ x1 x2) (+ y1 y2) reuse))))

(defmethod compose-transformations ((xf1 srt-transformation)
				    (xf2 srt-transformation)
				    &key reuse)
  (with-slots ((xf1-00 m00) 
	       (xf1-01 m01) 
	       (xf1-10 m10) 
	       (xf1-11 m11) 
	       (xf1-20 m20) 
	       (xf1-21 m21)) xf1
    (with-slots ((xf2-00 m00) 
		 (xf2-01 m01) 
		 (xf2-10 m10) 
		 (xf2-11 m11) 
		 (xf2-20 m20) 
		 (xf2-21 m21)) xf2 
      (reset-to-srt-transformation
       (+ (* xf1-00 xf2-00) (* xf1-01 xf2-10))
       (+ (* xf1-00 xf2-01) (* xf1-01 xf2-11))
       (+ (* xf1-10 xf2-00) (* xf1-11 xf2-10))
       (+ (* xf1-10 xf2-01) (* xf1-11 xf2-11))
       (+ (* xf1-20 xf2-00) (* xf1-21 xf2-10) xf2-20)
       (+ (* xf1-20 xf2-01) (* xf1-21 xf2-11) xf2-21)
       reuse))))

(defmethod compose-transformations ((xf1 st-transformation)
				    (xf2 st-transformation)
				    &key reuse)
  (with-slots ((xf1-00 m00) 
	       (xf1-11 m11) 
	       (xf1-20 m20) 
	       (xf1-21 m21)) xf1
    (with-slots ((xf2-00 m00) 
		 (xf2-11 m11) 
		 (xf2-20 m20) 
		 (xf2-21 m21)) xf2 
      (reset-to-st-transformation 
       (* xf1-00 xf2-00)
       (* xf1-11 xf2-11)
       (+ (* xf1-20 xf2-00) xf2-20)
       (+ (* xf1-21 xf2-11) xf2-21)
       reuse))))

(defmethod compose-transformations ((xf1 srt-transformation)
				    (xf2 translation)
				    &key reuse)

  (with-slots ((xf1-00 m00) 
	       (xf1-01 m01) 
	       (xf1-10 m10) 
	       (xf1-11 m11) 
	       (xf1-20 m20) 
	       (xf1-21 m21)) xf1
    (with-slots ((xf2-20 m20) 
		 (xf2-21 m21)) xf2 
      (reset-to-srt-transformation xf1-00 xf1-01
				   xf1-10 xf1-11
				   (+ xf1-20 xf2-20)
				   (+ xf1-21 xf2-21)
				   reuse))))

(defmethod compose-transformations ((xf1 translation)
				    (xf2 srt-transformation)
				    &key reuse)
  (with-slots ((xf1-20 m20) 
	       (xf1-21 m21)) xf1
    (with-slots ((xf2-00 m00) 
		 (xf2-01 m01) 
		 (xf2-10 m10) 
		 (xf2-11 m11) 
		 (xf2-20 m20) 
		 (xf2-21 m21)) xf2 
      (reset-to-srt-transformation 
       xf2-00 xf2-01
       xf2-10 xf2-11
       (+ (* xf1-20 xf2-00) (* xf1-21 xf2-10)
	  xf2-20)
       (+ (* xf1-20 xf2-01) (* xf1-21 xf2-11) xf2-21)
       reuse))))

(defmethod compose-transformations ((xf1 st-transformation)
				    (xf2 translation)
				    &key reuse)
  (with-slots ((xf1-00 m00) 
	       (xf1-11 m11) 
	       (xf1-20 m20) 
	       (xf1-21 m21)) xf1
    (with-slots ((xf2-20 m20) 
		 (xf2-21 m21)) xf2 
      (reset-to-st-transformation  
       xf1-00 xf1-11
       (+ xf1-20 xf2-20)
       (+ xf1-21 xf2-21)
       reuse))))

(defmethod compose-transformations ((xf1 translation)
				    (xf2 st-transformation)
				    &key reuse)
  (with-slots ((xf1-20 m20) 
	       (xf1-21 m21)) xf1
    (with-slots ((xf2-00 m00) 
		 (xf2-11 m11) 
		 (xf2-20 m20) 
		 (xf2-21 m21)) xf2 
      (reset-to-st-transformation 
       xf2-00 xf2-11
       (+ (* xf1-20 xf2-00) xf2-20)
       (+ (* xf1-21 xf2-11) xf2-21)
       reuse))))

(defmethod compose-transformations ((xf1 st-transformation)
				    (xf2 srt-transformation)
				    &key reuse)
  (with-slots ((xf1-00 m00) 
	       (xf1-11 m11) 
	       (xf1-20 m20) 
	       (xf1-21 m21)) xf1
    (with-slots ((xf2-00 m00) 
		 (xf2-01 m01) 
		 (xf2-10 m10) 
		 (xf2-11 m11) 
		 (xf2-20 m20) 
		 (xf2-21 m21)) xf2 
      (reset-to-srt-transformation 
       (+ (* xf1-00 xf2-00))
       (+ (* xf1-00 xf2-01))
       (+ (* xf1-11 xf2-10))
       (+ (* xf1-11 xf2-11))
       (+ (* xf1-20 xf2-00) (* xf1-21 xf2-10) xf2-20)
       (+ (* xf1-20 xf2-01) (* xf1-21 xf2-11) xf2-21)
       reuse))))

(defmethod compose-transformations ((xf1 srt-transformation)
				    (xf2 st-transformation)
				    &key reuse)
  (with-slots ((xf1-00 m00) 
	       (xf1-01 m01) 
	       (xf1-10 m10) 
	       (xf1-11 m11) 
	       (xf1-20 m20) 
	       (xf1-21 m21)) xf1
    (with-slots ((xf2-00 m00)
		 (xf2-11 m11) 
		 (xf2-20 m20) 
		 (xf2-21 m21)) xf2 
      (reset-to-srt-transformation
       (+ (* xf1-00 xf2-00))
       (+ (* xf1-01 xf2-11))
       (+ (* xf1-10 xf2-00))
       (+ (* xf1-11 xf2-11))
       (+ (* xf1-20 xf2-00) xf2-20)
       (+ (* xf1-21 xf2-11) xf2-21)
       reuse))))


(defmethod invert-transformation ((xf identity-transformation) 
				  &key reuse)
  (declare (ignore reuse))
  +identity-transformation+)

(defmethod invert-transformation ((xf translation) &key reuse)
  (with-slots (m20 m21) xf
    (if reuse
	(reset-to-translation (- m20) (- m21) reuse)
	(%make-translation (- m20) (- m21)))))

(defmethod invert-transformation ((xf srt-transformation) &key reuse)
  (with-slots ((xf-00 m00) 
	       (xf-01 m01) 
	       (xf-10 m10) 
	       (xf-11 m11) 
	       (xf-20 m20) 
	       (xf-21 m21)) xf 
    (let ((det (- (* xf-00 xf-11) (* xf-01 xf-10))))
      (reset-to-srt-transformation
       (/ xf-11 det) (- (/ xf-01 det))
       (- (/ xf-10 det)) (/ xf-00 det)
       (/ (- (* xf-10 xf-21) (* xf-11 xf-20)) det)
       (/ (- (* xf-01 xf-20) (* xf-00 xf-21)) det)
       reuse))))

(defmethod invert-transformation ((xf st-transformation) &key reuse)
  (with-slots ((xf-00 m00) 
	       (xf-11 m11) 
	       (xf-20 m20) 
	       (xf-21 m21)) xf 
    (let ((det (* xf-00 xf-11)))
      (reset-to-st-transformation (/ xf-11 det) (/ xf-00 det)
			    (/ (- (* xf-11 xf-20)) det)
			    (/ (- (* xf-00 xf-21)) det)
			    reuse))))

;;;
;;; Modification by post-multiplication
;;;

(defmethod compose-with-translation ((xf identity-transformation)
				 dx dy &key reuse)
  (reset-to-translation dx dy reuse))

(defmethod compose-with-translation ((xf translation) dx dy &key reuse)
  (with-slots (m20 m21) xf
    (reset-to-translation (+ m20 dx) (+ m21 dy) reuse)))

(defmethod compose-with-translation ((xf st-transformation) dx dy &key reuse)
  (with-slots (m00 m11 m20 m21) xf
    (reset-to-st-transformation m00 m11 (+ m20 dx) (+ m21 dy) reuse)))

(defmethod compose-with-translation ((xf srt-transformation)
				 dx dy &key reuse)
  (with-slots (m00 m01 m10 m11 m20 m21) xf
    (reset-to-srt-transformation m00 m01 m10 m11 (+ m20 dx) (+ m21 dy)
				 reuse)))

(defmethod compose-with-scaling ((xf identity-transformation) 
			     sx sy &key reuse)
  (reset-to-st-transformation sx sy 0 0 reuse))

(defmethod compose-with-scaling ((xf translation) sx sy &key reuse)
  (with-slots (m20 m21) xf
    (reset-to-st-transformation sx sy (* sx m20) (* sy m21) reuse)))

(defmethod compose-with-scaling ((xf st-transformation) 
			     sx sy &key reuse)
  (with-slots (m00 m11 m20 m21) xf
    (reset-to-st-transformation 
     (* sx m00) (* sy m11) (* sx m20) (* sy m21) reuse)))

(defmethod compose-with-scaling ((xf srt-transformation) sx sy &key reuse)
  (with-slots (m00 m01 m10 m11 m20 m21) xf
    (reset-to-srt-transformation
     (* m00 sx) (* m01 sy) (* m10 sx) (* m11 sy) (* m20 sx) (* m21 sy)
     reuse)))

(defmethod compose-with-rotation ((xf identity-transformation)
			     rotation &key reuse)
  (let ((cos (cos rotation))
	(sin (sin rotation)))
    (reset-to-srt-transformation cos sin (- sin) cos 0 0 reuse)))

(defmethod compose-with-rotation ((xf translation) rotation &key reuse)
  (with-slots (m20 m21) xf
    (let ((cos (cos rotation))
	  (sin (sin rotation)))
      (reset-to-srt-transformation
       cos                          sin
       (- sin)                      cos
       (- (* cos m20) (* sin m21))  (+ (* sin m20) (* cos m21))
       reuse))))

(defmethod compose-with-rotation ((xf st-transformation) rotation &key reuse)
  (with-slots (m00 m11 m20 m21) xf
    (let ((cos (cos rotation))
	  (sin (sin rotation)))
      (reset-to-srt-transformation
       (* cos m00)                  (* sin m00)
       (- (* sin m11))              (* cos m11)
       (- (* cos m20) (* sin m21))  (+ (* sin m20) (* cos m21))
       reuse))))

(defmethod compose-with-rotation ((xf srt-transformation) rotation &key reuse)
  (with-slots (m00 m01 m10 m11 m20 m21) xf
    (let ((cos (cos rotation))
	  (sin (sin rotation)))
      (reset-to-srt-transformation
       (- (* cos m00) (* sin m01)) (+ (* sin m00) (* cos m01))
       (- (* cos m10) (* sin m11)) (+ (* sin m10) (* cos m11))
       (- (* cos m20) (* sin m21))  (+ (* sin m20) (* cos m21))
       reuse))))
     
;;;
;;; Transformation application and removal
;;;

(defmethod transform-region (transformation (r everywhere) &key reuse)
  ;;; ??? What should I do with reuse in this case.  
  (declare (ignore transformation reuse))
  r)

(defmethod transform-region (transformation (r nowhere) &key reuse)
  (declare (ignore transformation reuse))
  r)

(defmethod transform-region (transformation (ca rectangle-set) &key reuse)
  (declare (ignore reuse))
  (make-rectangle-set :rectangles
		      (mapcar #'(lambda (a)
				  (transform-region transformation a))
			      (rectangle-set-rectangles ca))))


(defmethod untransform-region (transformation (r everywhere) &key reuse)
  (declare (ignore transformation reuse))
  r)

(defmethod untransform-region (transformation (r nowhere) &key reuse)
  (declare (ignore transformation reuse))
  r)


(defmethod untransform-region (transformation (ca rectangle-set) &key reuse)
  (declare (ignore reuse))
  (make-rectangle-set :rectangles
		      (mapcar #'(lambda (a)
				  (untransform-region transformation a))
			      (rectangle-set-rectangles ca))))

(defmethod transform-region 
	   ((transformation identity-transformation) (point standard-point)
	    &key reuse)
  (with-slots (x y) point
    (make-point x y :reuse reuse)))

(defmethod transform-region
	   ((translation translation) (point standard-point) &key reuse)
  (with-slots (x y) point
    (with-slots (m20 m21) translation
      (make-point (+ x m20) (+ y m21) :reuse reuse))))

(defmethod transform-region 
	   ((xform st-transformation) (point standard-point) &key reuse)
  (with-slots (m00 m11 m20 m21) xform
    (with-slots (x y) point
      (make-point (+ (* x m00) m20)
		  (+ (* y m11) m21)
		  :reuse reuse))))

(defmethod transform-region 
	   ((xform srt-transformation) (point standard-point) &key reuse)
  (with-slots (m00 m01 m10 m11 m20 m21) xform
    (with-slots (x y) point
      (make-point
       (+ (* x m00) (+ (* y m10) m20))
       (+ (* y m11) (+ (* x m01) m21))
       :reuse reuse))))

;;; There used to be methods for points for each ofthe different kinds
;;; of transformations, but that was a maintenance nightmare. --- rsl
(defmethod untransform-region (transformation (point standard-point) &key reuse)
  (with-slots (x y) point
    (multiple-value-bind (new-x new-y) (untransform-point* transformation x y)
      (make-point new-x new-y :reuse reuse))))

;;;
;;; Transforming rectangles
;;;

(defmethod transform-region
	   ((transformation identity-transformation) (rect standard-rectangle)
	    &key reuse)
  (with-bounding-rectangle* (min-x min-y max-x max-y) rect
    (make-rectangle* min-x min-y max-x max-y :reuse reuse)))

(defmethod transform-region
	   ((translation translation) (rect standard-rectangle)
	    &key reuse)
  (with-slots (m20 m21) translation
    (with-bounding-rectangle* (min-x min-y max-x max-y) rect    
      (make-rectangle* (+ min-x m20) (+ min-y m21)
		      (+ max-x m20) (+ max-y m21)
		      :reuse reuse))))

(defmethod transform-region 
	   ((xform srt-transformation) (rect standard-rectangle)
	    &key reuse)
  (with-bounding-rectangle* (min-x min-y max-x max-y) rect    
    (with-slots (m00 m01 m10 m11 m20 m21) xform
      (let ((new-x1 (+ (* min-x m00) (* min-y m10) m20))
	    (new-y1 (+ (* min-y m11) (* min-x m01) m21))
	    (new-x2 (+ (* max-x m00) (* max-y m10) m20))
	    (new-y2 (+ (* max-y m11) (* max-x m01) m21)))
	(make-rectangle* (min new-x1 new-x2) (min new-y1 new-y2)
			(max new-x1 new-x2) (max new-y1 new-y2)
			:reuse reuse)))))

(defmethod transform-region 
	   ((xform st-transformation) (rect standard-rectangle) &key reuse)
  (with-slots (min-x min-y max-x max-y) rect    
    (with-slots (m00 m11 m20 m21) xform
      (let ((new-x1 (+ (* min-x m00) m20))
	    (new-y1 (+ (* min-y m11) m21))
	    (new-x2 (+ (* max-x m00) m20))
	    (new-y2 (+ (* max-y m11) m21)))
	(make-rectangle* (min new-x1 new-x2) (min new-y1 new-y2)
			 (max new-x1 new-x2) (max new-y1 new-y2)
			 :reuse reuse)))))

;;; Another maintenance nightmare removed. --- rsl
(defmethod untransform-region (transformation (rect standard-rectangle) &key reuse)
  (with-slots (min-x min-y max-x max-y) rect
    (multiple-value-bind (new-min-x new-min-y new-max-x new-max-y)
	(untransform-rectangle* transformation min-x min-y max-x max-y)
      (make-rectangle* (min new-min-x new-max-x)
		       (min new-min-y new-max-y)
		       (max new-min-x new-max-x)
		       (max new-min-y new-max-y)
		       :reuse reuse))))

;;;
;;; Spread Application 
;;;

(defmethod transform-rectangle* ((transformation identity-transformation)
				 min-x min-y max-x max-y)
  (values min-x min-y max-x max-y))

(defmethod transform-rectangle* ((translation translation) 
				 min-x min-y max-x max-y)
  (with-slots (m20 m21) translation
    (let ((min-x (+ min-x m20))
	  (min-y (+ min-y m21)))
      (values min-x min-y (+ max-x m20) (+ max-y m21)))))

(defmethod transform-rectangle* ((xform st-transformation) 
				 min-x min-y max-x max-y)
  (with-slots (m00 m11 m20 m21) xform
    (let ((new-x1 (+ (* min-x m00) m20))
	  (new-y1 (+ (* min-y m11) m21))
	  (new-x2 (+ (* max-x m00) m20))
	  (new-y2 (+ (* max-y m11) m21)))
      (values (min new-x1 new-x2) (min new-y1 new-y2)
	      (max new-x1 new-x2) (max new-y1 new-y2)))))

(defmethod transform-rectangle* ((xform srt-transformation) 
				 min-x min-y max-x max-y)
  (with-slots (m00 m01 m10 m11 m20 m21) xform
    (let ((new-x1 (+ (* min-x m00) (* min-y m10) m20))
	  (new-y1 (+ (* min-y m11) (* min-x m01) m21))
	  (new-x2 (+ (* max-x m00) (* max-y m10) m20))
	  (new-y2 (+ (* max-y m11) (* max-x m01) m21)))
      (values (min new-x1 new-x2) (min new-y1 new-y2)
	      (max new-x1 new-x2) (max new-y1 new-y2)))))

(defmethod untransform-rectangle* ((transformation identity-transformation)
				   min-x min-y max-x max-y)
  ;; Canonicalize resulting rectangle
  (when (< max-x min-x) (rotatef min-x max-x))
  (when (< max-y min-y) (rotatef min-x max-y))
  (values min-x min-y max-x max-y))

(defmethod untransform-rectangle* ((translation translation) 
				   min-x min-y max-x max-y)
  ;; Canonicalize rectangle
  (when (< max-x min-x) (rotatef min-x max-x))
  (when (< max-y min-y) (rotatef min-x max-y))
  (let ((m20 (slot-value translation 'm20))
	(m21 (slot-value translation 'm21)))
    (values (- min-x m20) (- min-y m21)
	    (- max-x m20) (- max-y m21))))

(defmethod untransform-rectangle* ((xform st-transformation) 
				   min-x min-y max-x max-y)
  (let ((m00 (slot-value xform 'm00))
	(m11 (slot-value xform 'm11))
	(m20 (slot-value xform 'm20))
	(m21 (slot-value xform 'm21)))
    (let ((x1 (/ (- min-x m20) m00))
	  (y1 (/ (- min-y m21) m11))
	  (x2 (/ (- max-x m20) m00))
	  (y2 (/ (- max-y m21) m11)))
      ;; Canonicalize rectangle
      (when (< x2 x1) (rotatef x1 x2))
      (when (< y2 y1) (rotatef y1 y2))
      (values x1 y1 x2 y2))))

(defmethod untransform-rectangle* ((xform srt-transformation) 
				   min-x min-y max-x max-y)
    (let ((m00 (slot-value xform 'm00))
	  (m01 (slot-value xform 'm01))
	  (m10 (slot-value xform 'm10))
	  (m11 (slot-value xform 'm11))
	  (m20 (slot-value xform 'm20))
	  (m21 (slot-value xform 'm21)))
      (let ((det (- (* m00 m11) (* m01 m10)))
	    (x1 (- min-x m20))
	    (y1 (- min-y m21))
	    (x2 (- max-x m20))
	    (y2 (- max-y m21)))
	(let ((nx1 (/ (- (* m11 x1) (* m10 y1)) det))
	      (ny1 (/ (- (* m00 y1) (* m01 x1)) det))
	      (nx2 (/ (- (* m11 x2) (* m10 y2)) det))
	      (ny2 (/ (- (* m00 y2) (* m01 x2)) det)))
	  ;; Canonicalize rectangle
	  (when (< nx2 nx1) (rotatef nx1 nx2))
	  (when (< ny2 ny1) (rotatef ny1 ny2))
	  (values nx1 ny1 nx2 ny2)))))

(defmethod transform-point* ((transformation identity-transformation)
			     min-x min-y)
  (values min-x min-y))

(defmethod transform-point* ((translation translation) min-x min-y)
  (with-slots (m20 m21) translation
    (let ((min-x (+ min-x m20))
	  (min-y (+ min-y m21)))
      (values min-x min-y))))

(defmethod transform-point* ((xform st-transformation) min-x min-y)
  (with-slots (m00 m11 m20 m21) xform
    (let ((new-x1 (+ (* min-x m00) m20))
	  (new-y1 (+ (* min-y m11) m21)))
      (values new-x1 new-y1))))

(defmethod transform-point* ((xform srt-transformation) min-x min-y)
  (with-slots (m00 m01 m10 m11 m20 m21) xform
    (let ((new-x1 (+ (* min-x m00) (* min-y m10) m20))
	  (new-y1 (+ (* min-y m11) (* min-x m01) m21)))
      (values new-x1 new-y1))))

(defmethod untransform-point* ((transformation identity-transformation) 
			       min-x min-y)
  (values min-x min-y))

(defmethod untransform-point* ((translation translation) min-x min-y)
  (with-slots (m20 m21) translation
    (let ((min-x (- min-x m20))
	  (min-y (- min-y m21)))
      (values min-x min-y))))

(defmethod untransform-point* ((xform st-transformation) min-x min-y)
  (with-slots (m00 m11 m20 m21) xform
    (values (/ (- min-x m20) m00)
	    (/ (- min-y m21) m11))))

(defmethod untransform-point* ((xform srt-transformation) min-x min-y)
    (let ((m00 (slot-value xform 'm00))
	  (m01 (slot-value xform 'm01))
	  (m10 (slot-value xform 'm10))
	  (m11 (slot-value xform 'm11))
	  (m20 (slot-value xform 'm20))
	  (m21 (slot-value xform 'm21)))
      (let ((det (- (* m00 m11) (* m01 m10)))
	    (x1 (- min-x m20))
	    (y1 (- min-y m21)))
	(values (/ (- (* m11 x1) (* m10 y1)) det)
		(/ (- (* m00 y1) (* m01 x1)) det)))))

(defmethod transform-dimensions 
	   ((transformation identity-transformation) width height)
  (values width height))

(defmethod transform-dimensions ((translation translation) width height)
  (values width height))

(defmethod transform-dimensions ((xform st-transformation) width height)
  (with-slots (m00 m11) xform
    (values (abs (* width m00))
	    (abs (* height m11)))))

(defmethod transform-dimensions ((xform srt-transformation) width height)
  (with-slots (m00 m01 m10 m11) xform
    (values (abs (+ (* width m00) (* height m10)))
	    (abs (+ (* height m11) (* width m01))))))

(defmethod untransform-dimensions 
	   ((transformation identity-transformation) width height)
  (values width height))


(defmethod untransform-dimensions ((transformation translation) width height)
  (values width height))

;;; --- unclear to me that this should be using ABS, but that's what Rao's code did. --- rsl
(defmethod untransform-dimensions ((xform st-transformation) width height)
  (with-slots (m00 m11) xform
      (values (abs (/ width m00))
	      (abs (/ height m11)))))

(defmethod untransform-dimensions ((xform srt-transformation) width height)
  (let ((m00 (slot-value xform 'm00))
	(m01 (slot-value xform 'm01))
	(m10 (slot-value xform 'm10))
	(m11 (slot-value xform 'm11)))
    (let ((det (- (* m00 m11) (* m01 m10))))
      (values (abs (+ (* width (/ m11 det)) (* height (- (/ m10 det)))))
	      (abs (+ (* height (/ m00 det)) (* width (- (/ m01 det)))))))))


