;;; -*- Mode:Lisp; Package: SILICA; Syntax:COMMON-LISP; Base:10 -*-
;;;
;;;
;;; A version of this code was sharked from Deli and modified to taste.
;;; Thanx, MCC.
;;; I added lucid stream support.
;;; 

(in-package "SILICA")

#-(or excl symbolics lucid ccl-2)
  (error "Sheet-based streams not implemented only for your Lisp")

(eval-when (compile load eval)
  (export
   '(terminal-stream-mixin
     terminal-stream
     open-terminal-stream
     close-terminal-stream
     terminal-stream-echoing-p
     terminal-stream-location
     terminal-stream-read-focus
     terminal-stream-write-focus
     )))

;;; -------
;;; ALLEGRO

#+excl
(progn
(defun %make-terminal-stream ()
  (let ((stm (excl::make-vanilla-stream)))
    (setf (excl::sm_describe Stm) #'window-stm-describe
	  (excl::sm_flags Stm) 3	; i/o
	  (excl::sm_type Stm)  :terminal-stream
	  (excl::sm_write-char Stm) #'window-stm-write-char
	  (excl::sm_write-byte Stm) #'window-stm-write-char
	  (excl::sm_write-string Stm) #'window-stm-write-string
	  (excl::sm_read-char Stm) #'window-stm-read-char
	  (excl::sm_read-byte Stm) #'window-stm-read-char
	  (excl::sm_listen Stm) #'window-stm-listen
	  (excl::sm_read-char-nh Stm) #'window-stm-illegal
	  (excl::sm_misc Stm) #'window-stm-misc)
    (setf (excl::sm_fio-fn Stm) (cons nil nil))
    Stm))

(defun window-stm-describe (stream)
  (declare (ignore stream))
  (write-string "sheet-based stream"))

(defun window-stm-illegal (stream &rest ignore)
  (declare (ignore ignore))
  (error "illegal operation for canvas stream ~s" stream))

(defun window-stm-misc (stream code &optional arg)
  (case code
    (:unread (window-stm-unread-char stream arg))
    (:input-edit-p nil)
    (:clear-input (window-stm-clear-input stream))
    ((:filename :charpos) nil)
    (:close (window-stm-close stream))
    (:force-output (window-stm-force-output stream))
    (:clear-output nil)
    (:element-type :string-char)
    (t (error "illegal code to canvas stream: ~s" code))))
) ;end of #+excl progn


;;; -----
;;; LUCID

#+lucid
(progn
  
(lucid:define-stream-type terminal-stream
    (lucid::basic-stream)
  ((read-focus nil)
   (write-focus nil)
   (location nil)
   (input-buffer nil)
   (un-tyi-char nil)
   (echo nil))
  (:constructor %%make-terminal-stream))

(defun %make-terminal-stream (&rest args)
  (let ((stream (apply #'%%make-terminal-stream args)))
    (setf (lucid::basic-stream-element-type stream) 'string-char
	  (lucid::basic-stream-direction stream) :io)
    stream))
   
(lucid::define-stream-method lucid::internal-write-char ((stream terminal-stream) char)
  (window-stm-write-char stream char))

(lucid::define-stream-method lucid::internal-write-string 
    ((stream terminal-stream) string start end)
  (window-stm-write-string stream string start (if end (- end start))))

(lucid::define-stream-method lucid::internal-read-char 
    ((stream terminal-stream) no-hang-p)
  (declare (ignore no-hang-p))
  (window-stm-read-char stream))

(lucid::define-stream-method lucid::internal-unread-char
    ((stream terminal-stream) char)
  (window-stm-unread-char stream char))

(lucid::define-stream-method lucid::internal-listen ((stream terminal-stream))
  (window-stm-listen stream))

(lucid::define-stream-method lucid::internal-clear-input ((stream terminal-stream))
  (window-stm-clear-input stream))

(lucid::define-stream-method lucid::internal-close ((stream terminal-stream))
  (window-stm-close stream))

(lucid::define-stream-method lucid::internal-force-output
    ((stream terminal-stream) wait-p)
  (declare (ignore wait-p))
  (window-stm-force-output stream))

) ;; end of LUCID


;;;;;;;;;;;;;;;;;;
;;;
;;; SYMBOLICS

#+symbolics
(progn
(scl:defflavor terminal-stream
	((un-tyi-char nil)
	 (input-buffer nil)
	 (read-focus nil)
	 (write-focus nil)
	 (location nil)
	 (echo nil))
	(si:BIDIRECTIONAL-STREAM)
  :settable-instance-variables)

(defun %make-terminal-stream ()
  (scl:make-instance 'terminal-stream))

(scl:defmethod (:tyo terminal-stream) (arg1 &rest rest)
  (declare (ignore rest))
  (window-stm-write-char scl:self arg1))

(scl:defmethod (:string-out terminal-stream)
	   (string &optional start end &rest rest)
  (declare (ignore rest))
  (window-stm-write-string scl:self
			   string start
			   (if end (- end start))))

(scl:defmethod (:tyi terminal-stream) (&optional arg1 &rest rest)
  (declare (ignore arg1 rest))
  (window-stm-read-char scl:self))

(scl:defmethod (:untyi terminal-stream) (arg1 &rest rest)
  (declare (ignore rest))
  (window-stm-unread-char scl:self arg1))

(scl:defmethod (:listen  terminal-stream) (&optional arg1 &rest rest)
  (declare (ignore arg1 rest))
  (window-stm-listen scl:self))

(scl:defmethod (:clear-input terminal-stream) (&rest ignore)
  (declare (ignore IGNORE))
  (window-stm-clear-input scl:self))

(scl:defmethod (:close terminal-stream) (&rest ignore)
  (declare (ignore IGNORE))
  (window-stm-close scl:self))

(scl:defmethod (:force-output terminal-stream) (&rest ignore)
  (declare (ignore IGNORE))
  (window-stm-force-output scl:self))

(scl:defmethod (:element-type terminal-stream) ()
  :string-char)

(scl:defmethod (:unclaimed-message terminal-stream) (operation &rest rest)
  (sys:stream-default-handler
   scl:self operation (car rest) (cdr rest)))

) ;end of #+symbolics progn

#+ccl-2
(progn
(defclass terminal-stream (ccl::stream)
  ((un-tyi-char :initform nil
                :accessor window-stm-un-tyi-char)
   (input-buffer :initform nil
                 :accessor window-stm-input-buffer)
   (read-focus :initform nil
               :accessor window-stm-read-focus)
   (write-focus :initform nil
                :accessor window-stm-write-focus)
   (location :initform nil
             :accessor window-stm-location)
   (echo :initform nil
         :accessor window-stm-echo)
   (last-output-char :initform #\Newline)))

(defun %make-terminal-stream ()
  (make-instance 'terminal-stream))

(defmethod ccl:stream-tyo ((stream terminal-stream) char)
  (with-slots (last-output-char) stream
    (setq last-output-char char)
    (window-stm-write-char stream char)))

(defmethod ccl:stream-fresh-line ((stream terminal-stream))
  (with-slots (last-output-char) stream
    (unless (eql last-output-char #\Newline)
      (ccl:stream-tyo stream #\Newline))))

(defmethod ccl:stream-tyi ((stream terminal-stream))
  (window-stm-read-char stream))

(defmethod ccl:stream-untyi ((stream terminal-stream) char)
  (window-stm-unread-char stream char))

(defmethod ccl:stream-listen ((stream terminal-stream))
  (window-stm-listen stream))

(defmethod ccl:stream-clear-input ((stream terminal-stream))
  (window-stm-clear-input stream))

(defmethod ccl:stream-close ((stream terminal-stream))
  (window-stm-close stream))

(defmethod ccl:stream-force-output ((stream terminal-stream))
  (window-stm-force-output stream))

(defmethod stream-element-type ((stream terminal-stream))
  'character)

) ;end of #+ccl-2 progn

;;
;;  The following macros allow access to fields of the canvas stream object
;;    in a machine independent way.  The fields are used to store fields such
;;    as the CANVAS associated with the stream and the current X,Y position.
;;    In EXCL the fields are stored in aribitrarily chosen empty fields
;;    of the stream object.  In Symbolics, there are iv's in the flavor
;;    object. 

(defmacro window-stm-p (stream)
  #+excl       `(eq (excl::sm_type ,stream)  :terminal-stream)
  #+(or 
     symbolics ccl-2
     lucid)    `(eq (type-of ,stream) 'terminal-stream))

#-ccl-2
(defmacro window-stm-read-focus (stream)
  #+excl       `(car (excl::sm_fio-fn ,stream))
  #+symbolics  `(scl:send ,stream :read-focus)
  #+lucid      `(terminal-stream-read-focus ,stream))

#-ccl-2
(defsetf window-stm-read-focus (stream) (new-canvas)
  #+excl       `(setf (car (excl::sm_fio-fn ,stream)) ,new-canvas)
  #+symbolics  `(scl:send ,stream :set-read-focus ,new-canvas)
  #+lucid      `(setf (terminal-stream-read-focus ,stream) ,new-canvas))

#-ccl-2
(defmacro window-stm-write-focus (stream)
  #+excl       `(cdr (excl::sm_fio-fn ,stream))
  #+symbolics  `(scl:send ,stream :write-focus)
  #+lucid      `(terminal-stream-write-focus ,stream))

#-ccl-2
(defsetf window-stm-write-focus (stream) (new-canvas)
  #+excl       `(setf (cdr (excl::sm_fio-fn ,stream)) ,new-canvas)
  #+symbolics  `(scl:send ,stream :set-write-focus ,new-canvas)
  #+lucid      `(setf (terminal-stream-write-focus ,stream) ,new-canvas))

#-ccl-2
(defmacro window-stm-input-buffer (stream)
  #+excl       `(excl::sm_fio-buffer ,stream)
  #+symbolics  `(scl:send ,stream :input-buffer)
  #+lucid      `(terminal-stream-input-buffer ,stream))

#-ccl-2
(defsetf window-stm-input-buffer (stream) (new-buffer)
  #+excl       `(setf (excl::sm_fio-buffer ,stream) ,new-buffer)
  #+symbolics  `(scl:send ,stream :set-input-buffer ,new-buffer )
  #+lucid      `(setf (terminal-stream-input-buffer ,stream) ,new-buffer))

#-ccl-2
(defmacro window-stm-location (stream)
  #+excl       `(excl::sm_fio-buffpos ,stream)
  #+symbolics  `(scl:send ,stream :location)
  #+lucid      `(terminal-stream-location ,stream))

#-ccl-2
(defsetf window-stm-location (stream) (new-value)
  #+excl       `(setf (excl::sm_fio-buffpos ,stream) ,new-value)
  #+symbolics  `(scl:send ,stream :set-location ,new-value)
  #+lucid      `(setf (terminal-stream-location ,stream) ,new-value))

#-ccl-2
(defmacro window-stm-char-to-unread (stream)
  #+excl       `(excl::sm_term-svchar ,stream)
  #+symbolics  `(scl:send ,stream :un-tyi-char)
  #+lucid      `(terminal-stream-un-tyi-char ,stream))

#-ccl-2
(defsetf window-stm-char-to-unread (stream) (new-char)
  #+excl       `(setf (excl::sm_term-svchar ,stream) ,new-char)
  #+symbolics  `(scl:send ,stream :set-un-tyi-char ,new-char)
  #+lucid      `(setf (terminal-stream-un-tyi-char ,stream) ,new-char))
  
#-ccl-2
(defmacro window-stm-echo (stream)
  #+excl       `(excl::sm_fio-mode ,stream)
  #+symbolics  `(scl:send ,stream :echo)
  #+lucid      `(terminal-stream-echo ,stream))

#-ccl-2
(defsetf window-stm-echo (stream) (new-char)
  #+excl       `(setf (excl::sm_fio-mode ,stream) ,new-char)
  #+symbolics  `(scl:send ,stream :set-echo ,new-char)
  #+lucid      `(setf (terminal-stream-echo ,stream) ,new-char))

;;
;;  The following macros access information about the canvas associated
;;    with a canvas stream.  Basically these just expand into messages
;;    to the canvas object that is accessed using the WINDOW-STM-CANVAS
;;    macro defined above.
;;

(defmacro stream-to-sheet (stream)
  `(window-stm-write-focus ,stream))

(defmacro stream-to-medium (stream)
  `(sheet-medium (stream-to-sheet ,stream)))

(defmacro stream-region (stream)
  `(sheet-region (stream-to-sheet ,stream)))

(defmacro window-stm-bottom (stream)
  `(bounding-rectangle-min-y (stream-region ,stream)))

(defmacro window-stm-left (stream)
  `(bounding-rectangle-min-x (stream-region ,stream)))

(defmacro window-stm-top (stream)
  `(rectangle-max-y (stream-region ,stream)))

(defmacro window-stm-right (stream)
  `(rectangle-max-x (stream-region ,stream)))

(defmacro window-stm-line-height (stream)
  (let ((medium (gensym)))
    `(let ((,medium (stream-to-medium ,stream)))
       (text-style-height (medium-text-style ,medium) ,medium))))

(defmacro window-stm-baseline (stream)
  (let ((medium (gensym)))
    `(let ((,medium (stream-to-medium ,stream)))
       (text-style-descent (medium-text-style ,medium) ,medium))))

;;;
;;;     Machine independent functions to do all the work!
;;;

(defun window-stm-write-char (stream ch)
  (if (or (char= ch #\newline) 
	  (char= ch #\return))		;for sun talking to a lispm
      (window-stm-newline stream)
      (let ((medium (stream-to-medium stream))
	    (location (window-stm-location stream)))
	(draw-text medium ch location)
	(incf (point-x (window-stm-location stream))
	      (char-width ch (medium-text-style medium) medium)))))

(defun window-stm-write-string (stream string &optional start length)
  (when (or start length) 
    (setq string (subseq string start (if length (+ start length)))))
  (let ((newline-index (or (position #\newline string)
			   (position #\return string))))
    (if newline-index
	(progn
	  (window-stm-write-string stream (subseq string 0 newline-index))
	  (window-stm-newline stream)
	  (window-stm-write-string stream (subseq string (1+ newline-index))))
	(let ((medium (stream-to-medium stream))
	      (location (window-stm-location stream)))
	  (draw-text medium string location)
	  (incf (point-x location)
		(string-width string (medium-text-style medium) medium))))))

(defun window-stm-newline (stream)
  (let* ((line-height (window-stm-line-height stream))
	 (y-pos (- (point-y (window-stm-location stream)) 
		   (* 1.2 line-height)))
	 (medium (stream-to-medium stream)))
    (when (< y-pos (window-stm-bottom stream))
      (setq y-pos (- (window-stm-top stream) line-height)))
    (setf (point-x (window-stm-location stream)) (window-stm-left stream)
	  (point-y (window-stm-location stream)) y-pos)
    (draw-rectangle* medium
		     (window-stm-left stream)
		     (- y-pos (window-stm-baseline stream)
			(* 0.05 line-height))
		     (window-stm-right stream)
		     (* 1.1 line-height)
		     :filled t
		     :ink *white*)))

(defun window-stm-newpage (stream)
  (make-point (window-stm-left stream)
	      (- (window-stm-top stream)
		 (window-stm-line-height stream))
	      :reuse (window-stm-location stream))
  (with-bounding-rectangle* (minx miny maxx maxy)
    (sheet-region (stream-to-sheet stream))
    (draw-rectangle* (stream-to-medium stream)
		     minx miny maxx maxy
		     :filled t
		     :ink *white*)))

(defun window-stm-force-output (stream)
  (medium-force-output (stream-to-medium stream)))

(defun window-stm-close (stream)
  (close-terminal-stream (window-stm-read-focus stream)) ;clean up window
  #+excl(excl::st-close-deon-stream stream)
  )
  
;;;
;;;    input comes from the buffer that was registered with the window server
;;;
  
(defun WINDOW-STM-READ-CHAR (STREAM)
  #-input (declare (ignore stream))
  #+input
  (cond
    ((window-stm-char-to-unread STREAM)
     (prog1 (window-stm-char-to-unread STREAM)
       (setf (window-stm-char-to-unread STREAM) nil)))
    (t (if (ue-lq:buffer-empty-p (window-stm-input-buffer STREAM))
	   (ue:process-wait "Waiting for ipc stream char"
			    #'listen STREAM))
       (ue-lq:buffer-pop (window-stm-input-buffer STREAM)))))

(defun WINDOW-STM-UNREAD-CHAR (STREAM CHAR)
  #-input (declare (ignore stream char))
  #+input
  (setf (window-stm-char-to-unread STREAM) CHAR))

(defun WINDOW-STM-LISTEN (STREAM)
  #-input (declare (ignore stream))
  #+input
  (not (ue-lq:buffer-empty-p (window-stm-input-buffer STREAM))))

(defun WINDOW-STM-CLEAR-INPUT (STREAM)
  #-input (declare (ignore stream))
  #+input
  (ue-lq:buffer-flush (window-stm-input-buffer STREAM)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; external functions for terminal-streams

;;; This function creates a window stream and associates it with a given window
;;;  Also registers the stream's input buffer with the window server.

(defun make-terminal-stream (read-focus write-focus 
					&key (buffer-size 512) (echo t))
  #-input (declare (ignore read-focus buffer-size))
  (let ((Stm (%make-terminal-stream)))
    ;; create the input buffer and set it in
    ;; place
    #+input
    (setf (window-stm-input-buffer Stm)
	  (ue-lq:make-buffer :fifo BUFFER-SIZE 'string-char))
    
    ;; set read-focus to be the associate read-focus for this stream
    (setf (window-stm-write-focus Stm) WRITE-FOCUS
	  #+input (window-stm-read-focus Stm)  
	  #+input READ-FOCUS
	  (window-stm-echo Stm) ECHO)
    
    ;; initialize the current position
    (setf (window-stm-location Stm)
	  (make-point (window-stm-left Stm)
		      (- (window-stm-top Stm)
			 (window-stm-line-height Stm))))
    #+input
    (register-canvas-io-buffer
     (connection read-focus) read-focus (window-stm-input-buffer Stm))
    Stm))

(defun terminal-stream-p (stream)
  (window-stm-p stream))

(defun terminal-stream-read-focus (stream)
  (when (window-stm-p stream)
    (window-stm-read-focus stream)))

(defun terminal-stream-write-focus (stream)
  (when (window-stm-p stream)
    (window-stm-write-focus stream)))

(defun terminal-stream-echoing-p (stream)
  (when (window-stm-p stream)
    (window-stm-echo stream)))

(defsetf terminal-stream-echoing-p (stream) (new-echoing-state)
  `(progn
     (when (window-stm-p stream)
       (error "~S is not a terminal-stream" stream))
     (setf (window-stm-echo ,stream) ,new-echoing-state)))

(defun terminal-stream-location (stream)
  "Return the write-focus window location where the next character
   will be output"
  (when (window-stm-p stream)
    (window-stm-location stream)))

(defsetf terminal-stream-location (stream) (new-location)
  "Set the write-focus window location where the next character
   will be output"
  `(progn
     (unless (window-stm-p ,stream)
       (error "~S is not a terminal-stream" ,stream))
     (setf (window-stm-location ,stream) ,new-location)))

(defun terminal-stream-clear (stream)
  (window-stm-newpage stream))
    
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; class support stuff

(defclass terminal-stream-mixin ()
    ((stream :initform nil :accessor terminal-stream))
  (:documentation
   "sheet-stream-mixin class allows a window to have a terminal io
    stream associated with it.
    "))

(defmethod open-terminal-stream
	   ((self terminal-stream-mixin) 
	    &key (read-focus self) (write-focus self) 
	    (echo t) (buffer-size 512)
	    &allow-other-keys)
  (when (terminal-stream read-focus)
    (error "window ~s is already the read focus for ~s"
	   read-focus (terminal-stream read-focus)))
  (prog1
      (setf (terminal-stream self)
	    (make-terminal-stream
	     read-focus write-focus :echo echo :buffer-size buffer-size))
    #+input
    (set-event-interests read-focus :input-keyboard)))

(defmethod close-terminal-stream ((self terminal-stream-mixin))
  #+input
  (setf (event-interest self :input-keyboard) nil)
  #+input
  (unregister-canvas-io-buffer (connection self) self)
  (setf (terminal-stream self) nil))

