;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;;
;;; Windowing and Sheet Parts
;;;

(defclass windowing-part () ())

(defclass parent-part (windowing-part) 
    ())
(defmethod parent-part-p ((part parent-part)) t)
(defmethod parent-part-p (object) (declare (ignore object)))

(defclass child-part (windowing-part)  
    ())

(defmethod child-part-p ((part child-part)) t)
(defmethod child-part-p (object) (declare (ignore object)))

;;;
;;; Default Sheet Functionality
;;;

(defmethod fetch-mirrored-sheet ((sheet child-part))
  (if (sheet-mirror sheet) sheet
      (fetch-mirrored-sheet (sheet-parent sheet))))

(defmethod sheet-siblings ((sheet child-part))
  (remove sheet (sheet-children (sheet-parent sheet))))

(defmethod sheet-enabled-children ((sheet parent-part))
  (with-collection
      (dolist (child (sheet-children sheet))
	(when (sheet-enabled-p sheet)
	  (collect child)))))

(defmethod sheet-viewable-p ((sheet child-part))
  (and (sheet-enabled-p sheet)
       (let ((parent (sheet-parent sheet)))
	 (when parent
	   (sheet-viewable-p parent)))))

(defmethod sheet-ancestor-p ((sheet child-part) ancestor)
  (let ((parent (sheet-parent sheet)))
    (when parent
      (or (eq parent ancestor)
	  (sheet-ancestor-p parent ancestor)))))

;;;
;;; Default Hit Detection Code
;;;

(defmethod child-at-point* ((sheet parent-part) x y &optional filter-fn)
  ;; Can assume that this is only called when x y are within parent.
  (let* ((children (sheet-enabled-children sheet)))
    (dolist (child children)
      (when (and (or (null filter-fn) (funcall filter-fn child))
		 (region-contains-point*-p
		  (fetch-clipping-region child sheet) x y))
	(return child)))))

(defmethod children-in-region ((sheet parent-part) region)
  ;; The Dumb Default method
  ;; Used to determine which children need repainting so it dumbly says all of
  ;; them.
  (declare (ignore region))
  (sheet-children sheet))

(defmethod descendant-at-point* ((sheet parent-part) x y &optional filter-fn)
  (let ((child (child-at-point* sheet x y filter-fn)))
    (when child
      (multiple-value-setq (x y)
	(untransform-point* (sheet-transformation child) x y))
      (or (descendant-at-point* child x y filter-fn)
	  child))))

(defmethod sheet-inside? ((sheet child-part) x y)
  (region-contains-point*-p (sheet-region sheet) x y))

;;;
;;; Tree Modification Protocol
;;;

(defmethod (setf sheet-parent) (new-parent (sheet sheet))
  (let ((old-parent (sheet-parent sheet)))
    (when old-parent (disown-child old-parent sheet)))
  (adopt-child new-parent sheet))

(defmethod (setf port) (port (sheet sheet) &key graft &allow-other-keys)
  (setf (slot-value sheet 'port)  port
	(slot-value sheet 'graft) graft)
  (sheet-grafted sheet)
  (dolist (child (sheet-children sheet))
    (setf (port child :graft graft) port)))

(defmethod (setf port) ((port null) (sheet sheet) &key &allow-other-keys)
  (sheet-ungrafted sheet)
  (setf (slot-value sheet 'port) nil
	(slot-value sheet 'graft) nil)  
  (dolist (child (sheet-children sheet))
    (setf (port child) port)))

(defmethod adopt-child :after ((parent parent-part) (child child-part)
			       &key &allow-other-keys)
  (let ((port (port parent)))
    (when port
      (setf (port child :graft (graft parent)) port)))
  (sheet-adopted child))

(defmethod disown-child :after ((parent parent-part) (child child-part)
				&key &allow-other-keys)
  (sheet-disowned child)
  (setf (port child) nil))

;;;
;;; Enabling Protocol
;;;
;;; Enabling Mixins --- not all windowing contract need "enabling"
;;; These mixin's get pulled into a child part that wants them.
;;;

(defclass enabled-slot-mixin ()
    ((enabled-p :initform nil :reader sheet-enabled-p)))

(defmethod enable-sheet ((sheet enabled-slot-mixin) 
			 &rest keys &key &allow-other-keys)
  (declare (dynamic-extent keys))
  (with-slots (enabled-p) sheet
    (unless enabled-p
      (setf enabled-p t)
      (apply #'sheet-enabled sheet keys))))

(defmethod disable-sheet ((sheet enabled-slot-mixin)
			  &rest keys &key &allow-other-keys)
  (declare (dynamic-extent keys))
  (with-slots (enabled-p) sheet
    (when enabled-p
      (setf enabled-p nil)
      (apply #'sheet-disabled sheet keys))))

(defclass permanently-enabled-mixin ()
    (
     #+new-pcl-class-bug-fixed
     ;; A normal method may be better than using this reader anyway.
     (enabled :allocation :class
	      :initform t :reader sheet-enabled-p)))

(defmethod sheet-adopted :after ((sheet permanently-enabled-mixin)
				 &key &allow-other-keys)
  (sheet-enabled sheet))

(defmethod sheet-disowned :before ((sheet permanently-enabled-mixin)
				   &key &allow-other-keys)
  (sheet-disabled sheet))

(defmethod sheet-enabled-p ((sheet permanently-enabled-mixin)) t)

(defmethod enable-sheet ((sheet permanently-enabled-mixin) &rest keys)
  (declare (dynamic-extent keys)
	   (ignore keys))
  (warn "This sheet's youth contract doesn't support enabling"))

(defmethod disable-sheet ((sheet permanently-enabled-mixin) &rest keys)
  (declare (dynamic-extent keys)
	   (ignore keys))
  (warn "This sheet's youth contract doesn't support enabling"))

;;;
;;; Change Notification Protocol 
;;; Default Methods 
;;;

(defmethod sheet-adopted (sheet &key &allow-other-keys)
  (declare (ignore sheet)))

(defmethod sheet-disowned (sheet &key &allow-other-keys) 
  (declare (ignore sheet)))

(defmethod sheet-grafted (sheet) (declare (ignore sheet)))
(defmethod sheet-ungrafted (sheet) (declare (ignore sheet)))

#+PCL
(pcl::do-standard-defsetf port-stamp) ; ???

(defmethod sheet-enabled ((sheet child-part) &key &allow-other-keys)
  (when (port sheet) (incf (port-stamp (port sheet)))))

(defmethod sheet-disabled ((sheet child-part) &key &allow-other-keys)
  (when (port sheet) (incf (port-stamp (port sheet)))))

(defmethod sheet-region-changed ((sheet child-part) &key &allow-other-keys)
  (when (port sheet) (incf (port-stamp (port sheet)))))

(defmethod sheet-transformation-changed ((sheet child-part) 
					 &key &allow-other-keys)
  (when (port sheet) (incf (port-stamp (port sheet)))))

;;;
;;; Mirroring Protocol
;;;
;;; The default is that sheets are unmirrored.  If they take the
;;; mirrored-sheet-mixin than they can be mirrored and this part will be pushed
;;; back in the class precedence list since mirrored-sheet-mixin inherits it.
;;;

(defclass mirror-protocol-part () ())
(defmethod sheet-mirror ((sheet mirror-protocol-part)) nil)
(defmethod sheet-native-transformation ((sheet mirror-protocol-part)) nil)

(defmethod sheet-mirror! ((sheet mirror-protocol-part))
  (or (sheet-mirror sheet)
      (sheet-mirror! (sheet-parent sheet))))

;;; 
;;; Contract and Part Mixins to handle xf & cr caches
;;;

;;; DTCR -> delta-transformation-clipping-region
(defclass dtcr-mixin (mirror-protocol-part) 
    (
     ;; Delta transformation caching and calculation.
     (%dtstamp 
      ;; Indicates last time when something has changed that would
      ;; effect a delta transformation calculation.  Namely last time
      ;; local transformation has changed
      :initform 0 :accessor %dtstamp)
     (%delta-transformation-cache :initform nil :accessor 
				  %delta-transformation-cache)
        
     ;; Clipping Region caching and calculation
     (%crstamp 
      ;; Last time anything has changed that would affect clipping
      ;; region of silica sheet.  Namely lots of events in the
      ;; sheet's youth realm.
      :initform 0 :accessor %crstamp)
     (%clipping-region-cache :initform nil :accessor %clipping-region-cache)))

;;; 
;;; Kernel Events
;;;
    
(defmethod sheet-enabled :after ((child dtcr-mixin) &key &allow-other-keys)
  (update-tables child))

(defmethod sheet-disabled :after ((child dtcr-mixin) &key &allow-other-keys)
  (update-tables child))

(defmethod sheet-region-changed :after ((child dtcr-mixin) 
					&key &allow-other-keys)
  (update-tables child))

(defmethod sheet-transformation-changed :after ((child dtcr-mixin)
						&key &allow-other-keys)
  (when (port child)
    (setf (%dtstamp child) (incf (port-dtstamp (port child))))
    (update-tables child)))

(defmethod update-tables ((sheet dtcr-mixin))
  ;;; ???  This whole scheme nees revamping to take advantage contract specific
  ;;; knowledge.  E.g. more than one child, overlapping children, etc.
  (when (port sheet)
    ;; ??? assuming children-intersect?
    (let ((global-stamp (incf (port-crstamp (port sheet))))
	  (parent (sheet-parent sheet)))
      (when parent
	(dolist (child (sheet-children parent))
	  (setf (%crstamp child) global-stamp))))))

;;;
;;; Silica Sheet Based COMPOSITION CACHES
;;;
;;; Mechanisms for calculating and caching of combined transformations and
;;; clipping regions
;;;
;;; Transformation Calculation and Caching Scheme:
;;;
;;;   All attempts to get a delta-transformation are made through a call to
;;; fetch-delta-transformation.
;;;   A global index is bumped whenever any local transformation is changed. 
;;; The value of the global index is associated with local transformation
;;; whenever they are set.  (So no two values associated with a
;;; local-transformations are the same and all values are ordered) 
;;;
;;;   When a delta-transformation is calculated it is cached in the child along
;;; with the value of the global index.  The composition cache looks like:
;;;  (ancestor (entry-stamp . cached-value) ancestor2 ....)
;;;
;;;   Cached delta-transformations are first
;;; checked against the global index, and if this fails against the local
;;; indexes of all ancestors (i.e. the dependencies of the transformation).

;;; Optimization:
;;; This optimization makes checking the dependencies of a cached
;;; transformation faster.  Improves cases which miss on the global index, but
;;; don't really require any recalculation.
;;; At each node in the tree you keep a list that looks like this:
;;;    ((this-stamp . this-node) . <list from parent>)
;;; When the transformation at a node is changed, it sets this-stamp.
;;; Since these lists all share structure, all the children's cache-lists see
;;; the change.  When you need to verify a cached computed transformation from
;;; a node to some ancestor, pull the list out of the node and walk the list
;;; until you get to that ancestor.  No need to touch the ancestor objects at
;;; all.
;;; This optimization would require some maintenance by the kernel protocols.

(defmacro caching-calculate
    (node key cache-accessor global-stamp verify-fn calculation-form)
  ;; Returns the calculated value, and whether it was available in the cache.
  (once-only (global-stamp)
    (let ((entry (gentemp))
	  (calculated-value (gentemp)))
      `(let* ((,entry (getf (,cache-accessor ,node) ,key))
	      ,calculated-value)
	(if (and ,entry 
		 ;; IF EITHER nothing has changed since last validation
		 ;; OR nothing
		 ;; that the entry depends on has changed since the entry was
		 ;; caculated.
		 (or (= (car ,entry) ,global-stamp)
		     (when (funcall ,verify-fn (car ,entry))
		       ;; Cuts off need to reconfirm this entry.
		       (setf (car ,entry) ,global-stamp))))
	    ;; THEN Return the cached value
	    (cdr ,entry)
	    ;; ELSE Calculate and cache a value.
	    (progn (setf (getf (,cache-accessor ,node) ,key)
			 (cons ,global-stamp 
			       (setq ,calculated-value ,calculation-form)))
		   ,calculated-value))))))

(defmacro check-dependencies (entry-stamp node terminal-node stamp-accessor 
					  &optional 
					  (next-node-fn 'sheet-parent))
  ;; Assuming that ancestor is really an ancestor of node.  
  (once-only (entry-stamp)
    (let ((current (gensym)))
      `(do ((,current ,node))
	   (;; FAIL if current's stamp is newer than the entry stamp
	    (> (,stamp-accessor ,current) ,entry-stamp))
	 (if (eq (setq ,current (,next-node-fn ,current)) ,terminal-node)
	     ;; Reaching the terminal node means that we have succeeded. 
	     ;; Notice that null ancestor will terminate properly too.  
	     (return t))))))

;;;
;;; DELTA TRANSFORMATIONS
;;;

(defmethod fetch-delta-transformation ((sheet dtcr-mixin) ancestor)
  (with-accessors ((parent sheet-parent)) sheet
    (cond ((eq parent ancestor) (sheet-transformation sheet))
	  ((null parent) 
	   ;; previous clause filtered both ancestor and parent being null
	   (error "~s should be an ancestor of ~s" ancestor sheet))
	  (t 
	   (caching-calculate 
	    sheet ancestor
	    %delta-transformation-cache
	    (port-dtstamp (port sheet))
	    #'(lambda (entry-stamp)
		(check-dependencies entry-stamp sheet ancestor %dtstamp))
	   
	    ;; Notice that this scheme calculates and caches intermediate delta
	    ;; transformations to the ancestor.  Alternatively, we could cache
	    ;; from this node to intermediate nodes.
	    (compose-transformations 
	      ;; --- swapped forms for CLIM 1 arg order (as below) but a month later!
	     (fetch-delta-transformation parent ancestor)
	     (sheet-transformation sheet)))))))

(defmethod fetch-native-transformation ((sheet dtcr-mixin))
  (caching-calculate 
    sheet :native
    %delta-transformation-cache
    (port-dtstamp (port sheet))
    #'(lambda (entry-stamp)
	(do ((current sheet (sheet-parent current)))
	    ;; some sheet must have a mirror
	    ((sheet-mirror current) 
	     (<= (%ntstamp current) entry-stamp))
	  ;; if some local transformation has changed after entry was calculated
	  ;; then fail
	  (if (> (%dtstamp current) entry-stamp) (return nil))))
    (compose-transformations 
      ;; --- swapped forms for CLIM 1 arg order, 10/9/91
      (fetch-native-transformation (sheet-parent sheet))
      (sheet-transformation sheet))))

;;;
;;; CLIPPING REGIONS  
;;;

(defmethod fetch-clipping-region ((sheet dtcr-mixin) ancestor)
  (with-accessors ((parent sheet-parent)) sheet
    (cond ((null parent)
	   (if (null ancestor)
	       (careful-transform-region
		(sheet-transformation sheet) 
		(sheet-region sheet))
	       (error "~s should be an ancestor of ~s" ancestor sheet)))
	  ((eq parent ancestor)
	   (caching-calculate 
	    sheet ancestor
	    %clipping-region-cache
	    (port-crstamp (port sheet))
	    #'(lambda (entry-stamp)
		(>= entry-stamp (%crstamp sheet)))
	    ;; Calculate it
	    (allocated-region parent sheet)))
	  (t 
	   (caching-calculate 
	    sheet ancestor
	    %clipping-region-cache
	    (port-crstamp (port sheet))
	    #'(lambda (entry-stamp)
		(check-dependencies entry-stamp sheet ancestor %crstamp))
	    ;; Calculate it
	    (region-intersection 
	     (careful-transform-region 
	      (fetch-delta-transformation parent ancestor)
	      (allocated-region parent sheet))
	     (fetch-clipping-region sheet parent)))))))

(defmethod fetch-native-clipping-region ((sheet dtcr-mixin))
  (with-accessors ((parent sheet-parent)) sheet
    (caching-calculate 
     sheet :native
     %clipping-region-cache
     (port-crstamp (port sheet))
     #'(lambda (entry-stamp)
	 (do ((current sheet (sheet-parent current)))
	     ;; some sheet must have a mirror
	     ((sheet-mirror current) 
	      (<= (%ncrstamp current) entry-stamp))
	   ;; if some local transformation has changed after entry was
	   ;; calculated then fail
	   (if (> (%crstamp current) entry-stamp) (return nil))))
     ;; Calculate it
     (region-intersection 
      (careful-transform-region 
       (fetch-native-transformation parent)
       (allocated-region parent sheet))
      (fetch-native-clipping-region parent)))))

(defmethod allocated-region ((parent parent-part) 
			     (child dtcr-mixin))
  (reduce #'region-difference (mapcar
			       #'(lambda (occluding-sheet)
				   (careful-transform-region
				    (sheet-transformation occluding-sheet)
				    (sheet-region occluding-sheet)))
			       (occluding-sheets parent child))
	  :initial-value
	  (careful-transform-region (sheet-transformation child)
				    (sheet-region child))))

(defmethod occluding-sheets ((parent parent-part) sheet)
  (let ((children (sheet-children parent)))
    (subseq children 0 (position sheet children))))


;;;
;;; Windowing Contract Library
;;;

;;;
;;; Macrology
;;;

(defmacro define-windowing-contract (class supers slots &rest options)
  ;; Process the extra options, resetting to the defclass options
  (let ((parent-part nil))
    (setq options
	  (with-collection
	      (dolist (opt options)
		(case (first opt)
		  (:contract-name
		   (push `(contract-name 
			    :allocation :class :initform ,(second opt)
			    :reader contract-name)
			 slots))
		  (:parent-part
		   (setq parent-part (second opt))
		   (push `(parent-part :allocation :class 
				       :initform ',parent-part
				       :reader contract-parent-part)
			 slots))
		  (:child-part
		   (push `(child-part :allocation :class
				      :initform ',(second opt)
				      :reader contract-child-part)
			 slots))
		  (otherwise (collect opt))))))
    `(define-group ,class define-windowing-contract
       ,@(if parent-part
	     `((defmethod sheet-adult-contract-class ((sheet ,parent-part)) 
		 (find-class ',class)))
	     (error 
	       "Parent-part option must be provided to def-windowing-part"))
       (eval-when (compile load eval)		;Sigh.
	 (defclass ,class ,supers ,slots ,@options)))))

;;;
;;; Mute parent part --- for childless sheets
;;;

(defclass mute-parent-part (parent-part) ())

(defmethod sheet-children ((sheet mute-parent-part)) 
  nil)

;;;
;;; Simple Windowing
;;;

(defclass simple-parent-part (parent-part) 
    ((children :initform nil :accessor sheet-children)))

(defclass simple-child-part (child-part)
    ((parent :initform nil :reader sheet-parent)))

(defmethod adopt-child ((parent simple-parent-part) 
			(child simple-child-part)
			&key &allow-other-keys)
  
  (when (slot-value child 'parent)
    (error  "Trying to adopt a sheet which already has a parent"))
	  
  (push child (slot-value parent 'children))
  (setf (slot-value child 'parent) parent)
  child)

(defmethod disown-child ((parent simple-parent-part) 
			 (child simple-child-part)
			 &key &allow-other-keys)
  
  (unless (eq parent (slot-value child 'parent))
    (error "Trying to disown ~s from ~s which is not its parent" child parent))
  
  (setf (slot-value parent 'children) 
	(delete child (slot-value parent 'children)))
  (setf (slot-value child 'parent) nil)
  child)

(defmethod sheet-child ((sheet simple-parent-part))
  (car (slot-value sheet 'children)))


;;;
;;; Standard Windowing
;;; 
;;;   Standard Windowing is a fairly general purpose windowing contract.
;;; It supports full overlapping and take care of general purpose hit
;;; hit detection, repaints, and region calculations.
;;;

(defclass standard-parent-part (simple-parent-part) ())
(defclass standard-child-part (mirrored-sheet-mixin
			       simple-child-part
			       dtcr-mixin
			       sheet-rectangle-translation-mixin
			       enabled-slot-mixin) 
    ())

(define-windowing-contract standard-windowing-contract ()
    ()
  (:contract-name "standard")
  (:parent-part standard-parent-part)
  (:child-part  standard-child-part))

(defmethod sheet-adopted :after ((sheet standard-child-part)
				 &key &allow-other-keys)
  ;; Unallocating mirrors is automatically 
  ;; taken care of by mirrored-sheet-mixin
  (when (graftp (sheet-parent sheet))
    (let ((port (port sheet)))
      (realize-mirror port sheet))))

#|| ??? Standard should provide typical X type functionality like this old code.

(defmethod move-sheet* :around ((self standard-child-part) x y)
  (declare (ignore x y))
  ;; Unpaint
  (repaint-sheet (sheet-parent self)
		 (transformation-apply transformation (sheet-region self)))
  ;; Change
  (call-next-method)
  ;; Paint
  ;; (repaint-sheet self +everywhere+)
  (repaint-sheet (sheet-parent self) +everywhere+))


(defmethod resize-sheet* :around ((self standard-child-part) width height)
  (declare (ignore width height))
  (call-next-method)
  (repaint-sheet (sheet-parent self) +everywhere+))

(defmethod move-and-resize-sheet* :around
	   ((self standard-child-part) left bottom width height)
  (declare (ignore left bottom width height))
  ;; Paint off

  (repaint-sheet (sheet-parent self)
		 (transformation-apply transformation region))
  ;; Change up
  (call-next-method)
  ;; Paint on
  #+wait
  (repaint-sheet (sheet-parent self) new-region-in-parent)
  #+wait
  (repaint-sheet (sheet-parent self) +everywhere+))

(defmethod raise-sheet ((sheet standard-child-part))
  (stack-sheet sheet :to-top))

(defmethod bury-sheet ((sheet standard-child-part))
  (stack-sheet sheet :to-bottom))

(defmethod stack-sheet ((self standard-child-part) stack-request)
  (let ((contract (sheet-youth-contract self)))
    (update-tables self)
    (case stack-request
      (:to-top (setf (windowing-children contract)
		     (cons self (remove self (windowing-children contract)
					:test #'eq)))
	       (repaint-sheet self (sheet-region self)))
      (:to-bottom (setf (windowing-children contract)
			(nconc (remove self (windowing-children contract)
				       :test #'eq)
			       (list self)))
		  (repaint-sheet 
		    (sheet-parent self)
		    (transformation-apply 
		      (sheet-transformation self)
		      (sheet-region self)))))))


||#



