;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;;
;;;  A library of distributors.  Currently I just added them as I go, but will
;;; eventually get to organizing them a little bit and maybe abstracting out
;;; distributor pieces.  

(defmacro with-distributor ((port distributor) &body body)
  (let ((name (gensymbol 'with-distributor)))
    `(flet ((,name ()
	     ,@body))
       (with-distributor-internal ,port ,distributor
				  #',name))))

(defmethod with-distributor-internal ((port port) distributor continuation)
  (with-distributor-locked (port)
    (letf-globally (((slot-value port 'distributor) distributor))
       (funcall continuation))))

(defmacro with-pointer-grab ((port) &body body)
  (once-only (port)
    `(do-with-pointer ,port #'(lambda () ,@body))))

;;; mix in with some input-contract, such as STANDARD-INPUT-CONTRACT or
;;; INVOKING-INPUT-CONTRACT 


;;;
;;; Self Distribution
;;; 
;;; This is a distributor mixin that distributes events to itself rather than
;;; to the "recipient sheet".  Useful when you want to snarf all events.
;;; This is the probably the case when you are doing tight mouse tracking in a
;;; special mode.   

(defclass self-distributor-base-mixin (standard-event-distributor 
				       input-handler)
    ;; Take crossing events.
    ((dispatch-crossings-p :initform t)))

(defmethod initialize-instance :after ((distributor self-distributor-base-mixin)
				       &key &allow-other-keys)
  (setf (slot-value distributor 'dispatch-focus) distributor))


;;; I don't think I need a RESET-DISTRIBUTOR method. 
;;; --- I don't understand this --- RR


;;; Why do I have to write this?  Shouldn't the defaults arrange for the event
;;; to be freed after it's been processed.
;;; ??? I'm confused by why the following method is here --- RR
(defmethod queue-event ((distributor self-distributor-base-mixin) event 
			&key &allow-other-keys)
  (queue-put (slot-value distributor 'event-queue)
	     (copy-instance event))
  event)


;;;
;;; Basic Tracker
;;;
;;;

(defclass tracking-distributor (self-distributor-base-mixin
				invoking-input-contract)
    ())  

;;;
;;; Sheet Selection Distributor 
;;;     for selecting one or more sheets (what else?)
;;;

(defclass sheet-selection-distributor (tracking-distributor)
    ((sheets :initform nil)))
  
(defmethod pointer-enter ((handler sheet-selection-distributor) 
			  &key sheet &allow-other-keys)
  (using-display-medium (dm :clim sheet)
    (with-bounding-rectangle* (left top right bottom) sheet
      (declare (ignore left top))
      (draw-rectangle* dm 0 0 right bottom ; (rectangle-max-x sheet) (rectangle-max-y sheet)
		       :ink +flipping-ink+))))

(defmethod pointer-exit ((handler sheet-selection-distributor) 
			 &key sheet &allow-other-keys)
  (using-display-medium (dm :clim sheet)
    (with-bounding-rectangle* (left top right bottom) sheet
      (declare (ignore left top))
      (draw-rectangle* dm 0 0 right bottom ; (rectangle-max-x sheet) (rectangle-max-y sheet)
		       :ink +flipping-ink+))))

(defmethod button-release ((handler sheet-selection-distributor) button-name
			   &key sheet &allow-other-keys)
  (declare (ignore button-name))
  (push sheet (slot-value handler 'sheets))
  (using-display-medium (dm :clim sheet)
    (with-bounding-rectangle* (left top right bottom) sheet
      (declare (ignore left top))
      (draw-rectangle* dm 1 1
		       (1- right #+Ignore (rectangle-max-x sheet))
		       (1- bottom #+Ignore (rectangle-max-y sheet))
		       :ink +flipping-ink+)))
  (setf (distributor-enabled handler) nil))

(defmethod button-click-click ((handler sheet-selection-distributor) button-name
			       &key sheet &allow-other-keys)
  (declare (ignore button-name))
  (push sheet (slot-value handler 'sheets)))

;;;
;;; Entry Function to Selecting Sheets
;;;

(defun which-sheet (&optional (port (find-port)))
  (let ((distributor (make-instance 'sheet-selection-distributor)))
    ;; Install the new "all sheets" distributor...
    (with-distributor (port distributor)
      ;; ...and wait for it to finish
      (process-wait "Selecting Sheets"
		    #'(lambda ()
			(not (distributor-enabled distributor))))
      (with-slots (sheets) distributor
	(when sheets
	  (if (= (length sheets) 1) (car sheets)
	      sheets))))))


;;; Rule 1: Distributor can NOT determine or rely on input contract policy.
;;; That means that all a distributor can do is eventually invoke dispatch 
;;; on some input handler.  Eventually fix the self-distributor-mixin above
;;; or rewrite tracking-pointer-distributor in the new way.
;;; ??? The above comment is defunct, I think, but I don't exactly understand
;;; it.  There is a new new way which I believe outmodes it.

(defclass menu-tracking-distributor (self-distributor-base-mixin
				     invoking-input-contract
				     )
    (
     ;; the menu sheet is the one root sheet on which events are processed.
     ;; We may want to support using multiple top menu sheets at once.
     (menu-sheet :initarg :menu-sheet)

     ;; one of :button-release or :button-press
     (exit-mode :initarg :exit-mode)

     ;; did the mouse move into the desired sheet yet?
     (received-enter :initform nil)

     ;; I got rid of while-checker, which I don't think will want to come back.
     ;; I also got rid of do-final-exit-p, preferring to unconditionally do the
     ;; final exit.  This fine-grained control may want to come back, though.
     )
  )

;;; No one will reset us, but if they do, make sure this slot gets reset.
(defmethod reset-distributor :after ((distributor menu-tracking-distributor))
  (setf (slot-value distributor 'received-enter) nil))

(define-constructor make-menu-tracking-distributor menu-tracking-distributor
  (&key menu-sheet enabled exit-mode)
  :menu-sheet menu-sheet
  :enabled enabled
  :exit-mode exit-mode
  )

;;; The idea is that we want to wait for a menu item to be selected.  
;;; The old code waited for the button to go down, then waited for it
;;; to go back up again.
;;;
;;; We'll put that information into state variables of the distributor, 
;;; and instead we'll wait until the distributor is no longer enabled.
;;;
;;;

;;; We ignore all events that don't happen on a child of the menu sheet, except
;;; for press/release events which cause us to exit.
(defmethod dispatch-event-using-port-keys
	   ((distributor menu-tracking-distributor) port &rest keys
	    &key sheet event-key direction &allow-other-keys)
  (declare (dynamic-extent keys))
  (with-slots (menu-sheet exit-mode)
      distributor
    (cond ((sheet-ancestor-p sheet menu-sheet) 
	   
	   ;; IF we're in there (so to speak)
	   ;; Mark the entry into menu sheet and dispatch normally
	   (when (and (eql event-key :pointer-enter)
		      (eql direction :down))
	     (setf (slot-value distributor 'received-enter) t))
	   (apply #'dispatch-event-using-port-keys sheet port keys))
	  
	  ;; ELSE check for exit modes
	  ;; The two clauses below can be combined into an or, right? --- RR
	  
	  ;; if you click/release off the menu, you're done!
	  ((eql event-key exit-mode)
	   (setf (distributor-enabled distributor) nil)
	   ;; --- This used to call pointer-exit and reset the trace.
	   ;; --- Is it bad that we don't??
	   )
	  ((slot-value distributor 'received-enter)
	   ;; Once we've entered, if we get any event off this 
	   ;; menu, we want to make it go away...
	   (setf (distributor-enabled distributor) nil)))))

(defun track-menu (menu sheet enable-fn &optional (exit-mode :button-release))
  (let ((port (port sheet))
	enabled)
    (with-pointer-grab (port)
      (let ((distributor (make-menu-tracking-distributor
			   :menu-sheet sheet
			   :exit-mode exit-mode
			   ;; enabled explicitly below
			   :enabled nil)))
	;; Install the menu distributor
	(with-distributor-locked (port)
	  ;; keep the lock around the WITH-DISTRIBUTOR and following form
	  ;; because we want to reset the existing distributor if it's a 
	  ;; menu tracker itself
	  (with-distributor (port distributor)
	    ;; Enable the distributor
	    (reset-distributor distributor)
	    (unless enabled (funcall enable-fn))
	    (port-event-wait port
			     #'(lambda () 
				 (or (menu-choice menu)
				     (not (distributor-enabled distributor))))
			     :wait-reason "Menu select"))
	  ;; --- If the previous distributor is a menu-tracking-distributor,
	  ;; --- tell it that we're awaiting enter again.
	  ;; --- Is there a more modular way of doing this?
	  (let ((old (slot-value port 'distributor)))
	    (when (typep old 'menu-tracking-distributor)
	      (setf (slot-value old 'received-enter) nil))))
	t))))


#||
Obsolete Code  
If needed, I'll have to port since input distribution cleanup --- RR

;;;
;;; Tracking for child of goods
;;;

(defclass child-finding-distributor (tracking-distributor)
    ((while-checker :initarg :while-checker :accessor while-checker)
     (goods :initarg :goods)
     (current :initform nil) 
     ;; ??? see below
     (do-final-exit-p :initform nil)))

(define-constructor make-child-finding-distributor child-finding-distributor
    (&key while-checker goods)
  :while-checker while-checker :goods goods)

(defmethod queue-input ((distributor child-finding-distributor) 
			&rest keys
			&key sheet state type button &allow-other-keys)
  (declare (ignore keys) (dynamic-extent keys))
  (with-slots (while-checker goods trace current) 
      distributor
    (labels ((twiddle (sheet)
	       (using-display-medium (medium :clim sheet)
		 (draw-rectangle* medium 0 0
				  (rectangle-max-x sheet)
				  (rectangle-max-y sheet)
				  :ink +flipping-ink+)))
	     (child-of-good (sheet)
	       (if (member (sheet-parent sheet) goods) sheet
		   (child-of-good (sheet-parent sheet)))))

      (if (and sheet 
	       (some #'(lambda (good) 
			 (sheet-ancestor-p sheet good))
		     goods))
	  (progn
	    (when current (twiddle current))
	    (setf current (child-of-good sheet)))
	  (when current 
	    (twiddle current)
	    (setf current nil)))
    
      (let ((after (state-after state type button)))
	(when after
	  (unless (funcall while-checker after)
	    ;; Time to exit.
	    (when current
	      (twiddle current))
	    (setf (distributor-enabled distributor) nil)
	    (setf (fill-pointer trace) 0)))))))
	
(defun prompt-for-child-of (sheet)
  ;;; Not for calling in event handler process.
  (let ((port (port sheet)))
    (with-pointer-grab (port)
      (multiple-value-bind (x y state) (poll-pointer sheet)
	(declare (ignore x y))
	(let ((distributor 
	       (make-child-finding-distributor
		:while-checker
		#'(lambda (state)
		    (state-match-p state (not (or :left :middle :right))))
		:goods (list sheet))))
	  ;; This distributor is enabled by default, and already has a while-checker
	  (with-distributor (port distributor)
	    (unless (state-match-p state (or :left :middle :right))
	      (process-wait "Tracking Mouse" 
			    #'(lambda () 
				(not (distributor-enabled distributor)))))
	    (setf (while-checker distributor)
		  #'(lambda (state) (state-match-p state
						   (or :left :middle :right)))
		  (slot-value distributor 'do-final-exit-p) t)
	    ;; Re-enable the distributor with the new while-checker
	    (reset-distributor distributor)
	    (process-wait "Tracking Mouse" 
			  #'(lambda () (not (distributor-enabled distributor)))))
	  ;; Return the last CURRENT sheet, if any.
	  (slot-value distributor 'current))))))


||#



