;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;;
;;; SILICA SHEET
;;;

#+ignore
(eval-when (load)
  (lisp:format *terminal-io* "~%~@{~A~%~}" "The SILICA Window System" "

sil|i|ca n. A white or colorless crystalline compound, SiO2, occurring 
abundantly as quartz, sand, flint, agate, and many other minerals, and used
to manufacture a wide variety of materials, notably glass and concrete. 
Also called \"silicon dioxide\" and, formerly, \"silex.\" 
[New Latin, from Latin silex (stem silic-), flint, silex.] 
--  (From the American Heritage Dictionary)"))

(eval-when (load)
  (lisp:format *terminal-io* 
	       "~2%;;;>>> ~A Copyright (c)~{ ~D,~} by ~A. ~%~
		   ;;;>>> All rights reserved.~2%" 
	       "SILICA" '(1988 --- 1990)
	       "Xerox Corporation")
  (lisp:force-output *terminal-io*))

(defclass sheet ()
    ((port :initform nil :reader port)
     (graft :initform nil :reader graft))
  (:documentation "An abstract surface for doing input and output"))

(defmethod sheetp (object) (declare (ignore object)) nil)
(defmethod sheetp ((object sheet)) object)

(defmethod initialize-instance :after
	   ((sheet sheet) 
	    &key region transformation parent 
	    &allow-other-keys)
  (when region 
    (assert (regionp region) (region)
	    "Not passed a valid region"))
  (when transformation
    (assert (transformationp transformation) (transformation)
	    "Not passed a valid transformation"))

  ;; Enact the windowing contracts.
  (when parent
    (assert (sheetp parent) (parent)
	    "Not passed a valid sheet")
    (adopt-child parent sheet)))

#+genera-release-8
;;; Genera 8 CLOS doesn't initialize the prototype slot until the first instance
;;; is made, so try to avoid the debugger by faking it.
;;; --- not needed in Genera 8.1
(defmethod slot-unbound ((class standard-class) instance-class (slot (eql 'clos-internals::prototype)))
  ;; hope that all these classes can be made with no init args.
  (make-instance instance-class))

(defmacro define-sheet-class (class supers slots &rest options)
  (let (output-contract-class 
	input-contract-class 
	youth-contract-class
	adult-contract-class)
    
    (setq options
	  (with-collection
	      (dolist (opt options)
		(case (first opt)
		  (:output-contract-class
		   (setq output-contract-class (second opt)))
		  (:input-contract-class
		   (setq input-contract-class (second opt)))
		  (:youth-contract-class
		   (setq youth-contract-class (second opt)))
		  (:adult-contract-class
		   (setq adult-contract-class (second opt)))
		  (otherwise (collect opt))))))
    
    (unless output-contract-class
      (setq output-contract-class 'standard-output-contract))
    
    (unless input-contract-class 
      (setq input-contract-class 'standard-input-contract))
  
    (unless youth-contract-class 
      (setq youth-contract-class 'standard-windowing-contract))

    (let ((pos (position :parts supers))
	  (parts (list 'sheet
		       (contract-child-part
			(class-prototype 
			 (find-class youth-contract-class)))
		       (if adult-contract-class 
			   (contract-parent-part
			    (class-prototype 
			     (find-class adult-contract-class)))
			   'mute-parent-part)
		       output-contract-class
		       input-contract-class)))
      (setq supers
	    (if pos
		(append (subseq supers 0 (position :parts supers))
			parts
			(subseq supers (1+ (position :parts supers))
				(length supers)))
		(nconc parts supers)))
    
      `(define-group ,class define-sheet-class (defclass ,class ,supers ,slots ,@options)))))

;;;
;;; Utilities
;;;

(defmethod sheet-force-output ((sheet sheet))
  (port-force-output (port sheet)))

(defun walk-tree (walk-fn sheet &optional (depth 0) (nth 0))
  (catch 'stop-walk
    (funcall walk-fn sheet depth nth)
    (let ((depth (1+ depth))
	  (nth 0))
      (dolist (child (sheet-children sheet))
	(walk-tree walk-fn child depth nth)
	(incf nth)))))
		     
(defun walk-sheets (sheet walk-fn accumulator-fn value)
  (setq value (funcall accumulator-fn sheet value))
  (funcall walk-fn sheet value)
  (dolist (child (sheet-children sheet))
    (walk-sheets child walk-fn accumulator-fn value)))

(defun count-sheets (sheet)
  (let ((count 0)
	(depth 0))
    (walk-sheets sheet #'(lambda (p d) 
			   (declare (ignore p))
			   (incf count)
			   (setq depth (max d depth)))
		 #'(lambda (p d)
		     (declare (ignore p))
		     (1+ d))
		 0)
    (values count depth)))

;;;
;;; Contract Objects
;;;   provide some functionality to interactive sheets.
;;;   either to a single sheet or to a set of sheets.

(defclass contract () ())
(defclass sheet-contract (contract) ())

(defclass output-contract (sheet-contract) ())
(defmethod output-contract-p ((contract output-contract)) t)
(defmethod output-contract-p (object) (declare (ignore object)))

(defclass input-contract (sheet-contract) ())
(defmethod input-contract-p ((contract input-contract)) t)
(defmethod input-contract-p (object) (declare (ignore object)))

(defclass relationship-contract (contract) ())
(defclass windowing-contract (relationship-contract) ())

;;;
;;; Sheet Contract Accessor
;;;

(defmethod sheet-output-contract ((sheet sheet)) 
  sheet)

(defmethod sheet-input-contract ((sheet sheet)) 
  sheet)

(defun search-for-part (instance predicate)
  (dolist (class (class-precedence-list (class-of instance)))
    (let ((prototype  (class-prototype class)))
      (when (and (not (sheetp prototype)) (funcall predicate prototype))
	(return class)))))
    
(defmethod sheet-output-contract-class ((sheet sheet))
  (search-for-part sheet #'output-contract-p))

(defmethod sheet-input-contract-class ((sheet sheet))
  (search-for-part sheet #'input-contract-p))

(defmethod sheet-parent-part ((sheet sheet))
  (search-for-part sheet #'parent-part-p))

(defmethod sheet-child-part ((sheet sheet))
  (search-for-part sheet #'child-part-p))

#||
;;;
;;; Contract Change Protocol
;;;

(defmethod change-input ((sheet sheet) input-contract-class)
  (declare (ignore input-contract-class)))
(defmethod change-output ((sheet sheet) output-contract-class)
  (declare (ignore output-contract-class)))

||#

;;;
;;; Sheet Region and Transformation Protocols
;;;

;;; The regions and transformations of sheets and realms should not be changed
;;; except by using the following setfs are the modification protocols that are
;;; applicable to the sheet via protocol trampolining.
;;;
;;; This allows reconfigurations of the sheets to be transmitted to neighboring
;;; windowing contracts as well as to the output and input contracts of the
;;; sheet. 
;;;



;;;
;;; Mixins for Representing Transformations and Regions
;;;

(defclass sheet-region-mixin ()
    ((region :initform (make-rectangle* 0 0 100 100)
	     :initarg :region
	     :reader sheet-region)))

(defclass sheet-transformation-mixin ()
    ((transformation :initform +identity-transformation+
		     :initarg :transformation
		     :reader sheet-transformation)))

(defmethod (setf sheet-region) (region 
				(sheet sheet-region-mixin)
				&rest keys
				&key &allow-other-keys)
  (declare (dynamic-extent keys))
  (let ((old (slot-value sheet 'region)))
    (setf (slot-value sheet 'region) region)
    (apply #'sheet-region-changed sheet :old old keys)))

(defmethod (setf sheet-transformation) (transformation 
					(sheet sheet-transformation-mixin)
					&rest keys
					&key &allow-other-keys)
  (declare (dynamic-extent keys))
  (let ((old (slot-value sheet 'transformation)))
    (setf (slot-value sheet 'transformation) transformation)
    (apply #'sheet-transformation-changed sheet :old old keys)))


;;; --- This whoe special-purpose sheet implementation needs to
;;; --- be reconsidered.  We need to support all of MOVE-SHEET*,
;;; --- RESIZE-SHEET* and MOVE-AND-RESIZE-SHEET* on all sheets.
;;; --- The fact that we don't blow out now indicates that we aren't
;;; --- acutally using much of the generality of Silica in the WS
;;; --- and PTK layers.

;;;
;;; Rectangular Sheets
;;;

(defclass sheet-rectangle-mixin (sheet-region-mixin)
    ()
  (:default-initargs :region (make-rectangle* 0 0 100 100)))

(defmethod bounding-rectangle* ((sheet sheet-rectangle-mixin))
  (multiple-value-bind (minx miny maxx maxy) 
      (bounding-rectangle* (sheet-region sheet))
    (untransform-rectangle* (sheet-transformation sheet)
			    minx miny maxx maxy)))

(defmethod resize-sheet* ((sheet sheet-rectangle-mixin) width height)
  (let ((new-region (bounding-rectangle (sheet-region sheet))))
    (when width (setf (rectangle-max-x new-region) 
		      (+ (rectangle-min-x new-region) width)))
    (when height (setf (rectangle-max-y new-region) 
		       (+ (rectangle-min-y new-region) height)))
    (when (or width height)
      (setf (sheet-region sheet)
	    new-region))))
    
;;;
;;; Sheets w/ Translations
;;;

(defclass sheet-translation-mixin (sheet-transformation-mixin)
    ()
  (:default-initargs :transformation (make-translation-transformation 0 0)))

(defmethod move-sheet* ((sheet sheet-translation-mixin) x y)
  (let (x0 y0) 
    (when (or x y)
      (unless (and x y)
	(multiple-value-setq (x0 y0)
	  (transform-point* (sheet-transformation sheet) 0 0)))
      (setf (sheet-transformation sheet)
	    (make-translation-transformation (or x x0) (or y y0))))))

;;;
;;; Sheets w/ translations and rectangles
;;;

(defclass sheet-rectangle-translation-mixin (sheet-rectangle-mixin
					     sheet-translation-mixin)
    ())

(defmethod move-and-resize-sheet* 
	   ((sheet sheet-rectangle-translation-mixin) min-x min-y width height)
  (let ((region (bounding-rectangle (sheet-region sheet)))
	(transformation (sheet-transformation sheet))) 
    (when (or min-x min-y)
      (multiple-value-bind (zero-x zero-y)
	  (transform-point* transformation 0 0)
	(unless min-x (setq min-x zero-x))
	(unless min-y (setq min-y zero-y)))
      (setf transformation
	    (make-translation-transformation min-x min-y 
					     ;; :reuse transformation
					     ;; --- No :REUSE protocol any more 9/30/91.
					     ;; --- This might be a good place to invent
					     ;; --- a specialized one.
					     )))
    (when width (setf (rectangle-max-x region) 
		      (+ (rectangle-min-x region) width)))
    (when height (setf (rectangle-max-y region) 
		       (+ (rectangle-min-y region) height)))
    (when (or min-x min-y) 
      (setf (sheet-transformation sheet) transformation))
    
    (when (or width height)
      (setf (sheet-region sheet) region))))





