;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;;
;;; SILICA PORT
;;;

;;;
;;; This file defines the "porting" architecture of Silica and provides basic
;;; support that can be used in the implementation of ports for particular
;;; display servers.
;;;

;;;
;;; Ports 
;;;

(defgeneric port-force-output (port)
  (:documentation "FORCES-OUTPUT on a port."))

(defgeneric port-finish-output (port)
  (:documentation "FINISH-OUTPUT on a port (after implicit PORT-FORCE-OUTPUT)."))



(defvar *ports* nil)

;;; Default port spec to use when none is specified in a call to
;;; FIND-PORT or FIND-GRAFT.  Set by each port implementation.
(defvar *default-server-path* nil)
;;; Defaults to use to fill in sub-fields of a port spec.  Each
;;; port implementation pushes a defaults spec onto this list.
;;; --- NYI
(defvar *server-path-defaults* nil)

(defclass port (basic-port)
    (
     ;; Identification
     (server-path :reader port-server-path)
     
     ;; Map to host
     (mirror->sheet-table :initform nil)
     
     ;; These props are used by other standard input and output.  They could
     ;; use the some more general mechanism (e.g. port-prop), but since,
     ;; afterall they are priveleged to be in the kernel they can get a special
     ;; in. 
     (dm-cache    :initform () :accessor dm-cache)
     (distributor :initform (make-instance 'standard-event-distributor)
		  :reader port-event-distributor)
     (distributor-lock :initform (initial-recursive-lock-value "Distributor Lock"))
     (cursor      :initform nil :accessor port-cursor)
     

     (port-props :initform nil)
     (watchers :initform nil)
     
     ;; Kernel Bookkeeping.
     ;; ??? stamps better stay in small number zone.
     ;;     need to make them do the right thing at rollover.
     (dtstamp :initform 0 :accessor port-dtstamp)
     (crstamp :initform 0 :accessor port-crstamp)
     (stamp :initform 0 :accessor port-stamp)
     
     ;; input processing
     (event-process   :initform nil :accessor event-process)
     (preempting-processes :initform nil)
     (canonical-gesture-spec-table :initform (make-hash-table :test #'equal))

     ;; CLIM pointers
     ;; --- In devo system, move all the pointer stuff into Silica.
     (pointer :initform nil :accessor port-pointer)

     ;; A place to store the pointer documentation handler.
     ;; This is an object which supports the SET-POINER-DOCUMENTATION generic function.
     (pointer-documentation-handler :initform nil :accessor pointer-documentation-handler)

     (text-style-mapping-table :initform (make-hash-table))

     ))

(def-property-slot-macros port-prop (port) port-props)
(def-property-slot-macros mirror->sheet (port) mirror->sheet-table)

(defun make-port (&rest keys &key server-path &allow-other-keys)
  (declare (dynamic-extent keys))
  (apply #'make-instance (find-port-type (car server-path)) keys))

(defmethod find-port-type ((type (eql nil)))
  (error "Apparently no port implementation has been loaded."))

(defun find-port (&key (server-path *default-server-path*) &allow-other-keys)
  (or (dolist (port *ports*) 
	(when (port-match port server-path) (return port)))
      (make-port :server-path server-path)))

(defmethod initialize-instance :around 
	   ((port port) &key server-path (pointer-documentation-handler :default)
	    font-for-undefined-style
			&allow-other-keys)
  (setf (slot-value port 'server-path) 
	(copy-list server-path))
  (call-next-method)
  (restart-input port)
  (push port *ports*)
  (when (member pointer-documentation-handler '(:default t))
    (setf pointer-documentation-handler (port-default-pointer-documentation-handler port)))
  (setf (slot-value port 'pointer-documentation-handler) pointer-documentation-handler)
  (when font-for-undefined-style
    (add-text-style-mapping
      port *standard-character-set* *undefined-text-style*
      font-for-undefined-style))
  ;;  (setf (slot-value port 'alive-p) t)
  t)

;;; By default, we don't implement a pointer documentation window.
;;;
;;; Eventually, we will define a type of frame which will have a documentation
;;; string, and when the string is changed the documentation will be blasted to the
;;; screen.  This method is on BASIC-PORT so the overriding implementation can be on
;;; PORT.  [It can't be defined here, because we don't have frames loaded yet...]

(defmethod port-default-pointer-documentation-handler ((port basic-port))
  nil)

(defmethod set-pointer-documentation ((handler null) documentation)
  (declare (ignore documentation)))

(defmethod port-finish-output :before ((port port))
  (port-force-output port))

(defmethod port-force-output ((port port))
  port nil)

(defmethod port-finish-output ((port port))
  port nil)

(defmethod restart-port ((port port))
  (restart-input port)
  (with-slots (watchers) port
    (dolist (watcher watchers)
      (reset-watcher watcher :restart))))

(defun restart-ports (&optional (port-type))
  (mapc #'(lambda (g)
	    (when (or (null port-type)
		      (eq port-type (class-name (class-of g))))
	      (restart-port g)))
	*ports*))

(defmethod destroy-port :before ((port port))
  (with-slots (mirror->sheet-table) port
    ;; It's a plist, sigh
    (do ((plist (copy-list mirror->sheet-table) (cddr plist)))
	((null plist))
      (let ((mirror (first plist))
	    (sheet (second plist)))
	(declare (ignore mirror))
	;; --- make sure there is a :before method that unlinks the association
	;; --- destroy-mirror should take a mirror, not a sheet, I think, maybe...
	(destroy-mirror port sheet)))))

(defmethod destroy-port ((port port))
  (with-slots (#+ignore alive-p watchers) port
    #+ignore (setf alive-p nil)
    (dolist (watcher watchers)
      (reset-watcher watcher :destroy)))
  (destroy-input port)
  (setq *ports* (delete port *ports*))
  t)

(defun destroy-ports (&optional (port-type))
  (setq *ports*
	(mapcan #'(lambda (g)
		    (if (or (null port-type)
			    (eq port-type (class-name (class-of g))))
			(prog1 nil (destroy-port g))
			(list g)))
		*ports*)))

(defvar *suspended-port-state* "Port suspended")
  
(defmethod standard-port-event-process-loop ((port port))
  (let ((*multiprocessing-p* nil))
    ;; Once we're inside the events process, we have to do everything synchronously.
    ;; (I.e. we can't "let the events process handle it" :-)
    (loop
      (catch :suspend-event-process
	(loop
	  ;; In Genera, if you type Control-Abort, this
	  ;; process drops dead, leaving you in a wedged
	  ;; state users should never see. Symbolics users
	  ;; cannot NOT type c-Abort; it's hard-wired. 
	  #+Genera
	  (scl:error-restart (sys:abort "Process next event")
	    (process-next-event port))
	  #-Genera
	  (process-next-event port)))
      ;; Only here if the port has decided to suspend the event process to allow
      ;; other processes privileged access to port event queues.
      (process-wait *suspended-port-state*
		    #'(lambda () (null (slot-value port 'preempting-processes)))))))
  
(defmethod restart-input ((port port))
  (with-slots (event-process preempting-processes distributor) port
    (destroy-input port)
    (setf event-process 
	  (make-process #'(lambda () (standard-port-event-process-loop port))
			:name (format nil "Events: ~s" port)))
    (setf preempting-processes nil)
    (setf distributor (make-instance 'standard-event-distributor))
    event-process))
    
(defmethod destroy-input ((port port))
  (with-slots (event-process) port
    (when event-process (destroy-process event-process))))

;;; Does this want to be a generic function?
(defun port-event-wait (port waiter &key (wait-reason "CLIM Input") timeout)
  ;; The waiter must have NO side effects.
  (cond (*multiprocessing-p*
	 ;; Wait until some other process does something for us.
	 ;; CLIM's process-wait-with-timeout handles TIMEOUT=NIL correctly.
	 (process-wait-with-timeout wait-reason timeout waiter) 
	 (values))
	;; single process, so run the event-processing loop in-line
	(t (let ((finish-time (when timeout (+ (* timeout internal-time-units-per-second)
					       (get-internal-real-time)))))
	     (loop
		 (when (or (and timeout (> (get-internal-real-time) finish-time))
			   (funcall waiter))
		   (return-from port-event-wait (values)))
	       (process-next-event port 
				   :wait-test waiter
				   :state wait-reason))))))

(defmacro with-port (port &body forms)
  `(do-with-port ,port #'(lambda () ,@forms)))

(defmethod do-with-port ((port port) function)
  (funcall function))

(defmethod port-name ((port port))
  (let ((keys (cdr (port-server-path port))))
    (format nil "~A:~D" 
	    (getf keys :host) 
	    (getf keys :display))))

(defmethod register-watcher ((port port) watcher)
  (with-slots (watchers) port
    (pushnew watcher watchers)))

(defmethod unregister-watcher ((port port) watcher)
  (with-slots (watchers) port
    (setf watchers (delete watcher watchers))))

(define-unimplemented-protocol-method display-medium-type port
  (port type))

(define-unimplemented-protocol-method pixmap-medium-type port
  (port type))

(defmethod get-port-canonical-gesture-spec :around (gesture-spec (port port))
  ;; this doesn't really take any time if the spec is already 
  ;; canonical, but maybe we should assume that the spec is?
  (setq gesture-spec (parse-gesture-spec gesture-spec))
  (with-slots (canonical-gesture-spec-table) port
    (multiple-value-bind (value found-p)
	(gethash gesture-spec canonical-gesture-spec-table)
      (if found-p 
	  value
	  (setf (gethash gesture-spec canonical-gesture-spec-table)
		(call-next-method gesture-spec port))))))


;;; Needed for implementations which allow changing the shift
;;; keys around, such as on the Macintosh.  --RWK

(defmethod invalidate-gesture-spec-cache ((port port))
  (with-slots (canonical-gesture-spec-table) port
    (clrhash canonical-gesture-spec-table)))

(defmacro with-distributor-locked ((port) &body body)
  `(with-slots (distributor-lock) ,port
     (with-recursive-lockf (distributor-lock "Distributor Lock")
       ,@body)))

(defgeneric (setf sheet-mirror) (mirror sheet))

(defgeneric mirror-origin (port mirrored-sheet))	;usually :NW, sometimes :SW
(defgeneric sheet-target-native-edges* (mirrored-sheet)
  #+Genera (declare (values min-x min-y lim-x lim-y)))
(defgeneric sheet-actual-native-edges* (port mirrored-sheet)
  #+Genera (declare (values min-x min-y lim-x lim-y)))
(defgeneric set-sheet-actual-native-edges* (port mirrored-sheet min-x min-y lim-x lim-y))
(defgeneric mirror-inside-region* (port mirrored-sheet))

;;; This must be mixed in ahead of sheet whenever you want mirroring.
(defclass mirrored-sheet-mixin (mirror-protocol-part)
    ((mirror :initform nil :accessor sheet-mirror)
     (native-transformation :initform nil
			    :initarg :native-transformation
			    :accessor sheet-native-transformation)
     (%ntstamp :initform 0 :accessor %ntstamp)
     (%ncrstamp :initform 0 :accessor %ncrstamp)))

(defmethod realize-mirror :around ((port port) sheet)
  (or (sheet-mirror sheet)
      (let ((mirror (call-next-method)))
	(setf (mirror->sheet port mirror) sheet)
	(setf (sheet-mirror sheet) mirror))))

(defmethod destroy-mirror :around ((port port) (sheet mirrored-sheet-mixin))
  (let ((mirror (sheet-mirror sheet)))
    (when mirror
      ;; unlink the mirror from the tables.  It would be ideal to SETF
      ;; the sheets slot now, but unless we change the protocol for
      ;; destroy mirror we can't really do that.
      (remf (slot-value port 'mirror->sheet-table) mirror)
      (unwind-protect
	  (call-next-method)
	(setf (sheet-mirror sheet) nil))))
  nil)

(defmethod (setf sheet-native-transformation) :after (xform (sheet mirrored-sheet-mixin))
  (declare (ignore xform))
  (let ((port (port sheet)))
    (setf (%ncrstamp sheet) (incf (port-crstamp port))
	  (%ntstamp sheet) (incf (port-dtstamp port)))))

(defmethod sheet-mirror-resource-id (sheet)
  (declare (ignore sheet))
  nil)

(defmethod sheet-mirror-resource-id ((sheet mirrored-sheet-mixin))
  (mirror-resource-id (port sheet) sheet))

(defmethod fetch-native-transformation :around ((sheet mirrored-sheet-mixin))
  (if (sheet-mirror sheet)
      (sheet-native-transformation sheet)
      (call-next-method)))

(defmethod fetch-native-clipping-region :around ((sheet mirrored-sheet-mixin))
  (if (sheet-mirror sheet)
      (careful-transform-region (sheet-native-transformation sheet)
				(sheet-region sheet))
      (call-next-method)))

(defmethod sheet-ungrafted ((sheet mirrored-sheet-mixin))
  (call-next-method)
  (when (sheet-mirror sheet)
    (let ((port (port sheet)))
      ;; ??? if this happens care should be taken to reinstall setting on
      ;; re-realization of x-window.
      ;; ??? also, we should un-enport the whole tree so that gcontext, etc
      ;; can be coughed up as well.
      (destroy-mirror port sheet)
      (setf (mirror->sheet port (sheet-mirror sheet)) nil)
      (setf (sheet-mirror sheet) nil)
      ;; Leaving the native transformation since this sheet may be on the
      ;; input trace.  It is needed to correctly generate exit events for the
      ;; sheets on the trace.  Nobody should trust this slot without first
      ;; seeing whether there is a mirror.
      #+comment (setf (sheet-native-transformation sheet) nil))))

(defmethod sheet-enabled ((sheet mirrored-sheet-mixin) 
			  &key port-trigger &allow-other-keys)
  (call-next-method)
  (when (and (sheet-mirror sheet)
	     (null port-trigger))
    (enable-mirror (port sheet) sheet)))

(defmethod sheet-disabled ((sheet mirrored-sheet-mixin)
			   &key port-trigger &allow-other-keys) 
  (call-next-method)
  (when (and (sheet-mirror sheet)
	     (null port-trigger))	     
    (disable-mirror (port sheet) sheet)))

(defmethod sheet-region-changed ((sheet mirrored-sheet-mixin)
				 &key port-trigger &allow-other-keys)
  (call-next-method)
  (when (and (sheet-mirror sheet)
	     (null port-trigger))
    (update-mirror-region (port sheet) sheet)))

(defmethod sheet-transformation-changed ((sheet mirrored-sheet-mixin)
					 &key port-trigger &allow-other-keys)
  (call-next-method)
  (when (and (sheet-mirror sheet)
	     (null port-trigger))
    (update-mirror-region (port sheet) sheet)))

(defmethod install-settings 
	   ((sheet mirrored-sheet-mixin) 
	    &rest args
	    &key left bottom width height 
	    title state plain save-under
	    &allow-other-keys)
  (declare (dynamic-extent args)
	   (ignore title state plain save-under))
  (let ((region nil))
    (when (or left bottom width height) 
      (move-and-resize-sheet* sheet left bottom width height)
      (setq region (sheet-region sheet)))
    (apply #'install-mirror-settings (port sheet) sheet
	   :region region args)))

;;;
;;; Grafts
;;;
;;; Currently grafts are only supported at top level where root window is
;;; immutable.

(defclass graft (sheet
		 mirrored-sheet-mixin
		 standard-parent-part
		 sheet-region-mixin
		 sheet-transformation-mixin)
    (
     ;; Class slots are losing ???
     (%dtstamp :initform 0 :reader %dtstamp)
     (%crstamp :initform 0 :reader %crstamp)
     (enabled-p :initform t :reader sheet-enabled-p)
     (youth-contract :initform nil)
     
     ;; These slots can be directly accessed by port implementors.
     (origin :initarg :origin :initform :sw :reader graft-origin)
     (units :initarg :units :initform :pixel :reader graft-units
	    :type (member :pixel :mm :homogenous))
     (width-pixel :reader graft-width-pixel)
     (height-pixel :reader graft-height-pixel)
     (width-mm :reader graft-width-mm)
     (height-mm :reader graft-height-mm)
     
     (pixels-per-point :reader graft-pixels-per-point)))

(defmethod graftp ((graft graft)) t)
(defmethod graftp ((object t)) nil)

(defmethod to-graft-transformation-pixel ((sheet sheet))
  (let ((graft (graft sheet))
	(xf (fetch-delta-transformation sheet nil)))
    (ecase (graft-units graft)
      (:pixel xf)
      (:homogenous 
       (scale-transformation
	xf (graft-width-pixel graft) (graft-height-pixel graft)
	;; :reuse xf
	;; --- No :REUSE protocol any more 9/30/91.
	;; --- This might be a good place to invent
	;; --- a specialized one.
	))
      (:mm
       (scale-transformation xf (/ (graft-width-pixel graft)
			       (graft-width-mm graft))
			 (/ (graft-height-pixel graft)
			    (graft-height-mm graft))
			 ;; :reuse xf
			 ;; --- No :REUSE protocol any more 9/30/91.
			 ;; --- This might be a good place to invent
			 ;; --- a specialized one.
			 )))))
    
(defmethod to-graft-transformation-mm ((sheet sheet))
  (let ((graft (graft sheet))
	(xf (fetch-delta-transformation sheet nil)))
    (ecase (graft-units graft)
      (:mm xf)
      (:homogenous 
       (scale-transformation xf (graft-width-mm graft) (graft-height-mm graft)
			     ;; :reuse xf
			     ;; No :REUSE protocol any more.
			     ))
      (:pixel 
       (scale-transformation xf (/ (graft-width-mm graft)
			       (graft-width-pixel graft))
			 (/ (graft-height-mm graft)
			    (graft-height-pixel graft)) 
			 ;; :reuse xf
			 ;; No :REUSE protocol any more.
			 )))))

(define-constructor make-graft graft (port &key (origin :sw) (units :pixel))
  :port port :origin origin :units units)

(defun find-graft (&key port (server-path *default-server-path*) (origin :sw))
  (unless port (setq port (find-port :server-path server-path)))
  (or (getf (port-prop port :grafts) origin)
      (setf (getf (port-prop port :grafts) origin)
	    (make-graft port :origin origin))))
		   
(defmethod initialize-instance :after 
	   ((sheet graft) &key port &allow-other-keys)
  ;(with-slots (units) sheet
    (setf (port sheet) port)
    (setf (slot-value sheet 'graft) sheet)
    (realize-graft port sheet)
    ;)
    )

(defmethod fetch-native-transformation ((sheet graft))
  ;; This method should never get invoked, but it is necessary since a primary
  ;; method is required even if it would not be invoked because of an around
  ;; method.  
  ;; ???  Maybe this code would be pronounced gregrious by gregor
  (error "This code is egregrious"))

(defmethod fetch-delta-transformation ((sheet graft) (ancestor null))
  +identity-transformation+)

(defmethod fetch-delta-transformation ((sheet graft) ancestor)
  (assert (eq ancestor sheet) ()
	  "~s should be an ancestor of ~s" ancestor sheet)
  +identity-transformation+)

(defmethod sheet-viewable-p ((sheet graft))
  (sheet-enabled-p sheet))

(defmethod sheet-ancestor-p ((sheet graft) maybe-ancestor) 
  (declare (ignore maybe-ancestor)))
  
(defmethod sheet-parent ((sheet graft))
  ;; ??? Should this method be provided.
  ;; Right now necessary to terminate check-dependency in dtcr when a composer
  ;; is called with nil. E.g. (fetch-delta-transformation sheet nil) 
  ;;
  nil)

(defmethod poll-pointer ((sheet graft))
  (multiple-value-bind (x y mask)
      (do-poll-pointer (port sheet) sheet)
    (multiple-value-setq (x y)
      (untransform-point* (fetch-native-transformation sheet) x y))
    (values x y mask)))


;; Not sure where this should go
(defun fit-region*-in-region* (min-x min-y max-x max-y min-xx min-yy max-xx max-yy)
  #+Genera (declare (values min-x min-y max-x max-y adjusted-p))
  (let* ((adjusted-p nil)
	 (w (- max-x min-x))
	 (h (- max-y min-y))
	 (ww (- max-xx min-xx))
	 (hh (- max-yy min-yy)))
    (when (> w ww)
      (let ((too-much (- w ww)))
	(decf w too-much)
	(decf max-x too-much)
	(setq adjusted-p t)))
    (when (> h hh)
      (let ((too-much (- h hh)))
	(decf h too-much)
	(decf max-y too-much)
	(setq adjusted-p t)))
    (when (< min-x min-xx)
      (let ((too-much (- min-xx min-x)))
	(incf min-x too-much)
	(incf max-x too-much)
	(setq adjusted-p t)))
    (when (< min-y min-yy)
      (let ((too-much (- min-yy min-y)))
	(incf min-y too-much)
	(incf max-y too-much)
	(setq adjusted-p t)))
    (when (> max-x max-xx)
      (let ((too-much (- max-x max-xx)))
	(decf min-x too-much)
	(decf max-x too-much)
	(setq adjusted-p t)))
    (when (> max-y max-yy)
      (let ((too-much (- max-y max-yy)))
	(decf min-y too-much)
	(decf max-y too-much)
	(setq adjusted-p t)))
    (values min-x min-y max-x max-y adjusted-p)))

;; varous sheet and port/sheet methods

(defmethod sheet-target-native-edges* ((sheet mirrored-sheet-mixin))
  (let* ((region (sheet-region sheet))
	 (sheet-to-parent (sheet-transformation sheet))
	 (parent-to-native (fetch-native-transformation (sheet-parent sheet))))
    (multiple-value-bind (sheet-min-x sheet-min-y sheet-lim-x sheet-lim-y)
	(bounding-rectangle* region)
      (multiple-value-bind (parent-min-x parent-min-y parent-lim-x parent-lim-y)
	  (careful-transform-rectangle* sheet-to-parent
					sheet-min-x sheet-min-y sheet-lim-x sheet-lim-y)
	(multiple-value-bind (native-min-x native-min-y native-lim-x native-lim-y)
	    (careful-transform-rectangle* parent-to-native
					  parent-min-x parent-min-y parent-lim-x parent-lim-y)
	  (values native-min-x native-min-y native-lim-x native-lim-y))))))

(defmethod update-mirror-region ((port port) (sheet mirrored-sheet-mixin))
  (multiple-value-bind (target-min-x target-min-y target-lim-x target-lim-y)
      (sheet-target-native-edges* sheet)
    (multiple-value-bind (actual-min-x actual-min-y actual-lim-x actual-lim-y)
	(sheet-actual-native-edges* port sheet)
      (unless (and (= target-min-x actual-min-x)
		   (= target-min-y actual-min-y)
		   (= target-lim-x actual-lim-x)
		   (= target-lim-y actual-lim-y))
	(set-sheet-actual-native-edges* port sheet
				 target-min-x target-min-y target-lim-x target-lim-y))
      (update-native-transformation port sheet))))

;; I don't know about the validity of the following comment.
;;
;; --- This really only works for top level (clim) sheets, since it
;; assumes the sheet's parent has a mirror, and that mirror is the same
;; as the sheet mirror's parent (got that?), which ain't necessarily so.
(defmethod mirror-region-updated ((port port) (sheet mirrored-sheet-mixin))
  (multiple-value-bind (native-min-x native-min-y native-max-x native-max-y)
      (sheet-actual-native-edges* port sheet)
    (let* ((region (sheet-region sheet))
	   (parent (sheet-parent sheet))
	   (sheet-to-parent (sheet-transformation sheet))
	   (parent-to-native (fetch-native-transformation parent))
	   (transformation-changed-p nil)
	   (region-changed-p nil))
      (multiple-value-bind (region-min-x region-min-y region-max-x region-max-y)
	  (bounding-rectangle* region)
	(multiple-value-bind (oparent-min-x oparent-min-y oparent-max-x oparent-max-y)
	    (careful-transform-rectangle* sheet-to-parent
					  region-min-x region-min-y region-max-x region-max-y)
	  (multiple-value-bind (nparent-min-x nparent-min-y nparent-max-x nparent-max-y)
	      (careful-untransform-rectangle* parent-to-native
					      native-min-x native-min-y native-max-x native-max-y)
	    ;; check for translation
	    (let ((dx (round (- nparent-min-x oparent-min-x)))
		  (dy (round (- nparent-min-y oparent-min-y))))
	      (unless (and (zerop dx) (zerop dy))
		(setf sheet-to-parent (translate-transformation sheet-to-parent dx dy)
		      transformation-changed-p t)))
	    ;; check for size change
	    (let* ((oparent-width  (- oparent-max-x oparent-min-x))
		   (oparent-height (- oparent-max-y oparent-min-y))
		   (nparent-width  (- nparent-max-x nparent-min-x))
		   (nparent-height (- nparent-max-y nparent-min-y))
		   (dw (round (- nparent-width  oparent-width)))
		   (dh (round (- oparent-height oparent-height))))
	      (unless (and (zerop dw) (zerop dh))
		(multiple-value-bind (nregion-width nregion-height)
		    (untransform-distance sheet-to-parent nparent-width nparent-height)
		  (with-bounding-rectangle* (x1 y1 x2 y2) region
		    (setf region
			  (make-rectangle* x1 y1
					   (+ x1 nregion-width) (+ y1 nregion-height)))
		    (setf region-changed-p t)
		    #+Ignore
		    (setf region (copy-region region)
			  (rectangle-width  region) nregion-width
			  (rectangle-height region) nregion-height
			  region-changed-p t)))))
	    ;; York moved the update-native-transformation call
	    ;; in front of the next two forms, 8/20/91
	    ;; I think that you have to update the native transformation
	    ;; before updating the region (or xform), since updating the
	    ;; region is going to try to resize any mirrored children, who
	    ;; will then invoke the mirror-region-updated code above,
	    ;; which relies on the native transformation of the parent.
	    ;; --- Of course, does the mirror's region/xform have to
	    ;; be updated before update-native-transformation works?
	    ;; Anyway, the adapting applications seem to work much better
	    ;; after the change.
	    (when (or transformation-changed-p region-changed-p)
	      (update-native-transformation port sheet))
	    (when transformation-changed-p 
	      (setf (sheet-transformation sheet :port-trigger t)
		    sheet-to-parent))
	    (when region-changed-p
	      (setf (sheet-region sheet :port-trigger t)
		    region))))))))


(defmethod update-native-transformation ((port port) (sheet mirrored-sheet-mixin))
  (multiple-value-bind (sheet-min-x sheet-min-y sheet-lim-x sheet-lim-y)
      (bounding-rectangle* (sheet-region sheet))
    (declare (ignore sheet-lim-x sheet-lim-y))
    (multiple-value-bind (mirror-min-x mirror-min-y mirror-lim-x mirror-lim-y)
	(mirror-inside-region* port sheet)
      (declare (ignore mirror-lim-x))
      (let ((sheet-to-mirror (make-translation-transformation
			       (- mirror-min-x sheet-min-x)
			       (- mirror-min-y sheet-min-y)))
	    (sheet-origin (graft-origin (graft sheet)))
	    (mirror-origin (mirror-origin port sheet)))
	(cond ((eq mirror-origin sheet-origin))
	      (t
	       ;; assume Y flipping is the only possibility
	       ;; --- No :REUSE protocol any more 9/30/91.
	       ;; --- This would be a good place to invent
	       ;; --- a specialized one.

	       (setq sheet-to-mirror (scale-transformation
				       sheet-to-mirror 1 -1
				       ;; :reuse sheet-to-mirror
				       ))
	       ;; (- mirror-min-y) wants to map to (1- mirror-lim-y)
	       (setq sheet-to-mirror (translate-transformation
				       sheet-to-mirror
				       0 (- (1- mirror-lim-y) (- mirror-min-y))
				       ;; :reuse sheet-to-mirror
				       ))))
	;; set it and return the xform
	(setf (sheet-native-transformation sheet)
	      sheet-to-mirror)))))

;; The main difference between this (graft) and above (non-graft) is the
;; scaling potential.
(defmethod update-native-transformation ((port port) (sheet graft))
  (multiple-value-bind (sheet-min-x sheet-min-y sheet-lim-x sheet-lim-y)
      (bounding-rectangle* (sheet-region sheet))
    (multiple-value-bind (mirror-min-x mirror-min-y mirror-lim-x mirror-lim-y)
	(mirror-inside-region* port sheet)
      (let* ((xfactor (/ (- mirror-lim-x mirror-min-x)
			 (- sheet-lim-x	 sheet-min-x)))
	     (yfactor (/ (- mirror-lim-y mirror-min-y)
			 (- sheet-lim-y	 sheet-min-y)))
	     (sheet-to-mirror (translate-transformation
				;; scale first, then adjust the scaled
				;; sheet min to the mirror min
				(make-scaling-transformation xfactor yfactor)
				(- mirror-min-x (* sheet-min-x xfactor))
				(- mirror-min-y (* sheet-min-y yfactor))))
	     (graft-origin (graft-origin (graft sheet))))
	(when (eq graft-origin :SW)
	  (setq sheet-to-mirror
		(scale-transformation
		  sheet-to-mirror 1 -1
		  ;; :reuse sheet-to-mirror
		  ;; --- No :REUSE protocol any more 9/30/91.
		  ;; --- This might be a good place to invent
		  ;; --- a specialized one.
		  ))
	  ;; --- Especially since sheet-to-mirror is often an intermediate
	  ;; --- result
	  ;; (- mirror-min-y) wants to map to (1- mirror-lim-y)
	  (setq sheet-to-mirror (translate-transformation
				  sheet-to-mirror
				  0 (- (1- mirror-lim-y) (- mirror-min-y))
				  ;; :reuse sheet-to-mirror
				  ;; --- No :REUSE protocol any more 9/30/91.
				  ;; --- This might be a good place to invent
				  ;; --- a specialized one.
				  )))
	;; set it and return the xform
	(setf (sheet-native-transformation sheet)
	      sheet-to-mirror)))))
