;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;;
;;; This File contains definition and implementations of the Silica REPAINT and
;;; OUTPUT Protocols.  
;;;



;;;
;;; SILICA Medium Protocol
;;;
;;;   Defines protocols for building display medium objects.  These objects
;;; essentially define the interface that has to implemented to incorporate a
;;; graphics package into Silica.
;;;
;;; No Graphics Operations are provided in the abstract class, but clearly
;;; any real DM will provide some way of doing graphics on it.
;;;
;;; Graphics Operations must respect the transformation and clipping region
;;; that the kernel imposes on dms.  DM classes should provide methods on the
;;; setf generic functions for setting the transformation and clipping region
;;; if it needs to change implementation state.  Or else each graphics call
;;; should clip and transformation user requests.
;;; In addition, use of these fields should be used within the scope of
;;; "output protection" to insure that they are valid and locked.

(defgeneric medium-force-output (medium)
  (:documentation "FORCE-OUTPUT on a medium."))

(defgeneric medium-finish-output (medium)
  (:documentation "FINISH-OUTPUT on a medium (after implicit FORCE-OUTPUT)."))


(defclass medium (basic-medium)
    (
     ;; A medium should clip and transformation using these values.
     ;;   clipping region is in output surface coordinates
     (device-transformation :initform +identity-transformation+
			    :accessor device-transformation)
     (device-clipping-region :initform +everywhere+ 
			     :accessor device-clipping-region )

     ;; This is needed by a pixmap and display medium, but needs thought for
     ;; printer medium, though some thing equivalent will be needed for that
     ;; too.  
     (port :initarg :port :reader port)))

;;; This has been co-opted to provide a "background window"
;;; capability.  You can draw on the pixmap-medium, then
;;; copy-area the results on to a visible window.
(defclass pixmap-medium (medium)
    ((medium-pixmap :initform nil :initarg :pixmap
		    :accessor medium-pixmap)))

;;; Kludge to make a "background window" medium associated with
;;; an existing medium.
;;; --- Should this take :WIDTH and :HEIGHT keywords?  The medium
;;; currently sized from the sheet associated with the display-medium.
;;; Maybe this should take a sheet, not a medium?
(defun make-pixmap-medium (&rest args &key display-medium port 
				 &allow-other-keys)
  (declare (dynamic-extent args))
  (unless port (setq port (port display-medium)))
  ;; --- The initialize-instance methods aren't expecting
  ;; :DISPLAY-MEDIUM
  (with-rem-keywords (pass-on-args args '(:display-medium))
    (apply #'make-instance 
	   (pixmap-medium-type 
	     port (graphics-package-key display-medium))
	   :port port
	   :associated-medium display-medium
	   pass-on-args)))

(defmethod medium-finish-output :before ((medium medium))
  (medium-force-output medium))

;; default methods, which for lack of anything better, trampoline to the port
(defmethod medium-force-output ((medium medium))
  (let ((port (slot-value medium 'port)))
    (when port
      (port-force-output port))))

(defmethod medium-finish-output ((medium medium))
  (let ((port (slot-value medium 'port)))
    (when port
      (port-finish-output port))))

;;;
;;; Display Media
;;; 

(defclass display-medium (medium)
    ((dmstamp :initform 0)
     (sheet :initarg :sheet :accessor medium-sheet)
     
     ;; Assuming that all display medium (i.e. all graphics packages) will want
     ;; to support the client view control (i.e. a client transformation and
     ;; clipping region in addition to silica's).  CLIM graphics certainly does
     ;; support this.  Once upon a time I had this facility separated into a
     ;; client-view-control-mixin and also provided a
     ;; simple-view-control-mixin which just forwarded the silica- calls to
     ;; device- calls, but for now I've folded it back in, since I
     ;; have a hard time imaging a graphics package that doesn't want to
     ;; support this anyway.
     
     (client-transformation :initform +identity-transformation+
			    :reader client-transformation)
     ;; Stored in device coordinates, though client sees and sets in medium
     ;; coordinate system.
     (%client-clipping-region :initform +everywhere+ )))

;;; Relying on Silica to cache its own native transformation and clipping
;;; regions for sheets, and indeed it does.

(defmethod (setf silica-transformation) (transformation (dm display-medium))
  (setf (device-transformation dm)
	;; --- swapped transform ordering below for gratuitous 1.0 argument order reversal
	;; --- doughty 12/22/91
	(compose-transformations transformation
				 (slot-value  dm 'client-transformation) 
				 ;; :reuse (device-transformation dm)
				 ;; --- No :REUSE protocol any more 9/30/91.
				 ;; --- This might be a good place to invent
				 ;; --- a specialized one.
				 )))

(defmethod (setf silica-clipping-region) ((region region) (dm display-medium))
  ;; region is expressed in device coordinate systems
  (setf (device-clipping-region dm)
	(region-intersection 
	 (slot-value dm '%client-clipping-region)
	 region)))


;;;
;;; Output Protection
;;;
;;; ALL Graphics Operations should be protected.
;;; Every graphics package has to check that the output-contract and display
;;; medium are ready, in case some window action has unreadied for output.  In
;;; addition, they have to grab a lock for the duration of the output.  

(defmacro with-output-synchronization (&body body)
  ;; So.  You think you're being protected.  Hah!
  `(progn ,@body))

(defmacro with-output-protection (dm &body body)
  `(with-output-synchronization 
       (when (validate-medium ,dm)
	 ,@body)))

(defmacro display-medium-valid-p (dm)
  ;; eventually switch eq to a =
  `(eq (slot-value ,dm 'dmstamp)
       (port-stamp (slot-value ,dm `port))))

(defmacro %mark-dm-valid (dm) 
  `(setf (slot-value ,dm 'dmstamp)
	 (port-stamp (slot-value ,dm `port))))

(defmacro %mark-dm-invalid (dm)
  `(setf (slot-value ,dm 'dmstamp) 0))

(defmethod validate-medium ((dm display-medium))
  (let ((sheet (slot-value dm 'sheet)))
    (and 
     (or (display-medium-valid-p dm)
	 (when       
	     ;; Make sure output is allowed to this sheet.
	     ;; Eventually could provide more flexibility in handling
	     ;; non-viewable output
	     ;; e.g. Hold Invisible Output, use backing store, etc.
	     (sheet-viewable-p sheet)

	   ;; Could be smarter about setting the transformation and
	   ;; clipping-region.  It isn't always necessary to reset these.
	   ;; Transformation should be set first, so that clipping region
	   ;; calculate can be guaranteed a good value of all transformations.
	   (setf (silica-transformation dm)
		 (fetch-native-transformation sheet))
	   (setf (silica-clipping-region dm)
		 (fetch-native-clipping-region sheet))
	
	   ;; FINALLY Mark the DM
	   (%mark-dm-valid dm)
	   t))
     (not (eq (device-clipping-region dm) +nowhere+)))))

(defmethod %validate-medium ((dm display-medium))
  (let ((sheet (slot-value dm 'sheet)))
    ;; Could be smarter about setting the transformation and clipping-region.
    ;; It isn't always necessary to reset these.
    ;; Transformation should be set first, so that clipping region calculate
    ;; can be guaranteed a good value of all transformations.
    (setf (silica-transformation dm)
	  (fetch-native-transformation sheet))
    (setf (silica-clipping-region dm)
	  (fetch-native-clipping-region sheet))
	
    ;; FINALLY Mark the DM
    (%mark-dm-valid dm)
    (values t)))

(defmethod insured-device-transformation ((dm display-medium))
  (let ((sheet (slot-value dm 'sheet)))
    (if (display-medium-valid-p dm)  
	(device-transformation dm)
	(setf (silica-transformation dm)
	      (fetch-native-transformation sheet)))))

;;;
;;; Client Transformation
;;;
;;; Here because %mark-dm-invalid macro needs to be defined.
;;;

(defmethod (setf client-transformation) 
	   (transformation (dm display-medium))
  (setf (slot-value dm 'client-transformation) transformation)
  (%mark-dm-invalid dm))

;; The client's clipping region is expressed in medium coordinate system, so
;; the current device-transformation is taken into to account, to store it in
;; device coordinate system.

(defmethod client-clipping-region ((dm display-medium))
  (untransform-region (insured-device-transformation dm)
		      (slot-value dm '%client-clipping-region)))

(defmethod (setf client-clipping-region) 
	   (new-clipping-region (dm display-medium))
  (setf (slot-value dm '%client-clipping-region)
	(transform-region (insured-device-transformation dm)
			  new-clipping-region))
  (%mark-dm-invalid dm))


;;;
;;; DM Reuse Caches 
;;;

(defmacro allocate-dm (dm-type-id port)
  `(or (let ((cache (getf (slot-value ,port 'dm-cache) ,dm-type-id)))
	 (and cache
	      (not (zerop (fill-pointer cache)))
	      (vector-pop cache)))
       (make-instance (display-medium-type ,port ,dm-type-id)
		      :port ,port)))

#+Ignore
(defmacro allocate-dm (dm-type-id port)
  `(or (pop (getf (slot-value ,port 'dm-cache) ,dm-type-id))
       (make-instance (display-medium-type ,port ,dm-type-id) 
		      :port ,port)))

(defmacro deallocate-dm (dm-type-id port dm)
  `(let ((cache (getf (slot-value ,port 'dm-cache) ,dm-type-id)))
     (unless cache
       (setq cache
	     (setf (getf (slot-value ,port 'dm-cache) ,dm-type-id)
		   (make-array 20 :fill-pointer 0 :adjustable t))))
     (vector-push-extend ,dm cache)))

#+Ignore
(defmacro deallocate-dm (dm-type-id port dm)
  `(push ,dm 
	 (getf (slot-value ,port 'dm-cache) ,dm-type-id)))

(defmethod cleanup-display-medium ((dm display-medium))
  nil)

(defmethod sheet-medium ((output-contract output-contract))
  nil)

(defmethod do-with-display-medium ((port port) sheet dm-type-id draw-closure)
  (let* ((local-dm (sheet-medium sheet))
	 dm)
    (unwind-protect
	 (progn
	   (unless local-dm 
	     (setq dm (allocate-dm dm-type-id port))
	     (%mark-dm-invalid dm)
	     (setf (medium-sheet dm) sheet)
	     ;; ??? Unoptimized slot-value
	     (setf (slot-value dm 'port) port))
	   ;; Do the Client stuff
	   (funcall draw-closure (or local-dm dm)))
      ;; Cleanup
      (unless local-dm
	(cleanup-display-medium dm)
	(deallocate-dm dm-type-id port dm)))))
		    
(defmacro using-display-medium ((dm dm-type-id sheet) &body forms)
  (let ((flet-name (make-symbol "DO-WITH-DISPLAY-MEDIUM-INTERNAL")))
    (once-only (sheet)
      `(flet ((,flet-name (,dm)
	       ,@forms))
	 (declare (dynamic-extent #',flet-name))
	 (do-with-display-medium (port ,sheet) ,sheet ,dm-type-id
				 #',flet-name)))))

(defmacro using-clim-medium ((dm sheet) &body forms)
  `(using-display-medium (,dm :clim ,sheet)
     ,@forms))


;;;
;;; Repaint Handlers Mixins
;;; 

(defclass repaint-handler ()
    ())

(define-unimplemented-protocol-method handle-repaint repaint-handler
    ((sheet repaint-handler) repaint-region
			   &key &allow-other-keys))

(defclass forwarding-repainter (repaint-handler)
    ((repaint-handler :initform nil :initarg :repaint-handler
		      :accessor repaint-handler
		      :type repaint-handler)))

(defmethod handle-repaint ((sheet forwarding-repainter) repaint-region
			   &rest keys
			   &key &allow-other-keys)
  (declare (dynamic-extent keys))
  (apply #'handle-repaint (slot-value sheet 'repaint-handler) repaint-region
	 :sheet sheet 
	 keys))

(defclass mute-repainter (repaint-handler)
    ())

(defmethod handle-repaint ((sheet mute-repainter) repaint-region
			   &key &allow-other-keys)
  (declare (ignore repaint-region)))

;;;
;;; Standard Output Contract
;;;

(defclass standard-output-contract (output-contract
				    queue-repaint-mixin
				    mute-repainter)
    ((medium :initform nil 
	     :initarg :medium-type
	     :initarg :medium
	     :accessor sheet-medium)))



(defmethod initialize-instance :after 
	   ((sheet standard-output-contract) 
	    &key medium medium-type port &allow-other-keys)
  
  (when (symbolp medium)
    (assert (null medium-type) ()
	    "Both medium and medium-type were passed in ~
             Note that medium-type is now deprecated.")
    (setq medium-type medium))
    
  (when (and medium-type port)
    ;; --- Shouldn't this just be (engraft-medium medium port sheet)
    ;; instead of all this basically equivalent stuff?  The main thing
    ;; this doesn't do is the T -> :CLIM hack that the ENGRAFT-MEDIUM
    ;; method on SYMBOL medium does.
    (let ((medium (make-instance (display-medium-type port medium-type)
				 :sheet sheet
				 :port port)))
      (setf (sheet-medium sheet) medium)
      (engraft-medium medium port sheet))))

(defmethod sheet-grafted :after ((sheet standard-output-contract))
  (let ((medium (sheet-medium sheet)))
    (when medium
      (engraft-medium medium (port sheet) sheet))))

(defmethod engraft-medium ((medium symbol) port (sheet standard-output-contract))
  
  (when (eq medium t) (setq medium :clim))
  (let ((medium (make-instance (display-medium-type port medium)
			       :sheet sheet
			       :port port)))
    (setf (slot-value sheet 'medium) medium)
    (engraft-medium medium port sheet)		;Plug into the other methods for this.
    medium))

(defmethod engraft-medium ((medium display-medium) 
			   port 
			   (sheet standard-output-contract))
  (setf (slot-value medium 'port) port)
  (setf (slot-value medium 'sheet) sheet))

(generate-trampolines clg-graphics clg-medium standard-output-contract
		      `(sheet-medium ,standard-output-contract))



;;;
;;; Other Output Contracts
;;;

(defclass mute-output-contract (output-contract)
    ())

(defclass providing-output-contract (output-contract
				     eager-queue-repaint-mixin
				     mute-repainter)
    ())

;;;
;;; REPAINT Protocol
;;;

(defclass queue-repaint-mixin ()
    ())
	 
(defmethod queue-repaint ((sheet queue-repaint-mixin) repaint-region)
  (queue-event 
   sheet 
   (make-repaint-event :sheet sheet :region repaint-region)))

(defclass eager-queue-repaint-mixin ()
    ())

(defmethod queue-repaint ((sheet eager-queue-repaint-mixin) repaint-region)
  (repaint-sheet sheet repaint-region))

(defmethod repaint-sheet ((sheet sheet) repaint-region)
  (when (sheet-viewable-p sheet)
    ;; ???? really should be locking, since everything below isn't going to
    ;; check for viewability of sheet.
    (let ((port (slot-value sheet 'port)))
      (when port 
	(repaint-sheet-internal sheet repaint-region)
	(port-force-output port)))))

(defmethod repaint-sheet ((sheet sheet) (repaint-region everywhere))
  (repaint-sheet sheet (sheet-region sheet)))

(defmethod repaint-sheet ((sheet sheet) (repaint-region nowhere))
  nil)

(defun repaint-children (sheet repaint-region)
  (let (child-repaint-region)
    (dolist (child (children-in-region sheet repaint-region))
      (when (sheet-enabled-p sheet)
	(setq child-repaint-region 
	      (careful-untransform-region (sheet-transformation child)
					  repaint-region))
	(setq child-repaint-region
	      (region-intersection child-repaint-region (sheet-region child)
				   ;; :reuse child-repaint-region
				   ;; --- No :REUSE protocol any more 9/30/91.
				   ;; --- This might be a good place to invent
				   ;; --- a specialized one.
				   ))
	(unless (eq child-repaint-region +nowhere+)
	  (repaint-sheet-internal child child-repaint-region))))))

(defmethod repaint-sheet-internal ((sheet mute-output-contract) repaint-region)
  (repaint-children sheet repaint-region))

;; Should always be nil at the top level
(defvar *medium* nil)

(defmethod repaint-sheet-internal :around ((sheet providing-output-contract)
					   repaint-region)
  (declare #-PCL (ignore repaint-region)
	   (special *medium*))
  (let* ((port (slot-value sheet 'port)))
    (when port
      (if *medium*
	  (call-next-method)
	  (let* ((sheet-medium (sheet-medium sheet))
		 (new-medium (or sheet-medium (allocate-dm :clim port))))
	    (unwind-protect 
		(let ((*medium* new-medium))
		  (call-next-method))
	      ;; Only clean up and deallocate if we allocated it
	      (unless sheet-medium
		(cleanup-display-medium new-medium)
		(deallocate-dm :clim port new-medium))))))))

(defmethod repaint-sheet-internal ((sheet providing-output-contract) 
				   repaint-region)
  (declare (special *medium*))
  (let ((medium *medium*))
    (setf (medium-sheet medium) sheet)
    (%validate-medium medium)
    (handle-repaint sheet repaint-region :medium medium)
    (repaint-children sheet repaint-region)))

(defmethod repaint-sheet-internal 
	   ((sheet standard-output-contract) repaint-region)
  (handle-repaint sheet repaint-region
		  ;; We depend on the INITIALIZE-INSTANCE method to set up the medium: 
		  :medium (sheet-medium sheet))
  (repaint-children sheet repaint-region))



