;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;;
;;; SILICA INPUT
;;;

;;;
;;; Standard Event Distribution
;;;

;;;  Dispatch mouse event to deepest (youngest) sheet.  In the future, will
;;;  consider trying more than one sheet based on interest of recipient sheet.
;;; 
;;;  Should do something for dispatching keyboard events differently.
;;;

(defclass standard-event-distributor ()
    ((handle-clicks? :allocation :class
		     :initform t :reader handle-clicks?)
     (dispatch-crossings-p :initform t)
     (trace :initform (make-array 10 :fill-pointer 0 :adjustable t))
     (trace-stamp :initform 0)
     
     (dispatch-focus :initform nil)
     (keyboard-focus :initform nil 
		     :accessor distributor-keyboard-focus)
     
      
     ;; For communication with event process.
     (enabled :initform t :initarg :enabled :accessor distributor-enabled)))

(defmethod reset-distributor ((distributor standard-event-distributor))
  (with-slots (trace trace-stamp enabled) distributor
    (setf (fill-pointer trace) 0
	  trace-stamp          0
	  enabled	       t)))

(defmethod top-of-trace ((distributor standard-event-distributor))
  (with-slots (trace) distributor
    (unless (zerop (fill-pointer trace))
      (aref trace (1- (fill-pointer trace))))))

(defclass standard-event-distributor-part ()
    ((propagate-events? :initform t
			:initarg :propagate-events?
			:accessor propagate-events?)

     #+ignore
     ;; Moved this into method so constructors can be optimized
     (handle-clicks? :allocation :class
		     :initform t :reader handle-clicks?)))

;; ??? Allow clicks.  This is necessary since the source sheet can also
;; cut off port level click processing.  Need to be rethought.
;; Necessary so that dws can cut off clicks.
(defmethod handle-clicks? ((sheet standard-event-distributor-part)) t)
  

#+???
(defmethod (setf propagate-events?) :after
	   (value (distributor standard-event-distributor-part))
  (declare (ignore value))
  ;; ??? need to do something better globally about stamps and possible
  ;; overflows, and other kinds of errors.  
  (when (port sheet)
    (setf (port-trace-stamp (port sheet)) 0)))

;;;
;;; Keyboard Focus Support
;;;

(defmethod enable-keyboard-focus ((sheet standard-event-distributor-part))
  (let* ((distributor (port-event-distributor (port sheet)))
	 (current (distributor-keyboard-focus distributor)))
    ;; Ensure non-primaries get a change to notice focus disabling
    (when current (disable-keyboard-focus current))
    (setf (distributor-keyboard-focus distributor) sheet)))

(defmethod disable-keyboard-focus ((sheet standard-event-distributor-part))
  (let ((distributor (port-event-distributor (port sheet))))
    (setf (distributor-keyboard-focus distributor) nil)))

(defmethod distributor-keyboard-focus ((sheet standard-event-distributor-part))
  (let ((distributor (port-event-distributor (port sheet))))
    (distributor-keyboard-focus distributor)))


;;;
;;; Event Distribution
;;;

(defmethod distribute-device-event
	   ((distributor standard-event-distributor) 
	    (port port)
	    source
	    &rest keys 
	    &key moved-p keyboard-p native-x native-y
	    &allow-other-keys)
  (declare (dynamic-extent keys))
  
  (with-slots (trace-stamp keyboard-focus dispatch-focus) distributor
    (if (null source) 
	(apply #'validate-trace distributor port nil keys)
      
	(let ((crossing-p nil) 
	      (same-tree-p (= trace-stamp (port-stamp port)))
	      recipient)
	  (when (or moved-p (not same-tree-p))
	    (setq crossing-p 
		  (apply #'validate-trace distributor port source
			 :same-tree-p same-tree-p keys)))

	  (if (and keyboard-p keyboard-focus)
	      (setq recipient keyboard-focus)
	      (unless (and moved-p crossing-p)
		;; null top of trace only happens when the mouse has left the
		;; top sheet and then the button or key goes up.  This happens
		;; because X distributes the up to whoever go the down (roughly
		;; speaking). 
		(setq recipient (top-of-trace distributor))))

	  (when recipient
	    (multiple-value-bind (sheet-x sheet-y)
		(untransform-point* (fetch-native-transformation recipient)
				    native-x native-y)

	      (apply #'dispatch-event-using-port-keys
		     (or dispatch-focus recipient) port
		     :sheet recipient :sheet-x sheet-x :sheet-y sheet-y
		     keys)))))))

(defmethod validate-trace ((distributor standard-event-distributor)
			   (port port)
			   source &rest keys
			   &key same-tree-p native-x native-y state
			   &allow-other-keys)
  (declare (dynamic-extent keys)
	   (ignore keys))
  (with-slots (trace trace-stamp dispatch-focus) distributor
    (let* ((trace-ptr (fill-pointer trace))
	   (crossing-p nil)
	   current currentx currenty direction
	   old-current)
      (labels ((cross (event-key) 
		 (unless crossing-p (setq crossing-p t))
		 ;; The sheets on the trace stack may no longer
		 ;; be grafted (e.g. if the frame layout has been
		 ;; changed), so be careful not to blow out.
		 ;; --- Is this the right test?
		 (when (and current (port current))
		   (dispatch-event-using-port-keys
		    (or dispatch-focus current) port 
		    :sheet current
		    :event-key event-key
		    :direction direction		  
		    :native-x native-x :native-y native-y
		    :sheet-x  currentx :sheet-y  currenty
		    :state state)))
	       (push-trace ()
		 (vector-push-extend current trace)
		 (incf trace-ptr))
	       (pop-trace ()
		 (vector-pop trace)
		 (decf trace-ptr))
	       (read-trace ()
		 (unless (zerop trace-ptr)
		   (setq current (aref trace (1- trace-ptr)))
		   t))
	       (set-current-pos ()
		 ;; If current is the same as last call, don't bother
		 ;; to recompute currentx,currenty
		 (unless (eql current old-current)
		   (setq old-current current)
		   (multiple-value-setq (currentx currenty) 
		     (untransform-point* 
		      (fetch-native-transformation current) 
		      native-x native-y)))))
	
	;; Check for having been degrafted
	(when (and (read-trace) (not same-tree-p))
	  
	  ;; Somebody made it so no crossing events are passed to degrafted
	  ;; sheets (cf. "cross" above), so this code doesn't do anything
	  ;; worthwhile anymore.
	  ;; --RR
	  
	  (let ((n
		 (dotimes (n trace-ptr trace-ptr)
		   (when (not (port (aref trace n)))
		     (return n)))))
	    
	    (setq direction :out)
	    (read-trace)
	    ;; ---We are going to generate exit events for (possibly
	    ;; ungrafted) sheets, and fill-event needs to have some
	    ;; non-nil value of X and Y to work with.  This code might
	    ;; be run before any call to set-current-pos.
	    
	    ;; Of course it doesn't matter anymore since cross checks for port
	    ;; and doesn't do anything if no port. --RR
	    
	    (when (null currentx)
	      (setq currentx 0 currenty 0))
	    (loop (when (= trace-ptr n) (return t))
		  (cross :pointer-exit)
		  (pop-trace)
		  (unless (read-trace) (return nil)))))

	(when source
	  
	  ;; Must check down to top of trace if it exists.
	  (when (read-trace)
	    (let ((n
		   (if (not (eq source (aref trace 0)))
		       0
		       ;; Else
		       ;; Have to worry about the following possibilities:
		       ;; -- Window events i.e. (null same-tree-p)
		       ;; -- Changes to propagate-events? on a member of trace
		       ;; -- Not inside member of trace.
		       (if same-tree-p 
			   ;; Can go up the tree and will since it's likely to
			   ;; be more efficient, and also don't have to check
			   ;; some the above things.
			   (do ((n trace-ptr (1- n)))
			       ((zerop n) 0)
			     (when (region-contains-point*-p
				    ;; ??? Not the right region if there's
				    ;; a mirror between sheet and the source.
				    (fetch-native-clipping-region
				     (aref trace (1- n)))
				    native-x native-y)
			       (return n)))
			   
			   ;; Have to worry about window events up high
			   ;; so go top down. Including changing of
			   ;; propagate-events? 
			   (dotimes (n trace-ptr trace-ptr)
			     (setq current (aref trace n))
			     (when (or (not (sheet-enabled-p current))
				       (progn
					 (set-current-pos)
					 (not (sheet-inside? 
					       current currentx currenty)))) 
			       (return n))
			     (unless (propagate-events? current)
			       (return (1+ n))))))))
	    
	      (setq direction :up)
	      (read-trace)
	      (set-current-pos)
	      (loop (when (= trace-ptr n) (return t))
		    (cross :pointer-exit)
		    (pop-trace)
		    (unless (read-trace) (return nil))
		    (set-current-pos)
		    (cross :pointer-enter))))
	  
	  ;; If no trace at this stage then see if we should restart it
	  (unless (read-trace)
	    (progn
	      (setq current source)
	      (when (and (sheet-enabled-p current)
			 (progn
			   (set-current-pos)
			   (sheet-inside? current currentx currenty)))
		(setq direction :down)
		(cross :pointer-enter)
		(push-trace))))
	
	  ;; If trace at this stage,
	  ;; go down the trace to deepest sheet containing cursor.
	  (when (read-trace)
	    (set-current-pos)
	    (setq direction :down)
	    (let (child) 
	      (loop
	       (unless (and 
			(propagate-events? current)
			(setq child (child-at-point* current currentx currenty)))
		 (return))
	       (cross :pointer-exit)
	       (setq current child)
	       (set-current-pos)
	       (cross :pointer-enter)
	       (push-trace))))
	
	  (setf trace-stamp (port-stamp port))
	
	  ;; Ensure cursor
	  (when crossing-p
	    (when (read-trace) 
	      (ensure-cursor current))
	    t))))))

;;;
;;; Cursor Support
;;;

(defclass standard-cursor-support ()
    ((cursor :initform :default
	     :initarg :cursor
	     :accessor sheet-cursor)))
    
(defmacro with-cursor ((cursor sheet) &body body)
  (let ((old (gentemp)))
    (once-only (sheet)
      `(let ((,old (sheet-cursor ,sheet)))
	 (unwind-protect
	      (progn
		(setf (sheet-cursor ,sheet) ,cursor)
		,@body)
	   (setf (sheet-cursor ,sheet) ,old))))))

(defmethod (setf sheet-cursor) :after 
	   (cursor (sheet standard-cursor-support))
  (declare (ignore cursor))
  (let ((port (port sheet)))
    (when port
      (when (eq sheet (top-of-trace (port-event-distributor port)))
	(ensure-cursor sheet)))))

(defmethod ensure-cursor ((sheet standard-cursor-support))
  (let ((port (port sheet)))
    (unless (eq (port-cursor port) (sheet-cursor sheet))
      (install-port-cursor port sheet (sheet-cursor sheet)))))

;;;
;;; Standard Polling Functions
;;;

(defclass standard-polling-support ()
    ())

(defmethod poll-pointer ((sheet standard-polling-support))
  (multiple-value-bind (x y state)
      (do-poll-pointer (port sheet) (fetch-mirrored-sheet sheet))
    (multiple-value-setq (x y)
      (untransform-point* (fetch-native-transformation sheet) x y))
    (values x y state)))

;;;
;;; Standard Input Contract
;;; 

(defclass standard-input-contract (input-contract 
				   standard-delivery
				   standard-event-distributor-part
				   standard-polling-support
				   standard-cursor-support)
    ())


(defmethod dispatch-event-using-port-keys
	   ((contract standard-input-contract) (port port) &rest keys)
  (declare (dynamic-extent keys))

  (flet ((queue-it (&rest keys 
			  &key event-key sheet time state
			  sheet-x sheet-y native-x native-y
			  &allow-other-keys)
	   (declare (dynamic-extent keys))
	   (let ((event (allocate-event (event-key->event-class event-key))))
	     (apply #'fill-event event 
		    sheet time sheet-x sheet-y native-x native-y state
		    keys)
	     (queue-event contract event))))
    (declare (dynamic-extent #'queue-it))    
    (apply #'filter-port-event-keys port contract #'queue-it keys)))


;;;
;;; Invoking Input Contract
;;;

(defclass invoking-input-contract (input-contract 
				   standard-event-distributor-part
				   standard-polling-support 
				   standard-cursor-support
				   input-handler)
    ())


(defmethod dispatch-event-using-port-keys
	   ((contract invoking-input-contract) (port port) &rest keys)
  (declare (dynamic-extent keys))
  (flet ((queue-it (&rest keys &key sheet-x sheet-y &allow-other-keys)
	   (declare (dynamic-extent keys))
	   ;; x, y for backwards compatibility.
	   (apply #'queue-input contract :x sheet-x :y sheet-y keys)))
    (declare (dynamic-extent #'queue-it))
    (apply #'filter-port-event-keys port contract #'queue-it keys)))


;;;
;;; Mute Input Contract
;;; 

(defclass mute-input-contract (input-contract
			       standard-event-distributor-part
			       standard-polling-support 
			       standard-cursor-support)
    ())

(defmethod dispatch-event-using-port-keys
	   ((contract mute-input-contract) (port port) &rest keys)
  (declare (dynamic-extent keys)
	   (ignore keys))
  nil)


(defmethod queue-event ((contract mute-input-contract) event 
			&key &allow-other-keys)
  (declare (ignore event)))


;;;
;;; Input Handler Protocol
;;;

(defclass input-handler ()
    ())

(defmethod queue-input ((input-handler input-handler) 
			&rest keys
			&key event-key button keysym char &allow-other-keys)
  (declare (dynamic-extent keys))
  (let ((handle-fn (ecase event-key
		     (:key-press 'key-press)
		     (:key-release 'key-release)
		     (:button-press 'button-press)
		     (:button-release 'button-release)
		     (:button-click 'button-click)
		     (:button-click-click 'button-click-click)
		     (:pointer-motion 'pointer-motion)
		     (:pointer-enter 'pointer-enter)
		     (:pointer-exit 'pointer-exit))))
    (ecase handle-fn
      ((button-press button-release button-click button-click-click) 
       (apply handle-fn input-handler button keys))
      ((key-press key-release)
       (apply handle-fn input-handler keysym char keys))
      ((pointer-motion pointer-enter pointer-exit)
       (apply handle-fn input-handler keys)))))

(defmethod pointer-enter ((input-handler input-handler) &key &allow-other-keys))

(defmethod pointer-exit ((input-handler input-handler) &key &allow-other-keys))

(defmethod pointer-motion
	   ((input-handler input-handler) &key &allow-other-keys))

(defmethod key-press
	   ((input-handler input-handler) keysym char &key &allow-other-keys)
  (declare (ignore keysym char)))

(defmethod key-release ((input-handler input-handler) keysym char
			&key &allow-other-keys)
  (declare (ignore keysym char)))

(defmethod button-press ((input-handler input-handler) button-name
			 &key &allow-other-keys)
  (declare (ignore button-name)))

(defmethod button-release ((input-handler input-handler) button-name
			   &key &allow-other-keys)
  (declare (ignore button-name)))

(defmethod button-click ((input-handler input-handler) button-name
			 &rest keys &key state &allow-other-keys)
  (declare (dynamic-extent keys))
  (apply #'button-press input-handler button-name keys)
  (apply #'button-release input-handler button-name 
	 :state (state-set state button-name) keys))

(defmethod button-click-click ((input-handler input-handler) button-name
			       &rest keys
			       &key &allow-other-keys)
  (declare (dynamic-extent keys))
  (apply #'button-click input-handler button-name keys)
  (apply #'button-click input-handler button-name keys))

