;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;;
;;;  EVENTS and EVENT QUEUES
;;;

;;;
;;; Event Queues
;;;

(defclass event-queue (locking-queue)
    ())

(define-constructor make-event-queue event-queue ())

(defmethod get-next-event ((queue event-queue) &key (timeout 0) last-event)
  (flet ((test-queue () (queue-next queue)))
    (when last-event (done-with-event last-event))
    (if (queue-next queue)
	(queue-get queue)
	(process-wait-with-timeout "Waiting for Event" timeout #'test-queue))))

(defmethod peek-event ((queue event-queue))
  (queue-next queue))

(defun done-with-event (event &optional (release-p t))
  (when release-p (free-event event)))

;;;
;;; Standard Delivery
;;;

(defclass standard-delivery ()
    ((event-queue :initform (make-event-queue) :initarg :event-queue
		  :accessor sheet-event-queue)))

(defmethod queue-event ((sheet standard-delivery) event 
			&key &allow-other-keys)
  (queue-put (slot-value sheet 'event-queue) event)
  event)

(defmethod get-next-event ((sheet standard-delivery) &key &allow-other-keys)
  (get-next-event (slot-value sheet 'event-queue)))

(defmethod peek-event ((sheet standard-delivery))
  (peek-event (slot-value sheet 'event-queue)))


;;;
;;; Simple Resource Facility for Events
;;;

(defvar *event-resource-table* (make-hash-table))

;;; A hash table associating vectors holding free events with the event type.
;;; The idea is to not cons on allocate/free.
(defun allocate-event (type)
  (multiple-value-bind (event-vector found-p)
      (gethash type *event-resource-table*)
    (unless found-p
      (setq event-vector 
	    (setf (gethash type *event-resource-table*)
		  (make-array 20 :fill-pointer 0 :adjustable t))))
    (if (zerop (fill-pointer event-vector))
	(make-instance type)
	(vector-pop event-vector))))

(defun free-event (event)
  ;; The gethash has to return an entry, since you have to allocate before
  ;; freeing.
  (let ((event-vector (gethash (type-of event) *event-resource-table*)))
    (vector-push-extend event event-vector)))

;;;
;;; Event Types
;;;

(defclass event ()
    ((key :allocation :class :reader event-key)
     (time :accessor event-time :reader event-timestamp)
     (sheet :accessor event-sheet)))

(defmacro define-event-type (name supers slots &rest options)
  (let ((slots (cons `(key :initform  
			   ,(let* ((str (string name))
				   (len (length str)))
			      (assert 
			       (equal "-EVENT" (subseq str (- len 6) len))
			       () "Event Type Name must end in -EVENT")
			      (intern (subseq (string name) 0 (- len 6))
				      "KEYWORD"))
			   :allocation :class)
		     slots)))
    `(defclass ,name ,supers ,slots ,@options)))

;;;
;;; Standard Windowing Events
;;;

(define-event-type windowing-event (event)
    ())

(define-event-type repaint-event (event)
    ((region :reader repaint-region)))

(defun make-repaint-event (&key sheet region)
  (let ((event (allocate-event 'repaint-event)))
    (%init-repaint event sheet region)
    event))

(defmethod %init-repaint ((event repaint-event) new-sheet new-region)
  (with-slots (sheet region) event
    (setf sheet new-sheet
	  region new-region)))

;;;
;;; Standard Input Events
;;;

(define-event-type device-event (event)
    ((native-x :accessor event-native-x)
     (native-y :accessor event-native-y)
     (x :accessor event-x)
     (y :accessor event-y)
     (state :accessor event-input-state)))

					; Keyboard Events
(define-event-type key-event (device-event) 
    ((char  :accessor event-char)
     (keysym :accessor event-keysym)))
(define-event-type key-press-event (key-event) ())
(define-event-type key-release-event (key-event) ())

					; Button Events
(define-event-type button-event (device-event) 
    ((button :accessor event-button)))
(define-event-type button-press-event (button-event) ())
(define-event-type button-release-event  (button-event) ())
(define-event-type button-click-event (button-event) ())
;;(define-event-type button-click-hold-event (button-event) ()) ???
(define-event-type button-click-click-event (button-event) ())

					; Pointer Events
(define-event-type pointer-event (device-event) 
    ((direction :accessor event-direction :initform :none)))
(define-event-type pointer-motion-event (pointer-event) ())
(define-event-type pointer-enter-event (pointer-event) ())
(define-event-type pointer-exit-event (pointer-event) ())

(defvar *event-key->event-class* 
  ;; These are now completely regular !!
  '(:key-press          key-press-event   
    :key-release        key-release-event
    :button-press       button-press-event
    :button-release     button-release-event
    :button-click       button-click-event
    :button-click-click button-click-click-event
    :pointer-motion     pointer-motion-event
    :pointer-enter      pointer-enter-event
    :pointer-exit       pointer-exit-event))

(defun-inline event-key->event-class (event-key)
  (getf *event-key->event-class* event-key))

(defmacro deffiller (type extra-fields)
  (let* ((extra-vars
	  (mapcar #'(lambda (field) (list (gensym) field))
		  extra-fields))
	 (extra-keyword-list
	  (mapcar #'(lambda (arg-name) 
		      (list arg-name nil (gensymbol arg-name "P")))
		  extra-fields)))
    `(progn (defmethod fill-event ((event ,type) 
				   sheet time
				   sheet-x sheet-y
				   native-x native-y
				   state 
				   &key ,@extra-keyword-list
				   &allow-other-keys)
	      (with-slots ((event-time time)
			   (event-native-x native-x)
			   (event-native-y native-y)
			   (event-x x)
			   (event-y y)
			   (event-input-state state)
			   (event-sheet sheet)
			   ,@extra-vars)
		  event
		(setf event-time time
		      event-native-x native-x
		      event-native-y native-y
		      event-x sheet-x
		      event-y sheet-y
		      event-sheet sheet
		      event-input-state state)
		,@(mapcar #'(lambda (slot-spec keyword-spec)
			      `(when ,(third keyword-spec)
				 (setf ,(first slot-spec) ,(second slot-spec))))
			  extra-vars extra-keyword-list))))))

(deffiller button-event (button))
(deffiller key-event (char keysym))
(deffiller pointer-event (direction))

;;;
;;; Mouse Buttons and Modifier States
;;;

(deftype button-name ()
  ""
  '(member :left :middle :right))
					; Button and Modifier State Masks
(deftype shift-keysym ()
  ""
  '(member :left-shift :right-shift :shift))

(deftype control-keysym ()
  ""
  '(member :left-control :right-control :control))

(deftype meta-keysym ()
  ""
  '(member :left-meta :right-meta :meta))

(deftype super-keysym ()
  ""
  '(member :left-super :right-super :super))

(deftype hyper-keysym ()
  ""
  '(member :left-hyper :right-hyper :hyper))

(deftype lock-keysym ()
  ""
  '(member :caps-lock :shift-lock :mode-lock))

(deftype modifier-keysym ()
  ""
  '(or shift-keysym control-keysym meta-keysym super-keysym hyper-keysym lock-keysym))

(deftype numeric-keysym ()
  ""
  '(member :\0 :\1 :\2 :\3 :\4 :\5 :\6 :\7 :\8 :\9))

(defun numeric-keysym-number (keysym)
  (position keysym #(:\0 :\1 :\2 :\3 :\4 :\5 :\6 :\7 :\8 :\9)))

(defconstant +pointer-left-bit+   8)
(defconstant +pointer-middle-bit+ 9)
(defconstant +pointer-right-bit+  10)
(defconstant +hyper-key-bit+    5)
(defconstant +super-key-bit+    4)
(defconstant +meta-key-bit+     3)
(defconstant +control-key-bit+  2)
(defconstant +shift-key-bit+    0)

(defconstant +pointer-left+   (ash 1 +pointer-left-bit+))
(defconstant +pointer-middle+ (ash 1 +pointer-middle-bit+))
(defconstant +pointer-right+  (ash 1 +pointer-right-bit+))
(defconstant +hyper-key+    (ash 1 +hyper-key-bit+))
(defconstant +super-key+    (ash 1 +super-key-bit+))
(defconstant +meta-key+     (ash 1 +meta-key-bit+))
(defconstant +control-key+  (ash 1 +control-key-bit+))
(defconstant +shift-key+    (ash 1 +shift-key-bit+))

(defvar symbolic->mask `(
			 :left    ,+pointer-left+
			 :right   ,+pointer-right+
			 :middle  ,+pointer-middle+
			 :hyper   ,+hyper-key+
			 :super   ,+super-key+
			 :meta    ,+meta-key+
			 :control ,+control-key+
			 :shift   ,+shift-key+))

(defvar *shifts* `(:hyper ,+hyper-key+
		   :super ,+super-key+
		   :meta  ,+meta-key+
		   :control ,+control-key+
		   :shift ,+shift-key+))


(defvar +n-shift-keys+ (floor (length symbolic->mask) 2))

(defun symbolic->mask (sym)
  (getf symbolic->mask sym sym))
    
(defmacro state-match-p (state &rest clauses)
  `(and ,@(with-collection
	      (dolist (clause clauses)
		(collect (compile-clause clause state))))))

(defmethod compile-clause ((clause symbol) state)
  (let ((constant (symbolic->mask clause)))
    (when constant (setq clause constant))
    `(= (logand ,state ,clause) ,clause)))

(defmethod compile-clause ((clause list) state)
  (ecase (first clause)
    (not       `(not ,(compile-clause (second clause) state)))
    ((or and)  `(,(first clause)
		  ,@(mapcar #'(lambda (clause) (compile-clause clause state))
			    (rest clause))))))

(defun state-set (state key)
  (logior state (symbolic->mask key)))

(defun state-unset (state key)
  (logand state (lognot (symbolic->mask key))))

(defun state-after (state type button)
  (case type
    (:key-press )
    (:key-release )
    ((:button-press :button-release :button-click-hold) 
     (logxor state (symbolic->mask button)))
    ((:button-click :button-click-click) state)
    (otherwise nil)))


;;;
;;; Canonical Gestures and shift masks
;;;

;;; Canonical gesture specs look like:
;;; (keysym . {shift-mask | shifts})
;;; where keysym is either a defined keysym name, such as :A or :RETURN,
;;; shift-mask is a bit-mask
;;; and shifts are keywords from the set :SHIFT, :CONTROL, :META, :HYPER,
;;; :SUPER. 
;;;

(defun-inline state->shift-mask (state)
  ;; Get just the shift-mask part of state
  (logand state #xFF))

(eval-when (eval load compile)

(defun shift-code (shift)
  (getf *shifts* shift))

(defun shift-code-shift (code)
  (let ((position (position code *shifts*)))
    (if position
	(elt *shifts* (1- position))
	(error "~S not a valid shift key" code))))


;;; Indicies are not bit numbers, but are dense, for
;;; use in looking up in tables.

(defun shift-index (shift)
  (let ((position (position shift *shifts*)))
    (if (and position (evenp position))
	(values (floor position 2))
	(error "~S is not a valid shift key." shift))))

(defun shift-index-shift (index)
  (elt *shifts* (* 2 index)))

;;; shift-masks can be compared with =
(defun make-shift-mask (&rest shifts)
  (declare (dynamic-extent shifts))
  (let ((mask 0))
    (dolist (shift shifts)
      (let ((code (shift-code shift)))
	(setf mask (logior mask code))))
    mask))

;;; This is so that wherever we say things like (make-shift-mask :shift)
;;; it gets expanded at compile time.  That's why the above is made
;;; available at compile time, too.
(define-compiler-macro make-shift-mask (&whole whole-form &rest shifts)
  (if (every #'constantp shifts)
      (apply #'make-shift-mask shifts)
      whole-form))

) ;; End of EVAL-WHEN

(defun shift-mask-shifts (shift-mask)
  ;; This is only called for debugging.
  (let ((shifts nil))
    (dotimes (i 7)
      (let ((code (ash 1 i)))
	(when (not (zerop (logand shift-mask code)))
	  (let ((shift (shift-code-shift code)))
	    (push shift shifts)))))
    (nreverse shifts)))

;;; Typically only called at COMPILE or DEBUG time, so it's ok if it conses.
(defun parse-gesture-spec (gesture-spec)
  (if (or (atom gesture-spec) 
	  (and (cdr gesture-spec)
	       (atom (cdr gesture-spec))))
      
      gesture-spec
      (let ((button (find-if-not #'shift-code gesture-spec)))
	(unless button
	  (cerror "Assume :LEFT and go on."
		  "Gesture spec missing a keysym: ~S" gesture-spec)
	  (setq button ':left))
	(cons button (apply #'make-shift-mask (remove button gesture-spec))))))

(defun gesture-spec-p (gesture-spec)
  ;; atoms are gesture specs, or lists of atoms consisting
  ;; of at most one non-shift.
  (or (atom gesture-spec)
      (let ((non-shift-count 0))
	(dolist (entry gesture-spec)
	  (unless (atom entry) (return-from gesture-spec-p nil))
	  (unless (typep entry 'modifier-keysym)
	    (incf non-shift-count)
	    (when (> non-shift-count 1) (return-from gesture-spec-p nil))))
	t)))

(defun print-gesture-spec (gesture-spec stream)
  ;; this is allowed to be slow?
  (setq gesture-spec (parse-gesture-spec gesture-spec))
  (cond ((atom gesture-spec)
	 (format stream "~:(~A~)" gesture-spec))
	(t
	 (print-shift-mask (cdr gesture-spec) stream)
	 (format stream "~:(~A~)" (car gesture-spec)))))

(defun print-shift-mask (shift-mask stream)
  (macrolet ((do-shift (shift-name pretty-name)
	       `(when (not (zerop (logand shift-mask (shift-code ,shift-name))))
		  (write-string ,pretty-name stream))))
    
    ;; --- This should really iterate over all the defined shifts in *shifts*,
    ;; rather than hard-wiring them here, but that's just too much trouble.
    
    (do-shift :hyper "Hyper-")
    (do-shift :super "Super-")
    (do-shift :control "Control-")
    (do-shift :meta "Meta-")
    (do-shift :shift "Shift-"))
  (values))

(defun key-event-matches-gesture-spec (key-event gesture-spec)
  (keysym-and-shift-mask-matches-gesture-spec
    (event-keysym key-event)
    (event-input-state key-event)
    gesture-spec
    (port (event-sheet key-event))))

(defun input-matches-gesture-spec-p
    (keysym shift-mask gesture-spec &optional port)
  ;; Otherwise it's just too damed hard to use.
  (setq gesture-spec (parse-gesture-spec gesture-spec))
  
  ;; nice if this could be under control of a speed/safety compiler optimizer.
  (unless (integerp shift-mask)
    (setq shift-mask (apply #'make-shift-mask shift-mask)))
  
  ;; just in case, we'll mask out the high-level state part of this.
  ;; How much extra work is these two forms?
  (setq shift-mask (state->shift-mask shift-mask))
  
  (fast-input-matches-gesture-spec-p keysym shift-mask gesture-spec port))
    
(defun fast-input-matches-gesture-spec-p
    (keysym shift-mask gesture-spec &optional port)

  ;; No ERROR Checking
  
  (or (if (atom gesture-spec)
	  (eql keysym gesture-spec)
	  (and (eql keysym (car gesture-spec))
	       (= shift-mask (cdr gesture-spec))))
      (and port
	   (let ((canonical-gesture-spec 
		  (get-port-canonical-gesture-spec gesture-spec port)))
	     (or (and (eql keysym (car canonical-gesture-spec))
		      (= shift-mask (cdr canonical-gesture-spec)))
		 (equal canonical-gesture-spec
			(get-port-canonical-gesture-spec
			 (cons keysym shift-mask) port)))))))

(defmacro gesture-case ((keysym state port) &rest clauses)
  ;; ??? Doesn't generate very efficient code, but fine for now

  (with-gensyms (state-var)
    
    (labels ((collect-test-part (test-part)
	       (cond
		 ((or (eq test-part t)
		      (eq test-part 'otherwise))
		  `t)
		 ((and (listp test-part)
		       (eq (car test-part) 'or))
		  `(or ,@(with-collection
			     (dolist (test (cdr test-part))
			       (collect (collect-test-part test))))))
		 (t `(fast-input-matches-gesture-spec-p
		      ,keysym ,state-var
		      ',(parse-gesture-spec test-part)
		      ,port)))))
							
      (let ((cond-clauses 
	     (with-collection
		 (dolist (clause clauses)
		   (collect (cons (collect-test-part (car clause))
				  (cdr clause)))))))
	`(let  ((,state-var (state->shift-mask ,state)))
	   (cond ,@cond-clauses))))))

#||

(gesture-case (keysym  state (port pane))
  ((or :backspace :rubout) clause-1 )
  (:return clause-2)
  ((:control :u) clause-3)
  (otherwise otherwise-clause))

||#










