;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;; Ports are responible for realizing images, inks, cursors, text styles and
;;; other such resources.
;;;
;;; The stuff below provides a number of generic facilities for supporting
;;; this.

;;;
;;; IMAGEs
;;;

(defmethod realize-pixmap :around (port (pixmap pixmap))
  (or (getf (port-pixmaps pixmap) port)
      (setf (getf (port-pixmaps pixmap) port)
	    (call-next-method))))

(defmethod read-pixmap (sheet rectangle &optional old)
  (with-bounding-rectangle* (min-x min-y max-x max-y) rectangle
    (read-pixmap* sheet min-x min-y (- max-x min-x) (- max-y min-y) old)))

(defmethod read-pixmap* (sheet min-x min-y width height &optional old)
  (if old 
      (unless (and (eq (pixmap-width old) width)
		   (eq (pixmap-height old) height))
	(setf (pixmap-width old) width)
	(setf (pixmap-height old) height)
	(do* ((tail (port-pixmaps old) (cddr tail))
	      (port (car tail) (car tail))
	      (pobj (cadr tail) (cadr tail)))
	     ((null tail))
	  (unrealize-pixmap port pobj))
	(setf (port-pixmaps old) nil))
      (setq old (make-pixmap :width width :height height)))
  
  (using-clim-medium (medium sheet)
    (copy-area old 0 0 medium min-x min-y width height))
  
  old)

;;;
;;; INKs
;;;

(defmethod realize-shade :around ((port port) shade)
  (or (cdr (assoc shade (port-prop port :shades)))
      (let ((pixmap (call-next-method)))
	(push (cons shade pixmap) (port-prop port :shades))
	pixmap)))

(defun color-equal (c1 c2)
  (or (eql c1 c2)
      (and 
       ;; do we know that they are colors?
       (typep c1 'color)
       (typep c2 'color)
       (= (color-red-value c1) (color-red-value c2))
       (= (color-blue-value c1) (color-blue-value c2))
       (= (color-green-value c1) (color-green-value c2)))))

(defmethod realize-color :around ((port port) color)
  ;; This method is large and complex because it optimizes for speed
  ;;
  ;; Can't use EQL in general because color objects aren't interned.  But
  ;; oftentimes the recent colors are used, so EQL is a good first test.
  ;; We implement this as a cache of fixed size, using RLU replacement.
  ;; The cache is filled from a backing store that is saved to avoid
  ;; possible recreation of colors in the port.  (Ideally, there should
  ;; be a way to determine if the underlying port has cheap colors or
  ;; expensive colors.  Ideally, that would be a port-specific around
  ;; method, but in this case the :around methods would be in the wrong
  ;; order.)
  ;;
  ;; The structure of the cache is
  ;;    (tick ((color tick host-color) (color tick host-color) ...) (backing-store bs-tail))
  ;; The tick is the RLU tick for the cache.  The backing store is an
  ;; ASSOC list, and bs-tail is the tail of that so that new colors get
  ;; put on the end quickly, on the assumption that early colors are
  ;; more common and want to be faster when it gets larger.
  ;;
  (macrolet ((cache (port)
	       `(or (port-prop ,port :realize-color-cache)
		    (setf (port-prop ,port :realize-color-cache)
			  (list 0
				(do ((count 20 (1- count))
				     (data '() (cons (list nil 0 nil) data)))
				    ((zerop count) data))
				(list nil nil)))))
	     (next-tick (cache) `(incf (first ,cache)))
	     (cache-data (cache) `(second ,cache))
	     (entry-color (entry) `(first ,entry))
	     (entry-tick (entry) `(second ,entry))
	     (entry-host-color (entry) `(third ,entry))
	     (cache-backing (cache) `(third ,cache))
	     (backing-alist (backing) `(first ,backing))
	     (backing-tail (backing) `(second ,backing))
	     (check-cache (data test next-tick)
	       `(do ((data ,data (cdr data)))
		    ((null data))
		  (let* ((entry (first data))
			 (test-color (entry-color entry)))
		    (when (and test-color (,test test-color color))
		      (setf (entry-tick entry) ,next-tick)
		      (return (entry-host-color entry))))))
	     (oldest-cache-entry (data)
	       `(let ((oldest nil)
		      (oldest-tick nil))
		  (do ((data ,data (cdr data)))
		      ((null data))
		    (let* ((entry (first data))
			   (tick (entry-tick entry)))
		      (when (or (null oldest-tick) (< tick oldest-tick))
			(setq oldest entry
			      oldest-tick tick))))
		  oldest))
	     (check-backing-store (alist test)
	       `(do ((alist ,alist (cdr alist)))
		    ((null alist))
		  (let ((entry (first alist)))
		    (when (,test (car entry) color)
		      (return (cdr entry))))))
	     (compute-and-add-to-backing-store (backing)
	       `(let* ((backing ,backing)
		       (tail (backing-tail backing))
		       (host-color (call-next-method))
		       (entry (cons color host-color))
		       (new-tail (cons entry nil)))
		  (if tail
		      (setf (cdr tail) new-tail
			    (backing-tail backing) new-tail)
		      (setf (backing-alist backing) new-tail
			    (backing-tail backing) new-tail))
		  host-color))
	     )
    (let* ((cache (cache port))
	   (data (cache-data cache))
	   (next-tick (next-tick cache)))
      (or (check-cache data EQL next-tick)
	  (check-cache data COLOR-EQUAL next-tick)
	  (let ((oldest (oldest-cache-entry data)))
	    (setf (entry-tick oldest) next-tick)	;own it now, in a sense
	    (let* ((backing (cache-backing cache))
		   (alist (backing-alist backing))
		   (host-color (or (check-backing-store alist EQL)
				   (check-backing-store alist COLOR-EQUAL)
				   (compute-and-add-to-backing-store backing))))
	      (setf (entry-color oldest) nil	;invalidate key 
		    (entry-host-color oldest) host-color	;fill in association
		    (entry-color oldest) color)	;validate key
	      host-color)))))
  #+Old-code
  ;; We can't just test for the EQ color object, since the application
  ;; may be consing new ones all the time.  So, compare the values.
  (let ((entry (assoc color (port-prop port :color-table)
		      :test #'color-equal)))
    (if entry
	(cdr entry)
      (let ((host-color (call-next-method)))
	(push (cons color host-color)
	      (port-prop port :color-table))
	host-color))))

;;;
;;; CURSORs
;;;

(defmethod available-cursors ((port port))
  *cursors*)




;;;
;;; TEXT STYLE REALIZATION Generic Machinery
;;;

#||
;;; This is what we want to be able to do:

(define-text-style-mappings genera-port *standard-character-set*
  (:font-for-undefined-style fonts:boxfont)
  (:family :fix (:face :roman (:size :very-large fonts:bigfnt
				     :large fonts:medfnt
				     :normal fonts:cptfont
				     :small fonts:tvfont
				     :very-small fonts:einy7
				     :tiny fonts:tiny)
		       :italic (:size :very-large fonts:bigfnti
				      :large fonts:medfnti
				      :normal fonts:cptfonti
				      :small fonts:tvfonti
				      :very-small fonts:einy7
				      :tiny fonts:tiny)
		       :bold (:size :very-large fonts:bigfntb
				    :large fonts:medfntb
				    :normal fonts:tvfontcb
				    :small fonts:tvfontb
				    :very-small fonts:einy7
				    :tiny fonts:tiny)
		       (:bold :italic) (:size :very-large fonts:bigfntbi
					      :large fonts:medfntbi
					      :normal fonts:cptfontbi
					      :small fonts:tvfontbi
					      :very-small fonts:einy7
					      :tiny fonts:tiny)
		       (:bold :extended) (:size :normal fonts:cptfontb
						:small fonts:tvfontb)
		       :condensed (:size :normal fonts:cptfontc)
		       (:extra :condensed) (:size :normal fonts:cptfontcc
						  :small fonts:tvfont))))
||#
;;; [ 1/4/90 naha -- determined by reading DEFINE-TEXT-STYLE-MAPPINGS-LOAD-TIME ]
;;; each element of SPECS can be one of
;;;     `(:style ,family ,face ,size)   [what will this do?]
;;;     `(:family ,family ,specs)       ==> recurse on specs for specified family
;;;     `(:face ,face ,specs)           ==> recurse on specs for specified face
;;;     `(:size ,size ,specs)           ==> recurse on specs for specified size
;;;     the name of a font to map the specified text style to.

(defmacro define-text-style-mappings (device character-set &body specs)
  `(define-text-style-mappings-load-time ,device ,character-set ',specs))

(defun define-text-style-mappings-load-time (device character-set specs)
  (labels ((load-specs (family face size specs)
	     (when (and (consp specs) (eql (first specs) :style))
	       (setf specs (apply #'make-text-style (rest specs))))
	     (if (and (consp specs) (not (eql (car specs) :style)))
		 (do* ((type (first specs))
		       (my-specs (cdr specs) (cddr my-specs))
		       (value (first my-specs) (first my-specs))
		       (rest (second my-specs) (second my-specs)))
		      ((null my-specs))
		   (case type
		     (:family (setf family value))
		     (:face (setf face value))
		     (:size (setf size value))
		     (otherwise (warn "Ill-formed mapping contains ~S" specs)))
		   (load-specs family face size rest))
		 (if (and family face size)
		     (add-text-style-mapping
		       device character-set
		       (make-text-style family face size) specs)
		     (error "Can't do [~A.~A.~A]" family face size)))))
    (dolist (spec specs)
      (load-specs nil nil nil spec))))

(defmethod add-text-style-mapping
    ((device port) character-set style result)
  (setq style (standardize-text-style device character-set style))
  (when (listp result)
    (assert (eql (first result) :style) ()
	    "Text style mappings must be atomic font names ~
	     or (:STYLE . (family face size))")
    (setf result (parse-text-style (cdr result))))
  (setf (gethash style (slot-value device 'text-style-mapping-table)) result))
 
;;; This is broken up into two methods so any :AROUND method will only
;;; be called on the outermost recursion.
(defmethod text-style-mapping
	   ((device port) character-set style)
  (text-style-mapping* device character-set style))

(defmethod text-style-mapping
	   ((device port) character-set (style device-font))
  (declare (ignore character-set))
  ;;--- What about character-set when using device fonts?
  ;;--- EQL? TYPE-EQUAL?  This is too restrictive as it stands
  (unless (eql device (device-font-display-device style))
    (error "An attempt was made to map device font ~S on device ~S, ~@
	    but it is defined for device ~S"
	   style device (device-font-display-device style)))
  (device-font-name style))

(defmethod text-style-mapping* ((device port) character-set style)
  (setq style (standardize-text-style device character-set (parse-text-style style)))
  (let* ((mapping-table (slot-value device 'text-style-mapping-table))
	 (result (or (gethash style mapping-table)
		     (gethash *undefined-text-style* mapping-table))))
    (when (typep result 'text-style)	;Logical translations
      (setf result (text-style-mapping* device character-set result)))
    result))

(defmethod text-style-mapping-exists-p ((device port) character-set style)
  (setq style (standardize-text-style device character-set (parse-text-style style)))
  (let* ((mapping-table (slot-value device 'text-style-mapping-table))
	 (result (gethash style mapping-table)))
    (cond ((null result) nil)
	  ((typep result 'text-style)				;logical translations
	   (text-style-mapping-exists-p device character-set style))
	  (t t))))

;;; This method allows the device to convert logical sizes into point
;;; sizes, etc.  The default method doesn't do anything.
(defmethod standardize-text-style ((device port) character-set style)
  (declare (ignore character-set))
  (unless (numberp (text-style-size style))
    (standardize-text-style-error style))
  style)

(defun standardize-text-style-error (style)
  (if (fully-merged-text-style-p style)
      (cerror "Use the undefined text style stand-in instead"
	      "The size component of ~S is not numeric.  This display-device does not know ~
	       how to map logical text style sizes"
	      style)
      (cerror "Use the undefined text style stand-in instead"
	      "The text style ~S must be a fully merged text style"
	      style))
  *undefined-text-style*)

;; For use by more specific STANDARDIZE-TEXT-STYLE methods
(defun-inline standardize-text-style-1 (display-device style character-set size-alist)
  (declare (ignore display-device character-set))
  (let ((size (text-style-size style)))
    (if (numberp size)
	style
	(let ((new-size (assoc size size-alist)))
	  (cond (new-size
		 (make-text-style (text-style-family style)
				  (text-style-face style)
				  (second new-size)))
		(t
		 (standardize-text-style-error style)))))))

;;; The common lisp character extensions are in the silica/text-style file for some reason.
