;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

(defgeneric regionp (object))
(defgeneric pointp (object))
(defgeneric rectanglep (object))

(defgeneric bounding-rectangle* (region))
(defgeneric* (setf bounding-rectangle*) (new-min-x new-min-y new-max-x new-max-y region))
(defgeneric bounding-rectangle (region)) ;;; What about (setf bounding-rectangle)?

;;; Compiler macro to optimize the case where the user asks for the
;;; bounding rectangle of a rectangle s/he has just constructed.  This
;;; is not entirely as ridiculuous as it may seem, since it could be the
;;; result of a compiler expansion.  Currently, the only thing which
;;; does this is DRAW-RECTANGLE-WITH-RECTANGLE, but there is another
;;; optimizer for POINT-POSITION (see below) which takes care of the
;;; (more common) case of the macro expansion of taking the position of
;;; a point the user has just consed.

;;; Comment on the code this compiler macro generates in the case of
;;; MAKE-POINT:  The contract of BOUNDING-RECTANGLE* returns four
;;; values.  In many cases, users call BOUNDING-RECTANGLE* on points and
;;; discard the first two values.  I hope the compilers are smart about
;;; MULTIPLE-VALUE-BIND.  They should turn
;;;
;;;   (MULTIPLE-VALUE-BIND (A B) (VALUES C D C D) ...)
;;; into
;;;   (LET ((A C) (B D)) ...)
;;;
;;; and drop the other two references on the floor, but I don't know if they actualy do this.

(define-compiler-macro bounding-rectangle* (&whole original-form region)
  (macrolet ((return-original ()
	       '(return-from bounding-rectangle* original-form))
	     (return (value)
	       `(return-from bounding-rectangle* ,value)))
    (when (atom region) (return-original))
    (case (first region)
      (make-rectangle*
	(unless (= (length region) 5) (return-original)) ;; Don't mess with :REUSE
	(let ((X1 (second region))
	      (Y1 (third region))
	      (X2 (fourth region))
	      (Y2 (fifth region)))
	  (return `(values ,X1 ,Y1 ,X2 ,Y2))))
      (make-point
	(unless (= (length region) 3) (return-original))
	(let ((X (second region))
	      (Y (third region))
	      (XV (gensymbol 'X))
	      (YV (gensymbol 'Y)))
	  (return `(let ((,XV ,X) (,YV ,Y))
		     (values ,XV ,YV ,XV ,YV)))))
      (make-rectangle
	(unless (= (length region) 3) (return-original))
	(let ((MIN-POINT (second region))
	      (MAX-POINT (third region))
	      (X1V (gensymbol 'X1))
	      (Y1V (gensymbol 'Y1))
	      (X2V (gensymbol 'X2))
	      (Y2V (gensymbol 'Y2)))
	  ;; If the program has (make-rectangle (make-point ...) ...),
	  ;; this will use the POINT-POSITION expander to improve that
	  ;; part of the code.
	  (return `(multiple-value-bind (,X1V ,Y1V)
		       (point-position ,MIN-POINT)
		     (multiple-value-bind (,X2V ,Y2V)
			 (point-position ,MAX-POINT)
		       (values ,x1v ,y1v ,x2v ,y2v))))))
      (otherwise (return-original)))))

(defgeneric copy-region (region &key reuse))

(defgeneric region-empty-p (region))
(defgeneric region-contains-region-p (region1 region2))
(defgeneric region-contains-point*-p (region point-x point-y))
(defgeneric region-intersects-region-p (region1 region2))

(defgeneric region-union (region1 region2 &key reuse))	;Should this be &KEY?  omitted?
(defgeneric region-intersection (region1 region2 &key reuse)) ; ditto.
(defgeneric region-difference (region1 region2 &key reuse))

;;;
;;; SILICA REGION
;;;
;;; This file contains the geometry kernel i.e. basic gemetric objects and
;;; operations needed by the Silica kernel and the ABC graphics package.  
;;;

;;;
;;; REGIONS
;;;

(defclass region ()
    ()
  (:documentation "An arbitrarily shaped region."))

(defmethod regionp ((region region)) t)
(defmethod regionp ((object t)) nil)

;;;
;;; Bounding Rectangle Protocol
;;;

(defmacro with-bounding-rectangle* ((x1 y1 &optional x2 y2) region &body body)
  `(multiple-value-bind (,x1 ,y1 ,@(when x2 (list x2 y2))) 
       (bounding-rectangle* ,region) 
     ,@body))

(define-unimplemented-protocol-method bounding-rectangle* region ((region region)))

(defmethod bounding-rectangle ((region region))
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (make-rectangle* x1 y1 x2 y2)))

(defun bounding-rectangle-position (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x2 y2))
    (%make-point x1 y1)))

(defun bounding-rectangle-position* (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x2 y2))
    (values x1 y1)))

(defun bounding-rectangle-min-point (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x2 y2))
    (%make-point x1 y1)))

(defun bounding-rectangle-min-x (region) 
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore y1 x2 y2))
    x1))

(defun bounding-rectangle-min-y (region) 
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x1 x2 y2))
    y1))

(defun bounding-rectangle-max-point (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x1 y1))
    (%make-point x2 y2)))

(defun bounding-rectangle-max-x (region) 
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x1 y1 y2))
    x2))

(defun bounding-rectangle-max-y (region) 
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x1 y1 x2))
    y2))

(defun bounding-rectangle-dimensions (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (values (- x2 x1) (- y2 y1))))

(defun bounding-rectangle-width (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore y1 y2))
    (- x2 x1)))

(defun bounding-rectangle-height (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x1 x2))
    (- y2 y1)))

;;; This latest implementation has some naming differences from the
;;; last printed spec (and therefore the 0.9 reference document).
;;; So, I have defined synonyms for the documented functions by way
;;; of "compatibility".  Hopefully we can straighten this out before
;;; release 1.0.  -York 7/16/90

(defun bounding-rectangle-left (region)
  (bounding-rectangle-min-x region))

(defun bounding-rectangle-top (region)
  (bounding-rectangle-min-y region))

(defun bounding-rectangle-right (region)
  (bounding-rectangle-max-x region))

(defun bounding-rectangle-bottom (region)
  (bounding-rectangle-max-y region))

(defun bounding-rectangle-size (region)
  (bounding-rectangle-dimensions region))

;;;
;;; Points
;;;

(defclass point (region)
    ())
    
(defclass standard-point (point)
    ((x :initform 0 :initarg :x :type number :accessor point-x)
     (y :initform 0 :initarg :y :type number :accessor point-y))
  (:documentation "A point in space"))

(define-constructor %make-point standard-point (x y) :x x :y y)

(defmethod pointp ((point point)) t)
(defmethod pointp ((object t)) nil)

(defmethod %load-point (x y (point standard-point))
  (with-slots ((px x) (py y)) point
    (setf px x py y)
    point))

(defun make-point (x y &key reuse)
  (if (or (null reuse) (not (pointp reuse)))
      (%make-point x y)
      (%load-point x y reuse)))

(defmethod copy-region ((point standard-point) &key reuse)
  (with-slots (x y) point
    (make-point x y :reuse reuse)))

(defmethod bounding-rectangle* ((point standard-point))
  (with-slots (x y) point
    ;; ??? Is this kind of speedup worth it.
    (let ((point-x x) (point-y y))
      (values point-x point-y point-x point-y))))

(defmethod point-position ((point standard-point))
  (with-slots (x y) point
    (values x y)))

;;; This compiler macro is so users can use the unspread versions of the
;;; graphics functions, CONS points as they go, and not actually pay the
;;; performance penalty of doing so.  Consider, for example,
;;;
;;;   (DRAW-LINE STREAM (MAKE-POINT X1 Y1) (MAKE-POINT X2 Y2) ...)
;;;
;;; The compiler macro on DRAW-LINE turns this into:
;;;   (LET ((#:MEDIUM-84 STREAM))
;;;     (MULTIPLE-VALUE-BIND (FROM-X FROM-Y)
;;;	    (POINT-POSITION (MAKE-POINT X1 Y1))
;;;       (MULTIPLE-VALUE-BIND (TO-X TO-Y)
;;;	      (POINT-POSITION (MAKE-POINT X2 Y2))
;;;	    (DRAW-LINE*-INTERNAL #:MEDIUM-84 FROM-X FROM-Y TO-X TO-Y ...))))
;;;
;;; This compiler macro will rewrite this code as:
;;;   (LET ((#:MEDIUM-84 STREAM))
;;;     (MULTIPLE-VALUE-BIND (FROM-X FROM-Y)
;;;         (VALUES X1 Y1)
;;;       (MULTIPLE-VALUE-BIND (TO-X TO-Y)
;;;           (VALUES X2 Y2)
;;;         (DRAW-LINE*-INTERNAL #:MEDIUM-84 FROM-X FROM-Y TO-X TO-Y))))
;;;
;;; This is still not as good as possible, but it's the best we can do
;;; in finite programmer time.

(define-compiler-macro point-position (&whole original-form point)
  (macrolet ((return-original ()
	       '(return-from point-position original-form))
	     (return (value)
	       `(return-from point-position ,value)))
    (when (atom point) (return-original))
    (case (first point)
      (make-point
	(unless (= (length point) 3) (return-original)) ;; Don't mess with :REUSE
	(let ((X (second point))
	      (Y (third point)))
	  (return `(values ,X ,Y))))
      (otherwise (return-original)))))

;;;
;;; Infinite Region
;;;

(defclass everywhere (region)
    ()
  (:documentation "All points on a 2-d everywhere."))

(defvar +everywhere+ (make-instance 'everywhere))

;;; Return the canonical region 
(defmethod copy-region ((region everywhere) &key reuse)
  (declare (ignore reuse))
  +everywhere+)

(defmethod bounding-rectangle ((region everywhere))
  region)

(defclass nowhere (region)
    ())

(defvar +nowhere+ (make-instance 'nowhere))

;;; Return the canonical region 
(defmethod copy-region ((region nowhere) &key reuse)
  (declare (ignore reuse))
  +nowhere+)

(defmethod bounding-rectangle ((region nowhere))
  region)

;;;
;;; Rectangular region.
;;;

(defclass rectangle (region)
    ())

(defclass standard-rectangle (rectangle)
    ((min-x :initform 0 :initarg :min-x :type number)
     (min-y :initform 0 :initarg :min-y :type number)
     (max-x :initform 0 :initarg :max-x :type number)
     (max-y :initform 0 :initarg :max-y :type number))
  (:documentation "A Rectangular Region."))

(defmethod rectanglep ((rectangle rectangle)) t)
(defmethod rectanglep ((object t)) nil)

(define-constructor %make-rectangle standard-rectangle (min-x min-y max-x max-y)
		    :min-x min-x :min-y min-y 
		    :max-x max-x :max-y max-y)

(defmethod %load-rectangle (x y x2 y2 (rectangle standard-rectangle))
  (with-slots (min-x min-y max-x max-y) rectangle
    (setf min-x x min-y y max-x x2 max-y y2)
    rectangle))

(defun make-rectangle* (min-x min-y max-x max-y &key reuse)
  (if (and reuse (rectanglep reuse))
      (%load-rectangle min-x min-y max-x max-y reuse)
      (%make-rectangle min-x min-y max-x max-y)))

(defun make-rectangle (min-point max-point &key reuse)
  (with-bounding-rectangle* (min-x min-y) min-point
    (with-bounding-rectangle* (max-x max-y) max-point
      (make-rectangle*  min-x min-y max-x max-y :reuse reuse))))

(defmethod copy-region ((rectangle standard-rectangle) &key reuse)
  (with-slots (min-x min-y max-x max-y) rectangle
    (if (and reuse (rectanglep reuse))
	(%load-rectangle min-x min-y max-x max-y reuse)
	(%make-rectangle min-x min-y max-x max-y))))

(defmethod bounding-rectangle* ((rectangle standard-rectangle))
  (with-slots (min-x min-y max-x max-y) rectangle
    (values min-x min-y max-x max-y)))

(defmethod* (setf bounding-rectangle*) 
	   (x y x2 y2 (rectangle standard-rectangle))
  (with-slots (min-x min-y max-x max-y) rectangle
    (when x  (setf min-x x))
    (when y  (setf min-y y))
    (when x2 (setf max-x x2))
    (when y2 (setf max-y y2))
    (values x y x2 y2)))

;;;
;;; Rectangle Convienence Functions
;;;

(defun rectangle-position (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x2 y2))
    (%make-point x1 y1)))

(defun rectangle-position* (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x2 y2))
    (values x1 y1)))

(defun rectangle-min-point (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x2 y2))
    (%make-point x1 y1)))

(defun rectangle-min-x (region) 
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore y1 x2 y2))
    x1))

(defun rectangle-min-y (region) 
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x1 x2 y2))
    y1))

(defun rectangle-max-point (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x1 y1))
    (%make-point x2 y2)))

(defun rectangle-max-x (region) 
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x1 y1 y2))
    x2))

(defun rectangle-max-y (region) 
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x1 y1 x2))
    y2))

(defun rectangle-dimensions (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (values (- x2 x1) (- y2 y1))))

(defun rectangle-width (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore y1 y2))
    (- x2 x1)))

(defun rectangle-height (region)
  (multiple-value-bind (x1 y1 x2 y2) (bounding-rectangle* region) 
    (declare (ignore x1 x2))
    (- y2 y1)))

(defmethod (setf rectangle-position) (point (rectangle standard-rectangle))
  (with-bounding-rectangle* (minx miny maxx maxy) rectangle
    (with-slots (x y) point
      (setf* (bounding-rectangle* rectangle)
	     (values x y (+ x (- maxx minx)) (+ y (- maxy miny)))))))

(defmethod* (setf rectangle-position*) (x y (rectangle standard-rectangle))
  (with-bounding-rectangle* (minx miny maxx maxy) rectangle
    (setf* (bounding-rectangle* rectangle)
	   (values x y (+ x (- maxx minx)) (+ y (- maxy miny))))))

(defmethod (setf rectangle-min-point) (point (rectangle standard-rectangle))
  (with-slots (x y) point
    (setf* (bounding-rectangle* rectangle)
	   (values x y nil nil))))

(defmethod (setf rectangle-min-x) (x (rectangle standard-rectangle))
  (setf* (bounding-rectangle* rectangle)
	 (values x nil nil nil))
  x)

(defmethod (setf rectangle-min-y) (y (rectangle standard-rectangle))
  (setf* (bounding-rectangle* rectangle)
	 (values nil y nil nil))
  y)

(defmethod (setf rectangle-max-point) (point (rectangle standard-rectangle))
  (with-slots (x y) point
    (setf* (bounding-rectangle* rectangle)
	   (values nil nil x y))))

(defmethod (setf rectangle-max-x) (x (rectangle standard-rectangle))
  (setf* (bounding-rectangle* rectangle)
	 (values nil nil x nil))
  x)

(defmethod (setf rectangle-max-y) (y (rectangle standard-rectangle))
  (setf* (bounding-rectangle* rectangle)
	 (values nil nil nil y))
  y)

(defmethod* (setf rectangle-dimensions)
	   (width height (rectangle standard-rectangle))
  (with-bounding-rectangle* (minx miny) rectangle
    (setf* (bounding-rectangle* rectangle)
	   (values nil nil (+ minx width) (+ miny height)))
    (values width height)))

(defmethod (setf rectangle-width) (width (rectangle standard-rectangle))
  (with-bounding-rectangle* (minx miny) rectangle
    (declare (ignore miny))
    (setf* (bounding-rectangle* rectangle)
	   (values nil nil (+ minx width) nil))
    width))
  
(defmethod (setf rectangle-height) (height (rectangle standard-rectangle))
  (with-bounding-rectangle* (minx miny) rectangle
    (declare (ignore minx))
    (setf* (bounding-rectangle* rectangle)
	   (values nil nil nil (+ miny height)))
    height))


;;;
;;; Rectangle Sets
;;;

(defclass rectangle-set (region)
    ((rectangles :type list 
		 :initform nil :initarg :rectangles
		 :accessor rectangles
		 :accessor rectangle-set-rectangles))
  (:documentation "A union of rectangles."))

(define-constructor make-rectangle-set rectangle-set (&key rectangles)
  :rectangles rectangles)

(defmethod bounding-rectangle* ((set rectangle-set))
  (let ((min-x most-positive-fixnum)
	(min-y most-positive-fixnum)
	(max-x most-negative-fixnum)
	(max-y most-negative-fixnum))
    (dolist (rect (rectangles set))
      (with-bounding-rectangle* (x y x2 y2) rect
	(when (< x min-x) (setf min-x x))
	(when (< y min-y) (setf min-y y))
	(when (> x2 max-x) (setf max-x x2))
	(when (> y2 max-y) (setf max-y y2))))
    (values min-x min-y max-x max-y)))

;;; ??? Need to implement normalization in x and y sorting.

(defmethod normalize ((rs rectangle-set))
  "Normailizes rectangle-set.
Ensures that all the rectangles in the set are disjoint."
  (labels ((collect-rectangles (x)
	     (etypecase x
	       (everywhere (list x))
	       (rectangle (list x))
	       (rectangle-set (mapcan #'collect-rectangles (rectangles x)))))
	   (reduce-rectangles (pending-rectangles processed-rectangles)
	     (cond ((null pending-rectangles) processed-rectangles)
		   ((region-empty-p (first pending-rectangles))
		    (reduce-rectangles (rest pending-rectangles)
				       processed-rectangles))
		   (t (let ((interecting-region 
			     (find-if
			      #'(lambda (a)
				  (region-intersects-region-p 
				   a (first pending-rectangles)))
			      (rest pending-rectangles))))
			(if (null interecting-region)
			    (reduce-rectangles 
			     (rest pending-rectangles)
			     (cons (first pending-rectangles)
				   processed-rectangles))
			    (reduce-rectangles 
			     (nconc (reduce-region-pair 
				     (first pending-rectangles)
				     interecting-region)
				    (delete interecting-region
					    (rest pending-rectangles)
					    :test #'eq))
			     processed-rectangles))))))
	   (reduce-region-pair (r1 r2)
	     (delete-if 
	      #'region-empty-p 
	      (let ((left1 (rectangle-min-x r1))
		    (bottom1 (rectangle-min-y r1))
		    (right1 (rectangle-max-x r1))
		    (top1 (rectangle-max-y r1))
		    (left2 (rectangle-min-x r2))
		    (bottom2 (rectangle-min-y r2))
		    (right2 (rectangle-max-x r2))
		    (top2 (rectangle-max-y r2)))
		(cond
		  ((region-contains-point*-p r1 left2 bottom2)
		   ;; Lower-left corner of R2 contained in R1
		   (list r1 (make-rectangle* left2 top1 right2 top2)
			 (make-rectangle* right1 bottom2 right2 top1)))
		  ((region-contains-point*-p r1 left2 top2)
		   ;; Upper-left corner of R2 contained in
		   ;; R1
		   (list r1 (make-rectangle* left2 bottom2 right2 bottom1)
			 (make-rectangle* right1 bottom1 right2 top2)))
		  ((region-contains-point*-p r1 right2 top2)
		   ;; Upper-right corner of R2 contained
		   ;; in R1
		   (list r1 (make-rectangle* left2 bottom2 right2 bottom1)
			 (make-rectangle* left2 bottom1 left1 top2 )))
		  ((region-contains-point*-p r1 right2 bottom2)
		   ;; Lower-right corner of R2 contained
		   ;; in R1
		   (list r1 (make-rectangle* left2 top1 right2 top2)
			 (make-rectangle* left2 bottom2 left1 top1)))
		  ((>= bottom1 bottom2)
		   ;; They cross like a "T", with R1 the
		   ;; vertical piece
		   (list r2 (make-rectangle* left1 bottom1 left2 top1)
			 (make-rectangle* right2 bottom1 right1 top1)))

		  (t 
		   ;; They cross like a "T", with R2 
		   (list r2 (make-rectangle* left1 bottom1 right1 bottom2)
			 (make-rectangle* left1 top2 right1 top1))))))))
    (setf (rectangles rs)
	  (reduce-rectangles (collect-rectangles rs) nil)))
  rs)

(defun make-normalized-rectangle-set (&key rectangles)
  (let ((rs (normalize (make-rectangle-set :rectangles rectangles))))
    (cond ((null (rectangles rs)) +nowhere+)
	  ((null (rest (rectangles rs))) (first (rectangles rs)))
	  (t rs))))


;;;
;;; REGION PREDICATES
;;;

;;;
;;; empty-p
;;;

(defmethod region-empty-p ((everywhere everywhere))
  nil)


(defmethod region-empty-p ((nowhere nowhere))
  t)

(defmethod region-empty-p ((rectangle standard-rectangle))
  (or (>= 0 (rectangle-width rectangle))
      (>= 0 (rectangle-height rectangle))))

(defmethod region-empty-p ((rs rectangle-set))
  (every #'region-empty-p (rectangles rs)))


;;;
;;; contains-p
;;;

(defmethod region-contains-region-p ((p everywhere) (point point)) 
  t)

(defmethod region-contains-region-p ((p nowhere) (point point)) 
  nil)

(defmethod region-contains-region-p ((rectangle standard-rectangle) 
				     (point standard-point))
  (with-slots (x y) point
    (with-slots (min-x min-y max-x max-y) rectangle
      (and (>= x min-x)
	   (<= x max-x)
	   (>= y min-y)
	   (<= y max-y)))))

(defmethod region-contains-region-p ((rs rectangle-set) (point point))
  (some #'(lambda (region)
	    (region-contains-region-p region point))
	(rectangles rs)))

(defmethod region-contains-point*-p ((region everywhere) x y)
  (declare (ignore x y))
  t)

(defmethod region-contains-point*-p ((region nowhere) x y)
  (declare (ignore x y))
  nil)

(defmethod region-contains-point*-p ((region standard-rectangle) x y)
  (with-slots (min-x min-y max-x max-y) region
    (and (>= x min-x)
	 (<= x max-x)
	 (>= y min-y)
	 (<= y max-y))))

(defmethod region-contains-point*-p ((rs rectangle-set) x y)
  (some #'(lambda (region)
	    (region-contains-point*-p region x y))
	(rectangles rs)))

;;;
;;; intesects-p
;;;

(defmethod region-intersects-region-p ((region region) (everywhere everywhere))
  (not (region-empty-p region)))

(defmethod region-intersects-region-p ((everywhere everywhere) (region region))
  (not (region-empty-p region)))

(defmethod region-intersects-region-p ((rs rectangle-set) (p everywhere))
  (not (region-empty-p rs)))

(defmethod region-intersects-region-p ((region region) (nowhere nowhere))
  nil)

(defmethod region-intersects-region-p ((nowhere nowhere) (region region))
  nil)

(defmethod region-intersects-region-p ((rs rectangle-set) (p nowhere))
  nil)

(defmethod region-intersects-region-p ((r1 standard-rectangle) 
				       (r2 standard-rectangle))
  (multiple-value-bind (r1x1 r1y1 r1x2 r1y2) (bounding-rectangle* r1)
    (multiple-value-bind (r2x1 r2y1 r2x2 r2y2) (bounding-rectangle* r2)
      (not (or (>= r1x1 r2x2)
	       (>= r2x1 r1x2)
	       (>= r1y1 r2y2)
	       (>= r2y1 r1y2))))))

(defmethod region-intersects-region-p ((region region) (rs rectangle-set))
  (some #'(lambda (rectangle)
	    (region-intersects-region-p region rectangle))
	(rectangles rs)))

(defmethod region-intersects-region-p ((rs rectangle-set) (region region))
  (some #'(lambda (rectangle)
	    (region-intersects-region-p rectangle region))
	(rectangles rs)))

;;;
;;; Region Operations
;;;

;;; ??? Need to support reuses

(defmethod region-union ((p everywhere) (a region) &key reuse)
  (declare (ignore reuse))
  p)

(defmethod region-union ((a region) (p everywhere) &key reuse)
  (declare (ignore reuse))
  p)

(defmethod region-union ((p nowhere) (a region) &key reuse)
  (declare (ignore reuse))
  a)

(defmethod region-union ((a region) (p nowhere) &key reuse)
  (declare (ignore reuse))
  a)

(defmethod region-union ((a1 standard-rectangle) 
			 (a2 standard-rectangle) &key reuse)
  (declare (ignore reuse))
  (make-normalized-rectangle-set :rectangles (list a1 a2)))

(defmethod region-union ((a1 rectangle-set) (a2 rectangle-set) &key reuse)
  (declare (ignore reuse))
  (make-normalized-rectangle-set :rectangles 
				 (append (rectangles a1) 
					 (rectangles a2))))

(defmethod region-union ((a1 rectangle-set) 
			 (a2 standard-rectangle) &key reuse)
  (declare (ignore reuse))
  (make-normalized-rectangle-set :rectangles (cons a2 (rectangles a1))))

(defmethod region-union ((a1 standard-rectangle) 
			 (a2 rectangle-set) &key reuse)
  (declare (ignore reuse))
  (make-normalized-rectangle-set :rectangles (cons a1 (rectangles a2))))

;;;
;;; Intersection
;;;

(defmethod region-intersection ((p everywhere) (a region) &key reuse)
  (declare (ignore reuse))
  a)

(defmethod region-intersection ((a region) (p everywhere) &key reuse)
  (declare (ignore reuse))
  a)

(defmethod region-intersection ((rs rectangle-set) (p everywhere) &key reuse)
  (declare (ignore reuse))
  rs)

(defmethod region-intersection ((p nowhere) (a region) &key reuse)
  (declare (ignore reuse))
  p)

(defmethod region-intersection ((a region) (p nowhere) &key reuse)
  (declare (ignore reuse))
  p)

(defmethod region-intersection ((rs rectangle-set) (p nowhere) &key reuse)
  (declare (ignore reuse))
  p)

(defmethod region-intersection ((x standard-rectangle) 
				(y standard-rectangle) &key reuse)
  (declare (ignore reuse))
  (with-bounding-rectangle* (minx1 miny1 maxx1 maxy1) x
    (with-bounding-rectangle* (minx2 miny2 maxx2 maxy2) y
      (let ((min-x (max minx1 minx2))
	    (min-y (max miny1 miny2))
	    (max-x (min maxx1 maxx2))
	    (max-y (min maxy1 maxy2)))
	(if (and (> max-x min-x)
		 (> max-y min-y))
	    (make-rectangle* min-x min-y max-x max-y)
	    +nowhere+)))))

(defmethod region-intersection ((rs rectangle-set) (region region) &key reuse)
  (declare (ignore reuse))
  (or (reduce #'region-union 
	      (mapcar #'(lambda (reg) (region-intersection reg region))
		      (rectangles rs)))
      +nowhere+))

(defmethod region-intersection ((region region) (rs rectangle-set) &key reuse)
  (declare (ignore reuse))
  (or (reduce #'region-union 
	      (mapcar #'(lambda (reg) (region-intersection reg region) )
		      (rectangles rs)))
      +nowhere+))
;;; 
;;; Difference
;;;

(defmethod region-difference ((p everywhere) (a region) &key reuse)
  (declare (ignore reuse))
  (error "~%Can't handle the resulting region type."))

(defmethod region-difference ((a region) (p everywhere) &key reuse)
  (declare (ignore reuse))
  +nowhere+)

(defmethod region-difference ((rs rectangle-set) (p everywhere) &key reuse)
  (declare (ignore reuse))
  +nowhere+)

(defmethod region-difference ((p nowhere) (a region) &key reuse)
  (declare (ignore reuse))
  +nowhere+)

(defmethod region-difference ((a region) (p nowhere) &key reuse)
  (declare (ignore reuse))
  a)

(defmethod region-difference ((a rectangle-set) (p nowhere) &key reuse)
  (declare (ignore reuse))
  a)

(defmethod region-difference ((rs rectangle-set) (a region) &key reuse)
  (declare (ignore reuse))
  (make-instance 'rectangle-set 
		 :rectangles (mapcar #'(lambda (x)
					 (region-difference x a))
				     (rectangles rs))))

(defmethod region-difference ((region region) (rs rectangle-set) &key reuse)
  (declare (ignore reuse))
  (reduce #'(lambda (x y) (region-difference x y))
	  (rectangles rs)
	  :initial-value region))

(defmethod region-difference ((r1 standard-rectangle) 
			      (r2 standard-rectangle) &key reuse)
  (declare (ignore reuse))
  ;; 17 cases covered.
  (if (region-intersects-region-p r1 r2)
      (let* ((left1 (rectangle-min-x r1))
	     (bottom1 (rectangle-min-y r1))
	     (right1 (rectangle-max-x r1))
	     (top1 (rectangle-max-y r1))
	     (left2 (rectangle-min-x r2))
	     (bottom2 (rectangle-min-y r2))
	     (right2 (rectangle-max-x r2))
	     (top2 (rectangle-max-y r2))
	     (sw-corner-p (region-contains-point*-p r1 left2 bottom2))
	     (nw-corner-p (region-contains-point*-p r1 left2 top2))
	     (ne-corner-p (region-contains-point*-p r1 right2 top2))
	     (se-corner-p (region-contains-point*-p r1 right2 bottom2)))
	(make-normalized-rectangle-set
	 :rectangles
	 (cond 
	   ;; R2 inside of R1
	   ((and sw-corner-p nw-corner-p ne-corner-p se-corner-p)
	    (list (make-rectangle* left1 bottom1 right1 bottom2)
		  (make-rectangle* left1 top2 right1 top1)
		  (make-rectangle* left1 bottom2 left2 top2)
		  (make-rectangle* right2 bottom2 right1 top2)))
	   ;; Left edge of R2 contained in R1
	   ((and sw-corner-p nw-corner-p)1
	    (list (make-rectangle* left1 bottom1 right1 bottom2)
		  (make-rectangle* left1 top2 right1 top1)
		  (make-rectangle* left1 bottom2 left2 top2)))
	   ;; Right edge of R2 contained in R1
	   ((and ne-corner-p se-corner-p)
	    (list (make-rectangle* left1 bottom1 right1 bottom2)
		  (make-rectangle* left1 top2 right1 top1)
		  (make-rectangle* right2 bottom2 right1 top2)))
	   ;; Bottom edge of R2 contained in R1
	   ((and sw-corner-p se-corner-p)
	    (list (make-rectangle* left1 bottom1 left2 top1)
		  (make-rectangle* left2 bottom1 right2 bottom2)
		  (make-rectangle* right2 bottom1 right1 top1)))
	   ;; Top edge of R2 contained in R1
	   ((and nw-corner-p ne-corner-p)
	    (list (make-rectangle* left1 bottom1 left2 top1)
		  (make-rectangle* left2 top2 right2 top1)
		  (make-rectangle* right2 bottom1 right1 top1)))
	   ;; If we got this far, no edge of R2 is contained in R1
	   ;; Only lower-left corner of R2 contained in R1
	   (sw-corner-p 
	    (list (make-rectangle* left1 bottom1 left2 top1)
		  (make-rectangle* left2 bottom1 right1 bottom2)))
	   ;; Only upper-left corner of R2 contained in R1
	   (nw-corner-p 
	    (list (make-rectangle* left1 bottom1 left2 top1)
		  (make-rectangle* left2 top2 right1 top1)))
	   ;; Only upper-right corner of R2 contained in R1
	   (ne-corner-p 
	    (list (make-rectangle* right2 bottom1 right1 top1)
		  (make-rectangle* left1 top2 right2 top1)))
	   ;; Only lower-right corner of R2 contained in R1
	   (se-corner-p 
	    (list (make-rectangle* right2 bottom1 right1 top1)
		  (make-rectangle* left1 bottom1 right2 bottom2)))
	   ;; 7 Cases Remaining.  In particular, R2 crosses R1 like a bar, 3
	   ;; case horizontally, 3 cases vertically, 1 case both.

	   ;; R2 is a horizontal bar across R1, 3 cases.
	   ((or (<= bottom1 bottom2)
		(>= top1 top2))
	    `(,@(if (< bottom1 bottom2)
		    (cons 
		     (make-rectangle* left1 bottom1 right1 bottom2)
		     nil))
		,@(if (> top1 top2)
		      (cons 
		       (make-rectangle* left1 top2 right1 top1)
		       nil))))
	   ;; R2 is a vertical bar across R1, 3 cases.
	   ((or (<= left1 left2)
		(>= right1 right2))
	    `(,@(if (< left1 left2)
		    (cons
		     (make-rectangle* left1 bottom1 left2 top1)
		     nil))
		,@(if (> right1 right2)
		      (cons
		       (make-rectangle* right2 bottom1 right1 top1)
		       nil))))

	   ;; R2 completely occludes R1
	   (t nil))))
      r1))

