;;; -*- Mode: LISP; Syntax: Common-lisp; Package: ON-POSTSCRIPT; Base: 10 -*-

"Copyright (c) 1991 by International Lisp Associates.  All rights reserved."

(in-package :on-postscript)

(defun approximately-gray-p (ink)
  (multiple-value-bind (red green blue) (color-rgb ink)
    (<= (+ (abs (- red green)) (abs (- red blue)))	;Red nearly the same as blue and green.
	.01)))

(defmethod prepare-printer-for-new-ink (printer-stream (ink color) color-printer-p)
  (if (or (not color-printer-p) (approximately-gray-p ink))
      (setgray printer-stream (silica::luminosity ink))
      (multiple-value-bind (red green blue) (color-rgb ink)
	(setcolor printer-stream red green blue)))
  nil)						;No raster.

;;; Raoian Shades of Gray
(defmethod prepare-printer-for-new-ink (printer-stream (ink number) color-printer-p)
  (declare (ignore color-printer-p))
  (assert (<= 0 ink 100) () "Shades must be between 0 and 100, inclusive")
  (setgray printer-stream (/ (- 100 ink) 100))
  nil)						;No raster

(defun prepare-original-printer-transformation (printer-stream ppi)
  (multiple-value-bind (left top right bottom) (ppi-device-inside-edges ppi)
    (progn
      (newpath printer-stream)
      (moveto printer-stream left top)
      (lineto printer-stream left bottom)
      (lineto printer-stream right bottom)
      (lineto printer-stream right top)
      (closepath printer-stream)
      (clip printer-stream))
    (translate printer-stream left bottom)
    (let ((scale-factor (/ (ppi-device-scale-factor ppi) 72.0)))
      (scale printer-stream scale-factor (- scale-factor)))))

;;; Note: This takes a transformation-vector, not a transformation.
(defun prepare-printer-for-new-transformation (printer-stream transformation-vector)
  (concat printer-stream transformation-vector))

;;; Maybe we should do something about non-rectangular clipping regions, but not yet.
(defmethod prepare-printer-for-new-clipping-region (printer-stream
						     (clipping-region region))
  (with-bounding-rectangle* (left top right bottom) clipping-region
    (newpath printer-stream)
    (moveto printer-stream left top)
    (lineto printer-stream left bottom)
    (lineto printer-stream right bottom)
    (lineto printer-stream right top)
    (closepath printer-stream)
    (clip printer-stream)))

(defmethod prepare-printer-for-new-clipping-region (printer-stream
						     (clipping-region everywhere))
  (declare (ignore printer-stream)))

(defmethod prepare-printer-for-new-clipping-region (printer-stream
						     (clipping-region nowhere))
  (newpath printer-stream)
  ;; Does this produce the empty clipping region?  I hope so.
  (clip printer-stream))

(defun prepare-printer-for-new-text-style (printer-stream text-style medium)
  (let ((font (realize-text-style (port medium) text-style)))
    (when (null font) (error "Text style mapping not found for ~S~*" text-style))
    (use-font medium printer-stream font)))

(defvar *default-dash-pattern* #(3))

#+Genera (zwei:defindentation (when-ls 1 1))

;;; --- This will need to be modified when line styles are cleaned up.
(defun prepare-for-line-style (printer-stream line-style gsaver)
  (macrolet ((when-ls ((property) &body body)
	       (let ((accessor (fintern "~A-~A" 'line-style property)))
		 `(let ((,property (,accessor line-style)))
		    (when ,property (gsave) ,@body))))
	     (gsave () `(funcall gsaver)))
    (when-ls (thickness)
      (setlinewidth printer-stream thickness))
    (when-ls (joint-shape)
      (setlinejoin printer-stream
		   (ecase joint-shape
		     ((:miter :none) 0)
		     ((:round) 1)
		     ((:bevel) 2))))
    (when-ls (cap-shape)
      (setlinecap printer-stream
		  (ecase cap-shape
		    ((:butt :no-end-point) 0)
		    ((:square) 2)
		    ((:round) 1))))
    (when-ls (dashed)
      (when (eql dashed 't)
	(setf dashed *default-dash-pattern*))
      (setdash printer-stream dashed (or (line-style-initial-dash-phase line-style) 0)))))


;;; Convert transformations from Silica values to Postscript vector order.

(defmethod decompose-transformation ((x silica::identity-transformation))
  (values 1 0 0 1 0 0))

(defmethod decompose-transformation ((x silica::translation))
  (with-slots ((m20 silica::m20) (m21 silica::m21)) x
    (values 1 0 0 1 m20 m21)))

(defmethod decompose-transformation ((x silica::st-transformation))
  (with-slots ((m00 silica::m00) (m11 silica::m11) (m20 silica::m20) (m21 silica::m21))
	      x
    (values m00 0 0 m11 m20 m21)))

(defmethod decompose-transformation ((x silica::srt-transformation))
  (with-slots ((m00 silica::m00) (m01 silica::m01) (m10 silica::m10) (m11 silica::m11)
	       (m20 silica::m20) (m21 silica::m21))
	      x
    (values m00 m01 m10 m11 m20 m21)))

;;; More breakage of proper protocol.  I'm not sure what this is for.

(defmethod realize-text-style ((port postscript-port) text-style)
  (let ((pfd (text-style-mapping port *standard-character-set* text-style)))
    pfd))

(defmethod text-style-ascent ((text-style text-style) (medium basic-postscript-medium))
  (let* ((port (port medium))
	 (pfd (realize-text-style port text-style))
	 (pface (pfd-face pfd))
	 (point-size (on-postscript::pfd-point-size pfd))
	 (height (* point-size (on-postscript::pface-height pface)))
	 (ascent (* (on-postscript::pface-ascent pface) height)))
    ascent))

(defmethod text-style-descent ((text-style text-style) (medium basic-postscript-medium))
  (let* ((port (port medium))
	 (pfd (realize-text-style port text-style))
	 (pface (pfd-face pfd))
	 (point-size (pfd-point-size pfd))
	 (height (* point-size (pface-height pface)))
	 (descent (* (- 1.0 (pface-ascent pface)) height)))
    descent))

(defmethod text-style-height ((text-style text-style) (medium basic-postscript-medium))
  (let* ((port (port medium))
	 (pfd (realize-text-style port text-style))
	 (pface (pfd-face pfd))
	 (point-size (pfd-point-size pfd))
	 (height (* point-size (pface-height pface))))
    height))

;;; Extremely stupid version of this for now:
(defmethod string-width (string (text-style text-style) (medium basic-postscript-medium)
				&key (start 0) end)
  (unless end (setf end (length string)))
  (multiple-value-bind (write-char next-char-index new-cursor-x new-baseline new-height font)
      (ci::stream-scan-string-for-writing nil medium string start end
					  text-style 0 most-positive-fixnum)
      (declare (ignore write-char next-char-index new-baseline new-height font))
      new-cursor-x))

(defmethod string-height (string (text-style text-style) (medium basic-postscript-medium))
  (declare (ignore string))
  (text-style-height text-style medium))
