;;; -*- Mode: LISP; Syntax: Common-lisp; Package: ON-POSTSCRIPT; Base: 10 -*-

"Copyright (c) 1991 by International Lisp Associates.  All rights reserved."

(in-package :on-postscript)

;;; Postscript Silica port implementation started January 1991 by Richard Lamson

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Issues:   
;;;
;;; How do we tell the port what the dimensions of the paper, margins,
;;; scale factor, pixels per point, etc., are?
;;;
;;; Various ways this doesn't seem to fit into the Silica model (e.g.,
;;; multiple media on a single graft, etc.)
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod find-port-type ((type (eql :postscript))) 
  (find-class 'postscript-port))

(defclass postscript-port (port)
    ((stream-realizer :accessor port-stream-realizer)
     (stream-closer)
     (port-matcher)
     (printer-description :accessor port-printer-description)
     (color-printer-p :accessor port-color-printer-p :initform nil)))

(defmethod port-type ((port postscript-port))
  :postscript)

(defmethod initialize-instance :after ((port postscript-port) &key server-path)
  (multiple-value-bind (realizer closer matcher description color-printer-p)
      (apply #'understand-postscript-server-path server-path)
    (setf (slot-value port 'stream-realizer) realizer
	  (slot-value port 'stream-closer) closer
	  (slot-value port 'port-matcher) matcher
	  (slot-value port 'printer-description) description
	  (slot-value port 'color-printer-p) color-printer-p))
  (initialize-postscript-display-device port))

(defmethod port-match ((port postscript-port) server-path)
  (funcall (slot-value port 'port-matcher) server-path))

(defvar *default-postscript-file* nil)

;;; Calculate this the first time it's needed, not at load time.
(defun default-postscript-file ()
  (or *default-postscript-file*
      (setf *default-postscript-file* (merge-pathnames "output.ps" (user-homedir-pathname)))))

(defun understand-postscript-server-path (ignored-keyword
					   &key (file :default)
						(if-exists :new-version) ;; Used with :FILE.
						(stream nil stream-p)
						(stream-maker nil maker-p)
						(printer-description :default)
						(color-printer-p nil))
  (declare (ignore ignored-keyword)
	   (values realizer-function closer-function matcher-function 
		   printer-description color-printer-p))
  (setf printer-description (find-postscript-printer-description printer-description))
  ;; (assert (eql ignored-keyword :postscript))
  (macrolet ((server-path-get (prop &optional (default ''#:server-path))
	       ;; Return something guaranteed to be not in the server path by default.
	       `(getf (cdr server-path) ,prop ,default)))
    (when stream-p
      (default-output-stream stream)
      (return-from understand-postscript-server-path
	(values
	  #'(lambda () stream)
	  #'(lambda (stream) (declare (ignore stream)))	;Ignore
	  #'(lambda (server-path) (and (eql stream (server-path-get :stream))
				       (eql color-printer-p (server-path-get 
							     :color-printer-p))))
	  printer-description
	  color-printer-p)))
    (when maker-p
      (return-from understand-postscript-server-path
	(values
	  #'(lambda () (funcall stream-maker))
	  #'close
	  #'(lambda (server-path) (and (eql stream-maker (server-path-get
							  :stream-maker))
				       (eql color-printer-p (server-path-get 
							     :color-printer-p))))
	  printer-description
	  color-printer-p)))
    ;; No FILE-P: FILE is the default (for now).
    (when file
      (when (eql file :default) (setf file (default-postscript-file)))
      (setq file (pathname file))
      (return-from understand-postscript-server-path
	(values
	  #'(lambda ()
	      (open file :direction :output :if-exists if-exists))
	  #'close
	  #'(lambda (server-path)
	      (let ((pathname (server-path-get :file :default)))
		(when (eql pathname :default) (setf pathname (default-postscript-file)))
		(and (equal pathname file)
		     (eql color-printer-p (server-path-get 
					   :color-printer-p)))))
	  printer-description
	  color-printer-p)))
    ;; This shouldn't happen at the moment, but put it here to remind me that we
    ;; need to handle this case if we decide to require one of the keywords...
    (error "A ~S server path must contain one of the keywords ~{~S~^, ~}."
	   ':postscript '(:stream :stream-maker :file))))

#+Genera
(scl:defprop define-postscript-noop "Postscript No-op method" si:definition-type-name)

(defmacro define-postscript-noop (name &optional rest-of-lambda-list)
  `(define-group ,name define-postscript-noop
     (defmethod ,name ((port postscript-port) ,@rest-of-lambda-list)
       (declare (ignore ,@rest-of-lambda-list))
       nil)))

(define-postscript-noop port-force-output)

(define-postscript-noop port-finish-output)

;;; Are these two necessary?
(defmethod do-with-port ((port postscript-port) function)
  (declare (dynamic-extent function))
  (funcall function))

(defmethod do-with-pointer ((port postscript-port) function)
  (declare (dynamic-extent function))
  (funcall function))

(defmethod realize-graft ((port postscript-port) (graft graft))
  ;; I don't quite know what to make of a graft in this implementation; 
  ;; A port doesn't have a device at the other end to ask about dimensions,
  ;; and a graft doesn't get any sort of mirror except the way any sheet would.
  (with-slots (units 
		width-pixel height-pixel width-mm height-mm 
		pixels-per-point region) 
	      graft
    (let ((ppi (port-printer-description port)))

      (setf width-pixel    (* (ppi-outside-width ppi) (ppi-device-scale-factor ppi))
	    height-pixel   (* (ppi-outside-height ppi) (ppi-device-scale-factor ppi))
	    width-mm	   (* (ppi-outside-width ppi) (ppi-metric-scale-factor ppi))
	    height-mm      (* (ppi-outside-height ppi) (ppi-metric-scale-factor ppi))
	    pixels-per-point 1)			;Don't know for sure what this means.
      
      (setf region
	      (ecase units
		(:pixel (make-rectangle* 0 0 width-pixel height-pixel))
		(:mm    (make-rectangle* 0 0 width-mm height-mm))
		(:homogenous (make-rectangle* 0.0 0.0 1.0 1.0))))
      
      (setf (sheet-native-transformation graft) 
	      +identity-transformation+)
;;  (setf (sheet-mirror graft) (realize-mirror port graft)) ;;Is the mirror already realized?
      (update-native-transformation port graft))))

(defmethod realize-mirror ((port postscript-port) sheet)
  (with-slots (stream-realizer) port
    (let ((mirror (funcall stream-realizer)))
      (setf (sheet-mirror sheet) mirror)
      ;; This doesn't work because the port isn't set yet;
      ;; can't set the port yet because it will try to make a medium too early.
      ;; (update-native-transformation port sheet)
      mirror)))

(defmethod destroy-mirror ((port postscript-port) sheet)
  (with-slots (stream-closer) port
    (funcall stream-closer (sheet-mirror sheet))))

(define-postscript-noop enable-mirror (sheet))

(define-postscript-noop disable-mirror (sheet))

(define-postscript-noop raise-mirror (sheet))

(define-postscript-noop bury-mirror (sheet))

(defmethod mirror-origin ((port postscript-port) sheet)
  (declare (ignore sheet))
  :SW)

(defmethod mirror-inside-region* ((port postscript-port) sheet)
  (declare (ignore sheet))
  (let ((ppi (port-printer-description port)))
    (ppi-device-inside-edges ppi)))

(defmethod sheet-actual-native-edges* ((port postscript-port) sheet)
  (declare (ignore sheet))
  (let ((ppi (port-printer-description port)))
    (ppi-device-outside-edges ppi)))

;;; --- You got me; Is this right?
(defmethod update-native-transformation ((port postscript-port) sheet)
  (setf (sheet-native-transformation sheet) +identity-transformation+))

(defmethod set-sheet-actual-native-edges* ((port postscript-port) sheet
					   min-x min-y max-x max-y)
  (declare (ignore sheet min-x min-y max-x max-y))
  (error "Can't set a Postscript sheet's native edges!"))

;;; Nobody calls these next two anyway, as far as I can tell.
(defmethod mirror-region ((port postscript-port) sheet)
  (multiple-value-bind (x1 y1 x2 y2) (mirror-region* port sheet)
    (when (and x1 y1 x2 y2)
      (make-rectangle* x1 x2 y1 y2))))

(defmethod mirror-region* ((port postscript-port) sheet)
  (declare (ignore sheet))
  (error "Don't know how to take ~S of a Postscript sheet." 'mirror-region*))

(defmethod process-next-event ((port postscript-port) &key timeout wait-test state)
  (declare (ignore timeout wait-test state))
  (error "No events on Postscript ports."))

(define-postscript-noop restart-input)		;; No events process on Postscript port.

;;; Probably not interesting, but seems to be part of the port protocol.
(defmethod get-port-canonical-gesture-spec (gesture-spec (port postscript-port))
  gesture-spec)

;;; The 8-1/2 by 11 inch printer.

(define-postscript-printer :8.5x11
  (:width 8.5)
  (:height 11)
  (:margins 0.5)
  (:device-scale-factor 72.0)
  (:metric-scale-factor 25.4))

;;; The default printer is the 8.5x11 printer, for now.

(define-postscript-printer :default
  (:like :8.5x11))

;;; Postscript font hacking.

(defstruct (postscript-font-family
	     (:conc-name pfam-)
	     (:print-function (lambda (pfam stream depth)
				(declare (ignore depth))
				(print-unreadable-object (pfam stream :type t)
				  (format stream "~A [~S]"
					  (pfam-name pfam)
					  (pfam-text-family pfam))))))
  name
  text-family
  (faces nil))

(defstruct (postscript-font-face
	     (:conc-name pface-)
	     (:print-function (lambda (pface stream depth)
				(declare (ignore depth))
				(print-unreadable-object (pface stream :type t)
				  (format stream "~A (~A.~A.*)"
					  (pface-name pface)
					  (pfam-text-family (pface-family pface))
					  (pface-text-face pface))))))
  family
  text-face
  name
  (size-table (make-array 20. :initial-element nil))
  (width-table nil)
  ;; These are correct for Courier:
  (height 0.951)
  (ascent 0.7))

(defstruct (postscript-font-descriptor
	     (:conc-name pfd-)
	     (:print-function (lambda (pfd stream depth)
				(declare (ignore depth))
				(print-unreadable-object (pfd stream :type t)
				  (format stream "\"~D point ~A\"~@[ (~S)~]"
					  (pfd-point-size pfd)
					  (pface-name (pfd-face pfd))
					  (pfd-logical-size pfd))))))
  face
  point-size
  logical-size
  (cached-pretty-name nil))

(defvar *postscript-font-families* nil)

(defun find-postscript-font-family (text-family)
  (find text-family *postscript-font-families* :key #'pfam-text-family))

(defun find-postscript-font-face (pfam text-face)
  (getf (pfam-faces pfam) text-face))

(defun find-postscript-font-logical-size (pface logical-size)
  (flet ((pfd-logical-size-no-error (pfd)
	   (if pfd
	       (pfd-logical-size pfd)
	       '#:this-is-not-in-the-table)))
    (find logical-size (pface-size-table pface) :key #'pfd-logical-size-no-error)))

(defun find-postscript-font-point-size (pface point-size)
  (let ((size-table (pface-size-table pface)))
    (and (integerp point-size)
	 (< 0 point-size (length size-table))
	 (aref size-table point-size))))

(defun find-postscript-font-descriptor (text-style)
  (multiple-value-bind (family face size)
      (text-style-components (parse-text-style text-style))
    (let ((pfam (find-postscript-font-family family)))
      (when (null pfam) (return-from find-postscript-font-descriptor nil))
      (let ((pface (find-postscript-font-face pfam face)))
	(when (null pface) (return-from find-postscript-font-descriptor nil))
	(etypecase size
	  (symbol (find-postscript-font-logical-size pface size))
	  ;; Always legal to introduce a new numerical size.
	  ;; --- Should sizes be only integers?  
	  ;; --- They have to be for now; they're array indices.
	  ((integer 1) (or (find-postscript-font-point-size pface size)
			   (create-postscript-font-size pface size nil))))))))

(defun pfd-pretty-name (pfd)
  (or (pfd-cached-pretty-name pfd)
      (setf (pfd-cached-pretty-name pfd)
	      (format nil "~A-~D" (pface-name (pfd-face pfd)) (pfd-point-size pfd)))))


;;; Initialization of font tables.

(defparameter *psftd-keywords* '(:tiny :very-small :small :normal :large :very-large :huge))

(defun create-postscript-font-family (text-family name)
  (let ((pfam (find-postscript-font-family text-family)))
    (unless pfam
      (push (setf pfam (make-postscript-font-family :name name :text-family text-family))
	    *postscript-font-families*))
    pfam))

(defun create-postscript-font-face (pfam text-face face-name sizes)
  (let ((pface (getf (pfam-faces pfam) text-face)))
    (if (null pface)
	(setf pface (make-postscript-font-face
		      :family pfam :text-face text-face :name face-name)
	      (getf (pfam-faces pfam) text-face) pface)
	(setf (pface-name pface) face-name))
    (mapcar #'(lambda (logical-size point-size)
		(create-postscript-font-size pface point-size logical-size))
	    *psftd-keywords* sizes)
	    
    pface))

(defun create-postscript-font-size (pface point-size logical-size)
  (check-type point-size (integer 1))
  (let ((size-table (pface-size-table pface)))
    (when (>= point-size (length size-table))
      (setf size-table (replace (make-array (+ point-size 10.)) size-table)
	    (pface-size-table pface) size-table))
    (let ((pfd (aref size-table point-size)))
      (when (null pfd)
	(when (and logical-size
		   (setf pfd (find-postscript-font-logical-size pface logical-size)))
	  (error "Trying to create multiple sizes for same logical size ~S.~S.~S: ~D and ~D"
		 (pfam-text-family (pface-family pface))
		 (pface-text-face pface)
		 logical-size
		 point-size
		 (pfd-point-size pfd)))
	(setf pfd (make-postscript-font-descriptor :face pface :point-size point-size
						   :logical-size logical-size)
	      (aref size-table point-size) pfd))
      pfd)))

(defparameter *postscript-font-translate-data*
	      '(("Courier" :fix (4 6 7 9 11 14)
		 (:roman .6 "Courier")
		 (:bold .6 "Courier-Bold")
		 (:italic .6 "Courier-Oblique")
		 ((:bold :italic) .6 "Courier-BoldOblique"))
		("Helvetica" :sans-serif (5 7 8 10 12 16)
		 (:roman nil "Helvetica")
		 (:bold nil "Helvetica-Bold")
		 (:italic nil "Helvetica-Oblique")
		 ((:bold :italic) nil "Helvetica-BoldOblique"))
		("Times" :serif (5 7 8 10 12 16)
		 (:roman nil "Times-Roman")
		 (:bold nil "Times-Bold")
		 (:italic nil "Times-Italic")
		 ((:bold :italic) nil "Times-BoldItalic"))))

(defmethod text-style-mapping :around ((device postscript-port) character-set text-style)
  (let ((pfd (call-next-method)))
    (when (null pfd)
      (setf pfd (find-postscript-font-descriptor text-style))
      (when (null pfd) (return-from text-style-mapping nil))
      (add-text-style-mapping device character-set text-style pfd))
    (when (null (pface-width-table (pfd-face pfd)))
      (read-postscript-font-width-table (pfd-face pfd)))
    pfd))

(defmethod initialize-postscript-display-device ((display-device postscript-port))
  (dolist (pftd *postscript-font-translate-data*)
    (let* ((family-name (pop pftd))
	   (text-family (pop pftd))
	   (text-sizes (pop pftd))
	   (pfam (create-postscript-font-family text-family family-name)))
      (dolist (face-stuff pftd)
	(let* ((text-face (pop face-stuff))
	       (width-table (pop face-stuff))
	       (font-name (pop face-stuff))
	       (text-style (make-text-style text-family text-face nil))
	       (canonical-face-code (text-style-face text-style))
	       (pface (create-postscript-font-face pfam canonical-face-code
						   font-name text-sizes)))
	  ;; Might only be the name of a width table:
	  (setf (pface-width-table pface) width-table)
	  (dolist (text-size text-sizes)
	    (let* ((text-style (make-text-style text-family text-face text-size))
		   (pfd (find-postscript-font-descriptor text-style)))
	      (add-text-style-mapping display-device *standard-character-set*
				      text-style pfd)))
	  (dolist (text-size *psftd-keywords*)
	    (let* ((text-style (make-text-style text-family text-face text-size))
		   (pfd (find-postscript-font-descriptor text-style)))
	      (when pfd
		(add-text-style-mapping display-device *standard-character-set*
					text-style pfd)))))))))

(defvar *postscript-font-width-table-pathname*
	#+Genera #P"CLIM:POSTSCRIPT-WIDTHS;"
	#-Genera nil)				;Figure out something for this.

(defun postscript-font-width-table-pathname (pface)
  (assert (pathnamep *postscript-font-width-table-pathname*)
	  (*postscript-font-width-table-pathname*)
	  "No pathname for postscript fonts has been supplied.")
  (make-pathname :name (pface-name pface) :type "PSWT"
		 :defaults *postscript-font-width-table-pathname*))

;;; A stub until we have real fonts widths tables.  We may want to
;;; continue to use this code after we have a font widths table reader;
;;; this can be the error clause.
(defun read-postscript-font-width-table (pface)
  (let ((fwt-path (postscript-font-width-table-pathname pface)))
    (with-open-file (fwt-file fwt-path :if-does-not-exist nil)
      (when (null fwt-file)
	(let ((fixed-width-font-face (find-postscript-font-face
				       (find-postscript-font-family :fix)
				       (pface-text-face pface))))
	  (cerror "Continue by using the width table for ~*~A"
		  "Postscript width table not found for font ~A~*"
		  (pface-name pface)
		  (pface-name fixed-width-font-face))
	  (return-from read-postscript-font-width-table
	    (setf (pface-width-table pface) (pface-width-table fixed-width-font-face)))))
      (let* ((scale-factor (float (read fwt-file)))
	     (ascent (/ (float (read fwt-file)) scale-factor))
	     (descent (/ (float (read fwt-file)) scale-factor))
	     (height (+ ascent descent))
	     (width-table (make-array 256. :initial-element nil))
	     char-code
	     width)
	(loop (setf char-code (read fwt-file nil :eof))
	      (when (eql char-code :eof) (return))
	      (setf width (float (read fwt-file)))
	      (setf (aref width-table char-code) (/ width scale-factor)))
	(setf (pface-height pface) height
	      (pface-ascent pface) ascent
	      (pface-width-table pface) width-table)))))
