;;; -*- Mode: LISP; Syntax: Common-lisp; Package: ON-POSTSCRIPT; Base: 10 -*-

"Copyright (c) 1991 by International Lisp Associates.  All rights reserved."

(in-package :on-postscript)

(defclass basic-postscript-medium (printer-medium)
    ((printer-description)
     (color-printer-p)
     (features-sent :initform nil)
     (ch1buf :initform (make-array 1 :element-type +string-array-element-type+))
     (font-map :initform (make-array 30 :fill-pointer 0) :reader printer-font-map)))

(defmethod initialize-instance :after ((medium basic-postscript-medium) &key)
  (setf (device-transformation medium) +identity-transformation+)
  (with-slots (printer-description color-printer-p printer-stream features-sent) medium
    (let ((port (port medium)))
      (setf features-sent nil
	    printer-description (port-printer-description port)
	    color-printer-p (port-color-printer-p port)))
    (emit-postscript-prologue medium)
    (initialize-graphics-state medium)))

(defmethod initialize-graphics-state ((medium basic-postscript-medium))
  (let ((printer-stream (slot-value medium 'printer-stream))
	(ppi (slot-value medium 'printer-description)))
    (prepare-original-printer-transformation printer-stream ppi)
    (prepare-printer-for-new-ink
     printer-stream (resolved-medium-ink medium (medium-ink medium))
     (slot-value medium 'color-printer-p))
    (prepare-printer-for-new-clipping-region
      printer-stream (medium-clipping-region medium))
    (use-font medium printer-stream
	      (realize-text-style (port medium) (medium-text-style medium)))))

(defmethod slot-unbound (class (medium basic-postscript-medium) (slot (eql 'printer-stream)))
  (declare (ignore class))
  (setf (slot-value medium 'printer-stream)
	  (realize-mirror (port medium) (medium-sheet medium))))

(defmethod maybe-send-feature ((medium basic-postscript-medium) feature-name code
			       &rest other-arguments)
  (declare (dynamic-extent other-arguments))
  (with-slots (features-sent printer-stream) medium
    (unless (member feature-name features-sent)
      (apply #'format printer-stream code other-arguments)
      (push feature-name features-sent))))

;;; --- This sure crosses a lot of modularity lines.  Can this really be right?
(defmethod close-printer-medium ((medium basic-postscript-medium) &key abort)
  (unless abort
    (emit-postscript-epilogue medium))
  (destroy-mirror (port medium) (medium-sheet medium)))

(defmacro with-gsave ((medium) &body body)
  `(flet ((with-gsave () ,@body))
     (declare (dynamic-extent #'with-gsave))
     (with-gsave-1 ,medium #'with-gsave)))

(defmethod with-gsave-1 ((medium basic-postscript-medium) continuation)
  (with-slots (printer-stream
		printer-ink printer-transformation printer-clipping-region printer-text-style)
	      medium
    (letf-globally ((printer-ink) (printer-transformation)
		    (printer-clipping-region) (printer-text-style))
      (ps-operation "gsave" printer-stream)
      (funcall continuation)
      (ps-operation "grestore" printer-stream))))

(defmethod resolved-medium-ink ((medium basic-postscript-medium) (ink (eql +foreground+)))
  (resolved-medium-ink medium (medium-foreground medium)))

(defmethod resolved-medium-ink ((medium basic-postscript-medium) (ink (eql +background+)))
  (resolved-medium-ink medium (medium-background medium)))

(defvar *already-warned-about-flipping-ink-for-postscript* nil)

(defmethod resolved-medium-ink ((medium basic-postscript-medium) (ink (eql +flipping-ink+)))
  (unless *already-warned-about-flipping-ink-for-postscript*
    (cerror "Continue using ~S for an ink; do this forever."
	    "Flipping ink is not implemented for Postscript media~*"
	    '+foreground+)
    (setf *already-warned-about-flipping-ink-for-postscript* t))
  (resolved-medium-ink medium (medium-foreground medium)))

;;; --- Don't ask.  This will go away when Rao shades go away.
;;; --- Actually, you should ask.  We need to do this so the ink 100 is
;;; canonicalized to +BLACK+ everywhere, so we are always comparing apples.
(defmethod resolved-medium-ink ((medium basic-postscript-medium) (ink (eql '100)))
  +black+)

(defmethod resolved-medium-ink ((medium basic-postscript-medium) (ink (eql '0)))
  +white+)

;;; This is the default method.
(defmethod resolved-medium-ink ((medium basic-postscript-medium) ink)
  ink)

(defmacro define-feature-method (name code &rest other-arguments)
  `(define-group ,name define-feature-method
     (defmethod ,name ((medium basic-postscript-medium))
       (maybe-send-feature medium ',name ,code ,@other-arguments))))

(define-feature-method emit-postscript-prologue *postscript-prologue*
  "(atend)"					;; DocumentFonts
  (or (w::printer-document-title medium) "Untitled document") ;; Title
  "Unknown user"  ;; Creator --- CL doesn't provide a portable user-name function!!
  (multiple-value-bind (second minute hour day month year) (get-decoded-time)
    ;; CreationDate
    (format nil "~2,'0D/~2,'0D/~2,'0D ~2,'0D:~2,'0D:~2,'0D"
	    year month day hour minute second))  )

(define-feature-method emit-postscript-epilogue *postscript-epilogue*
  (format nil "~{~A~^ ~}~%" (map 'list #'pfd-pretty-name (printer-font-map medium))))

(define-feature-method emit-mmcm-pattern-program *mmcm-code*)

(define-feature-method emit-ellipse-program *ps-ellipse-code*)

(defmethod show-page ((medium basic-postscript-medium))
  (showpage (slot-value medium 'printer-stream)))

(defmethod use-font ((medium basic-postscript-medium) printer-stream pfd)
  (let* ((font-map (slot-value medium 'font-map))
	 (font-index (position pfd font-map)))
    (block new-font-index
      (unless font-index
	(let ((new-font-index (vector-push pfd font-map)))
	  (when (null new-font-index)
	    (cerror "Use font ~2*~S instead"
		    "Font map overflow for ~S; can't use font ~S"
		    medium pfd (aref font-map 0))
	    (setf font-index 0)
	    (return-from new-font-index))
	  (estfont printer-stream new-font-index
		   (pfd-point-size pfd) (pface-name (pfd-face pfd)))
	  (setf font-index new-font-index))))
    (setfont printer-stream font-index)))
