;;; -*- Mode: LISP; Syntax: Common-lisp; Package: ON-POSTSCRIPT; Base: 10 -*-

"Copyright (c) 1991 by International Lisp Associates.  All rights reserved."

(in-package :on-postscript)

;;; Functions for sending operations to the stream

;;; Note: unlike the previous version, we send delimiters AFTER the operands/operators.

(defun ps-operation (operation stream)
  (write-string operation stream)
  (if (or (string= operation "closepath")
	  (<= (length operation) 1))
      (write-char #\space stream)
      (terpri stream)))

(defun ps-decimal (value stream)
  (format stream "~D " value))

(defun ps-vector (value stream &optional optimal-flonize)
  (write-char #\[ stream)
  (doseq (element value)
    (if optimal-flonize
	(ps-optimal-flonize element stream)
	(ps-decimal element stream)))
  (write-char #\] stream)
  (write-char #\space stream))

(defun ps-symbol (value stream)
  (format stream "/~A " value))

;;; Output a number as compactly as possible.
;;; This prints an optional minus sign, the integer part unless it's
;;; zero, a decimal point if required, and up to three decimal digits
;;; after it.  This might have some floating-point pathology to it, but
;;; no worse than the previous version.
(defun ps-optimal-flonize (x stream)
  (declare (inline digprint afp-intprint))
  (etypecase x
    (integer (ps-decimal x stream))
    (ratio (ps-optimal-flonize (float x 1.0) stream))
    (float
      (locally
	  (declare (float x))
	(when (minusp x)
	  (write-char #\- stream)
	  (setq x (- x)))
	;; pathological case: almost zero.
	;; --- Hmm.  The Genera version of this seems to print the exact value when <= 1.
	(when (< x .0005)
	  (write-char #\0 stream) (write-char #\space stream)
	  (return-from ps-optimal-flonize))
	(multiple-value-bind (intpart floatpart) (floor (+ x .0005) 1.0)
	  (declare (float floatpart)
		   (fixnum intpart))
	  (unless (zerop intpart) (format stream "~D" intpart))
	  (let ((floatpart (floor (* floatpart 1000.0))))
	    (declare (fixnum floatpart))
	    (unless (zerop floatpart)
	      (write-char #\. stream)
	      (multiple-value-bind (d12 d3) (floor floatpart 10)
		(declare (fixnum d12 d2))
		(multiple-value-bind (d1 d2) (floor d12 10)
		  (declare (fixnum d1 d2))

		  ;; Under Genera, (DIGIT-CHAR digit) takes approximately 2.5
		  ;; times as long as (CODE-CHAR (+ (CHAR-CODE #\0) digit)), even
		  ;; after the compiler optimization which rewrites it to be
		  ;; (AREF *WEIGHT-DIGITS* digit).  Enforce this better optimization.
		  (macrolet ((digit-char (digit)
					 `(code-char (+ (char-code #\0) ,digit))))

		    (if (zerop d3)
			(if (zerop d2)
			    (write-char (digit-char d1) stream)
			    (progn (write-char (digit-char d1) stream)
				   (write-char (digit-char d2) stream)))
			(progn (write-char (digit-char d1) stream)
			       (write-char (digit-char d2) stream)
			       (write-char (digit-char d3) stream))))))))))
      (write-char #\space stream))))

(defun ps-carefully-show-string (printer-stream string-or-char start end)
  (write-char #\( printer-stream)
  (etypecase string-or-char
    (string
      (do-delimited-substrings ((string-or-char :start start :end end)
				(substring-start substring-end delimiter))
			       (write-string string-or-char printer-stream
					     :start substring-start :end substring-end)
	((#\( #\) #\\)
	 (write-char #\\ printer-stream)
	 (write-char delimiter printer-stream))))
    (character (when (or (eql string-or-char #\()
			 (eql string-or-char #\))
			 (eql string-or-char #\\))
		 (write-char #\\ printer-stream)
		 (write-char string-or-char printer-stream))))
  (write-char #\) printer-stream)
  (write-char #\space printer-stream)
  (ps-operation "show" printer-stream))

#+Genera
(progn
  (scl:defprop define-postscript-operation lisp:defun zwei:definition-function-spec-type))

;;; Syntax of the operator is:
;;; () or absent means use the name of the function.
;;; A symbol means use string-downcase on its name
;;; A list means to apply FORMAT to the list (at compile time...)
(defmacro define-postscript-operation (name &optional operator &rest operands)
  (when (null operator) (setf operator name))
  (if (listp operator)
      (setf operator (apply #'format nil operator))
      (setf operator (string-downcase (string operator))))
  (labels ((parse-operand (operand)
	     (if (listp operand) (values (first operand) (second operand))
		 (values operand 'integer)))
	   (generate-operand (operand)
	     (multiple-value-bind (operand type) (parse-operand operand)
	       (ecase type
		 (integer `(ps-decimal ,operand stream))
		 (float   `(ps-optimal-flonize ,operand stream))
		 (vector  `(ps-vector ,operand stream))
		 (fvector `(ps-vector ,operand stream t))
		 (symbol  `(ps-symbol ,operand stream))))))
    `(define-group ,name define-postscript-operation
       (defun ,name (stream ,@(mapcar #'parse-operand operands))
	 ,@(mapcar #'generate-operand operands)
	 (ps-operation ,operator stream)))))

(define-postscript-operation setgray ()
  (gray-level float))

(define-postscript-operation setcolor setrgbcolor
  (red float)
  (green float)
  (blue float))

(define-postscript-operation concat ()
  (transformation-matrix fvector))

(define-postscript-operation setlinewidth ()
  width)
(define-postscript-operation setlinejoin ()
  join-type-number)
(define-postscript-operation setlinecap ()
  cap-type-number)

(define-postscript-operation setdash ()
  (dash-pattern vector)
  initial-dash-phase)

(define-postscript-operation moveto m
  (x float)
  (y float))

(define-postscript-operation lineto ()
  (x float)
  (y float))

(define-postscript-operation arc ()
  (x float)
  (y float)
  (radius float)
  from-angle
  to-angle)

(define-postscript-operation clip)
(define-postscript-operation stroke)
(define-postscript-operation ps-fill fill)
(define-postscript-operation patfill)
(define-postscript-operation closepath)
(define-postscript-operation newpath)
(define-postscript-operation showpage ("showpage~%%%Page: ? ?")) ;Put %%Page: ? everywhere.

(define-postscript-operation estfont ()
  font-index
  point-size
  (font-name symbol))

(define-postscript-operation setfont f
  font-index)

(define-postscript-operation scale ()
  (x float)
  (y float))

(define-postscript-operation translate ()
  (x float)
  (y float))
			     

;;; Some postscript code fragments.

;;; Prologue.  Always sent to any postscript output device.
(defparameter *postscript-prologue*
    #.
    (multiple-value-bind (major minor description)
	(clim-version)
      (multiple-value-bind (second minute hour day month year) (get-decoded-time)
	(format nil "%!~A~%~
		     ~{%%~A: ~~A~%~}~
		     %%EndComments~2%~
		     %%% Postscript output from CLIM ~D, ECO ~D~@[ (~A)~]~%~
		     %%% Postscript compiled ~D/~2,'0D/~2,'0D ~2,'0D:~2,'0D:~2,'0D~%~
		     ~{~A~%~}"
		"PS-Adobe-2.0" ;; Magic Unix string.
		'("DocumentFonts" "Title" "Creator" "CreationDate")
		major minor description
		year month day hour minute second
		;; Magic incantations:
		'("/saveobj save def"
		  "statusdict /waittimeout 30 put"
		  "/fontarray 30 array def"
		  "/f {fontarray exch get setfont} bind def"
		  "/fontscale [1.0 0.0 0.0 -1.0 0.0 0.0] def"
		  ;; Can't use scalefont: Need to scale by -1 in the vertical direction.
		  "/estfont {findfont exch 1 copy 
	  fontscale 0 3 2 roll put
	  fontscale 3 3 2 roll neg put
	  fontscale makefont
	  fontarray 3 1 roll put} bind def"
		  "/m {moveto} bind def")))))

(defparameter *postscript-epilogue*
	      (format nil "~{~A~%~}~
			   %%Trailer~%~
			   ~{%%~A: ~~A~%~}"
		      '("showpage"
			"saveobj restore")
		      '("DocumentFonts")))



;; "Not suitable for ritual use."
(defparameter *ps-ellipse-code*
"
/emtrx matrix def
/elpd 8 dict def
/ellipse {{arc} ellipsei} def
/ellipsen {{arcn} ellipsei} def
/ellipsei {elpd begin /arcp exch def /ea exch def /sa exch def /yra exch def /xra exch def
      /y exch def /x exch def
emtrx currentmatrix
x y translate xra yra scale 0 0 1 sa ea arcp setmatrix end} def
")

(defparameter *mmcm-code*
	;; for drawing filled patterns
	"/imgdict 12 dict def
%draw image.  One source pixel to one user space unit.
%width width-rounded-up height
/img { imgdict begin
	[/height /bitwidth /width ] {exch def} forall
	/nbits bitwidth height mul def
	/str 100 string def
	nbits 0 ne {
	  gsave width height scale
	  bitwidth height true [bitwidth 0 0 height neg 0 height] 
	    {	nbits 800 ge {/nbits nbits 800 sub def str} 
			       {nbits 8 idiv string /nbits 0 def}
			    ifelse 
		  currentfile exch readhexstring pop}
	  imagemask grestore
        } if end
	} def
/fmod { 2 copy div floor mul sub } bind def
%draw patterned rectangle.  One source pixel to scale device units (ignoring user scale).
%width height pattern scale
/pat { imgdict begin gsave
	[/scal /patseq ] {exch def} forall
	/patheight patseq length def
	/patwidth patseq 0 get length 8 mul def
%back up to an even phase boundary
	/pswidth patwidth scal mul def
	/psheight patheight scal mul def
	pswidth psheight idtransform
	0 0 transform psheight fmod neg exch pswidth fmod neg exch idtransform
        3 -1 roll exch dup 0 gt {add} {exch pop} ifelse
        3 1 roll dup 0 gt {add} {exch pop} ifelse exch 2 copy translate
	3 -1 roll exch abs add 3 1 roll abs add exch dtransform
	psheight div abs ceiling cvi patheight mul /height exch def
	pswidth div abs ceiling cvi patwidth mul /width exch def
	width 0 ne { height 0 ne {
	/scanline -1 def /linebits 0 def
	width height idtransform abs scale scal dup scale
	width height true [width 0 0 height neg 0 height] 
	{ linebits 0 le { /linebits width def
			  /scanline scanline 1 add patheight mod def
			  /linepat patseq scanline get def
			} if
	/linebits linebits patwidth sub def linepat }
	imagemask } if } if grestore end
      } def
%draw pattern in all of visible area.
%pattern scale opaque-p
/patfill1 { initmatrix clippath
%condition-case for nocurrentpoint, returning empty rectangle
             errordict begin
               /nocurrentpoint dup dup load exch { pop 0 0 0 0 } def 
                 pathbbox
               6 -2 roll def end
             4 2 roll 2 copy translate 4 -2 roll
             3 -1 roll sub 3 1 roll exch sub exch
             3 -1 roll { 2 copy gsave 1 setgray newpath
                         0 0 moveto 0 exch lineto 0 rlineto currentpoint pop 0 lineto
                         closepath fill grestore } if
             4 -2 roll pat } def
%like fill, etc. but with pattern, scale and opaque-p options.
/patfill { gsave clip patfill1 grestore newpath } def
/pateofill { gsave eoclip patfill1 grestore newpath } def
/patstroke { gsave strokepath clip patfill1 grestore newpath } def
 ")

;;; Copied verbatim from Y-windows postscript-implementation.
;;; I don't think we're currently using this for anything. --- rsl

;;; Moved the MAYBE-SEND-FEATURE call to SEND-PATTERN-AND-FILL in POSTSCRIPT-CLG.

(defun send-pattern (printer-stream pattern)
  (write-string "[ " printer-stream)
  (let ((height (array-dimension pattern 0))
	(width (array-dimension pattern 1)))
    (dotimes (j height)
      (write-char #\< printer-stream)
      (send-raster printer-stream pattern 0 j width (1+ j) nil)
      (write-string "> " printer-stream)))
  (write-char #\] printer-stream)
  (terpri printer-stream)
  (write-string " 4 true " printer-stream))

(defun send-raster (stream raster left top right bottom &optional (terpri t))
  (assert (= bottom (1+ top)))
  (unless (zerop (rem right 8))
    (error "Sorry, can't hack right /= 0 (mod 8); you have ~D" right))
  ;; this really wants with-stack-array.  Oh well.  Cons city...
  (let ((arr (make-array (array-total-size raster)
			 :element-type '(unsigned-byte 8) :displaced-to raster)))
    (ci::with-temporary-string (buf :length 100)
      (let ((bytes-per-row (truncate (array-dimension raster 1) 8))
	    (bytes-per-raster (ceiling (- right left) 8)))
	(let ((toprow (* top bytes-per-row))
	      (botrow (* bottom bytes-per-row))
	      (bigend-digit-char "084c2a6e195d3b7f")
	      (j 0)
	      #+Genera (buf buf)
	      #+Genera (arr arr)
	      )
	  #+Genera
	  (declare (sys:array-register bigend-digit-char buf arr))
	  (flet ((force-buf ()
		   (setf (fill-pointer buf) j)
		   (ci::with-temp-substring (subbuf buf 0 j)
		     (write-string subbuf stream)
		     (when terpri
		       (terpri stream)))))
	    (do ((index (- botrow bytes-per-row) (- index bytes-per-row)))
		((< index toprow))
	      (do ((counter 0 (1+ counter))
		   (i index (1+ i)))
		  ((>= counter bytes-per-raster))
		(let ((byte (aref arr i)))
		  (setf (aref buf j) 
		    (aref bigend-digit-char
			  (ldb (byte 4 0) byte)))
		  (setf (aref buf (1+ j))
		    (aref bigend-digit-char
			  (ldb (byte 4 4) byte)))
		  (incf j 2)
		  (when (> j 80)
		    (force-buf)))))
	    (when (> j 0) (force-buf))))))))
