;;; -*- Mode: LISP; Syntax: Common-lisp; Package: ON-POSTSCRIPT; Base: 10 -*-

"Copyright (c) 1991 by International Lisp Associates.  All rights reserved."

(in-package :on-postscript)

(defstruct (postscript-printer-information
	     (:conc-name ppi-)
	     (:print-function (lambda (ppi stream depth)
				(declare (ignore depth))
				(print-unreadable-object (ppi stream :type t :identity t)
				  (format stream "~A" (ppi-name ppi))))))
  name
  outside-width
  outside-height
  left-margin
  top-margin
  right-margin
  bottom-margin
  device-scale-factor
  metric-scale-factor
  font-table)

#+Genera (scl:defprop define-postscript-printer "Postscript printer description"
		      si:definition-type-name)

(defmacro define-postscript-printer (name &body options)
  (let ((like nil)
	(outside-width nil)
	(outside-height nil)
	(margins nil)
	(left-margin nil)
	(top-margin nil)
	(right-margin nil)
	(bottom-margin nil)
	(device-scale-factor nil)
	(metric-scale-factor nil))
    (dolist (option options)
      (multiple-value-bind (option value value-p)
	  (if (listp option)
	      (values (car option) (cdr option) t)
	      (values option nil nil))
	(macrolet ((option-error (format-string &rest format-args)
		     `(warn "For ~S ~S: ~@?." 'define-postscript-printer name ,format-string
			    option ,@format-args))
		   (one-value () `(if value-p
				      (progn (when (cdr value)
					       (option-error "Option ~S only takes one value"))
					     (first value))
				      (progn (option-error "No value supplied for option ~S")
					     1)))
		   (float-one-value () `(float (one-value) 1.0s0))
		   (no-values () `(when value (option-error "Option ~S takes no values")))
		   (n-values () `(if value-p value 
				     (progn (option-error "No values supplied for option ~S")
					    nil))))
	  (case option
	    (:like (setf like (one-value)))
	    (:width (setf outside-width (float-one-value)))
	    (:height (setf outside-height (float-one-value)))
	    (:margins (setf margins (float-one-value)))
	    (:left-margin (setf left-margin (float-one-value)))
	    (:top-margin (setf top-margin (float-one-value)))
	    (:right-margin (setf right-margin (float-one-value)))
	    (:bottom-margin (setf bottom-margin (float-one-value)))	
	    (:device-scale-factor (setf device-scale-factor (float-one-value)))	
	    (:metric-scale-factor (setf metric-scale-factor (float-one-value)))
	    (otherwise (option-error "Option ~S not recognized"))))))
    `(define-group ,name define-postscript-printer
       (define-postscript-printer-load-time
	 ',name
	 :like ',like
	 :outside-width ',outside-width
	 :outside-height ',outside-height
	 :left-margin ',(or left-margin margins)
	 :top-margin ',(or top-margin margins)
	 :right-margin ',(or right-margin margins)
	 :bottom-margin ',(or bottom-margin margins)
	 :device-scale-factor ',device-scale-factor
	 :metric-scale-factor ',metric-scale-factor))))

(defvar *postscript-printer-descriptions* nil)

(defun find-postscript-printer-description (name &key (if-does-not-exist :error))
  (or (and (typep name 'postscript-printer-information) name)
      (find name *postscript-printer-descriptions* :key #'ppi-name)
      (ecase if-does-not-exist
	(:error (error "Can't find postscript printer description ~S" name))
	(:create (merge-postscript-printer-information
		   (make-postscript-printer-information :name name :font-table nil)
		   (find-postscript-printer-description :default :if-does-not-exist nil)))
	((nil) nil))))

(defun ppi-device-margins (ppi)
  (let ((device-scale-factor (ppi-device-scale-factor ppi)))
    (values (* (ppi-left-margin ppi) device-scale-factor)
	    (* (ppi-top-margin ppi) device-scale-factor)
	    (* (ppi-right-margin ppi) device-scale-factor)
	    (* (ppi-bottom-margin ppi) device-scale-factor))))

(defun ppi-device-inside-edges (ppi)
  (let* ((left (ppi-left-margin ppi))
	 (top (ppi-top-margin ppi))
	 (right (- (ppi-outside-width ppi) (ppi-right-margin ppi)))
	 (bottom (- (ppi-outside-height ppi) (ppi-bottom-margin ppi)))
	 (scale-factor (ppi-device-scale-factor ppi)))
    (values (round (* left scale-factor)) (round (* top scale-factor))
	    (round (* right scale-factor)) (round (* bottom scale-factor)))))

(defun ppi-device-outside-edges (ppi)
  (let ((scale-factor (ppi-device-scale-factor ppi)))
    (values 0 0
	    (round (* (ppi-outside-width ppi) scale-factor))
	    (round (* (ppi-outside-height ppi) scale-factor)))))

(defun ppi-device-metric-scale-factor (ppi)
  (* (ppi-device-scale-factor ppi) (ppi-metric-scale-factor ppi)))

(defun ppi-metric-margins (ppi)
  (let ((metric-scale-factor (ppi-metric-scale-factor ppi)))
    (values (* (ppi-left-margin ppi) metric-scale-factor)
	    (* (ppi-top-margin ppi) metric-scale-factor)
	    (* (ppi-right-margin ppi) metric-scale-factor)
	    (* (ppi-bottom-margin ppi) metric-scale-factor))))

(defun define-postscript-printer-load-time (name &key
						 like
						 outside-width
						 outside-height
						 left-margin
						 top-margin
						 right-margin
						 bottom-margin
						 device-scale-factor
						 metric-scale-factor)
  (let ((ppi (find-postscript-printer-description name :if-does-not-exist nil)))
    (unless ppi
      (push 
	(setf ppi (find-postscript-printer-description name :if-does-not-exist :create))
	*postscript-printer-descriptions*))
    (when like
      (merge-postscript-printer-information
	ppi (find-postscript-printer-description like)))
    (setf (ppi-outside-width ppi) (or outside-width (ppi-outside-width ppi) 0.0)
	  (ppi-outside-height ppi) (or outside-height (ppi-outside-height ppi) 0.0)
	  (ppi-left-margin ppi) (or left-margin (ppi-left-margin ppi) 0.0)
	  (ppi-top-margin ppi) (or top-margin (ppi-top-margin ppi) 0.0)
	  (ppi-right-margin ppi) (or right-margin (ppi-right-margin ppi) 0.0)
	  (ppi-bottom-margin ppi) (or bottom-margin (ppi-bottom-margin ppi) 0.0)
	  (ppi-device-scale-factor ppi) (or device-scale-factor
					    (ppi-device-scale-factor ppi) 0.0)
	  (ppi-metric-scale-factor ppi)
	    (or metric-scale-factor (ppi-metric-scale-factor ppi) 1.0)))
  name)

(defun merge-postscript-printer-information (into-ppi from-ppi)
  (when from-ppi
    ;; Copy data from one PPI into another.
    (setf (ppi-outside-width into-ppi) (ppi-outside-width from-ppi)
	  (ppi-outside-height into-ppi) (ppi-outside-height from-ppi)
	  (ppi-left-margin into-ppi) (ppi-left-margin from-ppi)
	  (ppi-top-margin into-ppi) (ppi-top-margin from-ppi)
	  (ppi-right-margin into-ppi) (ppi-right-margin from-ppi)
	  (ppi-bottom-margin into-ppi) (ppi-bottom-margin from-ppi)
	  (ppi-device-scale-factor into-ppi) (ppi-device-scale-factor from-ppi)
	  (ppi-metric-scale-factor into-ppi) (ppi-metric-scale-factor from-ppi)
	  ;; Merge font table
	  (ppi-font-table into-ppi) (append (ppi-font-table into-ppi)
					    (ppi-font-table from-ppi))))
  into-ppi)
