;;; -*- Mode: LISP; Syntax: Common-lisp; Package: ON-POSTSCRIPT; Base: 10 -*-

"Copyright (c) 1991 by International Lisp Associates.  All rights reserved."

(in-package :on-postscript)

(defclass postscript-clg-medium (basic-clg-medium basic-postscript-medium)
    ((printer-ink :initform +black+)
     (printer-transformation :initform (make-array 6 :initial-contents '(1 0 0 1 0 0)))
     (printer-clipping-region :initform +everywhere+)
     (printer-text-style :initform *default-text-style*)))

(defclass postscript-clg-display-medium
	  (postscript-clg-medium basic-clg-display-medium)
    ())

(defclass postscript-clg-pixmap-medium
	  (postscript-clg-medium basic-clg-pixmap-medium)
    ())

(defmethod display-medium-type ((port postscript-port) (type (eql :clim)))
  'postscript-clg-display-medium)

#+Ignore ;; --- Can this be right?
(defmethod pixmap-medium-type ((port postscript-port) (type (eql :clim)))
  'postscript-clg-pixmap-medium)

(defmethod initialize-graphics-state :after ((medium postscript-clg-medium))
  (with-slots (printer-ink printer-transformation printer-clipping-region printer-text-style)
	      medium
    (replace printer-transformation
	     (multiple-value-list (decompose-transformation
				    (medium-transformation medium))))
    (setf printer-ink (resolved-medium-ink medium (medium-ink medium))
	  printer-clipping-region (medium-clipping-region medium)
	  printer-text-style (medium-text-style medium))))

;;; This is how we make sure that the "Graphics Context" is correct.
;;; This is required to return a raster when one is needed.
(defmethod prepare-for-graphics ((medium postscript-clg-medium) gsaver
				 ink transformation clipping-region text-style)
  (declare (values gsave-done? raster))
  (let ((printer-stream (slot-value medium 'printer-stream))
	gsave-done? raster)
    (macrolet ((gsave () `(funcall gsaver)))
      (with-slots (printer-ink color-printer-p printer-transformation
		   printer-clipping-region printer-text-style)
		  medium
	;; --- Assume: INKs are immutable.
	(unless (or (null ink) (eql ink printer-ink))
	  (gsave)
	  (setf raster (prepare-printer-for-new-ink printer-stream ink color-printer-p)
		printer-ink ink))
	;; Transformations are not immutable, but even if they were, we
	;; would probably want this representation for Postscript anyway:
	;;  [??? Maybe not....]
	(when transformation
	  (multiple-value-bind (mxx mxy myx myy tx ty)
	      (decompose-transformation transformation)
	    (let ((printer-transformation printer-transformation)
		  (good-ty nil))
	      (macrolet ((transform-element (name)
			   `(svref printer-transformation
				   ,(second
				      (assoc name '((mxx 0) (mxy 1) (myx 2)
						    (myy 3) ( tx 4) ( ty 5)))))))
		(unless (and (eql mxx (transform-element mxx))
			     (eql mxy (transform-element mxy))
			     (eql myx (transform-element myx))
			     (eql myy (transform-element myy))
			     (eql  tx (transform-element  tx))
			     (eql  ty (transform-element  ty)))
		  (unwind-protect
		      (progn 
			(setf (transform-element mxx) mxx
			      (transform-element mxy) mxy
			      (transform-element myx) myx
			      (transform-element myy) myy
			      (transform-element  tx)  tx
			      (transform-element  ty)  ty)
			(gsave)
			(prepare-printer-for-new-transformation
			  printer-stream printer-transformation)
			(setf good-ty ty))
		    ;; If we fail to transmit the transformation, make sure we
		    ;; transmit a good one next time.
		    (setf (transform-element ty) good-ty)))))))
	;; --- Assume: CLIPPING-REGIONs are immutable (almost certainly false!).
	(unless (or (null clipping-region) (eql clipping-region printer-clipping-region))
	  (gsave)
	  (prepare-printer-for-new-clipping-region printer-stream clipping-region)
	  (setf printer-clipping-region clipping-region))
	;; --- Assume: TEXT-STYLEs are immutable.
	(unless (or (null text-style) (eql text-style printer-text-style))
	  (gsave)
	  (prepare-printer-for-new-text-style printer-stream text-style medium)
	  (setf printer-text-style text-style)))
      (values gsave-done? raster))))

(defmethod with-graphics-context-1 ((medium postscript-clg-medium)
				    ink transformation clipping-region text-style line-style
				    continuation)
  (let ((printer-stream (slot-value medium 'printer-stream))
	(printer-transformation (slot-value medium 'printer-transformation))
	(gsave-done? nil))
    (flet ((gsave ()
	     (unless gsave-done?
	       (ps-operation "gsave" printer-stream)
	       (setf gsave-done? t)))
	   (grestore ()
	     (when gsave-done?
	       (ps-operation "grestore" printer-stream)
	       (setf gsave-done? nil))))
    (declare (dynamic-extent #'gsave #'grestore))
    (with-slots (printer-ink printer-clipping-region printer-text-style)
		medium
      (letf-globally ((printer-ink)
		      ;; Have to save each slot here...
		      ((aref printer-transformation 0))
		      ((aref printer-transformation 1))
		      ((aref printer-transformation 2))
		      ((aref printer-transformation 3))
		      ((aref printer-transformation 4))
		      ((aref printer-transformation 5))
		      (printer-text-style)
		      (printer-clipping-region))
	(multiple-value-bind (raster)
	    (prepare-for-graphics medium #'gsave
				  ink transformation clipping-region text-style)
	  (when line-style
	    (prepare-for-line-style printer-stream line-style #'gsave))
	  (unwind-protect (funcall continuation medium printer-stream raster)
	    (grestore))))))))

(defmacro with-graphics-context ((medium &key ink transformation clipping-region
					 text-style line-style raster medium-required)
				 &body body &environment environment)
  (multiple-value-bind (documentation declarations body)
      (extract-declarations body environment)
    (declare (ignore documentation))
    (unless medium-required
      (push `(declare (ignore ,medium)) declarations))
    (when (null raster)
      (setf raster (gensymbol 'raster))
      (push `(declare (ignore ,raster)) body))
    (flet ((make-accessor (argument accessor)
	     (cond ((eql argument t) `(,accessor ,medium))
		   (argument)
		   (t nil)))
	   (make-ink-accessor ()
	     (cond ((eql ink t) `(resolved-medium-ink ,medium (medium-ink ,medium)))
		   (ink)
		   (t nil))))
      `(flet ((with-graphics-context (,medium printer-stream ,raster)
		,@declarations
		,@body))
	 (declare (dynamic-extent #'with-graphics-context))
	 (with-graphics-context-1
	   ,medium
	   ,(make-ink-accessor)
	   ,(make-accessor transformation 'medium-transformation)
	   ,(make-accessor clipping-region 'medium-clipping-region)
	   ,(make-accessor text-style 'medium-text-style)
	   ,(make-accessor line-style 'medium-line-style)
	   #'with-graphics-context)))))

(defmacro define-postscript-graphics-method (name medium-slots
					     options
					     &body body)
  (assert (null (set-difference options '(:line-style :text-style))) ()
	  "The options supported in ~S are only ~{~S~^, ~}."
	  'define-postscript-graphics-method '(:line-style :text-style))
  ;; We will be binding this in WITH-GRAPHICS-CONTEXT:
  (setf medium-slots (remove 'printer-stream medium-slots))
  (let* ((gf (find-graphics-function name))
	 (name* (gf-spread-function-name gf))
	 (type (gf-type gf))
	 (raster (and (eql type ':area) (gensymbol 'raster)))
	 (epilogue (ecase type
		     (:text ())
		     ((:point :line)
		      `((stroke printer-stream)))
		     ((:area) ;; Implies FILLED argument to the method!
		      `((if filled
			    (if ,raster
				(send-pattern-and-fill medium ,raster printer-stream)
				(ps-fill printer-stream))
			    (stroke printer-stream)))))))
    `(define-group ,name define-postscript-graphics-methods
       (define-graphics-function-method ,name* ((medium postscript-clg-medium)
						,@(gf-method-lambda-list gf))
	 (with-graphics-context
	   (medium :ink t :transformation t
		   :clipping-region t :raster ,raster
		   :text-style ,(not (null (member ':text-style options)))
		   :line-style ,(not (null (member ':line-style options)))
		   :medium-required ,(or raster medium-slots (eql type ':text)))
	   (newpath printer-stream)
	   (let ,(with-collection
		   (dolist (slot-name medium-slots)
		     (collect `(,slot-name (slot-value medium ',slot-name)))))
	     (block ,name
	       ,@body)
	     ,@epilogue))))))


(defun send-pattern-and-fill (medium pattern printer-stream)
  (maybe-send-feature medium 'mmcm-pattern-program *mmcm-code*)
  (send-pattern printer-stream pattern)
  (ps-fill printer-stream))

(define-postscript-graphics-method draw-point (printer-stream)
				   (:line-style)
  (moveto printer-stream point-x point-y)
  (lineto printer-stream point-x point-y))

(define-postscript-graphics-method draw-line (printer-stream)
				   (:line-style)
  (moveto printer-stream from-x from-y)
  (lineto printer-stream to-x to-y))

(define-postscript-graphics-method draw-rectangle (printer-stream)
				   (:line-style)
  (moveto printer-stream x1 y1)
  (lineto printer-stream x1 y2)
  (lineto printer-stream x2 y2)
  (lineto printer-stream x2 y1)
  (closepath printer-stream))

(define-postscript-graphics-method draw-polygon (printer-stream)
				   (:line-style)
  (if (listp coordinate-sequence)
      (let* ((x (pop coordinate-sequence))
	     (y (pop coordinate-sequence)))
	(moveto printer-stream x y)
	(loop (when (null coordinate-sequence) (return))
	      (setf x (pop coordinate-sequence)
		    y (pop coordinate-sequence))
	      (lineto printer-stream x y)))
      (do ((i 2 (+ i 2))
	   (length (length coordinate-sequence)))
	  ((< i length))
	(lineto printer-stream
		(aref coordinate-sequence i) (aref coordinate-sequence (1+ i)))))
  (if closed (closepath printer-stream)
      (setf filled nil)))			;; --- This might be controversial.

(defun radians->degrees (radians)
  (* radians (/ 360.0 2pi)))

(define-postscript-graphics-method draw-circle (printer-stream)
				   (:line-style)
  (when (< end-angle start-angle) (incf end-angle 2pi)) ;; ?? Cribbed from Dennis' code.
  (arc printer-stream center-x center-y radius
	     (radians->degrees start-angle) (radians->degrees end-angle)))
  
;;; Here's a coup: we define ellipses in terms of circles, instead of vice-versa.
;;; We draw a circle at 0,0 with radius 1, under a transformation which transforms
;;; 0,0 to (CX,CY), 0,1 to (CX+R1DX,CY+R1DY) and 1,0 to (CX+R2DX,CY+R2DY).
;;; Neat, huh?  
;;; --- I wonder if this works adequately for non-zero/2pi start/end angles.
(define-graphics-function-method draw-ellipse*
				 ((medium postscript-clg-medium)
				  center-x center-y
				  radius-1-dx radius-1-dy radius-2-dx radius-2-dy
				  &key (filled t) (start-angle 0) (end-angle 2pi))
  (draw-circle* medium 0 0 1
		:filled filled :start-angle start-angle :end-angle end-angle
		:transformation (make-3-point-transformation*
				  0 0 center-x center-y
				  0 1 (+ center-x radius-1-dx) (+ center-y  radius-1-dy)
				  1 0 (+ center-x radius-2-dx) (+ center-y  radius-2-dy))))

;;; string-or-char point-x point-y start end align-x align-y toward-point transform-glyphs
(define-postscript-graphics-method draw-text (printer-stream)
				   (:text-style)
  (if (characterp string-or-char)
      (let ((ch1buf (slot-value medium 'ch1buf)))
	(setf (aref ch1buf 0) string-or-char)
	(setf string-or-char ch1buf start 0 end 1))
      (when (null end) (setf end (length string-or-char))))
  (let ((text-style (medium-text-style medium)))
    (ecase align-x
      (:left)
      (:right (decf point-x (string-width string-or-char text-style medium
					  :start start :end end)))
      (:center (decf point-x (/ (string-width string-or-char text-style medium
					      :start start :end end)
				2))))
    (ecase align-y
      (:baseline)
      (:top (incf point-y (text-style-ascent text-style medium)))
      (:bottom (decf point-y (text-style-descent text-style medium)))
      (:center (incf point-y (/ (- (text-style-ascent text-style medium)
				   (text-style-descent text-style medium))
				2))))
    (moveto printer-stream point-x point-y)
    (ps-carefully-show-string printer-stream string-or-char start end)))

