;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  DWS PIXRECT File Formats
;;;

(defmacro pixmap-internal-write-encoded-byte (value file)
  `(format ,file "~d " ,value))

(defmacro pixmap-internal-update-output-file (count value file)
  `(if (< ,count 3)
       (if (= ,value #x80)
	   (if (= ,count 1)
	       (progn (pixmap-internal-write-encoded-byte #x80 ,file)
		      (pixmap-internal-write-encoded-byte #x00 ,file))
	       (progn (pixmap-internal-write-encoded-byte #x80 ,file)
		      (pixmap-internal-write-encoded-byte (1- ,count) ,file)
		      (pixmap-internal-write-encoded-byte #x80 ,file)))
	   (dotimes (i ,count)
	     #+(or ansi-90 Genera-Release-8) (declare (ignore i))
	     (pixmap-internal-write-encoded-byte ,value ,file)))
       (progn (pixmap-internal-write-encoded-byte #x80 ,file)
	      (pixmap-internal-write-encoded-byte (1- ,count) ,file)
	      (pixmap-internal-write-encoded-byte ,value ,file))))

(defun write-pixmap-to-file (pixmap filename &key (format 1) &allow-other-keys)
  (unless (pixmapp pixmap)
    (error "first argument must be wsii:pixmap clos object."))
  (when (and (= format 2) (> (color-count pixmap) 256))
    (error "pixmaps with more than 256 colors can not be written in format 2")
    )
  (with-open-file (file filename :direction :output)
    (let* ((color-count (color-count pixmap))
	   (format-string (format nil "~~~d,'0x" (ceiling (/ color-count 16))))
	   (width (pixmap-width pixmap))
	   (height (pixmap-height pixmap))
	   (pixels (pixmap-data pixmap)))
      (format file "; ~a~%" filename)
      (format file ";--------- version ----------------------------------------------------~%")
      (format file "~d~%" format)
      (format file ";--------- width, height, color-count ---------------------------------~%")
      (format file "~d ~d ~d~%" width height color-count)
      (format file ";--------- color-table: rgb [0.0 to 1.0] values, one per line ---------~%")
      (dolist (color (color-table pixmap))
	(format file "~d ~d ~d~%"
		(color-red-value color)
		(color-green-value color)
		(color-blue-value color)))
      (cond
	((= format 1)
	 (format file ";--------- data: indices to color-table entries -----------------------~%")
	 (dotimes (line height)
	   (let* ((index (* line width))
		  (max (* (+ line 1) width)))
	     (loop
	       (when (>= index max) (return))
	       (format file format-string (elt pixels index))
	       (incf index)
	       ))
	   (format file "~%")))
	((= format 2)
	 (format file ";--------- data: run-length-encoded indices to color-table entries ----~%")
	 (format file ";; 1, 2, & 3 byte runs.  first value == 128 indicates run, next value~%")
	 (format file ";; (length - 1) of run, 3rd value of run.  second value of zero indicates~%")
	 (format file ";; single pixel, value 128.  first value not 128 indicates single pixel.~%")
	 (let ((l (length pixels))
	       (index 1)
	       (value (elt pixels 0))
	       (count 1))
	   (loop (unless (< index l) (return))
		 (if (and (= value (elt pixels index)) (< count 255))
		     (incf count)
		     (progn
		       (pixmap-internal-update-output-file count value file)
		       (setq value (elt pixels index))
		       (setq count 1)))
		 (incf index))
	   (pixmap-internal-update-output-file count value file)))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Reading a DWS Pixrect file
;;;

(defun pixmap-internal-read-line (file)
  "read line from file, discarding comments."
  (let (line)
    (loop
      (setq line (read-line file))
      (if (not (char-equal (elt line 0) #\;))
	  (return line)))))


(defmethod pixmap-internal-read-pixmap-from-file ((pixmap pixmap) filename)
  (with-open-file (file filename :direction :input)
    (let* ((line (pixmap-internal-read-line file))
	   (version (read-from-string line)))
      (cond ((= version 1)
	     (pixmap-internal-read-file-version-1 pixmap file))
	    ((= version 2)
	     (pixmap-internal-read-file-version-2 pixmap file))
	    (t
	     (error "unknown pixmap file version ~s~%" version))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Reading a version one file
;;;

(defun pixmap-internal-parse-count-line (line)
  (let ((start 0)
	(a 0)
	(b 0)
	(c 0))
    (multiple-value-setq (a start) (read-from-string line nil 0 :start start))
    (multiple-value-setq (b start) (read-from-string line nil 0 :start start))
    (multiple-value-setq (c start) (read-from-string line nil 0 :start start))
    (list a b c)))

(defun pixmap-internal-read-pixels-from-line 
    (destination-vector line chars-per-pixel)
  "decode hex coding and push pixels into vector"
  (let ((index 0)
	(max (length line))
	(chars-per-pixel-minus-1 (1- chars-per-pixel)))
    (loop
      (when (>= index max) (return))
      (vector-push
	(let ((value (digit-char-p (elt line index) 16)))
	  (dotimes (i chars-per-pixel-minus-1)
	     #+(or ansi-90 Genera-Release-8) (declare (ignore i))
	    (setq value (+ (* 16 value)
			   (digit-char-p (elt line (incf index)) 16))))
	  value) 
	destination-vector)
      (incf index))))

(defmethod pixmap-internal-read-file-version-1 ((pixmap pixmap) file)
  (let* ((line (pixmap-internal-read-line file))
	 (stats (pixmap-internal-parse-count-line line))
	 (pixels ())
	 (width 0)
	 (chars-per-pixel 0)
	 (color-list ()))
    (setf (pixmap-width pixmap) (first stats))
    (setq width (first stats))
    (setf (pixmap-height pixmap) (second stats))
    (setf (color-count pixmap) (third stats))
    (setf pixels (make-array (* width (pixmap-height pixmap))
			     :element-type 'integer
			     :fill-pointer 0))
    (if (> (color-count pixmap) 0)
	(progn
	  (dotimes (index (color-count pixmap))
	    #+(or ansi-90 Genera-Release-8) (declare (ignore index))
	    (setq line (pixmap-internal-read-line file))
	    (setq stats (pixmap-internal-parse-count-line line))
	    (push (make-color-rgb (first stats) (second stats) (third stats))
		  color-list))
	  (setf (color-table pixmap) (nreverse color-list)))
	(progn
	  (setf (color-table pixmap) (list +white+ +black+))
	  (setf (color-count pixmap) 2)))
    (setq chars-per-pixel (ceiling (log (color-count pixmap) 16)))
    (dotimes (index (pixmap-height pixmap))
      #+(or ansi-90 Genera-Release-8) (declare (ignore index))
      (setq line (pixmap-internal-read-line file))
      (pixmap-internal-read-pixels-from-line pixels
					     line
					     chars-per-pixel))
    (setf (pixmap-data pixmap) pixels)
    pixmap))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Reading a version 2 file  (run-length-encoded)
;;;

(defmacro pixmap-internal-insert-pixels (buf index count pixel)
  `(dotimes (i (1+ ,count))
     #+(or ansi-90 Genera-Release-8) (declare (ignore i))
     (setf (elt ,buf ,index) ,pixel)
     (incf ,index)))     
     
(defun pixmap-internal-read-rle-pixels (pixmap file)
  (let* ((length;; number of pixels in pixmap
	  (* (pixmap-width pixmap)
	     (pixmap-height pixmap)))
	 (index;; current pixel in pixels
	  0)
	 (pixels;; input buffer of pixels
	  (make-array length :element-type '(unsigned-byte 8)))
	 value;; pixel value (#x80 indicated run of pixels)
	 count;; # of pixels in run
	 )
    (loop 
      (unless (< index length)
	(return))
      (setq value (read file))
      (setq count 1)
      (if (= value #x80)
	  (progn
	    (setq count (read file))
	    (when (not (zerop count))
	      (setq value (read file)))
	    (pixmap-internal-insert-pixels pixels index count value))
	  (pixmap-internal-insert-pixels pixels index 0 value)))
    (setf (pixmap-data pixmap) pixels)))

(defmethod pixmap-internal-read-file-version-2 ((pixmap pixmap) file)
  (let* ((line (pixmap-internal-read-line file))
	 (stats (pixmap-internal-parse-count-line line))
	 ;; below was not used 1/10/91 -- Doughty
	 ;; (width 0)
	 (color-list ()))
    (setf (pixmap-width pixmap) (first stats))
    ;; Well, ok, WIDTH was used here, but only set, which isn't really
    ;; used...
    ;; (setq width (first stats))
    (setf (pixmap-height pixmap) (second stats))
    (setf (color-count pixmap) (third stats))
    (if (> (color-count pixmap) 0)
	(progn
	  (dotimes (index (color-count pixmap))
	    #+(or ansi-90 Genera-Release-8) (declare (ignore index))
	    (setq line (pixmap-internal-read-line file))
	    (setq stats (pixmap-internal-parse-count-line line))
	    (push (make-color-rgb (first stats) (second stats) (third stats))
		  color-list))
	  (setf (color-table pixmap) (nreverse color-list)))
	(progn
	  (setf (color-table pixmap) 
		(list +white+ +black+))
	  (setf (color-count pixmap) 2)))
    (pixmap-internal-read-rle-pixels pixmap file)
    pixmap))