;;; -*- Mode:Lisp; Package: SILICA; Syntax:COMMON-LISP; Base:10 -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;;
;;; Indirect Inks
;;;

(defconstant +foreground+ :foreground)

(defconstant +background+ :background)

(defconstant +flipping-ink+ :flipping-ink)

;;;
;;; Standard Opacities
;;;

(defvar +clear-opacitiy+      1.0)
(defvar +extra-light-opacity+  .875)
(defvar +light-opacity+        .75)
(defvar +dark-light-opacity+   .625)
(defvar +medium-opacity+       .5)
(defvar +light-dark-opacity+   .375)
(defvar +dark-opacity+         .25)
(defvar +extra-dark-opacity+   .125)
(defvar +opaque-opacity+      0.0)
	  
;;;
;;; COLOR is the color object for graphics/text operations
;;; Contents:
;;;   RED-VALUE, GREEN-VALUE, BLUE-VALUE: rgb triplet
;;;
;;;  This implementation was derived from work done on the Aria project by
;;;  Franz, Xerox, and MCC.
;;;

(defstruct (color (:constructor %make-color)
		  (:print-function
		   (lambda (self stream level)
		     (declare (ignore level))
		     (format stream "#<color: ~sr ~sg ~sb>"
			     (color-red-value self)
			     (color-green-value self)
			     (color-blue-value self)))))
  (red-value 0 )
  (green-value 0 )
  (blue-value 0 ))

(defun make-color-rgb (red green blue)
  (%make-color :red-value red :green-value green :blue-value blue))

(defun make-color-ihs (intensity hue saturation)
  (multiple-value-bind (red green blue) (ihs-to-rgb intensity hue saturation)
    (%make-color :red-value red :green-value green :blue-value blue)))

(defun make-gray-color (whiteness)
  (%make-color :red-value whiteness
	       :green-value whiteness
	       :blue-value whiteness))
  
(defun externalize-color (COLOR)
  (format nil "#~4,'0X~4,'0X~4,'0X"
	  (round (* (color-red-value COLOR) 16384))
	  (round (* (color-green-value COLOR) 16384))
	  (round (* (color-blue-value COLOR) 16384))))

(defun color-rgb (color)
  (values (color-red-value color) 
	  (color-green-value color) 
	  (color-blue-value color)))

(defun color-ihs (color)
  (multiple-value-bind (r g b) (color-rgb color)
    (multiple-value-bind (i h s)
	(rgb-to-ihs r g b)
      (values i h s))))

(defmacro luminosity (COLOR)
  ;;   The following code is based on Foley and Van Dam, page 613, 
  ;;   Section 17.4.3 The YIQ Color Model.  There they describe the calculation
  ;;   of the light intensity of an RGB color by the equation
  ;;             Luminosity = .299*R + .587*G + .114*B
  `(+ (* 0.299 (color-red-value   ,color))
      (* 0.587 (color-green-value ,color))
      (* 0.114 (color-blue-value  ,color))))

(defun almost-white-p (COLOR)
  " Return non-NIL if COLOR should be condidered white on mono display
    else nil "
  ;;   To determine "whiteness", we use a threshold of .6 to separate light
  ;;   from dark.   0 <= dark <= .6 <= light <= 1.0
  ;;
  (< 0.6 (luminosity color)))

;;; Comment out RGB-TO-IHS by saying (PUSHNEW :IHS-NYI *FEATURES*) if you care. --- rsl

#+IHS-NYI (progn

(defun rgb-to-ihs (r g b)
  (declare (ignore r g b))
  (error "IHS color is not yet implemented"))

(defun ihs-to-rgb (i h s)
  (declare (ignore i h s))
  (error "IHS color is not yet implemented"))
) ;; #+IHS-NYI

#-IHS-NYI (progn

;;; RGB/IHS conversion algorithm; converted from MIT algorithm to Common
;;; Lisp 23 July 1990 by Richard Lamson after he decided that the HSV
;;; algorithms didn't implement IHS at all.

(defconstant *ihs-rgb-c1* (sqrt (/ 1.0 6.0)))
(defconstant *ihs-rgb-c2* (sqrt (/ 1.0 2.0)))
(defconstant *ihs-rgb-c3* (sqrt (/ 1.0 3.0)))

(defun rgb-to-ihs (r g b)
  (let* ((red   (float r 1.0))
	 (green (float g 1.0))
	 (blue  (float b 1.0))
	 (x (* *ihs-rgb-c1* (- (+ red red) blue green)))
	 (y (* *ihs-rgb-c2* (- green blue)))
	 (z (* *ihs-rgb-c3* (+ red green blue)))
	 (q (+ (* x x) (* y y)))
	 (intensity (sqrt (+ q (* z z)))))	;== (sqrt (+ r^2 g^2 b^2))!
    (declare (single-float red green blue x y z q intensity))
    (if (zerop q)
	(values intensity 0.0 0.0)		;A totally unsaturated color.
	(let* ((hue (mod (/ (atan y x) 2pi) 1.0))
	       (f1 (/ z intensity))
;;; The Genera version contains a comment saying that RG thinks that
;;; F1^2 might be larger than 1.0, causing an attempt to take a negative
;;; square root.  I don't believe it, especially since we are ensuring
;;; that the input numbers are not complex by using FLOAT. -- rsl
	       (f2 (sqrt (- 1.0 (* f1 f1))))
	       (saturation (atan f2 f1)))
	  (declare (single-float hue f1 f2 saturation))
	  (values intensity hue saturation)))))

(defun ihs-to-rgb (intensity hue saturation)
  (macrolet ((rgb-values (r g b)
	       `(values (max 0.0 (min ,r 1.0))
			(max 0.0 (min ,g 1.0))
			(max 0.0 (min ,b 1.0)))))
    (let* ((intensity (float intensity 1.0))
	   (hue (float hue 1.0))
	   (saturation (float saturation 1.0))
	   (hh (* hue 2pi))
	   (ss (sin saturation))
	   (x (* *ihs-rgb-c1* ss (cos hh) intensity))
	   (y (* *ihs-rgb-c2* ss (sin hh) intensity))
	   (z (* *ihs-rgb-c3* (cos saturation) intensity)))
      (declare (single-float intensity hue saturation hh ss x y z))
      (rgb-values (+ x x z) (+ y z (- x)) (- z x y)))))

(defun-inline rgb-to-intensity (r g b)
  (let ((red   (float r 1.0))
	(green (float g 1.0))
	(blue  (float b 1.0)))
    (declare (single-float red green blue))
    (sqrt (+ (* red red) (* green green) (* blue blue)))))

#|| ;;; Test function

(defun test-rgb-to-ihs (&optional (step-size 0.1))
  (do ((max-error 0.0)
       (r step-size (+ r step-size)))
      ((>= r 1.0) (values max-error))
    (do ((g step-size (+ g step-size)))
	((>= g 1.0))
      (do ((b step-size (+ b step-size)))
	  ((>= b 1.0))
	(multiple-value-bind (new-r new-g new-b)
	    (multiple-value-bind (i h s) (rgb-to-ihs r g b)
	      (ihs-to-rgb i h s))
	  (let ((this-error (max (abs (- r new-r))
				 (abs (- g new-g))
				 (abs (- b new-b)))))
	    (setf max-error (max max-error this-error))
	    (when (> this-error 0.0001)
	      (error "~S is not invertible for (~D ~D ~D)"
		     'rgb-to-ihs r g b))))))))

||#

) ;; #-IHS-NYI

;;;
;;;  Standard (X see R4/mit/rgb/rgb.txt) Colors
;;;

;;; Basic Colors

(defvar +white+ (make-color-rgb 1.0 1.0 1.0))
(defvar +black+ (make-color-rgb 0.0 0.0 0.0))
(defvar +red+ (make-color-rgb 1.0 0.0 0.0))
(defvar +blue+ (make-color-rgb 0.0 0.0 1.0))
(defvar +green+ (make-color-rgb 0.0 1.0 0.0))
(defvar +cyan+ (make-color-rgb 0.0 1.0 1.0))
(defvar +magenta+ (make-color-rgb 1.0 0.0 1.0))
(defvar +yellow+ (make-color-rgb 1.0 1.0 0.0))

;;; Standard Grays

(defvar +dim-gray+ (make-color-rgb 0.329 0.329 0.329))
(defvar +gray+ (make-color-rgb .50 .50 .50))
(defvar +light-gray+ (make-color-rgb 0.659 0.659 0.659))

;;; the big X crayon box

(defvar +unknown-color+ nil)

(defvar +snow+ +unknown-color+)
(defvar +dodger-blue+ +unknown-color+)
(defvar +dark-goldenrod+ +unknown-color+)
(defvar +ghost-white+ +unknown-color+)
(defvar +deep-sky-blue+ +unknown-color+)
(defvar +rosy-brown+ +unknown-color+)
(defvar +white-smoke+ +unknown-color+)
(defvar +sky-blue+ (make-color-rgb 0.196 0.6 0.8))
(defvar +indian-red+ (make-color-rgb 0.31 0.184 0.184))
(defvar +gainsboro+ +unknown-color+)
(defvar +light-sky-blue+ +unknown-color+)
(defvar +saddle-brown+ +unknown-color+)
(defvar +floral-white+ +unknown-color+)
(defvar +steel-blue+ (make-color-rgb 0.137 0.42 0.557))
(defvar +sienna+ (make-color-rgb 0.557 0.42 0.137))
(defvar +old-lace+ +unknown-color+)
(defvar +light-steel-blue+ (make-color-rgb 0.561 0.561 0.737))
(defvar +peru+ +unknown-color+)
(defvar +linen+ +unknown-color+)
(defvar +light-blue+ (make-color-rgb 0.749 0.847 0.847))
(defvar +burlywood+ +unknown-color+)
(defvar +antique-white+ +unknown-color+)
(defvar +powder-blue+ +unknown-color+)
(defvar +beige+ +unknown-color+)
(defvar +papaya-whip+ +unknown-color+)
(defvar +pale-turquoise+ +unknown-color+)
(defvar +wheat+ (make-color-rgb 0.847 0.847 0.749))
(defvar +blanched-almond+ +unknown-color+)
(defvar +dark-turquoise+ (make-color-rgb 0.439 0.576 0.859))
(defvar +sandy-brown+ +unknown-color+)
(defvar +bisque+ +unknown-color+)
(defvar +medium-turquoise+ (make-color-rgb 0.439 0.859 0.859))
(defvar +tan+ (make-color-rgb 0.859 0.576 0.439))
(defvar +peach-puff+ +unknown-color+)
(defvar +turquoise+ (make-color-rgb 0.678 0.918 0.918))
(defvar +chocolate+ +unknown-color+)
(defvar +navajo-white+ +unknown-color+)
(defvar +firebrick+ (make-color-rgb 0.557 0.137 0.137))
(defvar +moccasin+ +unknown-color+)
(defvar +light-cyan+ +unknown-color+)
(defvar +brown+ +unknown-color+)
(defvar +cornsilk+ +unknown-color+)
(defvar +cadet-blue+ (make-color-rgb 0.373 0.624 0.624))
(defvar +dark-salmon+ +unknown-color+)
(defvar +ivory+ +unknown-color+)
(defvar +medium-aquamarine+ (make-color-rgb 0.196 0.8 0.6))
(defvar +salmon+ (make-color-rgb 0.435 0.259 0.259))
(defvar +lemon-chiffon+ +unknown-color+)
(defvar +aquamarine+ (make-color-rgb 0.439 0.859 0.576))
(defvar +light-salmon+ +unknown-color+)
(defvar +seashell+ +unknown-color+)
(defvar +dark-green+ (make-color-rgb 0.184 0.31 0.184))
(defvar +orange+ (make-color-rgb 0.8 0.196 0.196))
(defvar +honeydew+ +unknown-color+)
(defvar +dark-olive-green+ (make-color-rgb 0.31 0.31 0.184))
(defvar +dark-orange+ +unknown-color+)
(defvar +mint-cream+ +unknown-color+)
(defvar +dark-sea-green+ +unknown-color+)
(defvar +coral+ (make-color-rgb 1.0 0.498 0.0))
(defvar +azure+ +unknown-color+)
(defvar +sea-green+ (make-color-rgb 0.137 0.557 0.42))
(defvar +light-coral+ +unknown-color+)
(defvar +alice-blue+ +unknown-color+)
(defvar +medium-sea-green+ (make-color-rgb 0.259 0.435 0.259))
(defvar +tomato+ +unknown-color+)
(defvar +lavender+ +unknown-color+)
(defvar +light-sea-green+ +unknown-color+)
(defvar +orange-red+ (make-color-rgb 1.0 0.0 0.498))
(defvar +lavender-blush+ +unknown-color+)
(defvar +pale-green+ (make-color-rgb 0.561 0.737 0.561))
(defvar +misty-rose+ +unknown-color+)
(defvar +spring-green+ (make-color-rgb 0.0 1.0 0.498))
(defvar +hot-pink+ +unknown-color+)
(defvar +lawn-green+ +unknown-color+)
(defvar +deep-pink+ +unknown-color+)
(defvar +pink+ (make-color-rgb 1.000 0.500 0.700))
(defvar +dark-slate-gray+ (make-color-rgb 0.184 0.31 0.31))
(defvar +chartreuse+ +unknown-color+)
(defvar +light-pink+ +unknown-color+)
(defvar +medium-spring-green+ (make-color-rgb 0.498 1.0 0.0))
(defvar +pale-violet-red+ +unknown-color+)
(defvar +slate-gray+ +unknown-color+)
(defvar +green-yellow+ (make-color-rgb 0.576 0.859 0.439))
(defvar +maroon+ (make-color-rgb 0.557 0.137 0.42))
(defvar +light-slate-gray+ +unknown-color+)
(defvar +lime-green+ (make-color-rgb 0.196 0.8 0.196))
(defvar +medium-violet-red+ (make-color-rgb 0.859 0.439 0.576))
(defvar +yellow-green+ (make-color-rgb 0.6 0.8 0.196))
(defvar +violet-red+ (make-color-rgb 0.8 0.196 0.6))
(defvar +forest-green+ (make-color-rgb 0.137 0.557 0.137))
;; Once had (defvar +medium-forest-green+ (make-color-rgb 0.42 0.557 0.137))
(defvar +midnight-blue+ (make-color-rgb 0.184 0.184 0.31))
(defvar +olive-drab+ +unknown-color+)
(defvar +violet+ (make-color-rgb 0.31 0.184 0.31))
(defvar +navy-blue+ (make-color-rgb 0.137 0.137 0.557))
(defvar +navy+ +navy-blue+) ; Not in spec, but should it be supported
(defvar +dark-khaki+ +unknown-color+)
(defvar +plum+ (make-color-rgb 0.918 0.678 0.918))
(defvar +cornflower-blue+ (make-color-rgb 0.259 0.259 0.435))
(defvar +khaki+ (make-color-rgb 0.624 0.624 0.373))
(defvar +orchid+ (make-color-rgb 0.859 0.439 0.859))
(defvar +dark-slate-blue+ (make-color-rgb 0.42 0.137 0.557))
(defvar +pale-goldenrod+ +unknown-color+)
(defvar +medium-orchid+ (make-color-rgb 0.576 0.439 0.859))
(defvar +slate-blue+ (make-color-rgb 0.0 0.498 1.0))
(defvar +light-goldenrod-yellow+ +unknown-color+)
(defvar +dark-orchid+ (make-color-rgb 0.6 0.196 0.8))
(defvar +medium-slate-blue+ (make-color-rgb 0.498 0.0 1.0))
(defvar +light-yellow+ +unknown-color+)
(defvar +dark-violet+ +unknown-color+)
(defvar +light-slate-blue+ +unknown-color+)
(defvar +blue-violet+ (make-color-rgb 0.624 0.373 0.624))
(defvar +medium-blue+ (make-color-rgb 0.196 0.196 0.8))
(defvar +gold+ (make-color-rgb 0.8 0.498 0.196))
(defvar +purple+ +unknown-color+)
(defvar +royal-blue+ +unknown-color+)
(defvar +light-goldenrod+ +unknown-color+)
(defvar +medium-purple+ +unknown-color+)
(defvar +goldenrod+ (make-color-rgb 0.859 0.859 0.439))
;; Once had (defvar +medium-goldenrod+ (make-color-rgb 0.918 0.918 0.678)) 
(defvar +thistle+ (make-color-rgb 0.847 0.749 0.847))

;;; Support for obsolete SHADE ink, an integer between 0 and 100
;;; representing a gray level.

(defvar *shade-color-array* (make-array 102 :initial-element nil))

(defun shade-to-color (shade)
  (macrolet ((get-gray-color (shade)
	       `(make-gray-color (/ (- 100.0 (float ,shade 1.0)) 100.0))))
    (if ;; (typep shade '(integer 0 100))  ;; Sigh: Genera optimizes this well.
	(and (integerp shade) (<= 0 shade 100)) ;; Sigh*.
	(locally (declare (fixnum shade))
	  (or (aref *shade-color-array* shade)
	      (setf (aref *shade-color-array* shade) (get-gray-color shade))))
      (let ((element (assoc shade (aref *shade-color-array* 101.))))
	(if element (cdr element)
	  (let* ((new-color (get-gray-color shade))
		 (new-element (cons shade new-color)))
	    (push new-element (aref *shade-color-array* 101.))
	    new-color))))))
